package com.sonicsw.buildtools;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Properties;

import org.apache.tools.ant.AntClassLoader;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.Path;

/**
 * @author jcrump
 * @author tsteinbo
 */
public class MakeVersion extends Task
{

    private String m_classname = null;

    private String m_outDirName = null;

    private String m_baseName = null;

    private Path m_classPath = null;

    private boolean m_override = false;

    private boolean m_addReleaseVersion = false;

    private String releaseVersion = "";

    /**
     * @param classname
     *            the classname for Version.java.
     */
    public void setClass(String classname)
    {
        m_classname = classname;
    }

    /**
     * @param toDir
     *            the output directory
     */
    public void setToDir(String toDir)
    {
        m_outDirName = toDir;
    }

    /**
     * @param baseName
     *            basename for the output file
     */
    public void setBaseName(String baseName)
    {
        m_baseName = baseName;
    }

    public Path createClasspath()
    {
        m_classPath = new Path(getProject());
        return m_classPath;
    }

    public void setOverride(boolean b)
    {
        m_override = b;
    }

    public void setAddReleaseVersion(boolean addReleaseVersion)
    {
        m_addReleaseVersion = addReleaseVersion;
    }

    @Override
    public void execute() throws BuildException
    {

        if (m_classname == null)
        {
            throw new BuildException("No Version class specified.");
        }
        if (m_outDirName == null)
        {
            throw new BuildException("No target directory file specified.");
        }
        if (m_baseName == null)
        {
            throw new BuildException("No basename specified.");
        }

        File outDir = new File(m_outDirName);
        if (!outDir.exists())
        {
            throw new BuildException("Output directory does not exist: "
                    + m_outDirName);
        }

        try
        {
            Class clazz = null;
            if (m_classPath != null)
            {
                AntClassLoader cl = new AntClassLoader(getProject(),
                        m_classPath);
                clazz = cl.findClass(m_classname);
            } else
            {
                clazz = Class.forName(m_classname);
            }
            String major = getVersionProperty(clazz, "getMajorVersion");
            String minor = getVersionProperty(clazz, "getMinorVersion");
            String point = getOptionalVersionProperty(clazz, "getPointVersion");
            point = (point != null ? point : "0");
            String build = getVersionProperty(clazz, "getBuildNumber");
            String revision = getOptionalVersionProperty(clazz, "getRevisionNumber");

            if (m_addReleaseVersion)
            {
                releaseVersion = getVersionProperty(clazz, "getReleaseVersion");
            }

            writeBasenameFile(outDir, major, minor, point, build, revision);
            writePropertiesFile(outDir, major, minor, point, build);

        } catch (BuildException be)
        {
            throw be;
        } catch (Exception e)
        {
            throw new BuildException(e);
        }
    }

    private void writeBasenameFile(File outDir, String major, String minor,
                                   String point, String build, String revision) throws IOException
    {
        String filename = m_baseName + major + "." + minor + "." + point
                + releaseVersion + ".txt";

        File outFile = new File(outDir, filename);

        log("Checking version file: " + outFile.getAbsolutePath(),
                Project.MSG_VERBOSE);

        if (!outFile.exists() || m_override)
        {
            PrintWriter pw = new PrintWriter(new FileWriter(outFile));
            pw.println("MAJOR_VERSION=" + major);
            pw.println("MINOR_VERSION=" + minor);
            pw.println("POINT_VERSION=" + (point != null ? point : "0"));

            pw.println(isSet(build) ? "BUILD_NUMBER=" + build : "DEVELOPER BUILD");
            if (isSet(revision)) {
                pw.println("REVISION=" + revision);
            }

            pw.flush();
            pw.close();
            log("Created version file: " + outFile.getAbsolutePath());
        }
    }

    private void writePropertiesFile(File outDir, String major, String minor,
            String point, String build) throws IOException
    {
        Properties properties = new Properties();
        properties.setProperty("progress.sonic.version", major + "." + minor + "."
                + point + "." + build);
        FileOutputStream out = new FileOutputStream(new File(outDir,
                "progress.sonic.version.properties"));
        try
        {
            properties.store(out, null);
        } finally
        {
            out.close();
        }
    }

    private String getVersionProperty(Class c, String methodname)
            throws Exception
    {
        String result = getOptionalVersionProperty(c, methodname);
        if (result == null)
        {
            throw new BuildException("Could not find method '" + methodname
                    + "' in class: " + c.getName());
        }
        return result;
    }

    private String getOptionalVersionProperty(Class c, String methodname)
            throws Exception
    {
        try
        {
            java.lang.reflect.Method m = c.getMethod(methodname, new Class[]
            {});
            Object o = m.invoke(null, new Object[]
            {});
            return o.toString();

        } catch (NoSuchMethodException excp)
        {
            return null;
        }
    }

    private boolean isSet(String property) {
        return property != null && !"0".equals(property);
    }

}
