/*
 * Copyright 2004 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.sonicsw.buildtools;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.jar.Pack200.Packer;
import java.util.zip.GZIPOutputStream;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.taskdefs.MatchingTask;

/**
 * An optional ant Task which emulates the Command Line Interface pack200(1).
 * Note, only the most commonly used command line options are implemented here,
 * users are encouraged to use the configuration file described below to set
 * other Pack200 options.
 * 
 * @author Kumar Srinivasan
 */
public class Pack200Task extends MatchingTask
{
    private static final String EXT_JAR = ".jar";

    private static final String ERRMSG_ZF = "zipfile attribute must end";

    protected static final String COM_PREFIX = "com.sun.java.util.jar.pack.";

    private boolean doRepack = false;

    private boolean doGZIP = false;

    private File dir;

    private boolean removeSource = false;

    // Storage for the properties used by the setters.
    private HashMap<String, String> propMap;

    public Pack200Task()
    {
        // Initialize our fields
        doRepack = false;
        doGZIP = false;
        propMap = new HashMap<String, String>();
    }

    /**
     * Execute the task.
     * 
     * @throws BuildException
     *             on error
     */
    @Override
    public void execute() throws BuildException
    {
        if (null == dir || !dir.exists() || !dir.isDirectory())
        {
            throw new BuildException(
                    "dir attribute is not set of not a directory.");
        }

        if (doRepack)
        {
            doGZIP = false;
            removeSource = false;
        }

        DirectoryScanner ds = getDirectoryScanner(dir);
        String[] files = ds.getIncludedFiles();
        for (String file : files)
        {
            String outFile;
            if (file.endsWith(EXT_JAR))
            {
                if (doRepack)
                {
                    outFile = file;
                } else
                {
                    outFile = file + ".pack";
                    if (doGZIP)
                    {
                        outFile += ".gz";
                    }
                }
            } else
            {
                throw new BuildException("File " + file + " must and with .jar");
            }
            pack(new File(dir, file), new File(dir, outFile));
        }
    }

    // We validate our own stuff
    private void validate(File outputFile) throws BuildException
    {
        if (doGZIP)
        {
            if (!outputFile.toString().toLowerCase().endsWith(".gz"))
            {
                throw new BuildException(ERRMSG_ZF + " with .gz extension",
                        getLocation());
            }
        } else if (doRepack)
        {
            if (!outputFile.toString().toLowerCase().endsWith(".jar"))
            {
                throw new BuildException(ERRMSG_ZF + " with .jar extension",
                        getLocation());
            }
        } else
        {
            if (!outputFile.toString().toLowerCase().endsWith(".pack")
                    && !outputFile.toString().toLowerCase().endsWith(".pac"))
            {
                throw new BuildException(ERRMSG_ZF
                        + "with .pack or .pac extension", getLocation());
            }
        }
    }

    /**
     * Sets the basedir for includes and excludes.
     * 
     * @param _dir
     */
    public void setDir(File _dir)
    {
        dir = _dir;
    }

    /**
     * Set if we should remove the source after packing.
     * 
     * @param _removeSource
     */
    public void setRemoveSource(boolean _removeSource)
    {
        removeSource = _removeSource;
    }

    /**
     * Sets the repack option, ie the jar will be packed and repacked.
     */

    public void setRepack(boolean value)
    {
        doRepack = value;
    }

    /**
     * Sets whether the pack archive is additionally deflated with gzip.
     */
    public void setGZIPOutput(boolean value)
    {
        doGZIP = value;
    }

    /**
     * Sets whether the java debug attributes should be stripped
     */
    public void setStripDebug(String value)
    {
        propMap.put(COM_PREFIX + "strip.debug", value);
    }

    /**
     * Sets the modification time for the archive
     */
    public void setModificationTime(String value)
    {
        propMap.put(Packer.MODIFICATION_TIME, value);
    }

    /**
     * Sets the deflate hint for the archive
     */
    public void setDeflateHint(String value)
    {
        propMap.put(Packer.DEFLATE_HINT, value);
    }

    /**
     * Sets the file ordering.
     */
    public void setKeepFileOrder(String value)
    {
        propMap.put(Packer.KEEP_FILE_ORDER, value);
    }

    /**
     * Sets the segment limit.
     */
    public void setSegmentLimit(String value)
    {
        propMap.put(Packer.SEGMENT_LIMIT, value);
    }

    /**
     * Sets the effort.
     */
    public void setEffort(String value)
    {
        propMap.put(Packer.EFFORT, value);
    }

    /**
     * Sets the action to be taken if an unknown attribute is encountered.
     */
    public void setUnknownAttribute(String value)
    {
        propMap.put(Packer.UNKNOWN_ATTRIBUTE, value);
    }

    /**
     * Set the verbosity level.
     */
    public void setVerbose(String value)
    {
        propMap.put(COM_PREFIX + "verbose", value);
    }

    private void pack(File source, File targetFile)
    {
        validate(targetFile);

        try
        {
            if (doRepack)
            {
                log("Repack Pack200 " + source);
                File packFile = new File(targetFile.toString() + ".tmp");
                try
                {
                    compress(source, packFile);
                    uncompress(packFile, targetFile);
                } finally
                {
                    packFile.delete();
                }
            } else
            {
                log("Pack200 " + source);
                compress(source, targetFile);
            }

        } catch (IOException ioe)
        {
            ioe.printStackTrace();
            throw new BuildException("Error in pack200");
        }

        if (removeSource)
        {
            source.delete();
        }
    }

    private void uncompress(File packFile, File targetFile)
            throws FileNotFoundException, IOException
    {
        InputStream previouslyPacked = new BufferedInputStream(
                new FileInputStream(packFile));

        JarOutputStream repackOutput = new JarOutputStream(
                new BufferedOutputStream(new FileOutputStream(targetFile)));

        Pack200.Unpacker unpkr = Pack200.newUnpacker();
        unpkr.properties().putAll(propMap);

        unpkr.unpack(previouslyPacked, repackOutput);
        previouslyPacked.close();
        repackOutput.close();
    }

    private void compress(File source, File target) throws IOException,
            FileNotFoundException
    {
        if (target.exists() && (target.lastModified() > source.lastModified()))
        {
            return;
        }
        Pack200.Packer pkr = Pack200.newPacker();
        pkr.properties().putAll(propMap);

        JarFile jarFile = new JarFile(source);
        FileOutputStream fos = new FileOutputStream(target);

        OutputStream os = (doGZIP) ? new BufferedOutputStream(
                new GZIPOutputStream(fos)) : new BufferedOutputStream(fos);

        pkr.pack(jarFile, os);
        os.close();
    }
}
