/*
 * Copyright 2004 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.sonicsw.buildtools;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.SortedMap;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.zip.GZIPInputStream;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.MatchingTask;

/**
 * An optional ant Task which emulates the Command Line Interface unpack200(1).
 * 
 * @author Kumar Srinivasan
 */
public class Unpack200Task extends MatchingTask
{

    private static final String EXT_PACK_GZ = ".pack.gz";

    private static final String EXT_PACK = ".pack";

    enum FileType
    {
        unknown, gzip, pack200, zip
    };

    private SortedMap<String, String> propMap;

    private Pack200.Unpacker unpkr;

    private File dir;

    public Unpack200Task()
    {
        unpkr = Pack200.newUnpacker();
        propMap = unpkr.properties();
    }

    public void setVerbose(String value)
    {
        propMap.put(Pack200Task.COM_PREFIX + "verbose", value);
    }

    public void setDir(File _dir)
    {
        dir = _dir;
    }

    /**
     * Execute the task.
     * 
     * @throws BuildException
     *             on error
     */
    @Override
    public void execute() throws BuildException
    {
        if (null == dir || !dir.exists() || !dir.isDirectory())
        {
            throw new BuildException(
                    "dir attribute is not set of not a directory.");
        }

        DirectoryScanner ds = getDirectoryScanner(dir);
        String[] files = ds.getIncludedFiles();
        for (String file : files)
        {
            String outFile;
            if (file.endsWith(EXT_PACK))
            {
                outFile = file.substring(0, file.length() - EXT_PACK.length());
            } else if (file.endsWith(EXT_PACK_GZ))
            {
                outFile = file.substring(0, file.length()
                        - EXT_PACK_GZ.length());
            } else
            {
                throw new BuildException("File " + file
                        + " must and with .pack[.gz]");
            }
            extract(new File(dir, file), new File(dir, outFile));
        }
    }

    private FileType getMagic(File in) throws IOException
    {
        DataInputStream is = new DataInputStream(new FileInputStream(in));
        int i = is.readInt();
        is.close();
        if ((i & 0xffffff00) == 0x1f8b0800)
        {
            return FileType.gzip;
        } else if (i == 0xcafed00d)
        {
            return FileType.pack200;
        } else if (i == 0x504b0304)
        {
            return FileType.zip;
        } else
        {
            return FileType.unknown;
        }
    }

    private void extract(File source, File dest)
    {
        log("Unpacking with Unpack200", Project.MSG_VERBOSE);
        log("Source File :" + source, Project.MSG_VERBOSE);
        log("Dest.  File :" + dest, Project.MSG_VERBOSE);

        try
        {
            FileInputStream fis = new FileInputStream(source);

            InputStream is = (FileType.gzip == getMagic(source)) ? new BufferedInputStream(
                    new GZIPInputStream(fis))
                    : new BufferedInputStream(fis);

            FileOutputStream fos = new FileOutputStream(dest);
            JarOutputStream jout = new JarOutputStream(
                    new BufferedOutputStream(fos));

            unpkr.unpack(is, jout);
            jout.close();

        } catch (IOException ioe)
        {
            throw new BuildException("Error in unpack200");
        }

    }

}
