package com.sonicsw.jmx.remote.sonic;

import java.io.IOException;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.util.Hashtable;
import java.util.Map;

import javax.management.ListenerNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXServiceURL;
import javax.security.auth.Subject;

import com.sonicsw.mx.util.URLUtil;

import com.sonicsw.mf.jmx.client.IConnectionListener;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;

public class SonicConnector
implements JMXConnector, Serializable
{
    private static final int CONNECTION_TIMEOUT = 10000; // may want to drive this through a property
    private static final String DEFAULT_SERVER_IDENTITY = "Domain1.AGENT MANAGER"; // default to the container hosting the AGENT MANAGER

    private static final long serialVersionUID = 0L;

    private JMXServiceURL m_jmxServiceURL;
    private String m_user = "";
    private String m_password = "";             //NOSONAR field change is not required.

    private boolean m_closed = false;
    private JMSConnectorClient m_jmsConnectorClient;
    private SonicConnection m_mBeanServerConnection;

    public SonicConnector(JMXServiceURL serviceURL)
    throws IOException
    {
        this(serviceURL, null);
    }

    public SonicConnector(JMXServiceURL serviceURL, Map environment)
    throws IOException
    {
        if (!serviceURL.getProtocol().startsWith("sonic+"))
        {
            throw new MalformedURLException("Protocol not supported: " + serviceURL.getProtocol());
        }

        m_jmxServiceURL = serviceURL;

        // extract the user name and password if present
        if (environment != null)
        {
            Object credentials = environment.get(JMXConnector.CREDENTIALS);

            if (credentials != null)
            {
                // for JConsole this will be a String[2] array { username, password }
                if (credentials instanceof String[])
                {
                    m_user = ((String[])credentials)[0];
                    m_password = ((String[])credentials)[1];
                }
                else
                {
                    throw new IOException("Unsupported connector credentials type: " + credentials.getClass());
                }
            }
        }
    }

    @Override
    public synchronized void close()
    throws IOException
    {
        if (m_mBeanServerConnection != null)
        {
            m_jmsConnectorClient.disconnect();
            m_jmsConnectorClient = null;
            m_mBeanServerConnection = null;
        }
        m_closed = true;
    }

    @Override
    public synchronized void connect()
    throws IOException
    {
        connect(null);
    }

    @Override
    public synchronized void connect(Map environment)
    throws IOException
    {
        if (m_closed)
        {
            throw new IOException("Connector has been closed");
        }

        if (m_mBeanServerConnection != null)
         {
            return; // already connected
        }

        Hashtable sonicAddressProperties = new Hashtable();

        String protocol = m_jmxServiceURL.getProtocol().substring(m_jmxServiceURL.getProtocol().indexOf("+") + 1);

        String host = m_jmxServiceURL.getHost();
        if (host == null || host.length() == 0)
        {
            host = "localhost";
        }

        int port = m_jmxServiceURL.getPort();
        if (port == 0)
        {
            port = 2506;
        }

        sonicAddressProperties.put("ConnectionURLs", URLUtil.buildUrlString(protocol, host, port));

        String user = m_user;
        String password = m_password;

        // extract an override of user name and password if present
        if (environment != null)
        {
            Object credentials = environment.get(JMXConnector.CREDENTIALS);

            if (credentials != null)
            {
                // for JConsole this will be a String[2] array { username, password }
                if (credentials instanceof String[])
                {
                    user = ((String[])credentials)[0];
                    password = ((String[])credentials)[1];
                }
                else
                {
                    throw new IOException("Unsupported connector credentials type: " + credentials.getClass());
                }
            }
        }

        sonicAddressProperties.put("DefaultUser", user);
        sonicAddressProperties.put("DefaultPassword", password);

        JMSConnectorAddress jmsConnectorAddress = new JMSConnectorAddress(sonicAddressProperties);

        String serverIdentity = m_jmxServiceURL.getURLPath();
        if (serverIdentity.length() > 0)
        {
            // then remove the leading '/' or ';'
            serverIdentity = serverIdentity.substring(1);
        }
        jmsConnectorAddress.setServerIdentity(serverIdentity.length() == 0 ? SonicConnector.DEFAULT_SERVER_IDENTITY : serverIdentity);

        m_jmsConnectorClient = new JMSConnectorClient();

        try
        {
            m_jmsConnectorClient.connect(jmsConnectorAddress, SonicConnector.CONNECTION_TIMEOUT);
            m_jmsConnectorClient.setConnectionListener(new ConnectionListener());
        }
        catch (Throwable e)
        {
            IOException io = new IOException("Failed to create MBean");
            io.initCause(e);
            throw io;
        }

        m_mBeanServerConnection = new SonicConnection(m_jmsConnectorClient);
    }

    @Override
    public String getConnectionId()
    throws IOException
    {
        if (m_closed)
        {
            throw new IOException("Connector has been closed");
        }

        SonicConnection mBeanServerConnection = m_mBeanServerConnection;
        if (mBeanServerConnection == null)
        {
            throw new IOException("Connection has not been established");
        }

        return mBeanServerConnection.toString(); // the unique identity if the client is not currently available
    }

    @Override
    public MBeanServerConnection getMBeanServerConnection()
    throws IOException
    {
        return getMBeanServerConnection(null);
    }

    @Override
    public MBeanServerConnection getMBeanServerConnection(Subject arg0)
    throws IOException
    {
        SonicConnection mBeanServerConnection = m_mBeanServerConnection;
        if (mBeanServerConnection == null)
        {
            throw new IOException("Connection has not been established");
        }

        return mBeanServerConnection;
    }

    @Override
    public void addConnectionNotificationListener(NotificationListener notificationListener, NotificationFilter notificationFilter, Object handback)
    {
    // TODO Auto-generated method stub

    }

    @Override
    public void removeConnectionNotificationListener(NotificationListener notificationListener)
    throws ListenerNotFoundException
    {
    // TODO Auto-generated method stub

    }

    @Override
    public void removeConnectionNotificationListener(NotificationListener notificationListener, NotificationFilter notificationFilter, Object handback)
    throws ListenerNotFoundException
    {
    // TODO Auto-generated method stub

    }

    private void notifyConnectionOpened()
    {

    }

    private void notifyConnectionClosed()
    {

    }

    private void notifyConnectionFailed()
    {

    }

    public void sendConnectionNotifsLost(Exception exception)
    {
        // TODO Auto-generated method stub

    }

    private class ConnectionListener
    implements IConnectionListener
    {
        @Override
        public void onDisconnect() {  SonicConnector.this.sendConnectionNotifsLost(null); } // connection is not really closed .. it may reconnect
        @Override
        public void onFailure(Exception arg0) { } // not implemented - this is viewed as a temporary failure
        @Override
        public void onNotificationListenerRenewalFailure(Exception exception) { SonicConnector.this.sendConnectionNotifsLost(exception); }
        @Override
        public void onReconnect(String arg0) { } // not implemented - this is after a temporary failure
    }
}
