package com.sonicsw.jndi.mfcontext;

import java.lang.reflect.Method;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;

import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.MFSecurityException;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.view.IDeltaView;

public final class MFConnection
{
    private MFConnectionManager m_manager = null;
    private ConnectorClient m_connector = null;
    private String m_dsName;
    private ObjectName m_dsObjectName;;

    private long m_lastUsageTime = 0;
    private long m_idleTimeout = 0;
    
    private boolean m_exiting = false;

    private static final String READER_ROLE = "READER";
    private static final String COMMS_TYPE = "jmx";
    private static final String INVOKE_OPERATION = "invoke";
    
    private static final String[] INVOKE_SIGNATURE = new String[]
    {
        ObjectName.class.getName(),
        String.class.getName(),
        Object[].class.getName(),
        String[].class.getName()
    };

    MFConnection(MFConnectionManager manager, String domainName, long idleTimeout)
    {
        m_manager = manager;
        m_dsName = domainName + ".DIRECTORY SERVICE:ID=DIRECTORY SERVICE";
        try
        {
            m_dsObjectName = new ObjectName(m_dsName);
        }
        catch (MalformedObjectNameException e) { }

        m_idleTimeout = idleTimeout;
        init();
    }

    synchronized void init()
    {
        m_connector = m_manager.getConnector();
        m_lastUsageTime = System.currentTimeMillis();  // set m_lastUsageTime before starting IdleStateVerifier
        IdleStateVerifier idleStateVerifier = new IdleStateVerifier();
        idleStateVerifier.setName("JNDI connection - idle state verifier " + idleStateVerifier.hashCode());
        idleStateVerifier.start();
    }

    synchronized void close()
    {
        m_exiting = true;
        this.notifyAll();
    }
    
    synchronized boolean isConnected()
    {
        return m_connector == null ? false : m_connector.isConnected();
    }

    private static final String[] SET_ELEMENT_SIGNATURE = new String[]
    {
        IBasicElement.class.getName(),
        IDeltaView.class.getName()
    };
    void setElement(IBasicElement element, IDeltaView view)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "setElement", new Object[] { element, view }, SET_ELEMENT_SIGNATURE }, INVOKE_SIGNATURE);
        }
        catch(Exception e)
        {
            throwException(e);
        }
    }

    private static final String[] GET_ELEMENT_SIGNATURE1 = new String[]
    {
        String.class.getName(),
        Boolean.class.getName()
    };
    IDirElement getElement(String elementName, boolean forUpdate)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IDirElement)m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "getElement", new Object[] { elementName, new Boolean(forUpdate) }, GET_ELEMENT_SIGNATURE1 }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] DELETE_ELEMENT_SIGNATURE = new String[]
    {
        String.class.getName(),
        IDeltaView.class.getName()
    };
    IElementIdentity deleteElement(String elementName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IElementIdentity)m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "deleteElement", new Object[] { elementName, view }, DELETE_ELEMENT_SIGNATURE }, INVOKE_SIGNATURE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] CREATE_DIRECTORY_SIGNATURE1 = new String[]
    {
        String.class.getName()
    };
    void createDirectory(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "createDirectory", new Object[] { dirName }, CREATE_DIRECTORY_SIGNATURE1 }, INVOKE_SIGNATURE);
        }
        catch(Exception e)
        {
            throwException(e);
        }
    }

    private static final String[] DELETE_DIRECTORY_SIGNATURE = new String[]
    {
        String.class.getName()
    };
    void deleteDirectory(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "deleteDirectory", new Object[] { dirName }, DELETE_DIRECTORY_SIGNATURE }, INVOKE_SIGNATURE);
        }
        catch(Exception e)
        {
            throwException(e);
        }
    }

    private static final String[] LIST_DIRECTORIES_SIGNATURE = new String[]
    {
        String.class.getName()
    };
    IDirIdentity[] listDirectories(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IDirIdentity[])m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "listDirectories", new Object[] { dirName }, LIST_DIRECTORIES_SIGNATURE }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] LIST_ELEMENTS_SIGNATURE = new String[]
    {
        String.class.getName()
    };
    IElementIdentity[] listElements(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IElementIdentity[])m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "listElements", new Object[] { dirName }, LIST_ELEMENTS_SIGNATURE }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] LIST_ALL_SIGNATURE = new String[]
    {
        String.class.getName()
    };
    IIdentity[] listAll(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IIdentity[])m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "listAll", new Object[] { dirName }, LIST_ALL_SIGNATURE }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] GET_ALL_ELEMENTS_SIGNATURE1 = new String[]
    {
        String.class.getName(),
        Boolean.class.getName()
    };
    IDirElement[] getAllElements(String dirName, boolean forUpdate)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IDirElement[])m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "getAllElements", new Object[] { dirName, new Boolean(forUpdate) }, GET_ALL_ELEMENTS_SIGNATURE1 }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private static final String[] GET_ALL_DIRECTORIES_SIGNATURE1 = new String[]
    {
        String.class.getName(),
    };
    IDirIdentity[] getAllDirectories(String dirName)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (IDirIdentity[])m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "listDirectories", new Object[] { dirName }, GET_ALL_DIRECTORIES_SIGNATURE1 }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }


    private static final String[] RESOLVE_URL_SIGNATURE = new String[]
    {
        String.class.getName(),
    };
    String resolveURL(String url)
    throws DirectoryServiceException, NamingException
    {
        checkIdleTimeout();
        try
        {
            return (String)m_connector.invoke(COMMS_TYPE, m_dsName, m_dsName, INVOKE_OPERATION, new Object[] { m_dsObjectName, "resolveURL", new Object[] { url }, RESOLVE_URL_SIGNATURE }, INVOKE_SIGNATURE, READER_ROLE);
        }
        catch(Exception e)
        {
            throwException(e);
        }

        return null;
    }

    private synchronized void checkIdleTimeout()
    throws NamingException
    {
        m_lastUsageTime = System.currentTimeMillis();
        
        if (m_manager.getConnector() == null)
        {
            m_manager.connect(); //restore connection
            init(); //reset this proxy
        }
    }

    private void throwException(Exception e)
    throws DirectoryServiceException, NamingException
    {
        if (e instanceof MFSecurityException)
        {
            throw new NoPermissionException(e.getMessage());
        }

        Exception mappedException = mapException(e);
        if (mappedException instanceof DirectoryServiceException)
        {
            throw (DirectoryServiceException)mappedException;
        }

        NamingException namingException = new NamingException();
        namingException.setRootCause(e);
        throw namingException;
    }

    private Exception mapException(Exception exception)
    {
        Class exceptionClass = exception.getClass();

        Method getTargetExceptionMethod = null;
        try
        {
            getTargetExceptionMethod = exceptionClass.getMethod("getTargetException", new Class[0]);
        }
        catch(NoSuchMethodException nsme)
        {
            return exception; // can't do anything
        }

        Object exceptionObject = null;
        try
        {
            exceptionObject = getTargetExceptionMethod.invoke(exception, new Object[0]);
        }
        catch(Exception e)
        {
            throw new RuntimeException("Error extracting target exception");
        }

        if (exceptionObject == null)
        {
            return exception;
        }

        return mapException((Exception) exceptionObject);
    }

    private class IdleStateVerifier
    extends Thread
    {
        private IdleStateVerifier()
        {
        }

        @Override
        public void run()
        {
            synchronized (MFConnection.this)
            {
                while (!MFConnection.this.m_exiting)
                {
                    // Calculate time to wait to take us to lastUsageTime + m_idleTimeout
                    long waitTime = (MFConnection.this.m_lastUsageTime + MFConnection.this.m_idleTimeout)
                                     - System.currentTimeMillis();
                    // CR71828
                    if (waitTime <= 0)
                    {
                        break;
                    }
                    
                    try
                    {
                        MFConnection.this.wait(waitTime);
                    }
                    catch (InterruptedException ie)
                    {
                        break;
                    }
                }
                
                MFConnection.this.m_manager.disconnect();
            }
        }
    }
}
