package com.sonicsw.jndi.mfcontext;

import java.util.HashMap;
import java.util.Hashtable;

import javax.naming.AuthenticationException;
import javax.naming.NamingException;

import com.sonicsw.mf.comm.IExceptionListener;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.MFSecurityException;

public final class MFConnectionManager
implements IExceptionListener
{
    private static Hashtable s_managers = new Hashtable();
    private static LockManager s_lockManager = new LockManager();

    private ConnectorClient m_connector = null;
    private Hashtable m_env;
    private long m_connectTimeout;
    private long m_socketConnectTimeout;
    private long m_initialConnectTimeout;
    private long m_requestTimeout;
    private String m_draNode;
    MFConnection m_connection = null;
    private int m_referenceCount;
    private String m_id;

    static MFConnectionManager getManager(String urls, String user, String password, String draNode, String domainName, long idleTimeout, String connectTimeout, String socketConnectTimeout,
    		String requestTimeout) throws NamingException
    {
    	String ID = urls + user + password + draNode + domainName + idleTimeout + connectTimeout +
    	    socketConnectTimeout + requestTimeout;

        synchronized (s_lockManager.getLock(ID))
        {
            MFConnectionManager manager = (MFConnectionManager)s_managers.get(ID);
            if (manager == null)
            {
                manager = new MFConnectionManager( urls,
                        user,
                        password,
                        draNode,
                        domainName,
                        idleTimeout,
                        new Long(connectTimeout).longValue(),
                        new Long(socketConnectTimeout).longValue(),
                        new Long(requestTimeout).longValue(),
                        ID);

                s_managers.put(ID, manager);
            }
            else
            {
                manager.incRefCount();
            }
            return manager;
        }
    }

    MFConnectionManager(String urls, String user, String password, String draNode, String domainName,
                        long idleTimeout, long connectTimeout, long socketConnectTimeout, long requestTimeout,
                        String ID)
    throws NamingException
    {
    	m_id = ID;
        m_connectTimeout = connectTimeout;
        m_socketConnectTimeout = socketConnectTimeout;
        m_initialConnectTimeout = connectTimeout;
        m_requestTimeout = requestTimeout;
        m_draNode = draNode;

        m_referenceCount = 1;

        m_env = new Hashtable();
        m_env.put("ConnectionURLs", urls);
        m_env.put("DefaultUser", (user == null ? "" : user));
        m_env.put("DefaultPassword", (password == null ? "" : password));

        connect();

        m_connection = new MFConnection(this, domainName, idleTimeout);
    }

    //establish JMS connection
    void connect()
    throws NamingException
    {
        synchronized (s_lockManager.getLock(m_id))
        {
            try
            {
                m_connector = new ConnectorClient("JNDICLIENT");
                m_connector.setDeliveryMode(javax.jms.DeliveryMode.NON_PERSISTENT);
                m_connector.setExceptionListener(this);
                m_connector.setConnectTimeout(m_connectTimeout);
                m_connector.setSocketConnectTimeout(m_socketConnectTimeout);
                m_connector.setRequestTimeout(m_requestTimeout);
                m_connector.setDurable(false);

                if (m_draNode != null && m_draNode.length() > 0)
                {
                    m_connector.setManagementNode(m_draNode);
                }

                m_connector.connect(m_env, m_initialConnectTimeout > 0 ? m_initialConnectTimeout : 0);
            }
            catch(MFSecurityException e)
            {
                throw new AuthenticationException(e.getMessage());
            }
            catch(Exception e) //usually not happen, Connection tries to reconnect
            {

                NamingException namingException = new NamingException();
                namingException.setRootCause(e);
                m_connector = null;
                throw namingException;
            }
        }
    }

    // free JMS connection
    void disconnect()
    {
        synchronized (s_lockManager.getLock(m_id))
        {
            if (m_connector != null)
            {
                m_connector.close();
                m_connector = null;
            }
        }
    }

    ConnectorClient getConnector()
    {
        return m_connector;
    }

    MFConnection getConnection()
    {
        return m_connection;
    }

    void incRefCount()
    {
        synchronized (s_lockManager.getLock(m_id))
        {
            m_referenceCount++;
        }
    }

    void close()
    {
        cleanup(false);	
    }
    
    //detects permanent (non-recoverable) exception
    //occurs in the connector
    @Override
    public void onException(Exception exception)
    {
        cleanup(true);        
    }    

    @Override
    protected void finalize()
    {
        cleanup(true);
    }
    
    private void cleanup(boolean forceCleanup)
    {
        MFConnection closeConnection = null;

        synchronized (s_lockManager.getLock(m_id))
        {
            m_referenceCount--;
            if (m_referenceCount <= 0 || forceCleanup)
            {
                m_referenceCount = 0;
                s_managers.remove(m_id);
                disconnect();

                // SNC00079875 - to avoid deadlock, null m_connection but defer
                // closing the MFConnection until lock has been been released
                closeConnection = m_connection;
                m_connection = null;
            }
        }

        if (closeConnection != null)
        {
            closeConnection.close();
        }
    }


    static class LockManager
    {
        private HashMap m_lockTable;

        LockManager()
        {
            m_lockTable = new HashMap();
        }

        synchronized Object getLock(String ID)
        {
            Object lock = m_lockTable.get(ID);
            if (lock == null)
            {
                lock = new String("L");
                m_lockTable.put(ID, lock);
            }

            return lock;
        }
    }

}
