package com.sonicsw.jndi.mfcontext;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.ContextNotEmptyException;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameClassPair;
import javax.naming.NameNotFoundException;
import javax.naming.NameParser;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoInitialContextException;
import javax.naming.OperationNotSupportedException;
import javax.naming.RefAddr;
import javax.naming.Referenceable;
import javax.naming.StringRefAddr;
import javax.naming.spi.NamingManager;

import com.sonicsw.mx.jndi.MFNameParser;
import com.sonicsw.mx.jndi.ObjHelper;

import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;

// import com.sonicsw.mx.jndi.MFNameParser;

/**
 *
 */
public class MFContext
implements Context
{
    /** An instance of Name Parser for this Context. */
    private MFNameParser m_parser = MFNameParser.getInstance();

    /** Context Environment. */
    Hashtable m_env;

    String m_ctxName;

    String m_fullCtxName; // context name is full name of MF DS directory mapped to this context, such as
                          // '/_MFContext/dir1/dirOfcurrentCtx'

    private MFConnectionManager m_connMng = null;

    private static final long MF_DEFAULT_IDLETIMEOUT = 5 * 60 * 1000; // 5 minutes in milliseconds format

    private static final long MF_MINIMUM_IDLETIMEOUT = 1 * 60 * 1000; // 1 minute in milliseconds format

    private static final String MF_DEFAULTDOMAINNAME = "Domain1";

    private static final String CONTEXT_CLASSNAME = Context.class.getName();

    public static final String DOMAIN = "com.sonicsw.jndi.mfcontext.domain";

    public static final String NODE = "com.sonicsw.jndi.mfcontext.node";

    /**
     * @deprecated The value is ignored. Only a single cluster is supported for JNDI communications.
     */
    public static final String SECONDARY_PROVIDER_URL = "com.sonicsw.jndi.mfcontext.secondaryProviderURL";

    /**
     * @deprecated The value is ignored. Only a single cluster is supported for JNDI communications.
     */
    public static final String SECONDARY_NODE = "com.sonicsw.jndi.mfcontext.secondaryNode";

    public static final String IDLE_TIMEOUT = "com.sonicsw.jndi.mfcontext.idleTimeout";

    public static final String CONNECT_TIMEOUT = "com.sonicsw.jndi.mfcontext.connectTimeout"; // note this is given in
                                                                                              // millis

    public static final String SOCKET_CONNECT_TIMEOUT = "com.sonicsw.jndi.mfcontext.socketConnectTimeout"; // note this
                                                                                                           // is given
                                                                                                           // in millis

    public static final String REQUEST_TIMEOUT = "com.sonicsw.jndi.mfcontext.requestTimeout"; // note this is given in
                                                                                              // millis

    private long m_requestTimeout = ConnectorClient.REQUEST_TIMEOUT_DEFAULT;

    public static final String LOOKUP_RETRIES = "com.sonicsw.jndi.mfcontext.lookupRetries";

    private int m_resolveRetries = 0;

    /**
     * @deprecated The value is ignored.
     */
    public static final String TRY_BACKUP_ON_FAILURE = "com.sonicsw.jndi.mfcontext.tryBackupOnFailure"; // deprecated

    private static final String VERBOSE_TRACING = "com.sonicsw.jndi.mfcontext.verbose";

    private boolean m_verboseTracing = false;

    /**
     * Constructs an instance of MFContext on createSubcontext(), lookup(), lookupLink() calls.
     * 
     * @see
     */
    MFContext(MFContext ctx, String ctxName)
    {
        m_connMng = ctx.m_connMng;
        m_env = ctx.m_env;
        if (m_env.get(VERBOSE_TRACING) != null)
        {
            m_verboseTracing = new Boolean((String)m_env.get(VERBOSE_TRACING)).booleanValue();
        }
        else
        {
            m_verboseTracing = Boolean.getBoolean(VERBOSE_TRACING);
        }

        if (!ctxName.startsWith("/"))
        {
            ctxName = '/' + ctxName;
        }
        m_fullCtxName = ctx.m_fullCtxName + ctxName;
        try
        {
            m_ctxName = (new EntityName(m_fullCtxName)).getBaseName();
        }
        catch (Exception e)
        {
        } // should not happen
        m_connMng.incRefCount();
    }

    /**
     * Constructs MFContext object with the provided JNDI environment properties.
     * 
     */
    public MFContext(String urls, String ctxDirName, Hashtable env)
    throws NamingException
    {
        if (env.get(VERBOSE_TRACING) != null)
        {
            m_verboseTracing = new Boolean((String)env.get(VERBOSE_TRACING)).booleanValue();
        }
        else
        {
            m_verboseTracing = Boolean.getBoolean(VERBOSE_TRACING);
        }
        String domainName = (String)env.get(DOMAIN);
        if (domainName == null)
        {
            domainName = MF_DEFAULTDOMAINNAME;
        }
        long idleTimeout = getIdleTimeout((String)env.get(IDLE_TIMEOUT));
        String node = (String)env.get(NODE);

        String connectTimeout = (String)env.get(CONNECT_TIMEOUT);
        if (connectTimeout == null)
        {
            connectTimeout = Long.toString(ConnectorClient.CONNECT_TIMEOUT_DEFAULT);
        }

        String socketConnectTimeout = (String)env.get(SOCKET_CONNECT_TIMEOUT);
        if (socketConnectTimeout == null)
        {
            socketConnectTimeout = Long.toString(ConnectorClient.SOCKET_CONNECT_TIMEOUT_DEFAULT);
        }
        else
        {
            if (new Long(socketConnectTimeout).longValue() < 0)
            {
                socketConnectTimeout = Long.toString(ConnectorClient.SOCKET_CONNECT_TIMEOUT_DEFAULT);
            }
        }

        String requestTimeout = (String)env.get(REQUEST_TIMEOUT);
        if (requestTimeout == null)
        {
            requestTimeout = Long.toString(ConnectorClient.REQUEST_TIMEOUT_DEFAULT);
        }
        m_requestTimeout = Long.parseLong(requestTimeout);

        String lookupRetries = (String)env.get(LOOKUP_RETRIES);
        if (lookupRetries != null)
        {
            m_resolveRetries = Integer.parseInt(lookupRetries);
        }

        m_connMng = MFConnectionManager.getManager(urls, (String)env.get(Context.SECURITY_PRINCIPAL), (String)env.get(Context.SECURITY_CREDENTIALS), node, domainName, idleTimeout, connectTimeout, socketConnectTimeout, requestTimeout);

        init(ctxDirName);
        m_env = env == null ? new Hashtable(11, 0.75F) : (Hashtable)env.clone();
        m_ctxName = ctxDirName;
    }

    // //////////////////////////////////////////////////////////////////////
    // Implementation-> private methods
    // /////////////////////////////////////////////////////////////////////

    private void init(String ctxDirName)
    throws NamingException
    {
        String ctxName = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR + (ctxDirName.length() == 0 ? "" : IMFDirectories.MF_DIR_SEPARATOR + ctxDirName);

        if (!checkExistanceofCtxDirectory(ctxName))
        {
            throw new NoInitialContextException(ctxDirName + " context doesn't exist");
        }

        m_fullCtxName = ctxName;
    }

    private long getIdleTimeout(String timeout)
    {
        if (timeout == null)
         {
            return MF_DEFAULT_IDLETIMEOUT; // 5 min
        }
        long idleTimeout = (new Long(timeout)).longValue();
        return ((idleTimeout < MF_MINIMUM_IDLETIMEOUT) ? MF_MINIMUM_IDLETIMEOUT : idleTimeout); // can't be less then 1
                                                                                                // min
    }

    private boolean checkExistanceofCtxDirectory(String ctxName)
    throws NamingException
    {
        try
        {
            // Sonic00036551: We must still perform some check on the DS, otherwise a NamingException will not be
            // thrown.
            // Cannot use the parent of /_MFContext because that would be the root dir and we may not have
            // permission on that directory (and thus we do not use the else logic below)
            if (ctxName.equals(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR))
            {
                m_connMng.getConnection().listDirectories(ctxName); // don't need to check results .. the mfcontext root
                                                                    // directory will always exist!
                return true;
            }
            else
            {
                String parent = (new EntityName(ctxName)).getParent();
                IDirIdentity[] list = m_connMng.getConnection().listDirectories(parent);
                for (int i = 0; i < list.length; i++)
                {
                    if (ctxName.equals(list[i].getName()))
                    {
                        return true;
                    }
                }
            }
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }

        return false;
    }

    // //////////////////////////////////////////////////////////////////////
    // public Methods
    // /////////////////////////////////////////////////////////////////////

    /**
     * Retrieves the named object. If <tt>name</tt> is empty, returns a new instance of this context.
     * 
     * @param name
     *            String the name of the object to look up, such as <tt>/factories/SonicTopicConnectionFactory</tt>
     * @return the object bound to <tt>name</tt>
     * @throws NamingException
     *             if a naming exception is encountered
     * @throws NameNotFoundException
     *             if any part of the name is not found in the underlying MF DS storage.
     * 
     * @see #lookup(Name)
     * @see #lookupLink(String)
     */
    @Override
    public final Object lookup(String name)
    throws NamingException, NameNotFoundException
    {
        Object obj = null;
        if (name == null)
        {
            return null;
        }
        if (name.length() == 0)
        {
            return this;
        }

        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }

        int retriesLeft = m_resolveRetries;

        long timeoutAt = 0;

        while (true)
        {
            try
            {
                timeoutAt = System.currentTimeMillis() + this.m_requestTimeout;

                String parent = (new EntityName(name)).getParent();
                String fullDSName = (m_fullCtxName + name);
                boolean bExist = false;

                IIdentity[] list = getLookupConnection(name).listAll(m_fullCtxName + parent);

                for (int i = 0; i < list.length; i++)
                {
                    if (fullDSName.equals(list[i].getName()))
                    {
                        bExist = true;
                        if (list[i] instanceof IDirIdentity)
                        {
                            obj = new MFContext(this, name);
                        }
                        else
                        {
                            IDirElement elmnt = getLookupConnection(name).getElement(fullDSName, false);

                            String elmntType = elmnt.getIdentity().getType();
                            if (elmntType.equals("SerializedObject"))
                            {
                                IAttributeSet attrs = elmnt.getAttributes();
                                byte[] bytes = (byte[])attrs.getAttribute("data");
                                obj = ObjHelper.deserializeObject(bytes);
                            }
                            else if (elmntType.equals("ReferenceObject"))
                            {
                                obj = getBindedRefObject(elmnt, fullDSName);
                                
                            	try
                            	{
                                    // make sure we have a p.m.j.ConnectionFactory or subclass before calling method
                                    Class baseclass = obj.getClass();
                                    while (baseclass != null)
                                    {
                                        if (baseclass.getName().equals("progress.message.jclient.ConnectionFactory"))
                                        {
                                        	Class parameters[] = new Class[] { javax.naming.Context.class };
                                    		Method m = obj.getClass().getMethod("setNamingContext",parameters);
                                    		Object[] args = new Object[] { this };
                                    		String result = (String)m.invoke(obj,args);
                                        	break;
                                        }
                                    	baseclass = baseclass.getSuperclass();
                                    }
                            	}
                            	catch (NoSuchMethodException ex)
                            	{
                            		// ignore; method doesn't exist in obj
                            	}
                            	catch (SecurityException ex1)
                            	{
                            		// shouldn't happen
                            	}
                            }
                        }
                    }
                }
                if (!bExist)
                {
                    throw (new NameNotFoundException(name + " not found in the specified context"));
                }

                return obj;
            }
            catch (Exception ex)
            {
                if (ex instanceof NamingException)
                {
                    Throwable cause = ex.getCause();
                    if (cause != null && cause instanceof InvokeTimeoutException && (retriesLeft > 0 || m_resolveRetries == -1))
                    {
                        if (m_resolveRetries > 0)
                        {
                            --retriesLeft;
                        }
                        if (cause instanceof InvokeTimeoutCommsException)
                        {
                            // since invoke timeout comms failure occur immediately, we should at least wait for
                            // a connection to be reestablished within the request timeout before executing the retry
                            while (!getLookupConnection(name).isConnected() && System.currentTimeMillis() < timeoutAt)
                            {
                                try
                                {
                                    Thread.sleep(500);
                                }
                                catch (Exception e)
                                {
                                }
                            }
                        }
                        continue;
                    }
                }

                throwNamingException(ex);
            }
        }
    }

    private MFConnection getLookupConnection(String lookupName)
    throws NamingException
    {
        try
        {
            return m_connMng.getConnection();
        }
        catch (NullPointerException e) // when the context is closed m_connMng gets set to null
        {
            throw new NamingException("Context has been closed; lookup of " + lookupName + " failed");
        }
    }

    /**
     * Retrieves the named object. See {@link #lookup(String)} for details.
     * 
     * @param name
     *            Name the name of the object to look up
     */

    @Override
    public final Object lookup(Name name)
    throws NamingException, NameNotFoundException
    {
        return lookup(m_parser.convertToString(name));
    }

    // Convert an element to the Reference of a referenceable object or to the object itself if it was Serialized.
    public static Object elementToReference(IElement element)
    throws NamingException
    {
        String elmntType = element.getIdentity().getType();
        if (elmntType.equals("SerializedObject"))
        {
            IAttributeSet attrs = element.getAttributes();
            byte[] bytes = (byte[])attrs.getAttribute("data");
            return ObjHelper.deserializeObject(bytes);
        }
        else if (elmntType.equals("ReferenceObject"))
        {
            return ObjHelper.convertElementToRef(element, false);
        }
        else
        {
            throw new NameNotFoundException(element.getIdentity().getName() + " is not of a ReferenceObject or a SerializedObject types");
        }
    }

    private Object getBindedRefObject(IDirElement elmnt, String fullDSName)
    throws NamingException
    {
        try
        {
            return NamingManager.getObjectInstance(ObjHelper.convertElementToRef(elmnt, m_verboseTracing), m_parser.parse(fullDSName), this, m_env);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
        return null;
    }

    /**
     * Binds a name to an object. See {@link #bind(String)} for details.
     * 
     * @param name
     *            (Name) the name to bind
     * @param obj
     *            the object to bind
     */
    @Override
    public final void bind(Name name, Object obj)
    throws NamingException, NameAlreadyBoundException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new InvalidNameException("Cannot bind empty name");
        }
        bind(m_parser.convertToString(name), obj);
    }

    /**
     * Binds a name to an object. All intermediate contexts and the target context must already exist.
     * 
     * @param name
     *            (String) the name to bind -> cannot be empty. The following objects types supported: Serializable,
     *            Reference and Referenceable
     * @param obj
     *            the object to bind -> can't be null
     * @throws NameAlreadyBoundException
     *             if name is already bound
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #bind(Name, Object)
     * @see #rebind(String, Object)
     */
    @Override
    public final void bind(String name, Object obj)
    throws NamingException, NameAlreadyBoundException
    {
        if (name.length() == 0)
        {
            throw new InvalidNameException("Cannot bind empty name");
        }
        if (obj == null)
        {
            throw new NamingException("Object can't be null");
        }

        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }
        try
        {
            IDirElement elmnt = null;
            obj = NamingManager.getStateToBind(obj, m_parser.parse(m_fullCtxName + name), this, m_env);
            if ((obj instanceof Referenceable) || (obj instanceof javax.naming.Reference))
            {
                elmnt = bindReferenceable(m_fullCtxName + name, obj, m_verboseTracing);
                m_connMng.getConnection().setElement(elmnt.doneUpdate(), null);
                return;
            }
            else if (obj instanceof java.io.Serializable)
            {
                elmnt = bindSerializable(m_fullCtxName + name, obj, m_verboseTracing);
                m_connMng.getConnection().setElement(elmnt.doneUpdate(), null);
                return;
            }
            else
            {
                throw new OperationNotSupportedException("Can only bind Serializable, References or Referenceable objects");
            }
        }
        catch (VersionOutofSyncException ex)
        {
            NamingException e = new NameAlreadyBoundException(name);
            if (m_verboseTracing)
            {
                e.setRootCause(ex);
            }
            throw e;
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    // Used by DS handlers to convert JNDI objects to DS elements the MF JNDI client
    public static IDirElement objectToElement(Object obj)
    throws DirectoryServiceException
    {
        if (obj == null)
        {
            throw new DirectoryServiceException("The Object can't be null");
        }

        try
        {
            if ((obj instanceof Referenceable) || (obj instanceof javax.naming.Reference))
            {
                return bindReferenceable("/NoNameIsNeeded", obj, false);
            }
            else if (obj instanceof java.io.Serializable)
            {
                return bindSerializable("/NoNameIsNeeded", obj, false);
            }
            else
            {
                throw new DirectoryServiceException("Can only bind Serializable, References or Referenceable objects");
            }
        }
        catch (NamingException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    private static IDirElement bindSerializable(String s, Object obj, boolean verboseTracing)
    throws NamingException
    {
        IDirElement elmnt = null;
        try
        {
            elmnt = ElementFactory.createElement(s, "SerializedObject", "1.0");
            IAttributeSet elmntAttributes = elmnt.getAttributes();
            elmntAttributes.setBytesAttribute("data", ObjHelper.serializeObject(obj));
            elmntAttributes.setStringAttribute("classname", obj.getClass().getName());
        }
        catch (Exception ex)
        {
            ObjHelper.throwNamingException(ex, verboseTracing);
        }

        return elmnt;
    }

    private static IDirElement bindReferenceable(String s, Object obj, boolean verboseTracing)
    throws NamingException
    {

        IDirElement elmnt = null;
        javax.naming.Reference reference = null;
        if (obj instanceof Referenceable)
        {
            reference = ((Referenceable)obj).getReference();
        }
        else if (obj instanceof javax.naming.Reference)
        {
            reference = (javax.naming.Reference)obj;
        }
        if (reference == null)
        {
            throw new NamingException("Reference can't be null");
        }
        try
        {
            elmnt = ElementFactory.createElement(s, "ReferenceObject", "1.0");
            IAttributeSet elmntAttributes = elmnt.getAttributes();
            String s1;
            if ((s1 = reference.getClassName()) != null)
            {
                elmntAttributes.setStringAttribute("classname", s1);
            }
            else
            {
                throw new NamingException("Reference has no class name");
            }
            if ((s1 = reference.getFactoryClassName()) != null)
            {
                elmntAttributes.setStringAttribute("factoryclassname", s1);
            }
            if ((s1 = reference.getFactoryClassLocation()) != null)
            {
                elmntAttributes.setStringAttribute("factorylocation", s1);
            }

            int i = reference.size();

            if (i > 0)
            {
                IAttributeList list = elmntAttributes.createAttributeList("RefAdresses");
                for (int j = 0; j < i; j++)
                {
                    RefAddr refaddr = reference.get(j);
                    String type = refaddr.getType();
                    Object obj1 = refaddr.getContent();
                    IAttributeSet set = list.addNewAttributeSetItem();
                    if (refaddr instanceof StringRefAddr)
                    {
                        set.setStringAttribute(type, (String)(obj1 != null ? obj1 : ""));
                    }
                    else
                    {
                        set.setBytesAttribute(type, (byte[])obj1);
                    }
                }
            }
        }
        catch (Exception ex)
        {
            ObjHelper.throwNamingException(ex, verboseTracing);
        }

        return elmnt;
    }

    /**
     * Binds a name to an object, overwriting any existing binding. All intermediate contexts and the target context
     * must already exist.
     * 
     * @param name
     *            (String) the name to bind; cannot be empty
     * @param obj
     *            the object to rebind; cannot be null, should be a same type as previously binded object
     * @throws NameNotFoundException
     *             if any part of the name is not found in the underlying MF DS storage.
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #rebind(Name, Object)
     * @see #bind(String, Object)
     */
    @Override
    public final void rebind(String name, Object obj)
    throws NamingException, NameNotFoundException
    {
        if (name.length() == 0 || name == null)
        {
            throw new NamingException("Cannot rebind an empty name");
        }
        if (obj == null)
        {
            throw new NamingException("Object can't be null");
        }

        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }
        try
        {
            IDirElement elmnt = null;
            // this logic here caters for the case that there is nothing to update
            while (elmnt == null)
            {
                elmnt = m_connMng.getConnection().getElement(m_fullCtxName + name, true);
                if (elmnt == null)
                {
                    try
                    {
                        bind(name, obj);
                        // if the bind worked then were done
                        return;
                    }
                    catch (NameAlreadyBoundException e)
                    {
                    } // fall through and rebind
                }
            }

            String elmntType = elmnt.getIdentity().getType();
            if (elmntType.equals("SerializedObject"))
            {
                if (obj instanceof java.io.Serializable)
                {
                    IAttributeSet elmntAttributes = elmnt.getAttributes();
                    elmntAttributes.setBytesAttribute("data", ObjHelper.serializeObject(obj));
                    elmntAttributes.setStringAttribute("classname", obj.getClass().getName());
                }
                else
                {
                    throw new NamingException("Can't rebind object of a different type. Object must be a Serializable.");
                }
            }
            else if (elmntType.equals("ReferenceObject"))
            {
                if ((obj instanceof javax.naming.Reference) || (obj instanceof Referenceable))
                {
                    rebindReferenceable(elmnt.getAttributes(), obj);
                }
                else
                {
                    throw new NamingException("Can't rebind object of a diffrent type. Object must be a Referenceable/Reference.");
                }
            }

            m_connMng.getConnection().setElement(elmnt.doneUpdate(), null);

        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    /**
     * Binds a name to an object, overwriting any existing binding. See {@link #rebind(String)} for details.
     * 
     * @param name
     *            (Name) the name to bind
     * @param obj
     *            the object to rebind
     * 
     */
    @Override
    public final void rebind(Name name, Object obj)
    throws NamingException
    {
        if (name == null || name.isEmpty())
        {
            throw new NamingException("Cannot unbind empty name");
        }
        rebind(m_parser.convertToString(name), obj);
    }

    /*
     * rebinds Reference object ==> if new Reference contains list of addresses, then the old list will be replaced with
     * the new one
     */
    private void rebindReferenceable(IAttributeSet elmntAttributes, Object obj)
    throws NamingException
    {
        javax.naming.Reference reference = null;
        if (obj instanceof Referenceable)
        {
            reference = ((Referenceable)obj).getReference();
        }
        else if (obj instanceof javax.naming.Reference)
        {
            reference = (javax.naming.Reference)obj;
        }
        try
        {
            String s1;
            if ((s1 = reference.getClassName()) != null)
            {
                elmntAttributes.setStringAttribute("classname", s1);
            }
            else
            {
                throw new NamingException("Reference has no class name");
            }
            if ((s1 = reference.getFactoryClassName()) != null)
            {
                elmntAttributes.setStringAttribute("factoryclassname", s1);
            }
            if ((s1 = reference.getFactoryClassLocation()) != null)
            {
                elmntAttributes.setStringAttribute("factorylocation", s1);
            }

            IAttributeList list = (IAttributeList)elmntAttributes.getAttribute("RefAdresses");
            if (list == null)
            {
                list = elmntAttributes.createAttributeList("RefAdresses");
            }
            else
            // delete old addresses
            {
                int count = list.getCount();
                for (int i = 0; i < count; i++)
                {
                    list.deleteAttributeItem(0);
                }
            }

            int i = reference.size();
            if (i > 0)
            {

                for (int j = 0; j < i; j++)
                {
                    RefAddr refaddr = reference.get(j);
                    String type = refaddr.getType();
                    Object obj1 = refaddr.getContent();
                    IAttributeSet set = list.addNewAttributeSetItem();
                    if (refaddr instanceof StringRefAddr)
                    {
                        set.setStringAttribute(type, (String)(obj1 != null ? obj1 : ""));
                    }
                    else
                    {
                        set.setBytesAttribute(type, ObjHelper.serializeObject(obj));
                    }
                }
            }
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    /**
     * Unbinds the named object(deletes object from the underlying MF storage) Removes the terminal name in
     * <code>name</code> from the target context.
     * 
     * @param name
     *            (String) the name to unbind; cannot be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * @see #unbind(Name)
     */

    @Override
    public final void unbind(String name)
    throws NamingException
    {
        if (name == null || name.length() == 0)
        {
            throw new NamingException("Cannot unbind empty name");
        }

        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }
        try
        {
            m_connMng.getConnection().deleteElement(m_fullCtxName + name, null);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    /**
     * Unbinds the named object(deletes object from the underlying MF storage). See {@link #unbind(String)} for details.
     * 
     * @param name
     *            (Name) the name to unbind
     */

    @Override
    public final void unbind(Name name)
    throws NamingException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new NamingException("Cannot unbind empty name");
        }
        unbind(m_parser.convertToString(name));
    }

    /**
     * Binds a new name to the object bound to an old name, and unbinds the old name(deletes old object location from
     * underlying storage). Both names are relative to this context. Intermediate contexts of the old name are not
     * changed. The underlyng MF storage doesn't support rename operation, therefore <tt>rename operation</tt> not
     * guaranteed to be atomic.
     * 
     * @param oldName
     *            (String) the name of the existing binding; cannot be empty
     * @param newName
     *            (String) the name of the new binding; cannot be empty
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #rename(Name, Name)
     * @see #bind(String, Object)
     * @see #rebind(String, Object)
     */

    @Override
    public final void rename(String oldname, String newname)
    throws NamingException
    {
        if (oldname.length() == 0 || newname.length() == 0)
        {
            throw new InvalidNameException("Cannot rename empty name");
        }
        if (!oldname.startsWith("/"))
        {
            oldname = '/' + oldname;
        }
        if (!newname.startsWith("/"))
        {
            newname = '/' + newname;
        }
        Object obj = null;
        try
        {
            IDirElement oldElemnt = m_connMng.getConnection().getElement(m_fullCtxName + oldname, false);
            String elmntType = oldElemnt.getIdentity().getType();
            if (elmntType.equals("SerializedObject"))
            {
                IAttributeSet attrs = oldElemnt.getAttributes();
                obj = ObjHelper.deserializeObject((byte[])attrs.getAttribute("data"));
            }
            else if (elmntType.equals("ReferenceObject"))
            {
                obj = getBindedRefObject(oldElemnt, m_fullCtxName + oldname);
            }

            bind(newname, obj); // bind object to the name
            m_connMng.getConnection().deleteElement(m_fullCtxName + oldname, null);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    /**
     * Binds a new name to the object bound to an old name, and unbinds the old name. See {@link #rename(String)} for
     * details.
     * 
     * @param oldName
     *            (Name) the name of the existing binding; cannot be null
     * @param newName
     *            (Name) the name of the new binding; cannot be null
     */
    @Override
    public final void rename(Name oldname, Name newname)
    throws NamingException
    {
        if (oldname == null || newname == null)
        {
            throw new InvalidNameException("Cannot rename empty name");
        }
        rename(m_parser.convertToString(oldname), m_parser.convertToString(newname));
    }

    /**
     * Enumerates the names bound in the named context, along with the class names of objects bound to them. The
     * contents of any subcontexts are not included.
     * 
     * <p>
     * If a binding is added to or removed from this context, its effect on an enumeration previously returned is
     * undefined.
     * 
     * @param name
     *            the name of the context to list
     * @return an enumeration of the names and class names of the bindings in this context. Each element of the
     *         enumeration is of type <tt>NameClassPair</tt>.
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #list(Name)
     * @see #listBindings(String)
     * @see javax.naming.NameClassPair
     */
    @Override
    public final NamingEnumeration list(String name)
    throws NamingException
    {
        String dsName;
        if (name.length() == 0)
        {
            dsName = m_fullCtxName; // list binded objects at this Context
        }
        else
        {
            if (!name.startsWith("/"))
            {
                name = '/' + name;
            }
            dsName = m_fullCtxName + name;
        }

        IDirIdentity[] contextList = null;
        IDirElement[] objectList = null;
        try
        {
            contextList = m_connMng.getConnection().getAllDirectories(dsName);
            objectList = m_connMng.getConnection().getAllElements(dsName, false);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }

        return new BindingEnumeration(contextList, objectList);
    }

    /**
     * Enumerates the names bound in the named context, along with the class names of objects bound to them. See
     * {@link #list(String)} for details.
     */
    @Override
    public final NamingEnumeration list(Name name)
    throws NamingException
    {
        if (name == null)
        {
            throw new InvalidNameException("Name object cannot be null");
        }
        return list(name.toString());
    }

    /**
     * Enumerates the names bound in the named context, along with the objects bound to them. The contents of any
     * subcontexts are not included.
     * 
     * <p>
     * If a binding is added to or removed from this context, its effect on an enumeration previously returned is
     * undefined.
     * 
     * @param name
     *            the name of the context to list
     * @return an enumeration of the bindings in this context. Each element of the enumeration is of type
     *         <tt>Binding</tt>.
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #listBindings(Name)
     * @see #list(String)
     * @see javax.naming.Binding
     */
    @Override
    public final NamingEnumeration listBindings(String name)
    throws NamingException
    {
        String dsName;
        if (name.length() == 0)
        {
            dsName = m_fullCtxName; // list binded objects at this Context
        }
        else
        {
            if (!name.startsWith("/"))
            {
                name = '/' + name;
            }
            dsName = m_fullCtxName + name;
        }

        IDirIdentity[] contextList = null;
        IDirElement[] objectList = null;
        try
        {
            contextList = m_connMng.getConnection().getAllDirectories(dsName);
            objectList = m_connMng.getConnection().getAllElements(dsName, false);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }

        return new ObjectBindingEnumeration(contextList, objectList);
    }

    /**
     * Enumerates the names bound in the named context, along with the class names of objects bound to them. See
     * {@link #listBindings(String)} for details.
     */

    @Override
    public final NamingEnumeration listBindings(Name name)
    throws NamingException
    {
        if (name == null)
        {
            throw new InvalidNameException("Name object cannot be null");
        }
        return listBindings(m_parser.convertToString(name));
    }

    /**
     * Destroys the named context and removes it from the underlying MF storage. Intermediate contexts are not
     * destroyed.
     * 
     * @param name
     *            the name of the context to be destroyed; cannot be empty
     * @throws NameNotFoundException
     *             if an intermediate context does not exist
     * @throws ContextNotEmptyException
     *             if the named context is not empty
     * @throws NamingException
     *             if a naming exception is encountered
     * 
     * @see #destroySubcontext(Name)
     */

    @Override
    public final void destroySubcontext(String name)
    throws NamingException, ContextNotEmptyException
    {
        if (name == null || name.length() == 0)
        {
            throw new NamingException("Cannot destroy subcontext with an empty name");
        }
        // If the name provided is not there or intermidiate contexts doesn't exist, throw an Exception.
        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }

        try
        {
            String fullname = m_fullCtxName + name;

            // is the directory empty?
            if (m_connMng.getConnection().listDirectories(fullname).length > 0)
            {
                throw new ContextNotEmptyException(fullname);
            }
            if (m_connMng.getConnection().getAllElements(fullname, false).length > 0)
            {
                throw new ContextNotEmptyException(fullname);
            }

            m_connMng.getConnection().deleteDirectory(fullname);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }
    }

    /**
     * Destroys the named context and removes it from the underlying MF storage. See {@link #destroySubcontext(String)}
     * for details.
     */
    @Override
    public final void destroySubcontext(Name name)
    throws NamingException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new NamingException("Cannot destroy subcontext with an empty name");
        }
        destroySubcontext(m_parser.convertToString(name));
    }

    /**
     * Creates and binds a new context. All intermediate contexts must already exist. See
     * {@link #createSubcontext(String)} for details.
     */

    @Override
    public final Context createSubcontext(Name name)
    throws NamingException, NameAlreadyBoundException, javax.naming.CommunicationException
    {
        if (name == null || name.toString().length() == 0)
        {
            throw new NamingException("Cannot create subcontext with an empty name");
        }
        return createSubcontext(m_parser.convertToString(name));
    }

    /**
     * Creates and binds a new context. All intermediate contexts must already exist.
     * 
     * @param name
     *            the name of the context to create; may not be empty
     * @return the newly created context
     * 
     * @throws NameAlreadyBoundException
     *             if name is already bound
     * @throws NamingException
     *             if a naming exception is encountered
     * @see #createSubcontext(String)
     */

    @Override
    public final Context createSubcontext(String name)
    throws NamingException, NameAlreadyBoundException, javax.naming.CommunicationException
    {
        MFContext subcontext = null;
        // If the name provided is already there or intermidiate contexts doesn't exist, throw an Exception.
        if (!name.startsWith("/"))
        {
            name = '/' + name;
        }
        try
        {
            String fullname = m_fullCtxName + name;

            // does a directory already exist?
            String parentDir = fullname.substring(0, fullname.lastIndexOf('/'));
            IDirIdentity[] dirs = m_connMng.getConnection().listDirectories(parentDir);
            for (int i = 0; i < dirs.length; i++)
            {
                if (dirs[i].getName().equals(fullname))
                {
                    throw new NameAlreadyBoundException(fullname);
                }
            }

            // is there an object already stored with that name
            if (m_connMng.getConnection().getElement(fullname, false) != null)
            {
                throw new NameAlreadyBoundException(fullname);
            }

            m_connMng.getConnection().createDirectory(fullname);
            subcontext = new MFContext(this, name);
        }
        catch (Exception ex)
        {
            throwNamingException(ex);
        }

        return subcontext;
    }

    /**
     * No support for Name Space Federation, so this method behaves as lookup See {@link #lookup(String)} for details.
     */
    @Override
    public final Object lookupLink(String name)
    throws NamingException
    {
        // No support for Name Space Federation, so this method behaves as lookup
        return lookup(name);
    }

    /**
     * No support for Name Space Federation, so this method behaves as lookup See {@link #lookup(Name)} for details.
     */
    @Override
    public final Object lookupLink(Name name)
    throws NamingException
    {
        // No support for Name Space Federation
        return lookupLink(m_parser.convertToString(name));
    }

    /**
     * Returns a Context name Parser. There is no support for Naming Federation all the Name Parsers are same.
     * 
     * @param name
     *            Name of the NameParser, may be null
     * @return parser Context Parser
     * @throws Naming
     *             Exception if there is any naming violation.
     */
    @Override
    public final NameParser getNameParser(String name)
    throws NamingException
    {
        return m_parser;
    }

    /**
     * Returns a Context name Parser. There is no support for Naming Federation all the Name Parsers are same. See
     * {@link #getNameParser(String)} for details.
     */
    @Override
    public final NameParser getNameParser(Name name)
    throws NamingException
    {
        return m_parser;
    }

    /**
     * Composes the name of this context with a name relative to this context. Given a name (name) relative to this
     * context,and the name (prefix) of this context relative to one of its ancestors, this method returns the
     * composition of the two names using the syntax appropriate for the naming system(s) involved. That is, if name
     * names an object relative to this context, the result is the name of the same object, but relative to the ancestor
     * context. None of the names may be null.
     * <p>
     * 
     * @param name
     *            Name string relative to this Context
     *            <p>
     * @param prefix
     *            Prefix string
     *            <p>
     * @return result Composition of the name with prefix.
     *         <p>
     * @throws NamingException
     *             If there is any naming violation
     */
    @Override
    public final String composeName(String name, String prefix)
    throws NamingException
    {
        if (name.length() == 0 || prefix.length() == 0)
        {
            throw new NamingException("Names can't be empty");
        }
        return (composeName(m_parser.parse(name), m_parser.parse(prefix))).toString();
    }

    /**
     * Composes the name of this context with a name relative to this context. See {@link #composeName(String, String)}
     * for details.
     */
    @Override
    public final Name composeName(Name name, Name prefix)
    throws NamingException
    {
        if (name == null || prefix == null)
        {
            throw new NamingException("Names can't be empty");
        }
        Name composition = (Name)prefix.clone();
        composition.addAll(name);
        return composition;
    }

    /**
     * Operation not supported
     * 
     * @throws OperationNotSupportedException
     */
    @Override
    public final Object addToEnvironment(String propName, Object propVal)
    throws OperationNotSupportedException
    {
        throw new OperationNotSupportedException("addToEnvironment");
    }

    /**
     * Operation not supported
     * 
     * @throws OperationNotSupportedException
     */
    @Override
    public final Object removeFromEnvironment(String propName)
    throws OperationNotSupportedException
    {
        throw new OperationNotSupportedException("addToEnvironment");
    }

    /**
     * Returns the current Context Environment.
     * 
     * @return m_env Context Enviroment.
     * @throws NamingException
     */
    @Override
    public final Hashtable getEnvironment()
    throws NamingException
    {
        if (this.m_env == null)
        {
            return new Hashtable();
        }

        return m_env;
    }

    /**
     * This method retrieves the MF DS full name of this context
     * 
     * @return String in MF DS format.
     */
    @Override
    public String getNameInNamespace()
    throws NamingException
    {
        if (m_fullCtxName.equals(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR))
        {
            return "";
        }
        else {
            return m_fullCtxName.substring(1 + IMFDirectories.MF_JNDI_DIR.length() + 1); // "/xxx"
        }
    }

    /**
     * Closes this context.
     * 
     * @throws NamingException
     *             if a naming exception is encountered
     */
    @Override
    public synchronized void close()
    throws NamingException
    {
        if (m_connMng != null)
        {
            m_connMng.close();
            m_connMng = null;
        }
    }
    
    /**
     * MFContext specific call to resolve a "sonicfs:///" URL
     */
    public String resolveURL(String url)
    throws URISyntaxException, IOException
    {
        Object obj = null;
        if (url == null)
        {
            return null;
        }
        if (url.length() == 0)
        {
            return "";
        }

        if (!url.startsWith("sonicrn:///"))
        {
            throw new URISyntaxException(url, "Not a sonicrn:/// URL", 0);
        }
        if (url.length() <= "sonicrn:///".length())
        {
            throw new URISyntaxException(url, "No routing node specified", 0);
        }

        int retriesLeft = m_resolveRetries;

        long timeoutAt = 0;

        while (true)
        {
            try
            {
                timeoutAt = System.currentTimeMillis() + this.m_requestTimeout;

                return getLookupConnection("").resolveURL(url);
            }
            catch (Exception ex)
            {
                if (ex instanceof NamingException)
                {
                    Throwable cause = ex.getCause();
                    if (cause != null && cause instanceof InvokeTimeoutException && (retriesLeft > 0 || m_resolveRetries == -1))
                    {
                        if (m_resolveRetries > 0)
                        {
                            --retriesLeft;
                        }
                        if (cause instanceof InvokeTimeoutCommsException)
                        {
                            // since invoke timeout comms failure occur immediately, we should at least wait for
                            // a connection to be reestablished within the request timeout before executing the retry
                            try
                            {
                                while (!getLookupConnection("").isConnected() && System.currentTimeMillis() < timeoutAt)
                                {
                                    try
                                    {
                                        Thread.sleep(500);
                                    }
                                    catch (Exception e)
                                    {
                                    }
                                }
                            }
                            catch (NamingException e)
                            {
                            }
                        }
                        continue;
                    }
                }

                IOException ioException = new IOException("Failed to resolve " + url + ", see cause...");
                ioException.initCause(ex instanceof NamingException ? ex.getCause() : ex);
                throw ioException;
            }
        }
    }

    @Override
    protected void finalize()
    {
        try
        {
            close();
        }
        catch (NamingException _ex)
        {
        }
    }

    private void throwNamingException(Exception e)
    throws NamingException
    {
        ObjHelper.throwNamingException(e, m_verboseTracing);
    }

    // ///////////////////////////////////////////////////////////////////////
    // Class for enumerating name/class pairs
    // ///////////////////////////////////////////////////////////////////////

    final class BindingEnumeration
    implements NamingEnumeration
    {
        HashSet set = new HashSet();

        Iterator m_iterator;

        BindingEnumeration(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            init(contextList, objectList);
            m_iterator = set.iterator();
        }

        /**
         * check for any more elements in the present Context.
         * 
         * @return true if there are any elements available in the set.
         */

        @Override
        public boolean hasMore()
        throws NamingException
        {
            return hasMoreElements();
        }

        /**
         * check for any more elements in the present Context.
         * 
         * @return true if there are any elements available in the set.
         */
        @Override
        public boolean hasMoreElements()
        {
            return m_iterator.hasNext();
        }

        /**
         * returns next element in the Context. The return value is of type NameClassPair.
         * 
         * @return Object NameClassPair object which represents the object name and class name pair of a binding found
         *         in a context.
         */
        @Override
        public Object next()
        throws NamingException
        {
            return nextElement();
        }

        /**
         * returns next element in the Context. The return value is of type NameClassPair.
         * 
         * @return Object NameClassPair object which represents the object name and class name pair of a binding found
         *         in a context.
         */
        @Override
        public Object nextElement()
        {
            return m_iterator.next();
        }

        /**
         * clears set
         */
        @Override
        public void close() // after close, NamingEnumeration
        {
            set.clear();
        }

        private void init(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            for (int i = 0; i < contextList.length; i++)
            {
                String fullname = contextList[i].getName();
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), CONTEXT_CLASSNAME));
                }
                catch (Exception ex)
                {
                } // should not happen
            }
            for (int i = 0; i < objectList.length; i++)
            {
                String fullname = objectList[i].getIdentity().getName();
                IAttributeSet attrs = objectList[i].getAttributes();
                String className = (String)attrs.getAttribute("classname");
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), className));
                }
                catch (Exception ex)
                {
                } // should not happen
            }
        }
    }//

    // ///////////////////////////////////////////////////////////////////////////////////
    // Class for enumerating bindings
    // ///////////////////////////////////////////////////////////////////////////////////

    final class ObjectBindingEnumeration
    implements NamingEnumeration
    {
        HashSet set = new HashSet();

        Iterator m_iterator;

        ObjectBindingEnumeration(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            init(contextList, objectList);
            m_iterator = set.iterator();
        }

        /**
         * check for any more elements in the present Context.
         * 
         * @return true if there are any elements available in the set.
         */

        @Override
        public boolean hasMore()
        throws NamingException
        {
            return hasMoreElements();
        }

        /**
         * returns next element in the Context. The return value is of type Binding.
         * 
         * @return Object Binding object which represents the object name, class name and object of a binding found in a
         *         context.
         */
        @Override
        public Object next()
        throws NamingException
        {
            return nextElement();
        }

        /**
         * returns next element in the Context. The return value is of type Binding.
         * 
         * @return Object Binding object which represents the object name, class name and object of a binding found in a
         *         context.
         */
        @Override
        public Object nextElement()
        {
            return m_iterator.next();
        }

        /**
         * check for any more elements in the present Context.
         * 
         * @return true if there are any elements available in the set.
         */
        @Override
        public boolean hasMoreElements()
        {
            return m_iterator.hasNext();
        }

        /**
         * clears set of elements
         */
        @Override
        public void close()
        {
            set.clear();
        }

        private void init(IDirIdentity[] contextList, IDirElement[] objectList)
        {
            for (int i = 0; i < contextList.length; i++)
            {
                String fullname = contextList[i].getName();
                try
                {
                    set.add(new NameClassPair((new EntityName(fullname)).getBaseName(), CONTEXT_CLASSNAME));
                }
                catch (Exception ex)
                {
                } // should not happen
            }
            for (int i = 0; i < objectList.length; i++)
            {
                String fullname = objectList[i].getIdentity().getName();
                String elmntType = objectList[i].getIdentity().getType();
                IAttributeSet attrs = objectList[i].getAttributes();
                String className = (String)attrs.getAttribute("classname");
                Object obj = null;
                try
                {
                    if (elmntType.equals("SerializedObject"))
                    {
                        obj = ObjHelper.deserializeObject((byte[])attrs.getAttribute("data"));
                    }
                    else if (elmntType.equals("ReferenceObject"))
                    {
                        obj = getBindedRefObject(objectList[i], fullname);
                    }
                    set.add(new Binding((new EntityName(fullname)).getBaseName(), className, obj));
                }
                catch (Exception ex)
                {
                } // should not happen
            }
        }

    } //
}//

