/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;

import modelobjects.framework.EditRule;
import modelobjects.framework.IntToStringConverter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelObjectPropagator;
import modelobjects.framework.model.ModelObjectValidator;
import modelobjects.framework.model.ModelPropagationException;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.swing.JCheckBoxAdapter;
import modelobjects.framework.swing.JSpinnerViewAspectAdapter;
import modelobjects.framework.swing.JTextFieldAdapter;

import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.messagelog.MessageTableModel;
import com.sonicsw.ma.gui.propsheets.JPropSheetTabbedDialog;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.JButtonPanel;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JPathField;
import com.sonicsw.ma.gui.util.JPathFieldAdapter;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;

public final class JPreferencesDialog extends JPropSheetTabbedDialog
{
    public static final String GENERAL_PREFS      = "preferences.general";
    public static final String METRIC_PREFS       = "preferences.metrics";
    public static final String NOTIFICATION_PREFS = "preferences.notifications";
    public static final String MESSAGE_PREFS      = "preferences.messages";

    // General Tab components.
    //
    public static final String SAVE_WINDOW_SIZES     = "saveWindowSizes";
    public static final String SAVE_WINDOW_POSITIONS = "saveWindowPositions";
    public static final String AM_POLL_FREQUENCY     = "amPollFrequencySecs";
    public static final String USE_SPECIFIC_EDITOR_FOR_PROPERTIES = "useSpecificEditorForProperties";
    public static final String PROPERTY_FILE_EDITOR_CMD = "propertyFileEditorCmd";

    public static final int DEFAULT_AM_POLL_FREQUENCY = 30;

    private JCheckBox m_saveWindowSizes;
    private JCheckBox  m_useSpecificEditorForProperties;
    private JCheckBox  m_saveWindowPositions;
    private JTextField m_amPollFrequency;
    private JTextField m_propertyFileEditorCmd;

    // metrics
    //
    private JTextField m_pollFrequency;
    private JTextField m_maxPlotPoints;
    private JTextField m_aggrPollFrequency;

    public static final String POLL_FREQUENCY      = "pollFrequencySecs";
    public static final String MAX_PLOT_POINTS     = "defaultMaxPlotPoints";
    public static final String AGGREGATE_POLL_FREQUENCY  = "aggrPollFrequencySecs";

    public static final int DEFAULT_POLL_FREQUENCY              = 30;
    public static final int DEFAULT_MAX_PLOT_POINTS             = 500;
    public static final int DEFAULT_AGGREGATE_POLL_FREQUENCY    = 60;

    // notifications
    //
    private JTextField m_maxNotifications;

    public static final String MAX_NOTIFICATIONS = "maxNotifications";

    public static final int DEFAULT_MAX_NOTIFICATIONS = 200;

    // message viewer
    //
    private JPathField m_logFilePath;
    private JCheckBox  m_logFile;
    private JSpinner m_logScreenShow;
    private JTextField m_connTimeout = new JTextField();
    private JTextField m_connCacheSize = new JTextField(2);

    public static final String LOG_FILE       = "logFile";
    public static final String LOG_FILEPATH   = "logFilePath";
    public static final String LOG_SCREENSHOW = "logScreenShowLast";

    // generic

    private Model m_model;

    public JPreferencesDialog(JMAFrame parent, PreferenceManager pManager)
    {
        super(parent, "preferences", JPreferencesDialog.Model.class);

        setValidator(new Validator());
        setPropagator(new Propagator());

        m_model = new Model(pManager);
    }

    @Override
    public void maInitialize()
    {
        super.maInitialize();

        try
        {
            editInstance(null, m_model, false);
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error initializing the dialog.", e, true);
        }
    }

    @Override
    public void editInstance(IPlugin plugin,Object model,boolean isNew)
        throws Exception
    {
        super.editInstance(plugin,model,isNew);
    }

    @Override
    public void initUI() throws Exception
    {
        addTabPanel("General",        createGeneralPanel());
        addTabPanel("Message Viewer", createMessagesPanel());
        addTabPanel("Metrics",        createMetricsPanel());
        addTabPanel("Notifications",  createNotificationsPanel());
    }

    @Override
    public void initForm() throws Exception
    {
        // General tab
        new JCheckBoxAdapter(ModelAspectId.forName("saveWindowSizes"),
                              m_saveWindowSizes,EditRule.ALWAYS,null,getMediator());
        new JCheckBoxAdapter(ModelAspectId.forName("saveWindowPositions"),
                              m_saveWindowPositions,EditRule.ALWAYS,null,getMediator());
        new JCheckBoxAdapter(ModelAspectId.forName("useSpecificEditorForProperties"),
                             m_useSpecificEditorForProperties,EditRule.ALWAYS,null,getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("propertyFileEditorCmd"),
                              m_propertyFileEditorCmd, EditRule.ALWAYS,
                              null, getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("amPollFrequency"),
                              m_amPollFrequency, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("connectionTimeout"),
                              m_connTimeout, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("maxConnections"),
                              m_connCacheSize, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());

        m_propertyFileEditorCmd.setEnabled(m_useSpecificEditorForProperties.isSelected());

        // metrics
        new JTextFieldAdapter(ModelAspectId.forName("pollFrequency"),
                              m_pollFrequency, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("maxPlotPoints"),
                              m_maxPlotPoints, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());
        new JTextFieldAdapter(ModelAspectId.forName("aggrPollFrequency"),
                              m_aggrPollFrequency, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());

        // notifications
        new JTextFieldAdapter(ModelAspectId.forName("maxNotifications"),
                              m_maxNotifications, EditRule.ALWAYS,
                              new IntToStringConverter(), getMediator());

        // message viewer
        new JSpinnerViewAspectAdapter(ModelAspectId.forName("logScreenShow"),
                                      m_logScreenShow, EditRule.ALWAYS,
                                      null, getMediator());
        new JCheckBoxAdapter(ModelAspectId.forName("logFile"),
                              m_logFile, EditRule.ALWAYS,
                              null, getMediator());
        new JPathFieldAdapter(ModelAspectId.forName("logFilePath"),
                              m_logFilePath, EditRule.ALWAYS,
                              null, getMediator());
    }

    private JPanel createMetricsPanel()
    {
        String unitSec = ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_UNIT, "seconds"});

        JPartitionPanel res = new JPartitionPanel(true, "p,p", null);

        JPartitionPanel plotPanel = new JPartitionPanel("Metric Data");

        m_pollFrequency      = new JTextField(3);
        m_maxPlotPoints      = new JTextField(5);
        m_aggrPollFrequency  = new JTextField(5);

        plotPanel.addRow("Poll Frequency",          m_pollFrequency, unitSec);
        plotPanel.addRow("Default Max Plot Points", m_maxPlotPoints);
        plotPanel.addRow("Aggregate Poll Frequency",      m_aggrPollFrequency, unitSec);

        res.add(plotPanel);

        return res;
    }

    private JPanel createNotificationsPanel()
    {
        JPartitionPanel res = new JPartitionPanel(true, "p", null);

        JPartitionPanel watchPanel = new JPartitionPanel("Watch Data");

        m_maxNotifications = new JTextField(3);

        watchPanel.addRow("Default Max Notifications", m_maxNotifications);

        res.add(watchPanel);

        return res;
    }

    private JPanel createMessagesPanel()
    {
        JPartitionPanel res = new JPartitionPanel(true, "p,p", null);

        JPartitionPanel screenPanel = new JPartitionPanel("Screen");

        //m_logScreenShow = new JTextField(3);
        m_logScreenShow = new JSpinner(new SpinnerNumberModel(1, 1, 1000, 1));

        screenPanel.addRow("Show last N messages", m_logScreenShow);

        res.add(screenPanel);

        JPartitionPanel filePanel = new JPartitionPanel("File");

        m_logFile     = new JCheckBox();
        m_logFilePath = new JPathField(16);

        m_logFilePath.setButtonAction(new LogFileAction());

        filePanel.addRow("Log to file", m_logFile);
        filePanel.addRow("Log path",    m_logFilePath);
        filePanel.addRow(createClearLogFilePanel());

        res.add(filePanel);

        return res;
    }

    private JPanel createClearLogFilePanel()
    {
        JPartitionPanel res = new JPartitionPanel(false, "p, r", null);

        res.add(new JButton(new LogFileClearAction()));
        res.add(new JPanel());

        return res;
    }

    private JPanel createGeneralPanel()
    {
        String unitSec = ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_UNIT, "seconds"});

        JPartitionPanel res = new JPartitionPanel(true,"p,p,p,p,p", null);

        JPartitionPanel generalPanel = new JPartitionPanel(true, "p,p", "Window Size / Position");

        m_saveWindowSizes = new JCheckBox(ResourceManager.getString(getClass(),
            "JPreferencesDialog.SaveWindowSizes"));
        generalPanel.add(m_saveWindowSizes);

        m_saveWindowPositions = new JCheckBox(ResourceManager.getString(getClass(),
            "JPreferencesDialog.SaveWindowPositions"));
        generalPanel.add(m_saveWindowPositions);

        JPartitionPanel pollPanel = new JPartitionPanel(false, "p,r,p", "Agent Manager");
        m_amPollFrequency = new JTextField(4);
        pollPanel.addRow(ResourceManager.getString(getClass(), "JPreferencesDialog.AMPollFrequency"),
                         m_amPollFrequency, unitSec);

        JPartitionPanel resetPanel = new JPartitionPanel(false, "r,p", "User Preferences");

        resetPanel.add(new JLabel("Delete all stored user preferences"));
        resetPanel.add(new JButton(new ResetAction()));

        m_useSpecificEditorForProperties = new JCheckBox();
        m_propertyFileEditorCmd = new JTextField(50);

        m_useSpecificEditorForProperties.addItemListener(new PropertyFileEditorCmdListener(m_propertyFileEditorCmd));

        JPartitionPanel propertyEditorPanel = new JPartitionPanel(false, "p,r",
                                                                  ResourceManager.getString(getClass(), "JPreferencesDialog.PropertyFileEditor") );

        propertyEditorPanel.addRow(ResourceManager.getString(getClass(), "JPreferencesDialog.UseSpecificFileEditor"),
                                   m_useSpecificEditorForProperties);
        propertyEditorPanel.addRow(ResourceManager.getString(getClass(), "JPreferencesDialog.FileEditorCmd"),
                                   m_propertyFileEditorCmd);

        res.add(generalPanel);
        res.add(pollPanel);
        res.add(createGeneralConnectionPanel());
        res.add(resetPanel);
        res.add(propertyEditorPanel);

        return res;
    }

    private JPanel createGeneralConnectionPanel()
    {
        JPartitionPanel panel = new JPartitionPanel("Connection");
        String unitSec = ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_UNIT, "seconds"});

        panel.addRow(ResourceManager.getString(getClass(), "JPreferencesDialog.Connection.Timeout"), m_connTimeout, unitSec);
        panel.addRow("Maximum cache size", m_connCacheSize);

        return panel;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class LogFileClearAction extends BasicGuiAction
    {
        LogFileClearAction()
        {
          super("tool.prefs.log.file.clear");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            String logFilePath = m_logFilePath.getText();

            JWaitCursor wc = new JWaitCursor(JPreferencesDialog.this);

            try
            {
                File logFile = new File(logFilePath);

                logFile.delete();

                Toolkit.getDefaultToolkit().beep();
                
                m_logFilePath.setText("");
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                     "Failed to clear log file: " + logFilePath, e, false);
            }
            finally
            {
                wc.release();
            }
        }
    }

    class PropertyFileEditorCmdListener implements ItemListener {

        private final JComponent propertyFileEditorCmdComponent;

        PropertyFileEditorCmdListener(JComponent propertyFileEditorCmdComponent)
        {
            this.propertyFileEditorCmdComponent = propertyFileEditorCmdComponent;
        }

        @Override
        public void itemStateChanged(ItemEvent e) {
            if (propertyFileEditorCmdComponent != null) {
                final Object evnSourceObj = e.getSource();
                if (evnSourceObj instanceof AbstractButton) {
                    boolean isSelected = ((AbstractButton) evnSourceObj).getModel().isSelected();
                    propertyFileEditorCmdComponent.setEnabled(isSelected);
                }
            }
        }
    }

    class LogFileAction extends BasicGuiAction
    {
        LogFileAction()
        {
          super("tool.prefs.log.file.browse");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JFileChooser chooser = new JFileChooser();

            chooser.setSelectedFile(new File(m_logFilePath.getText()));
            chooser.setApproveButtonText("OK");
            chooser.setDialogTitle("Choose log path");
            int res = chooser.showSaveDialog(JPreferencesDialog.this);

            if (res == JFileChooser.APPROVE_OPTION)
            {
                m_logFilePath.setText(chooser.getSelectedFile().getPath());
            }
        }
    }

    public class Model
    {
        protected PreferenceManager m_pManager;

        // General tab.
        private boolean m_saveWindowSizes = false;
        private boolean m_saveWindowPositions = false;
        private boolean m_useSpecificEditorForProperties = false;
        private int m_amPollFrequency;
        private int m_connTimeout;
        private int m_maxConnections;
        private String m_propertyFileEditorCmd;

        // metrics

        private int m_pollFrequency;
        private int m_maxPlotPoints;
        private int m_aggrPollFrequency;
        private int m_secondsPerPixel;
        private int m_width;
        private int m_height;

        // notifications

        private int m_maxNotifications;

        // message viewer

        private int m_logScreenShow;
        private String  m_logFilePath;
        private boolean m_logFile;


        private Model(PreferenceManager pManager)
        {
            m_pManager = pManager;

            // General tab.
            m_saveWindowSizes      = pManager.getBoolean(GENERAL_PREFS, SAVE_WINDOW_SIZES,     true);
            m_saveWindowPositions  = pManager.getBoolean(GENERAL_PREFS, SAVE_WINDOW_POSITIONS, true);
            m_useSpecificEditorForProperties = pManager.getBoolean(GENERAL_PREFS, USE_SPECIFIC_EDITOR_FOR_PROPERTIES, false);
            m_propertyFileEditorCmd = pManager.getString(GENERAL_PREFS, PROPERTY_FILE_EDITOR_CMD, "");

            m_amPollFrequency      = pManager.getInt(GENERAL_PREFS, AM_POLL_FREQUENCY, DEFAULT_AM_POLL_FREQUENCY);
            m_connTimeout          = pManager.getInt(GENERAL_PREFS, DomainConnectionModel.PREFS_TIMEOUT, DomainConnectionModel.DEF_TIMEOUT);
            m_maxConnections       = pManager.getInt(GENERAL_PREFS, DomainConnectionModel.PREFS_MAX_CONNECTIONS, DomainConnectionModel.DEF_MAX_CONNECTIONS);

            // metrics
            m_pollFrequency        = pManager.getInt(METRIC_PREFS, POLL_FREQUENCY,      DEFAULT_POLL_FREQUENCY);
            m_maxPlotPoints        = pManager.getInt(METRIC_PREFS, MAX_PLOT_POINTS,     DEFAULT_MAX_PLOT_POINTS);
            m_aggrPollFrequency    = pManager.getInt(METRIC_PREFS, AGGREGATE_POLL_FREQUENCY, DEFAULT_AGGREGATE_POLL_FREQUENCY);

            // notifications
            m_maxNotifications = pManager.getInt(NOTIFICATION_PREFS, MAX_NOTIFICATIONS, DEFAULT_MAX_NOTIFICATIONS);

            // messages
            m_logFile       = pManager.getBoolean(MESSAGE_PREFS, LOG_FILE,       MessageTableModel.DEFAULT_LOGFILE);
            m_logFilePath   = pManager.getString(MESSAGE_PREFS,  LOG_FILEPATH,   System.getProperty("user.home") + File.separator + "console.log");
            m_logScreenShow = pManager.getInt(MESSAGE_PREFS,     LOG_SCREENSHOW, MessageTableModel.DEFAULT_LOGSHOW);
        }

        // General tab.
        public boolean getSaveWindowSizes() { return (m_saveWindowSizes); }
        public void setSaveWindowSizes(boolean value) { m_saveWindowSizes = value; }

        public boolean getSaveWindowPositions() { return (m_saveWindowPositions); }
        public void setSaveWindowPositions(boolean value) { m_saveWindowPositions = value; }

        public boolean getUseSpecificEditorForProperties() { return (m_useSpecificEditorForProperties); }
        public void setUseSpecificEditorForProperties(boolean value) { m_useSpecificEditorForProperties = value; }

        public String getPropertyFileEditorCmd() { return (m_propertyFileEditorCmd); }
        public void setPropertyFileEditorCmd(String value) { m_propertyFileEditorCmd = value; }

        public int getAmPollFrequency() { return m_amPollFrequency; }
        public void setAmPollFrequency(int pollFrequency) { m_amPollFrequency = pollFrequency; }

        public int getConnectionTimeout() { return m_connTimeout; }
        public void setConnectionTimeout(int value) { m_connTimeout = value; }

        public int getMaxConnections() { return m_maxConnections; }
        public void setMaxConnections(int value) { m_maxConnections = value; }

        // metrics
        public int getPollFrequency() { return m_pollFrequency; }
        public void setPollFrequency(int pollFrequency) { m_pollFrequency = pollFrequency; }

        public int getMaxPlotPoints() { return m_maxPlotPoints; }
        public void setMaxPlotPoints(int maxPlotPoints) { m_maxPlotPoints = maxPlotPoints; }

        public int getAggrPollFrequency() { return m_aggrPollFrequency; }
        public void setAggrPollFrequency(int aggrPollFrequency) { m_aggrPollFrequency = aggrPollFrequency;}

        public int getSecondsPerPixel() { return m_secondsPerPixel; }
        public void setSecondsPerPixel(int secondsPerPixel) { m_secondsPerPixel = secondsPerPixel; }

        public int getWidth() { return m_width; }
        public void setWidth(int width) { m_width = width; }

        public int getHeight() { return m_height; }
        public void setHeight(int height) { m_height = height; }

        // notifications
        public int getMaxNotifications() { return m_maxNotifications; }
        public void setMaxNotifications(int maxNotifications) { m_maxNotifications = maxNotifications; }

        // message viewer
        public boolean getLogFile() { return m_logFile; }
        public void setLogFile(boolean logFile) { m_logFile = logFile; }

        public String getLogFilePath() { return m_logFilePath; }
        public void setLogFilePath(String logFilePath) { m_logFilePath = logFilePath; }

        public Integer getLogScreenShow() { return new Integer(m_logScreenShow); }
        public void setLogScreenShow(Integer logScreenShow) { m_logScreenShow = logScreenShow.intValue(); }
    }

    private static class Validator implements ModelObjectValidator
    {
        @Override
        public void validateModelEdit(ModelObjectAdapter adapter)
            throws ModelValidationException
        {
            JPreferencesDialog.Model model = (JPreferencesDialog.Model)adapter.getModelObject();

            // General
            //
            if (model.getAmPollFrequency() < 5)
            {
                throw new ModelValidationException("The Agent Manager poll frequency must be >= 5 seconds");
            }

            if (model.getConnectionTimeout() < 0)
            {
                throw new ModelValidationException("The connection timeout must be a positive number");
            }

            if (model.getMaxConnections() < 1)
            {
                throw new ModelValidationException("The recently used cache size must be >= 1");
            }
            if (model.getMaxConnections() > 100)
            {
                throw new ModelValidationException("The recently used cache size must be <= 100");
            }

            // metrics
            //
            if (model.getPollFrequency() < 10)
            {
                throw new ModelValidationException("The poll frequency must be >= 10 seconds");
            }
            if (model.getPollFrequency() > 900)
            {
                throw new ModelValidationException("The poll frequency must be <= 900 seconds");
            }
            if (model.getMaxPlotPoints() < 2)
            {
                throw new ModelValidationException("The default maximum number of plot points must be >= 2");
            }
            if (model.getMaxPlotPoints() > 10000)
            {
                throw new ModelValidationException("The default maximum number of plot points must be <= 10000");
            }

            if (model.getAggrPollFrequency() < 60)
            {
                throw new ModelValidationException("The aggregate poll frequency must be >= 60 seconds");
            }

            // notifications
            //
            if (model.getMaxNotifications() < 1)
            {
                throw new ModelValidationException("The maximum number of notifications per watch window must be > 0");
            }
            if (model.getMaxNotifications() > 1000)
            {
                throw new ModelValidationException("The maximum number of notifications per watch window must be <= 1000");
            }
        
            //message log
            //
            if (model.getLogFile())
            {
                String path = model.getLogFilePath();
                if (path == null || path.length() == 0)
                {
                    throw new ModelValidationException("Log file path can't be an empty string");
                }
            }
        }

        @Override
        public void validateModelCreation(ModelObjectAdapter adapter)
            throws ModelValidationException
        {
            throw new ModelValidationException("Create invalid");
        }

        @Override
        public void validateModelDeletion(ModelObjectAdapter adapter)
            throws ModelValidationException
        {
            throw new ModelValidationException("Delete invalid");
        }
    }

    private static class Propagator implements ModelObjectPropagator
    {
        /**
         * Propagate edits to a model object.
         *
         * @param adapter the ModelObjectAdapter that holds the model
         * @exception ModelPropagationException if the edits cannot be propagated
         */
        @Override
        public void propagateModelEdit(ModelObjectAdapter adapter)
            throws ModelPropagationException
        {
            JPreferencesDialog.Model model = (JPreferencesDialog.Model)adapter.getModelObject();

            // Note: The last parameter on each "set()" call is always set to
            // false to prohibit flushing the attribute when caching it.  If
            // a flush is done for every attribute being set, it slows down
            // the performance of it.  Once all the parameters are set, one
            // flush is done on the root node to write them out.

            model.m_pManager.setBoolean(GENERAL_PREFS,SAVE_WINDOW_SIZES, model.getSaveWindowSizes(),false);
            model.m_pManager.setBoolean(GENERAL_PREFS,SAVE_WINDOW_POSITIONS, model.getSaveWindowPositions(),false);
            model.m_pManager.setBoolean(GENERAL_PREFS,USE_SPECIFIC_EDITOR_FOR_PROPERTIES, model.getUseSpecificEditorForProperties(), false);
            model.m_pManager.setString(GENERAL_PREFS,PROPERTY_FILE_EDITOR_CMD, model.getPropertyFileEditorCmd(), false);

            model.m_pManager.setInt(GENERAL_PREFS, AM_POLL_FREQUENCY, model.getAmPollFrequency(), false);
            model.m_pManager.setInt(GENERAL_PREFS, DomainConnectionModel.PREFS_TIMEOUT, model.getConnectionTimeout(), false);
            model.m_pManager.setInt(GENERAL_PREFS, DomainConnectionModel.PREFS_MAX_CONNECTIONS, model.getMaxConnections(), false);

            // metrics
            model.m_pManager.setInt(METRIC_PREFS, POLL_FREQUENCY, model.getPollFrequency(), false);
            model.m_pManager.setInt(METRIC_PREFS, MAX_PLOT_POINTS, model.getMaxPlotPoints(), false);
            model.m_pManager.setInt(METRIC_PREFS, AGGREGATE_POLL_FREQUENCY , model.getAggrPollFrequency(), false);

            // notifications
            model.m_pManager.setInt(NOTIFICATION_PREFS, MAX_NOTIFICATIONS, model.getMaxNotifications(), false);

            // message viewer
            model.m_pManager.setBoolean(MESSAGE_PREFS, LOG_FILE,       model.getLogFile(), false);
            model.m_pManager.setString(MESSAGE_PREFS,  LOG_FILEPATH,   model.getLogFilePath(), false);
            model.m_pManager.setInt(MESSAGE_PREFS,     LOG_SCREENSHOW, model.getLogScreenShow().intValue(), false);
            
            //always reset internal logToFile to the default 
            MgmtConsole.resetLogToFile(MessageTableModel.DEFAULT_LOGFILE);
            
            model.m_pManager.flush();
        }

        /**
         * Propagate a newly created model object.
         *
         * @param modelObjectAdapter the ModelObjectAdapter that holds the model
         * @exception ModelPropagationException if the create cannot be propagated
         */
        @Override
        public void propagateModelCreation(ModelObjectAdapter modelObjectAdapter)
            throws ModelPropagationException
        {
        }

        /**
         * Propagate deletion of a model object.
         *
         * @param modelObjectAdapter the ModelObjectAdapter that holds the model
         * @exception ModelPropagationException if the delete cannot be propagated
         */
        @Override
        public void propagateModelDeletion(ModelObjectAdapter modelObjectAdapter)
            throws ModelPropagationException
        {
        }
    }

    public class ResetAction extends BasicGuiAction
    {
        public ResetAction()
        {
            super("JPreferencesDialog.reset");
        }

        @Override
        public void actionPerformed(ActionEvent event)
        {
            String message = ResourceManager.getString(getClass(),
                "JPreferencesDialog.ResetWarning");

            int result = JOptionPane.showConfirmDialog(null, message,
                                                       getValue(Action.NAME) + " " + getTitle(), JOptionPane.YES_NO_OPTION,
                                                       JOptionPane.WARNING_MESSAGE);

            if (result == JOptionPane.YES_OPTION)
            {
                setVisible(false);

                Thread t = new Thread(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        // Reset the User Preferences to their default values.
                        PreferenceManager prefMgr = PreferenceManager.getInstance();
                        prefMgr.resetUserPreferences();

                        // Persist the default User Preferences.
                        m_model = new Model(prefMgr);

                        try
                        {
                            editInstance(null,m_model,false);
                            initForm();

                            Action action = getDefaultOKAction();
                            if (action != null)
                            {
                                action.setEnabled(true);
                                JButton button = ((JButtonPanel)getButtonPanel()).getButton(
                                    (String)action.getValue(Action.ACTION_COMMAND_KEY));

                                if (button != null)
                                {
                                    button.doClick();
                                }
                            }
                        }
                        catch (Exception e)
                        {
                            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
                        }
                    }
                } );

                t.start();
            }
        }
    }
}
