// Copyright (c) 2002, 2007 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.ma.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.util.Map;
import java.util.StringTokenizer;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.ToolTipManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.sonicsw.ma.gui.config.AbstractConfigPlugin;
import com.sonicsw.ma.gui.table.AbstractTableContentPane;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.table.ITableContentPane;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.PluginMouseClickListener;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.gui.util.SwingWorker;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IFolderPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginChangeListener;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigChange;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigChangeListener;

public abstract class JWorkspacePanel extends JPanel
    implements TreeSelectionListener, ListSelectionListener, FocusListener, IConfigChangeListener
{
    protected IPluginContext m_context;
    protected JSplitPane     m_split;
    protected JTree          m_tree;
    protected JRowTable      m_tableCurrent       = null;
    protected boolean        m_bIsInitialized     = false;
    private   JComponent     m_defaultContentPane = null;
    private   Component      m_lastFocus          = null;
    private   PluginTreeExpansionListener m_tel   = null;

    public JWorkspacePanel()
    {
        super(new BorderLayout());
        
        setWSPanelBorder();
        m_split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true);
        m_split.setDividerLocation(250);

        m_tree = new JTree()
        {
            @Override
            public boolean isFixedRowHeight()
            {
                return true;
            }

            @Override
            public String getToolTipText(MouseEvent evt)
            {
                TreePath path = getPathForLocation(evt.getX(), evt.getY());

                String text = null;

                if(path != null && path.getLastPathComponent() instanceof IPlugin)
                {
                    text = ((AbstractGUIPlugin)path.getLastPathComponent()).getToolTipText();
                }

                return text;
            }
        };

        // Register the expansion listener first because subsequent operations
        // are going to issue expansion events that we want to track...
        m_tree.addTreeWillExpandListener(m_tel = new PluginTreeExpansionListener());

        // Register the menu shower that will automatically generate and display
        // a popup menu appropriate for the selected tree node(s)...
        m_tree.addMouseListener(new PluginMouseClickListener(m_tree));

        ToolTipManager.sharedInstance().registerComponent(m_tree);

        m_split.setLeftComponent(new ExtendedJScrollPane(m_tree));
        m_split.setFocusable(false);   // we don't want the splitter to get the focus.

        m_tree.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        m_tree.addTreeSelectionListener(this);
        m_tree.addFocusListener(this);

        setContentPane(null);

        addSplitProperty();
    }

    private void setWSPanelBorder() {
        setBorder(BorderFactory.createEmptyBorder(JPartitionPanel.BORDER_SIZE, JPartitionPanel.BORDER_SIZE, JPartitionPanel.BORDER_SIZE, JPartitionPanel.BORDER_SIZE));
    }
    
    private void addSplitProperty() {
        add(m_split, BorderLayout.CENTER);
    }
    
    public boolean isInitialized() { return m_bIsInitialized; }

    public boolean getTreeExpansionMode()
    {
        return m_tel.getAsyncMode();
    }

    public void setTreeExpansionMode(boolean bAsync)
    {
        m_tel.setAsyncMode(bAsync);
    }

    public Icon getIcon()
    {
        return null;
    }

    public void initialize(IPluginContext context)
    {
        if (isInitialized() == false)
        {
            m_context = context;

            m_tree.addTreeWillExpandListener(new TreeWillExpandListener()
            {
                @Override
                public void treeWillExpand(TreeExpansionEvent evt)
                    throws ExpandVetoException
                {
                }

                @Override
                public void treeWillCollapse(TreeExpansionEvent evt)
                    throws ExpandVetoException
                {
                    Object node = evt.getPath().getLastPathComponent();
                    Object root = m_tree.getModel().getRoot();

                    if (node.equals(root))
                    {
                        throw new ExpandVetoException(evt, "Can't collapse the root");
                    }
                }
            });

            // This is a bit of a hack so that setParent will be called in the
            // AbstractGuiPlugin. SetParent can then add in the dummy Boolean.TRUE
            // child node if it is required.
            ((DefaultMutableTreeNode)m_tree.getModel().getRoot()).setParent(null);

            context.getConfigContext().getConfigServer().addConfigChangeListener(null, this, null, null);

            boolean bOldMode = getTreeExpansionMode();
            setTreeExpansionMode(false);
            // Se we will collapse the root nodes and then expand it
            // to cause a TreeExpansionEvent to be fired...we want this because
            // the event is used to load/unload child nodes.
            //
            for (int i = m_tree.getVisibleRowCount() - 1; i >= 0; i--)
            {
                m_tree.collapseRow(i);
                m_tree.expandRow(i);
            }
            setTreeExpansionMode(bOldMode);

            m_bIsInitialized = true;
        }
    }

    public void dispose()
    {
        if (m_tree != null)
        {
            m_tree.removeFocusListener(this);
            m_tree.removeTreeSelectionListener(this);
            ToolTipManager.sharedInstance().unregisterComponent(m_tree);

            m_tree = null;
        }

        if (m_tableCurrent != null)
        {
            m_tableCurrent.cleanup();
        }
        removeAll();

        m_context            = null;
        m_defaultContentPane = null;
        m_lastFocus          = null;
    }

    public abstract Object[] getSelectedItems();
    public abstract Object   getSelectedItem();
    public abstract AbstractButton[] getToolbarItems();
    public abstract JComponent[] getMenuItems(int type);

    public boolean isPluginActive(IPlugin plugin)
    {
        IContentPane pane = getContentPane();

        return pane.getPlugin() == plugin;
    }

    public IContentPane getContentPane()
    {
        Component c = m_split.getRightComponent();

        return (c instanceof IContentPane) ? (IContentPane)c : null;
    }

    public IContentPane getContentPaneIfShown(IPlugin plugin)
    {
        IContentPane pane = getContentPane();

        return (pane != null && pane.getPlugin() == plugin) ? pane : null;
    }

    /**
     * If the specified plugins content pane is currently shown then update
     * the childPlugin row in the content pane
     *
     * @param plugin the plugin to check
     * @param childPlugin the plugin to update in the content pane
     * @return the content pane or null if the content pane isn't shown
     */
    public IContentPane refreshContentPaneIfShown(IPlugin plugin, IPlugin childPlugin)
    {
        IContentPane pane = getContentPaneIfShown(plugin);

        if(pane != null)
        {
            if (pane instanceof IPluginChangeListener)
            {
                ((IPluginChangeListener)pane).onUpdated(childPlugin);
            }
            else
            {
                pane.refresh();
            }
        }
        return pane;
    }

    /**
     * If the specified plugins content pane is currently shown then refresh
     * it.
     *
     * @param plugin the plugin to check
     * @return the content pane or null if the content pane isn't shown
     */
    public IContentPane refreshContentPaneIfShown(IPlugin plugin)
    {
        IContentPane pane = getContentPaneIfShown(plugin);

        if(pane != null)
        {
            pane.refresh();
        }

        return pane;
    }

    public JTree getTree()
    {
        return m_tree;
    }

    public DefaultMutableTreeNode getSelectedNode()
    {
        DefaultMutableTreeNode node = null;

        if (m_tree.getSelectionCount() > 0)
        {
            node = (DefaultMutableTreeNode)m_tree.getSelectionPath().getLastPathComponent();
        }

        return node;
    }

    public Component getFocusComponent()
    {
        return m_lastFocus;
    }


    // ------------------------------------------------------------------------
    // IConfigChangeListener implementation
    //
    @Override
    public void elementAdded(ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::elementAdded " + evt.getName());

        _elementAdded(evt.getName());
    }

    private void _elementAdded(final String name)
    {
        SwingWorker sw = new SwingWorker()
        {
            IPlugin m_parentPlugin = getPathToPluginParent(name);

            @Override
            public Object construct()
            {
                // Get the parent plugin
                Map attributes = null;

                if (m_parentPlugin != null)
                {
                    if(getPathToPlugin(name) != null)
                    {
                        return null;
                    }

                    try
                    {
                        // Fetch the meta attributes for this element.
                        attributes = m_context.getConfigContext().
                            getConfigServer().getMetaAttributes(name);
                    }
                    catch(Exception e)
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
                        e.printStackTrace();
                    }
                }
                return attributes;
            }

            @Override
            public void finished()
            {
                Map attributes = (Map)getValue();

                // Double check that the plugin doesn't already exist
                if(getPathToPlugin(name) != null)
                {
                    return;
                }

                if (attributes != null && m_parentPlugin != null)
                {
                    m_parentPlugin.onAdded(name, attributes);
                }
            }
        };

        sw.start();
    }

    @Override
    public void elementUpdated(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::elementUpdated " + evt.getName());

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                // Get the plugin that has been deleted
                IPlugin plugin = getPathToPlugin(evt.getName());

                if (plugin != null)
                {
                    plugin.onUpdated();
                }
                else
                {
                    _elementAdded(evt.getName());
                }
            }
        });
    }

    @Override
    public void elementReplaced(ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::elementReplaced " + evt.getName());

        System.out.println("JWorkspacePanel::elementReplaced not implemented");
    }

    @Override
    public void elementDeleted(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::elementDeleted " + evt.getName());

        _elementDeleted(evt.getName());
    }

    private void _elementDeleted(final String name)
    {
        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                IPlugin plugin = getPathToPlugin(name);

                if(plugin != null)
                {
                    IPlugin parentPlugin = (IPlugin)((AbstractGUIPlugin)plugin).getParent();

                    // Call delete on the parent plugin to delete the child
                    if (parentPlugin != null)
                    {
                        parentPlugin.onDeleted(plugin);
                    }
                }
            }
        });
    }

    @Override
    public void folderAdded(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::folderAdded " + evt.getName());

        _elementAdded(evt.getName());
    }

    @Override
    public void folderDeleted(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::folderDeleted " + evt.getName());

        _elementDeleted(evt.getName());
    }

    @Override
    public void rename(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::rename " + evt.getName() + " to " + evt.getNewName());

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                IPlugin plugin = getPathToPlugin(evt.getName());

                if (plugin != null)
                {
                    plugin.onRenamed(evt.getNewName());
                }
                else
                {
                    // The plugin to be renamed doesn't exist in our tree so lets
                    // assume that it should be there and create it
                    _elementAdded(evt.getNewName());
                }
            }
        });
    }

    private boolean productVersionSupported(IPlugin plugin,
    		                                String  productVersion)
    {
        // Sonic00027532 (AHJ)
        // Not guaranteed a product version (it is optional).
        // If it is missing then we assume it is supported.
        if (productVersion == null)
        {
            return true;
        }

    	PluginAttributes pa = plugin.getAttributes().getParent();
    	String[] pVersions = pa.getProductVersions();

    	for (int i = 0; i < pVersions.length; i++)
    	{
    		if (productVersion.equals(pVersions[i]))
            {
                return true;
            }
    	}

    	return false;
    }

    @Override
    public void metaAttributesChanged(final ConfigChange evt)
    {
        Helper.logDebugMessage(getClass().getName() + "::metaAttributeChanged " + evt.getName());

        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                String  name   = evt.getName();
                IPlugin plugin = getPathToPlugin(name);
                boolean basicUpdate = false;

                if (plugin == null)
                {
            		// If we have a meta attribute change notification and the plugin
                    // doesn't already exist, check to see if the parent plugin exists
                    IPlugin parentPlugin = getPathToPluginParent(name);

                    // ... and if it does treat this as a new plugin
                    if (parentPlugin != null)
                    {
                        parentPlugin.onAdded(name, evt.getMetaAttributes());
                    }
                }
                else
                if (plugin instanceof IConfigPlugin)
                {
                	if (plugin instanceof IFolderPlugin)
                	{
                        // If the plugin is a folder plugin then we are probably
                        // receiving this notification because the folder is
                        // actually a folder element
                        _elementReplaced(name, plugin, getPathToPluginParent(name), evt.getMetaAttributes());
                	}
                	else
                    if(plugin.getAttributes().isUpgradeable())
                	{
                		String oldVersion = (String)plugin.getAttributes().getConfigVersion();
                		String newVersion = (String)evt.getMetaAttributes().get(PluginAttributes.CONFIG_VERSION);

                        if (!oldVersion.equals(newVersion) ||
                            !productVersionSupported(plugin, (String)evt.getMetaAttributes().get(ConfigServerUtility.PRODUCT_VERSION)))
                        {
                            _elementUpgraded(name);
                        }
                        else
                        {
                            basicUpdate = true;
                        }
                	}
                }
                else
                {
                    basicUpdate = true;
                }

                if (basicUpdate && (plugin != null))
                {
                	// Just set the new meta attributes on the existing plugin
                	//
                	PluginAttributes pa = ((AbstractGUIPlugin)plugin).getAttributes();
                	pa.put(evt.getMetaAttributes());
                }
            }
        });
    }

    protected void _elementUpgraded(final String name)
    {
        Helper.logDebugMessage(getClass().getName() + "::elementUpgraded " + name);

        SwingWorker sw = new SwingWorker()
        {
           IPlugin m_plugin       = getPathToPlugin(name);
           IPlugin m_parentPlugin = getPathToPluginParent(name);

           @Override
        public Object construct()
           {
               // Get the plugin upgraded meta-attributes
               Map attributes = null;

               if (m_parentPlugin != null)
               {
                    try
                    {
                        // Fetch the meta attributes for this element.
                        attributes = m_context.getConfigContext().
                            getConfigServer().getMetaAttributes(name);
                    }
                    catch(Exception e)
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
                    }
                }
                return attributes;
            }

            @Override
            public void finished()
            {
                Map attributes = (Map)getValue();

                _elementReplaced(name, m_plugin, m_parentPlugin, attributes);

                //if(name.endsWith(ConfigServerUtility.DEFAULT_SUFFIX))
                    //m_plugin.refresh();

                if (m_plugin != null && m_parentPlugin != null && attributes != null)
                {
                    _upgradeRuntimePlugin(name);
                }
            }
        };
        sw.start();
    }

    private void _elementReplaced(String name, IPlugin plugin, IPlugin parentPlugin, Map attributes)
    {
        Helper.logDebugMessage(getClass().getName() + "::_elementReplaced " + name);

        if (plugin != null && parentPlugin != null)
        {
            // Remember whether this node was already selected
            boolean isSelected = getTree().isPathSelected(new TreePath
                (((DefaultMutableTreeNode)plugin).getPath()));

            // Call delete on the parent plugin to delete the upgraded child
            parentPlugin.onDeleted(plugin);

            // Call add on the parent plugin
            if(attributes != null)
            {
                ((AbstractGUIPlugin)parentPlugin).onAdded(name, attributes);

                if(isSelected)
                {
                    String tail = null;

                    if(name.endsWith(ConfigServerUtility.DEFAULT_SUFFIX))
                    {
                        tail = name.substring(name.lastIndexOf('/') + 1);
                    }

                    tail = name.substring(name.lastIndexOf('/') + 1);

                    TreeNode node = ((AbstractGUIPlugin)parentPlugin).findChildNode(tail);

                    if(node != null)
                    {
                        getTree().setSelectionPath(new TreePath(((DefaultMutableTreeNode)node).getPath()));
                        refreshContentPaneIfShown((IPlugin)node);
                        
                        /*
                         *  Bug Fix # Sonic00040393 : had to refresh the bean to load the bean attributes properly 
                         */
                        if ( node instanceof AbstractConfigPlugin )
                        {
                        	try {
                        		AbstractConfigPlugin newPlugin = (AbstractConfigPlugin) node;
                        		if ( newPlugin.getModel() instanceof ConfigBeanModel ){
                        			ConfigBeanModel model = (ConfigBeanModel) newPlugin.getModel();
                        			if ( model != null && model.getData() instanceof IConfigBean )
                                    {
                                        ((IConfigBean)model.getData()).refresh();
                                    }
                        		}
                        	} catch (Exception e) {
                        	    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
							}
                        }
                        
                    }
                    else
                    {
                        getTree().setSelectionPath(new TreePath(((DefaultMutableTreeNode)parentPlugin).getPath()));
                    }
                }
            }
        }
    }

    // Refresh entire root: if runtime panel is initialized and
    // upgraded plugin is locally cached by mx layer.(See bug Sonic00019242)
    private void _upgradeRuntimePlugin(String name)
    {
        JWorkspacePanel panel = m_context.getWorkspace().getWorkspacePanel(JWorkspaceRuntimePanel.PANEL_NAME);

        if (!panel.isInitialized())
        {
            return;
        }

        IPlugin rootNode = (IPlugin)((DefaultTreeModel)panel.getTree().getModel()).getRoot();

        if (rootNode != null)
        {
            rootNode.refresh();
        }
    }

    public void doExpansion(final DefaultMutableTreeNode node,
        final boolean bForceRefresh,
        final boolean isSelectionChange)
    {
        doExpansion(node, bForceRefresh, isSelectionChange, true);
    }

    public void doExpansion(final DefaultMutableTreeNode node,
        final boolean bForceRefresh,
        final boolean isSelectionChange,
        final boolean bAsync)
    {
        if (bAsync)
        {
            Helper.asyncExec(
                new Runnable()
                {
                    @Override
                    public void run()
                    {
                        doExpansionImpl(node, bForceRefresh, isSelectionChange);
                    }
                });
        }
        else
        {
            doExpansionImpl(node, bForceRefresh, isSelectionChange);
        }
    }

    public void doCollapse(final DefaultMutableTreeNode node)
    {
        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            final DefaultMutableTreeNode flag = node.isLeaf() ? null :
                (DefaultMutableTreeNode)node.getFirstChild();

            if ((flag != null) && !(flag.getUserObject()instanceof Boolean))
            {
                m_tree.setSelectionPath(new TreePath(node.getPath()));
                node.removeAllChildren();
                node.add((new javax.swing.tree.DefaultMutableTreeNode(Boolean.TRUE)));
                node.add(flag);
                ((DefaultTreeModel)m_tree.getModel()).nodeStructureChanged(node);

            }
        }
        finally
        {
            wc.release();
        }
    }

    private void doContentPane(final DefaultMutableTreeNode node)
    {
        // Now load the content pane...we do this after (potentially)
        // adding child nodes so that the content pane can be built
        // from the (new) child nodes...in most cases this will result
        // in simplier, faster content panes.
        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            final IContentPane contentPane;

            if (node instanceof IPlugin)
            {
                contentPane = ((IPlugin)node).getPluginContentPane();

                if (contentPane != null)
                {
                    contentPane.refresh();
                }
            }
            else
            {
                contentPane = null;
            }

            setContentPane(contentPane);
        }
        finally
        {
            wc.release();
        }
    }

    private void debugPluginAttributes(IPlugin plugin)
    {
        if (plugin != null)
        {
            StringBuffer buffer = new StringBuffer();

            if(plugin.getPluginName() != null)
            {
                buffer.append(plugin.getPluginName());
            }
            else
            {
                buffer.append("<PLUGIN NAME NOT DEFINED>");
            }

            buffer.append(" > ");
            buffer.append(plugin.getAttributes().toString());

            PluginAttributes parent = plugin.getAttributes().getParent();

            if(parent != null)
            {
                buffer.append(" [ParentAttributes ");
                buffer.append(parent.toString());
                buffer.append("]");
            }

            Helper.logDebugMessage(buffer.toString());
        }
    }

    //-------------------------------------------------------------------------
    //
    // TreeSelectionListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void valueChanged(TreeSelectionEvent evt)
    {
        TreePath               path = evt.getNewLeadSelectionPath();
        DefaultMutableTreeNode node = (path != null) ? (DefaultMutableTreeNode)path.getLastPathComponent() : null;

        if (node != null)
        {
            if (node instanceof AbstractGUIPlugin)
            {
                IPlugin plugin = (IPlugin) node;

                // debugPluginAttributes(plugin);

                MgmtConsole.getMgmtConsole().getToolBar().setActions(node);

                plugin.getPluginContext().getWorkspace().updateDescription(plugin.getPluginPath());
            }

            doExpansion(node, false, true);
        }
    }

    private JComponent getDefaultContentPane()
    {
        if (m_defaultContentPane == null)
        {
            JComponent c = new JLabel(ResourceManager.getApplicationLogo(getClass(), "logo"));
            c.setOpaque(true);
            c.setBackground(Color.white);

            m_defaultContentPane = c;
        }

        return m_defaultContentPane;
    }

    private void setContentPane(final IContentPane contentPane)
    {
        Helper.invoke(new Runnable()
        {
            @Override
            public void run()
            {
                if (m_split.getRightComponent()instanceof ITableContentPane)
                {
                    ITableContentPane oldContentPane = (ITableContentPane)m_split.getRightComponent();
                    m_tableCurrent = oldContentPane.getTable();

                    if (m_tableCurrent != null)
                    {
                        m_tableCurrent.removeFocusListener(JWorkspacePanel.this);
                        m_tableCurrent.getSelectionModel().removeListSelectionListener(JWorkspacePanel.this);
                        m_tableCurrent.cleanup();
                    }
                }

                int nPos = m_split.getDividerLocation();
                m_split.setRightComponent((contentPane != null) ? contentPane.getComponent() :
                    getDefaultContentPane());
                m_split.setDividerLocation(nPos);

                if (contentPane instanceof ITableContentPane)
                {
                    m_tableCurrent = ((ITableContentPane)contentPane).getTable();

                    // Add a focus listener to the content panes' table so that we can
                    // keep track of the selected plugin in the content pane, for
                    // updating the toolbar and edit menu actions correctly when switching
                    // between the Configure and Manage tabs.  Yikes!
                    //
                    m_tableCurrent.addFocusListener(JWorkspacePanel.this);
                    m_tableCurrent.getSelectionModel().addListSelectionListener(JWorkspacePanel.this);
                    m_tableCurrent.loadColumnPrefs();
                }

                if (m_split.getLastDividerLocation() > m_split.getMinimumDividerLocation())
                {
                    m_split.setDividerLocation(m_split.getLastDividerLocation());
                }
            }
        });
    }

    protected void doExpansionImpl(final DefaultMutableTreeNode node,
        final boolean bForceRefresh,
        final boolean isSelectionChange)
    {
        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            if (node == null)
            {
                System.out.println("We have a null node");
                return;
            }

            final DefaultMutableTreeNode flag = (node.getChildCount() > 0) ?
                (DefaultMutableTreeNode)node.getFirstChild() : null;

            if (bForceRefresh || (flag != null) && (flag.getUserObject() instanceof Boolean))
            {
                synchronized (node)
                {
                    Helper.invoke(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            node.removeAllChildren();
                        }
                    }, true);

                    if (!((AbstractGUIPlugin)node).expand())
                    {
                    	m_tree.collapsePath(new TreePath(node.getPath()));
                    }

                    Helper.invoke(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            ((DefaultTreeModel)m_tree.getModel()).nodeStructureChanged(node);
                        }
                    }, true);
                }
            }

            if (isSelectionChange)
            {
                doContentPane(node);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
            e.printStackTrace();
        }
        finally
        {
            wc.release();
        }
    }

    //-------------------------------------------------------------------------

    /**
     * Given a / delimited path traverse the tree to determine the plugin
     * at that location.
     */
    public IPlugin getPathToPlugin(String path)
    {
        return _getPathToPlugin(getCorrectConfigPath(path));
    }

    protected String getCorrectConfigPath(String name) {
        return name;
    }

    protected IPlugin _getPathToPlugin(String path) {
        TreeNode        parent    = (TreeNode)m_tree.getModel().getRoot();
        StringTokenizer tokenizer = new StringTokenizer(path, "/");

        while (tokenizer.hasMoreTokens())
        {
            String   token     = tokenizer.nextToken();
            TreeNode newParent = null;

            for (int i = 0; i < parent.getChildCount(); i++)
            {
                TreeNode child = parent.getChildAt(i);

                if (child instanceof AbstractGUIPlugin)
                {
                    if (token.equals(((AbstractGUIPlugin)child).getPluginName()) ||
                        token.equals(((AbstractGUIPlugin)child)._getPluginName()))
                    {
                        newParent = child;
                        break;
                    }
                }
            }

            if(newParent == null)
            {
                return null;
            }

            parent = newParent;
        }
        return (IPlugin)parent;
    }

    public IPlugin getPathToPluginParent(String path)
    {
        return getPathToPlugin(stripTerminalNode(path));
    }

    //-------------------------------------------------------------------------
    //
    // FocusListener implementation
    //
    //-------------------------------------------------------------------------

    /**
     * Method to save off the component with the focus.
     *
     * @param event The focus event that caused the action
     */
    @Override
    public void focusGained(FocusEvent evt)
    {
        // If the focus is fleeting or the focus hasn't changed since it was
        // last set then we don't need to do anything :)
        //
        if (evt.isTemporary() && (m_lastFocus == evt.getSource()))
        {
            return;
        }

        m_lastFocus = (Component)evt.getSource();

        MgmtConsole console = MgmtConsole.getMgmtConsole();

        console.getToolBar().clearActions();

        if (evt.getSource() == m_tree)
        {
            console.getToolBar().setActions(getSelectedNode());
        }
        else
        {
            Container container = (Container)evt.getSource();

            while (container != null)
            {
                if (container instanceof IContentPane)
                {
                    console.getToolBar().setActions((IContentPane)container);
                    break;
                }
                container = container.getParent();
            }
        }
    }

    /**
     * Unused method.
     *
     * @param event The focus event that caused the action
     */
    @Override
    public void focusLost(FocusEvent event)
    {
    }

    //-------------------------------------------------------------------------
    //
    // ListSelectionListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void valueChanged(ListSelectionEvent evt)
    {
        if (evt.getValueIsAdjusting())
        {
            return;
        }

        // If the content pane contains a table then we should track the
        // changes in selection, updating the toolbar/menubar actions...
        IContentPane cp = getContentPane();

        if ((cp != null) && (cp instanceof AbstractTableContentPane))
        {
            MgmtConsole.getMgmtConsole().getToolBar().setActions(cp);
        }
    }

    //-------------------------------------------------------------------------

    // Make sure expansion is threaded and updating the tree model
    // only occurs within the event dispatching thread.
    //
    class PluginTreeExpansionListener implements TreeWillExpandListener
    {
        private boolean m_bAsync = true;

        @Override
        public void treeWillExpand(TreeExpansionEvent evt)
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)evt.getPath().getLastPathComponent();

            doExpansion(node, false, false, m_bAsync);
        }

        @Override
        public void treeWillCollapse(TreeExpansionEvent evt)
        {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)evt.getPath().getLastPathComponent();

            doCollapse(node);
        }

        public void setAsyncMode(boolean bAsync)
        {
            m_bAsync = bAsync;
        }

        public boolean getAsyncMode()
        {
            return m_bAsync;
        }
    }

    public void updateToolbarIfRequired(IPlugin plugin)
    {
        if(m_tree == this.getFocusComponent() && isShowing())
        {
            if (m_tree.getSelectionCount() > 0)
            {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode)m_tree.getSelectionPath().getLastPathComponent();

                if (node == plugin)
                {
                    MgmtConsole.getMgmtConsole().getToolBar().setActions(node);
                }
            }
        }
    }

    private String stripTerminalNode(String strPath)
    {
        String strNewPath = strPath;

        int index = strPath.lastIndexOf('/');

        if(index != -1)
        {
            strNewPath = strPath.substring(0, index);
        }

        return strNewPath;
    }
}
