/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.text.DateFormat;
import java.util.Date;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

import javax.management.MBeanException;
import javax.management.ReflectionException;
import javax.management.RuntimeErrorException;
import javax.management.RuntimeMBeanException;
import javax.management.RuntimeOperationsException;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.ToolTipManager;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.model.ModelPropagationException;
import modelobjects.framework.model.ModelValidationException;

import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.domain.JDomainConnectionDialog;
import com.sonicsw.ma.gui.messagelog.JMessageInternalFrame;
import com.sonicsw.ma.gui.messagelog.MessageTableModel;
import com.sonicsw.ma.gui.perms.ManagementSecurityUtils;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JMADesktopPane;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JMAStatusBar;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JProgressDialog;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.gui.util.SwingWorker;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IFilePlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginContext;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IdentityOnlyConfigBean;

import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFProxyException;
import com.sonicsw.mf.common.MFProxyRuntimeException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;


public class MgmtConsole extends JMAFrame implements IApplication
{
    public static final boolean DEVELOPER_MODE = System.getProperty("developer", "false").equalsIgnoreCase("true");
    public static final boolean LOCAL_CLASSES  = System.getProperty("localclasses", "false").equalsIgnoreCase("true");
    public static final boolean FLUSH_PREFS    = System.getProperty("flushprefs", "false").equalsIgnoreCase("true");
    public static final Level   LOG_LEVEL      = getLogLevel(System.getProperty("loglevel"), Level.INFO);

    public static final int CLIP_ACTION_CUT  = 0;
    public static final int CLIP_ACTION_COPY = 1;

    private static boolean m_logToFile = false;

    private static volatile boolean s_initialized = false;

    private int                   m_clipAction;
    private Object                m_clipObject;
    private JMADesktopPane        m_desktopPane;
    private ToolBar               m_toolbar;
    private JMAStatusBar          m_statusbar;
    private MessageTableModel     m_messageModel;
    private JMessageInternalFrame m_messageWindow;
    private MouseHandler          m_mouseHandler;
    private ViewMenu              m_viewMenu;
    private EditMenu              m_editMenu;
    private PreferenceManager     m_prefManager;

    private ApplicationListenerSupport m_appListenerSupport = new ApplicationListenerSupport();

    protected IPluginContext m_context;

    protected static final String SEE_MSGVIEWER_MESSAGE = "\nSee the Message Viewer for details and trace information about this notice.";

    private static class MgmtConsoleHolder {
        private static final MgmtConsole INSTANCE = new MgmtConsole();
        
        static
        {
            INSTANCE.initialize();
            s_initialized = true;
        }
    }

    /**
     * Get the singleton instance
     */
    public static MgmtConsole getMgmtConsole()
    {
        return MgmtConsoleHolder.INSTANCE;
    }
    
    public static boolean isInitialized()
    {
        return s_initialized;
    }

    /**
     * Used to determine the correct log level to use when logging system events
     *
     * @param logLevelText  text representation of the log level
     * @param defaultLevel  the level to use if the supplied text cannot be
     *                      converted to a valid log level
     * @return              the threshold level for logging events
     */
    private static Level getLogLevel(String logLevelText, Level defaultLevel)
    {
        Level res = defaultLevel;

	try {
		if (logLevelText != null) {
			res = Level.parse(logLevelText.toUpperCase());
		}
	} catch (Exception e) {
	    displayMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
	}

        return res;
    }

    public static void resetLogToFile(boolean value)
    {
        m_logToFile = value;
    }

    public static boolean isLogToFileReset()
    {
        return m_logToFile;
    }

    public WorkspaceWindow getSelectedWorkspaceWindow()
    {
        if (getDesktopPane().getSelectedFrame() != null && getDesktopPane().getSelectedFrame() instanceof WorkspaceWindow)
        {
            return (WorkspaceWindow)getDesktopPane().getSelectedFrame();
        }

        return null;
    }

    private MgmtConsole()
    {
        super("console");
    }
    
    private void initialize()
    {
        m_messageModel = new MessageTableModel();

        Helper.getLogger().setLevel(LOG_LEVEL);
        if(DEVELOPER_MODE)
        {
            Helper.getLogger().setUseParentHandlers(false);
            Helper.getLogger().addHandler(new DebugMessageHandler());
        }
        Helper.getLogger().addHandler(new SMCMessageHandler());
        Helper.logInfoMessage("Sonic Management Console started.");

        m_prefManager = PreferenceManager.getInstance();

        if(FLUSH_PREFS)
        {
            System.out.println("Reset the user preferences in the preference manager");
            m_prefManager.resetUserPreferences();
        }

        // Init DesktopPane
        m_desktopPane = new JMADesktopPane();

        // Init mouse handle
        m_mouseHandler = new MouseHandler();
        prepareMgmtConsole();
    }

    private void prepareMgmtConsole() {
        
        if(DEVELOPER_MODE)
        {
            setTitle(getTitle() + " - DEVELOPER");
        }
        setImageIcon((ImageIcon)ResourceManager.getApplicationIcon(getClass(), "Application",36));
        // Init Menubar
        setJMenuBar(initMenuBar());

        m_toolbar   = new ToolBar();
        m_statusbar = new JMAStatusBar(null);

        getContentPane().add(m_toolbar, BorderLayout.NORTH);
        getContentPane().add(m_statusbar, BorderLayout.SOUTH);
        getContentPane().add(new ExtendedJScrollPane(m_desktopPane), BorderLayout.CENTER);

        showStatusBar(m_viewMenu.getShowStatus());
        showToolBar(m_viewMenu.getShowToolBar());
        IdentityOnlyConfigBean.setEnabled(true);
    }
    
     @Override
    public boolean canClose()
    {
        if (DEVELOPER_MODE)
        {
            return true;
        }

        return (showConfirmDialog("Are you sure you want to exit the Sonic Management Console?",
                                  "Exit Application",
                                  JOptionPane.QUESTION_MESSAGE,
                                  JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION);
    }

    public void connectToDefaultDomain()
    {
        connectToDomain(new DomainConnectionModel(m_prefManager));
    }

    /**
     * Connect to a domain
     */
    public void connectToDomain(DomainConnectionModel model)
    {
        if (model == null)
        {
            model = new DomainConnectionModel(m_prefManager);
        }

        JDomainConnectionDialog dialog = new JDomainConnectionDialog(this);
        dialog.pack();

        try
        {
            dialog.editInstance((IPlugin)null, model, true);
            dialog.setVisible(true);
        }
        catch (Exception e)
        {
            notifyMessage(MESSAGE_ERROR, "Error in the connect dialog.", e, true);
        }

        if (dialog.getCloseCommand() != JMADialog.CLOSE_CANCEL)
        {
            DomainConnectionModel dcm = (DomainConnectionModel)dialog.getModel();

            // save the users input to preferences regardless of whether the connection
            // attempt works or fails.
            dcm.saveToPrefs(m_prefManager);

            JProgressDialog waitDlg = new JProgressDialog(this, "Connecting...", true);
            waitDlg.setContent(createConnectingPanel(dcm));

            MgmtConsole.getMgmtConsole().notifyMessage(MESSAGE_STATUS,
                "Connecting to " + dcm.getDomainDescription(), false);

            waitDlg.setWorker(new ConnectWorker(dcm, waitDlg));
            waitDlg.setVisible(true);
        }

        dialog.dispose();
    }

    private JPanel createConnectingPanel(DomainConnectionModel dcm)
    {
        JPartitionPanel dataPanel = new JPartitionPanel(false, "p,r", null);
        String connectionName = ResourceManager.getString(getClass(), "dialog.domain.general.connection_name.label");
        String domainName     = ResourceManager.getString(getClass(), "dialog.domain.general.domain_name.label");
        String url            = ResourceManager.getString(getClass(), "dialog.domain.general.connection_url.label");

        dataPanel.addRow(connectionName, new JLabel(dcm.getConnectionName()));
        dataPanel.addRow(domainName,     new JLabel(dcm.getDomainName()));
        dataPanel.addRow(url,            new JLabel(dcm.getUrl()));

        JLabel icon = new JLabel("", ResourceManager.getIcon(getClass(), "Connect"), JLabel.CENTER);
        icon.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));

        JPanel topPanel = new JPartitionPanel(false, "p,r", "");
        topPanel.add(icon);
        topPanel.add(dataPanel);

        return topPanel;
    }

    class ConnectWorker extends SwingWorker
    {
        JProgressDialog       m_dialog;
        DomainConnectionModel m_model;
        IPluginContext        m_context;
        boolean               m_cancelled = false;

        public ConnectWorker(DomainConnectionModel model, JProgressDialog dialog)
        {
            super();

            m_model  = model;
            m_dialog = dialog;
        }

        @Override
        public Object construct()
        {
            Object res = null;

            m_context = new PluginContext(m_model);

            try
            {
                m_context.create();

                res = m_context;
            }
            catch (Throwable e)
            {
                res = e;
            }
            return res;
        }

        @Override
        public void finished()
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.this);

            try
            {
                if (m_dialog != null)
                {
                    m_dialog.setVisible(false);
                }

                if (m_cancelled)
                {
                    disconnectConnection();
                    return;
                }

                if (getValue() == null)
                {
                    return;
                }

                m_dialog.getCancelButton().setEnabled(false);

                if (getValue()instanceof Throwable)
                {
                    Throwable th = (Throwable)getValue();

                    MgmtConsole.getMgmtConsole().notifyMessage(MESSAGE_ERROR,
                        "Failed to connect to " + m_context.getConnectionInfo().getDomainDescription(),
                        new Exception("Could not establish a connection on the specified URL(s).\r\n" + th.getMessage(), th),
                        true);

                    disconnectConnection();
                }
                else if (!m_cancelled)
                {
                    Thread t = new Thread(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            launchWorkspaceWindow(m_context);

                            MgmtConsole.getMgmtConsole().notifyMessage(MESSAGE_STATUS,
                                "Successfully connected to " +
                                m_context.getConnectionInfo().getDomainDescription(),
                                false);
                        }
                    });

                    t.start();
                }
            }
            finally
            {
                if (m_dialog != null)
                {
                    m_dialog.setWorker(null);
                    m_dialog.dispose();
                    m_dialog = null;
                }

                wc.release();
            }
        }

        @Override
        public void interrupt()
        {
            m_cancelled = true;

            m_context.destroy();

            MgmtConsole.getMgmtConsole().notifyMessage(MESSAGE_INFO,
                    "Connection attempt to " +
                    m_context.getConnectionInfo().getDomainDescription() +
                    " cancelled by user",
                    false);

            super.interrupt();
        }

        private void disconnectConnection()
        {
            try
            {
                m_model.disconnect();
            }
            catch(Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MESSAGE_ERROR,
                    "Failed to disconnect " + e.getMessage(), e, false);
            }
        }
    }

    //-------------------------------------------------------------------------
    // JMAFrame abstract implementation
    //-------------------------------------------------------------------------

    @Override
    protected void maInitialize()
    {
    }

    @Override
    protected void maCleanup()
    {
        if (m_desktopPane != null)
        {
            m_desktopPane.maCleanup();
        }
    }

    //-------------------------------------------------------------------------

    public void launchWorkspaceWindow(IPluginContext context)
    {

    	this.m_context = context;

        try
        {
            String windowTitle = context.getConnectionInfo().getDomainDescription();

            showInternalFrame(new WorkspaceWindow(windowTitle, context), true);
        }
        catch (Exception e)
        {
            notifyMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
        }
    }

    /**
     * Method to disconnect the user belonging to the active workspace window.
     */
    public void disconnect()
    {
        JInternalFrame selectedFrame = m_desktopPane.getSelectedFrame();

        if (selectedFrame != null && selectedFrame instanceof WorkspaceWindow)
        {
            selectedFrame.dispose();
        }
    }

    public void showInternalFrame(JInternalFrame internalFrame, boolean visible)
    {
        getDesktopPane().add(internalFrame);

        internalFrame.setVisible(visible);
    }

    /**
     * Get the toolbar used by the application
     */
    public ToolBar getToolBar()
    {
        return m_toolbar;
    }

    /**
     * Initializes the menu bar.
     */
    private JMenuBar initMenuBar()
    {
        JMenuBar menuBar = new JMenuBar();

        menuBar.add(new ActionMenu(m_prefManager, m_mouseHandler));

        m_editMenu = new EditMenu(m_mouseHandler);
        menuBar.add(m_editMenu);

        m_viewMenu = new ViewMenu(m_prefManager, m_mouseHandler);
        menuBar.add(m_viewMenu);

        menuBar.add(new ToolsMenu(m_prefManager, m_mouseHandler, m_desktopPane));

        // Window Menu
        JMenu windowMenu = m_desktopPane.getInternalFramesMenu();
        windowMenu.setMnemonic('W');
        menuBar.add(windowMenu);

        menuBar.add(new HelpMenu(m_mouseHandler));

        return menuBar;
    }

    public ViewMenu getViewMenu()
    {
        return m_viewMenu;
    }

    //-------------------------------------------------------------------------

    class JCancelThreadButton extends JButton
    {
        public JCancelThreadButton(Action threadAction)
        {
            super(new StatusStopAction(threadAction));

            setMargin();
        }

        private void setMargin() {
            setMargin(new Insets(1, 1, 1, 1));
        }
        
        @Override
        public String getText()
        {
            return null;
        }
    }

    public static class StatusStopAction extends BasicGuiAction
    {
        public StatusStopAction(Action target)
        {
            super("statusbar.stop", target);
        }
    }

    public JMADesktopPane getDesktopPane()
    {
        return m_desktopPane;
    }

    public JMAStatusBar getStatusBar()
    {
        return m_statusbar;
    }

    public void showStatusBar(boolean state)
    {
        getStatusBar().setVisible(state);
        invalidate();
        validate();
    }

    public void showToolBar(boolean state)
    {
        getToolBar().setVisible(state);
        invalidate();
        validate();
    }

    /**
     * Shows the messages window, creating it if necessary.
     */
    public void showMessageLog(boolean show)
    {
        if (show)
        {
            if (m_messageWindow == null)
            {
                m_messageWindow = new JMessageInternalFrame(m_messageModel);
                m_messageWindow.addInternalFrameListener(new InternalFrameAdapter()
                {
                    @Override
                    public void internalFrameClosing(InternalFrameEvent evt)
                    {
                        m_messageWindow = null;
                        ((BasicAction)m_viewMenu.getMessageLogAction()).setSelected(false);
                    }
                });
            }

            showInternalFrame(m_messageWindow, show);
        }
        else
        {
            if (m_messageWindow != null)
            {
                m_messageWindow.dispose();
                m_messageWindow = null;
            }
        }
    }

    public JMessageInternalFrame getMessageLog()
    {
        return m_messageWindow;
    }

    /**
     * This adapter is constructed to handle mouse over component events.
     */
    class MouseHandler extends MouseAdapter
    {
        /**
         * On mouse enter events, write the actions description to the
         * StatusBar
         */
        @Override
        public void mouseEntered(MouseEvent evt)
        {
            if (evt.getSource() instanceof AbstractButton)
            {
                AbstractButton button = (AbstractButton)evt.getSource();
                Action action = button.getAction();

                if ((action != null) && (!(button instanceof JMenu)))
                {
                    String message = (String)action.getValue(Action.LONG_DESCRIPTION);
                    getStatusBar().showStatusMessage(message);
                }
            }
        }
    }

    public String showInputDialog(String msg, String title)
    {
        return JOptionPane.showInputDialog(this, msg, title, JOptionPane.QUESTION_MESSAGE);
    }

    /**
     * Displays a confirmation with the specified <i>JOptionPane</i> options
     * and get back the option that was selected.
     */
    public int showConfirmDialog(String msg, String title, int msgType, int optionType)
    {
        return showConfirmDialog(this, msg, title, msgType, optionType);
    }

    public int showConfirmDialog(Component parent, String msg, String title, int msgType, int optionType)
    {
        return JMAOptionPane.showConfirmDialog(parent, msg, title, optionType, msgType);
    }

    public static void displayMessage(int type, String message, boolean allowDialog)
    {
        displayMessage(type, message, null, allowDialog);
    }

    // If there is no MgmtConsole open (for instance, this is called from under the JMS client tool,
    // use JOptionPane to display a dialog with the message

    public static void displayMessage(int type, String message, Throwable throwable, boolean allowDialog)
    {
    	displayMessage(null, type, message, throwable, allowDialog);
    }

    public static void displayMessage(Component parent, int type, String message, Throwable throwable, boolean allowDialog)
    {
        if (parent == null)
        {
            getMgmtConsole().notifyMessage(type, message, throwable, allowDialog);
        }
        else
        {
            String outputString = "";
            if ((message != null) && (message.length() > 0))
            {
                outputString = outputString + message + " ";
            }
            if (throwable != null)
            {
                outputString = outputString + throwable.toString();
            }
            JOptionPane.showMessageDialog(parent, outputString);
        }
    }

    @Override
    public void notifyMessage(int type, String message, boolean allowDialog)
    {
        notifyMessage(type, message, null, allowDialog);
    }

    @Override
    public void notifyMessage(int type, String message, Throwable throwable, boolean allowDialog)
    {
        notifyMessage(this, type, message, throwable, allowDialog);
    }

    public void notifyMessage(Component parent,
                              int       type,
                              String    message,
                              Throwable throwable,
                              boolean   allowDialog)
    {
        if (m_messageModel != null)
        {
            int msgType = JOptionPane.PLAIN_MESSAGE;
            DisplayMessage msg = null;
            if (allowDialog)
            {
                /*
                     if, upon analysis, the getNotifyMessage(...) decides that the message type should differ from what
                     the client opted for, it would populate the  DisplayMessage's  messageType with  the appropriate value
                     from amoung MESSAGE_PROGRESS , MESSAGE_STATUS and such. That is given precedence here.
                */
                msg = getNotifyMessage(message, throwable);
                if(msg.messageType!=DisplayMessage.NONE)
                {
                    type = msg.messageType;
                }
            }

            switch (type)
            {
                case MESSAGE_PROGRESS:
                case MESSAGE_STATUS  : msgType = JOptionPane.PLAIN_MESSAGE;       break;
                case MESSAGE_INFO    : msgType = JOptionPane.INFORMATION_MESSAGE; break;
                case MESSAGE_WARNING : msgType = JOptionPane.WARNING_MESSAGE;     break;
                case MESSAGE_ERROR   : msgType = JOptionPane.ERROR_MESSAGE;       break;
            }

            if (type != MESSAGE_PROGRESS)
            {
                m_messageModel.addMessage(type, null, message, throwable);
            }

            if (type == MESSAGE_STATUS || type == MESSAGE_PROGRESS)
            {
                getStatusBar().showStatusMessage(message);
            }

            if (allowDialog)
            {

                showConfirmDialog(parent,
                                  msg.messageString,
                                  "Attention",
                                  msgType,
                                  JOptionPane.DEFAULT_OPTION);

                getStatusBar().showStatusMessage(null);
            }
        }
    }

    /**
     * Here we do some munging of the message text and the actual throwable.
     * If a message text has been supplied then we don't want to display any
     * part of the throwable...the throwable is a last resort when trying to
     * find something to display to the user.
     */
    private DisplayMessage getNotifyMessage(String message, Throwable throwable)
    {
        if (throwable == null)
        {
            return new DisplayMessage(message);
        }

        if (DEVELOPER_MODE)
        {
            Helper.logDebugMessage((message == null ? "(no message)" : message) + ", trace follows...");
            printExceptionInfo(throwable);
        }

        DisplayMessage xmsg = new DisplayMessage(throwable.getMessage());

        if (throwable instanceof ManagementPermissionDeniedException) {
            return new DisplayMessage(ManagementSecurityUtils.getErrorMessageFor(((ManagementPermissionDeniedException) throwable)) + SEE_MSGVIEWER_MESSAGE,MESSAGE_WARNING);
        }
        if (throwable instanceof com.sonicsw.mf.comm.InvokeTimeoutCommsException)
        {
            xmsg = new DisplayMessage("This connection is no longer active. Try reconnecting.");
        }
        else if (throwable instanceof com.sonicsw.mf.comm.InvokeTimeoutException)
        {
            xmsg = new DisplayMessage("A timeout occured before the latest operation completed.");
        }
        else if (throwable instanceof ModelValidationException)
        {
            xmsg = getLinkedExceptionMessage(throwable);
        }
        else if (throwable instanceof ConfigServiceException)
        {
            xmsg = getLinkedExceptionMessage(throwable);
        }
        else if(throwable instanceof ViewValueConversionException)
        {
            xmsg = null;
        }
        else if (xmsg.messageString == null)
        {
            xmsg = getLinkedExceptionMessage(throwable);
        }

        if (xmsg == null || xmsg.messageString==null)
        {
            return new DisplayMessage(message);
        }

        if ((message == null) || message.equals("") || message.equals(xmsg.messageString))
        {
            return xmsg;
        }

        xmsg.prepend(message+'\n');
        xmsg.append(SEE_MSGVIEWER_MESSAGE);
        return xmsg;
    }

    private static DisplayMessage getLinkedExceptionMessage(Throwable e)
    {
        if (e == null)
        {
            return null;
        }

        if(e instanceof IllegalArgumentException){
            Throwable cause = e.getCause();
            if(cause ==null)
            {
                return null;
            }
            return getLinkedExceptionMessage(cause);
        }

        if (e instanceof ModelValidationException && ((ModelValidationException)e).getNestedException() != null)
        {
            return getLinkedExceptionMessage(((ModelValidationException)e).getNestedException());
        }

        if (e instanceof ModelPropagationException && ((ModelPropagationException)e).getNestedException() != null)
        {
            return getLinkedExceptionMessage(((ModelPropagationException)e).getNestedException());
        }

        if(e instanceof ManagementPermissionDeniedException){
            String errorMessage = ManagementSecurityUtils.getErrorMessageFor((ManagementPermissionDeniedException) e);
            DisplayMessage displayMessage = new DisplayMessage(errorMessage,MESSAGE_WARNING); // all permission denied exceptions are warnings
            displayMessage.append(SEE_MSGVIEWER_MESSAGE);
            displayMessage.isComplete = true;
            return displayMessage;
        }

        if (e instanceof ConfigServiceException && ((ConfigServiceException)e).getLinkedException() != null)
        {
            return getLinkedExceptionMessage(((ConfigServiceException)e).getLinkedException());
        }

        if (e.getMessage() != null)
        {
            return new DisplayMessage(e.getMessage());
        }

        if (e instanceof MFException && ((MFException)e).getLinkedException() != null)
        {
            return getLinkedExceptionMessage(((MFException)e).getLinkedException());
        }

        if (e instanceof MFRuntimeException && ((MFRuntimeException)e).getLinkedException() != null)
        {
            return getLinkedExceptionMessage(((MFRuntimeException)e).getLinkedException());
        }

        if (e instanceof ReflectionException)
        {
            return getLinkedExceptionMessage(((ReflectionException)e).getTargetException());
        }

        if (e instanceof MBeanException)
        {
            return getLinkedExceptionMessage(((MBeanException)e).getTargetException());
        }

        if (e instanceof InvocationTargetException)
        {
            return getLinkedExceptionMessage(((InvocationTargetException)e).getTargetException());
        }

        if (e instanceof RuntimeOperationsException)
        {
            return getLinkedExceptionMessage(((RuntimeOperationsException)e).getTargetException());
        }

        if (e instanceof RuntimeMBeanException)
        {
            return getLinkedExceptionMessage(((RuntimeMBeanException)e).getTargetException());
        }

        if (e instanceof RuntimeErrorException)
        {
            return getLinkedExceptionMessage(((RuntimeErrorException)e).getTargetError());
        }

        if (e instanceof ProxyRuntimeException)
        {
            return getLinkedExceptionMessage(((ProxyRuntimeException)e).getTargetException());
        }

        return new DisplayMessage(e.getMessage());
    }

    public static void printExceptionInfo(Throwable e)
    {
        PrintWriter pw = new PrintWriter(System.err);

        printExceptionInfo(e, pw, false);
    }

    public static void printExceptionInfo(Throwable e, PrintWriter pw)
    {
        printExceptionInfo(e, pw, false);
    }

    public static void printExceptionInfo(Throwable e, PrintWriter pw, boolean isCause)
    {
        // don't trace wrappers
        if (e instanceof MFProxyException && ((MFProxyException)e).getActualException() != null)
        {
            e = ((MFProxyException)e).getActualException();
        }
        else
        if (e instanceof MFProxyRuntimeException && ((MFProxyRuntimeException)e).getActualException() != null)
        {
            e = ((MFProxyRuntimeException)e).getActualException();
        }
        else
        if (e instanceof ProxyRuntimeException && ((ProxyRuntimeException)e).getTargetException() != null)
        {
            e = ((ProxyRuntimeException)e).getTargetException();
        }
        
        // Output the stacktrace information ourselves to avoid having the
        // java.lang.Throwable output the JDK 1.4 cause information too.
        // We are going to handle the cause ourselves.
        synchronized (pw)
        {
            if (isCause)
            {
                pw.print("Caused by: ");
            }

            pw.println(e);
            StackTraceElement[] trace = e.getStackTrace();
            for (int i = 0; i < trace.length; i++)
            {
                pw.println("\tat " + trace[i]);
            }
            pw.flush();
        }

        if (e instanceof ModelValidationException && ((ModelValidationException)e).getNestedException() != null)
        {
            printExceptionInfo(((ModelValidationException)e).getNestedException(), pw, true);
        }
        else
        if (e instanceof ModelPropagationException && ((ModelPropagationException)e).getNestedException() != null)
        {
            printExceptionInfo(((ModelPropagationException)e).getNestedException(), pw, true);
        }
        else
        if (e instanceof ConfigServiceException && ((ConfigServiceException)e).getLinkedException() != null)
        {
            printExceptionInfo(((ConfigServiceException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFException && ((MFException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof MFRuntimeException && ((MFRuntimeException)e).getLinkedException() != null)
        {
            printExceptionInfo(((MFRuntimeException)e).getLinkedException(), pw, true);
        }
        else
        if (e instanceof ReflectionException)
        {
            printExceptionInfo(((ReflectionException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof MBeanException)
        {
            printExceptionInfo(((MBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof InvocationTargetException)
        {
            printExceptionInfo(((InvocationTargetException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeOperationsException)
        {
            printExceptionInfo(((RuntimeOperationsException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeMBeanException)
        {
            printExceptionInfo(((RuntimeMBeanException)e).getTargetException(), pw, true);
        }
        else
        if (e instanceof RuntimeErrorException)
        {
            printExceptionInfo(((RuntimeErrorException)e).getTargetError(), pw, true);
        }
        else
        if (e.getCause() != null)
        {
            printExceptionInfo(e.getCause(), pw, true);
        }
    }

    /**
     * Registers an <code>ApplicationListener</code>.
     */
    @Override
    public void addApplicationListener(ApplicationListener listener)
    {
        m_appListenerSupport.addApplicationListener(listener);
    }

    /**
     *  Unregisters an <code>ApplicationListener</code>.
     */
    @Override
    public void removeApplicationListener(ApplicationListener listener)
    {
        m_appListenerSupport.removeApplicationListener(listener);
    }

    /**
     * Shuts down the application after notifying <code>ApplicationListener</code>
     * about the impending shutdown.
     */
    @Override
    public void shutDown() throws ShutDownCancelledException
    {
        m_appListenerSupport.fireApplicationShuttingDown(this);
        setVisible(false);
        dispose();
    }

    /**
     * Notifies <code>ApplicationListener</code> that the application is
     * low on memory so that they can free up resources, clear caches, etc.,
     * to allow * the application to continue execution.
     */
    @Override
    public void lowOnMemory()
    {
        m_appListenerSupport.fireApplicationLowOnMemory(this);
    }

    //-------------------------------------------------------------------------

    private static class JMAOptionPane extends JOptionPane
    {
        public JMAOptionPane(String msg, int msgType, int optionType)
        {
            super(msg, msgType, optionType);
        }

        /**
         * Overridden method to perform additional size (layout) control.
         *
         * The code in this method is very similar to the JOptionPane method
         * but with the addition of some minimum size restrictions.
         */
        public static int showConfirmDialog(Component parentComponent,
                                            Object    message,
                                            String    title,
                                            int       optionType,
                                            int       messageType)
            throws HeadlessException
        {
            JMAOptionPane pane   = new JMAOptionPane(message.toString(), messageType, optionType);
            JDialog       dialog = pane.createDialog(parentComponent, title);

            pane.selectInitialValue();

            Dimension sz = dialog.getPreferredSize();

            if ((sz.width < 50) || (sz.height < 50))
            {
                dialog.setSize(400, 100);
            }

            dialog.validate();
            sz = dialog.getPreferredSize();
            dialog.setSize(sz.width + 8, sz.height + 8);

            dialog.show();

            Object selectedValue = pane.getValue();

            dialog.dispose();

            if ((selectedValue != null) && (selectedValue instanceof Integer))
            {
                return ((Integer)selectedValue).intValue();
            }

            return JOptionPane.CLOSED_OPTION;
        }

        @Override
        public int getMaxCharactersPerLineCount()
        {
            return 60;
        }
    }

    //-------------------------------------------------------------------------
    //
    // Clipboard Handling
    //
    //-------------------------------------------------------------------------

    public static final DataFlavor fileFlavor   = new DataFlavor(IFilePlugin.class,   "Sonic File Object");
    public static final DataFlavor configFlavor = new DataFlavor(IConfigPlugin.class, "Sonic Configuration Object");

    public int getClipboardAction()
    {
        return m_clipAction;
    }

    public Object getClipboardObject()
    {
        return m_clipObject;
    }

    public void setClipboard(int action, Object object)
    {
        m_clipAction = action;
        m_clipObject = object;
    }

    public void clearClipboard()
    {
        setClipboard(getClipboardAction(), null);
    }

    public boolean isClipboardDataFlavorSupported(DataFlavor flavor)
    {
        boolean res    = false;
        Object  object = getClipboardObject();

        if (object != null)
        {
            if (flavor.equals(fileFlavor) && (object instanceof IFilePlugin))
            {
                res = true;
            }

            if (flavor.equals(configFlavor) && (object instanceof IConfigPlugin))
            {
                res = true;
            }
        }

        return res;
    }

    //-------------------------------------------------------------------------

    @Override
    public void setVisible(boolean visible)
    {
        super.setVisible(visible);

        // Display Connect dialog here after making the Management Console frame
        // visible so that it gets the window focus.  Otherwise it won't always
        // end up with it.
        if (visible)
        {
            connectToDefaultDomain();
        }
    }

    public static void main(String[] args)
    {
        try
        {
            String strLaf = getPreferredLookAndFeel();

            if (strLaf != null)
            {
                UIManager.setLookAndFeel(strLaf);
            }
            String lafName = UIManager.getLookAndFeel().getClass().getName();
            if (lafName.equals("com.sun.java.swing.plaf.windows.WindowsLookAndFeel")) {
                UIDefaults uiDefaults = UIManager.getLookAndFeelDefaults();
                Color c = uiDefaults.getColor("TextField.background");
                if (!(c instanceof javax.swing.plaf.ColorUIResource)) {
                    // Bug not fixed
                    for (final Object key : uiDefaults.keySet()) {
                        Object o = uiDefaults.get(key);
                        if (o instanceof Color &&
                                !(o instanceof javax.swing.plaf.UIResource) &&
                                o == UIManager.get(key)) {

                            UIManager.put(key,
                                    new UIDefaults.ActiveValue() {
                                        @Override
                                        public Object createValue(UIDefaults table) {
                                            Color c = UIManager.getLookAndFeelDefaults().getColor(key);
                                            if (c != null &&
                                                    !(c instanceof javax.swing.plaf.UIResource)) {

                                                return new javax.swing.plaf.ColorUIResource(c);
                                            } else {
                                                return c;
                                            }
                                        }
                                    });
                        }
                    }
                }
            }


            // fixes disparity between label and text field fonts
            Object fixFont = UIManager.get("Label.font");
            UIManager.put("TextField.font", fixFont);
            UIManager.put("TextArea.font",  fixFont);

            // register extended UI's for JTextField and JTextArea to provide
            // popup menu's with cut, copy and paste support.
            UIManager.put("TextFieldUI", "com.sonicsw.ma.gui.plaf.EditTextFieldUI");
            UIManager.put("TextAreaUI",  "com.sonicsw.ma.gui.plaf.EditTextAreaUI");

            // Tooltips provide important information so we want them to stay
            // around alittle longer than the default 5 seconds.
            ToolTipManager.sharedInstance().setDismissDelay(20000);
        }
        catch (Exception e)
        {
            displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
        }

        MgmtConsole frame = MgmtConsole.getMgmtConsole();

        if (frame != null)
        {
            frame.setVisible(true);
        }
    }

    public class DebugMessageHandler extends ConsoleHandler
    {
        public DebugMessageHandler()
        {
            setFormatter();
        }

        private void setFormatter() {
            setFormatter(new DebugMessageFormatter());
        }
        
        public class DebugMessageFormatter extends SimpleFormatter
        {
            private Date date = new Date();
            private DateFormat df = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT);
            private String lineSeparator = (String) java.security.AccessController.doPrivileged(
                       new sun.security.action.GetPropertyAction("line.separator"));

            @Override
            public synchronized String format(LogRecord record)
            {
                StringBuffer sb = new StringBuffer();

                // Minimize memory allocations here.
                date.setTime(record.getMillis());
                sb.append(df.format(date));
                sb.append(" ");

                sb.append(record.getLevel().getLocalizedName());
                sb.append(": ");
                sb.append(formatMessage(record));
                sb.append(lineSeparator);

                return sb.toString();
            }
        }
    }

    public class SMCMessageHandler extends Handler
    {
        public SMCMessageHandler()
        {
            setLevel();
        }
        
        private void setLevel() {
            setLevel(Level.INFO);
        }
        
        @Override
        public void publish(LogRecord logRecord)
        {
            if (isLoggable(logRecord))
            {
                MgmtConsole.this.notifyMessage(getMessageLevel(logRecord.getLevel()),
                                               logRecord.getMessage(),
                                               logRecord.getThrown(),
                                               false);
            }
        }

        @Override
        public void flush() {}

        @Override
        public void close() {}

        protected int getMessageLevel(Level level)
        {
            int nMessageLevel = MESSAGE_STATUS;

            if (level == Level.SEVERE)
            {
                nMessageLevel = MESSAGE_ERROR;
            }
            else if (level == Level.WARNING)
            {
                nMessageLevel = MESSAGE_WARNING;
            }
            else if (level == Level.INFO)
            {
                nMessageLevel = MESSAGE_INFO;
            }

            return nMessageLevel;
        }
    }

    private static String getPreferredLookAndFeel()
    {
        String strResult = null;
        final String strLaf = System.getProperty("Laf");

        if (strLaf != null)
        {
            if (strLaf.equals("windows"))
            {
                strResult = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";
            }
            else if (strLaf.equals("motif"))
            {
                strResult = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";
            }
        }
        else
        {
            strResult = UIManager.getSystemLookAndFeelClassName();
        }
        
        return strResult;
    }

	public IPluginContext getPluginContext() {
		return m_context;
	}

    private static class DisplayMessage {
        static int NONE = -1;
        String messageString;
        /**
         * whether the message type provided by the client should be overriden
         */
        int messageType;
        /**
         * set by the maker to avert further tampering.
         */
        private boolean isComplete;

        public DisplayMessage(String messageString) {
            this(messageString, NONE);
        }

        public DisplayMessage(String messageString, int messageTypePreference) {
            this.messageString = messageString;
            this.messageType = messageTypePreference;
        }

        public void prepend(String moreInfo) {
            if (!isComplete)
            {
                messageString = moreInfo + messageString;
            }
        }

        public void append(String moreInfo) {
            if (!isComplete)
            {
                messageString = messageString + moreInfo;
            }
        }

        @Override
        public String toString() {
            return messageString;
        }

    }
}