/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui;

import java.util.HashMap;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

/**
 *
 * <p>Title: PreferenceManager</p>
 * <p>Description: Singleton class to manage the User Preferences feature.</p>
 * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Charlie Cocchiaro
 * @version 1.0
 */
public enum PreferenceManager
{
    INSTANCE;
    private HashMap m_nodeMap = new HashMap();
    private Preferences m_root;

    /**
     * Static method to create a single instance of this class.
     *
     * @returns An instance of this class
     */
    public static PreferenceManager getInstance()
    {
        return INSTANCE;
    }

    /**
     * Default constructor called by the public static method to create a single
     * instance of this class.
     */
    private PreferenceManager()
    {
        m_root = Preferences.userNodeForPackage(this.getClass());
    }

    public void resetUserPreferences()
    {
        try
        {
            m_root.removeNode();
            m_root = Preferences.userNodeForPackage(this.getClass());
            m_nodeMap = new HashMap();
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                        "Failed to reset User Preferences", e, true);
        }
        return;
    }

    public String getString(String path, String name, String defVal)
    {
        Preferences node = findNode(path);

        return (node != null) ? node.get(createPreferredKey(name), defVal) : null;
    }

    public boolean getBoolean(String path, String name, boolean defVal)
    {
        Preferences node = findNode(path);

        return (node != null) ? node.getBoolean(createPreferredKey(name), defVal) : defVal;
    }

    public int getInt(String path, String name, int defVal)
    {
        Preferences node = findNode(path);

        return (node != null) ? node.getInt(createPreferredKey(name), defVal) : defVal;
    }

    public void setBoolean(String path, String name, boolean value, boolean flush)
    {
        Preferences node = findNode(path);

        if (node != null)
        {
            node.putBoolean(createPreferredKey(name), value);

            if (flush)
            {
                flush(node);
            }
        }
    }

    public void setInt(String path, String name, int value, boolean flush)
    {
        Preferences node = findNode(path);

        if (node != null)
        {
            node.putInt(createPreferredKey(name), value);

            if (flush)
            {
                flush(node);
            }
        }
    }

    public void setString(String path, String name, String value, boolean flush)
    {
        Preferences node = findNode(path);

        if (node != null)
        {
            node.put(createPreferredKey(name), value);

            if (flush)
            {
                flush(node);
            }
        }
    }

    public void remove(String path, String name, boolean flush)
    {
        Preferences node = findNode(path);

        if ((node != null) && (node.get(createPreferredKey(name), null) != null))
        {
            node.remove(name);

            if (flush)
            {
                flush(node);
            }
        }
    }

    public void flush()
    {
        flush(m_root);
    }

    public void flush(String path)
    {
        flush(findNode(path));
    }

    public String[] getChildrenNames(String path)
    {
        try
        {
            Preferences node = findNode(path);

            return node.childrenNames();
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                        "Failed to get children names: " + path, e, true);
        }
        return null;
    }

    public void removeNode(String path)
    {
        try
        {
            Preferences node = findNode(path);

            node.removeNode();
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                        "Failed to remove node: " + path, e, true);
        }
    }

    private void flush(Preferences node)
    {
        try
        {
            node.flush();
        }
        catch (BackingStoreException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                        "Failed to flush preferences", e, true);
        }
    }

    private String createPreferredKey(String strKeyName)
    {
        // ensure key is no longer than 80 characters, as per Preferences......
        if (strKeyName != null && strKeyName.length() > Preferences.MAX_KEY_LENGTH)
        {
            strKeyName = strKeyName.substring(strKeyName.length() - Preferences.MAX_KEY_LENGTH, strKeyName.length());
        }

        return strKeyName;
    }

    private synchronized Preferences findNode(String path)
    {
        String pathPreferred = createPreferredKey(path);
        Preferences node = (Preferences)m_nodeMap.get(pathPreferred);

        if(node == null)
        {
            // replace all occurences of . with / in the path

            node = m_root.node(pathPreferred.replace('.', '/'));

            m_nodeMap.put(pathPreferred, node);
        }
        return node;
    }
}