/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui;

import java.awt.event.ActionEvent;
import java.beans.PropertyChangeListener;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.tree.DefaultMutableTreeNode;

import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.BasicResourceAction;
import com.sonicsw.ma.gui.util.ButtonActionPropertyChangeListener;
import com.sonicsw.ma.gui.util.ExtendedImageIcon;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;

/**
 * <p>Title: ToolBar</p>
 * <p>Description: Class to create and manage the workspace toolbar.</p>
 * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Charlie Cocchiaro
 * @version 1.0
 */
public class ToolBar extends JToolBar
{
    protected BasicResourceAction fileConnectAction = new FileConnectAction();
    protected BasicResourceAction fileNewFolderAction = new FileNewFolderAction();
    protected BasicResourceAction fileNewAction = new FileNewAction();
    protected BasicResourceAction editCutAction = new EditCutAction();
    protected BasicResourceAction editCopyAction = new EditCopyAction();
    protected BasicResourceAction editPasteAction = new EditPasteAction();
    protected BasicResourceAction editDeleteAction = new EditDeleteAction();
    protected BasicResourceAction editPropertiesAction = new EditPropertiesAction();
    protected BasicResourceAction viewRefreshAction = new ViewRefreshAction();
    protected BasicResourceAction viewGotoInstanceAction = new ViewGotoInstanceAction();
    protected BasicResourceAction viewGotoConfigurationAction = new ViewGotoConfigurationAction();
    protected BasicResourceAction runtimeMetricsAction = new RuntimeMetricsAction();
    protected BasicResourceAction runtimeNotificationsAction = new RuntimeNotificationsAction();
    protected BasicResourceAction viewLogAction = new ViewLogAction();

    private Object m_lastObject;

    public ToolBar()
    {
        super();
        prepareToolBar();
    }
    
    private void prepareToolBar() {
        setFloatable(false);
        setRollover(true);
        addDefaultItems();
    }

    @Override
    public JButton add(Action a)
    {
        JToolBarButton b = new JToolBarButton(a);
        add(b);
        return b;
    }

    /**
     * Method to add the default tollbar buttons to the toolbar.
     */
    public void addDefaultItems()
    {
        if(getComponentCount() > 0)
        {
            removeAll();
        }

        add(fileConnectAction);
        addSeparator();
        add(fileNewFolderAction);
        add(fileNewAction);
        addSeparator();

        add(editCutAction);
        add(editCopyAction);
        add(editPasteAction);
        add(editDeleteAction);
        addSeparator();

        add(viewRefreshAction);
        super.add(new JToolBarToggleButton(MgmtConsole.getMgmtConsole().getViewMenu().getMessageLogAction()));
        add(viewGotoInstanceAction);
        add(viewGotoConfigurationAction);
        addSeparator();

        add(editPropertiesAction);
        addSeparator();

        add(runtimeMetricsAction);
        add(runtimeNotificationsAction);

        addSeparator();
        add(viewLogAction);

        clearActions();
    }

    /**
     * Method to clear the toolbar buttons actions.
     */
    public void clearActions()
    {
        m_lastObject = null;

        clearAction(fileNewFolderAction);
        clearAction(fileNewAction);
        clearAction(editCutAction);
        clearAction(editCopyAction);
        clearAction(editPasteAction);
        clearAction(editDeleteAction);
        clearAction(viewRefreshAction);
        clearAction(viewGotoInstanceAction);
        clearAction(viewGotoConfigurationAction);
        clearAction(editPropertiesAction);
        clearAction(runtimeMetricsAction);
        clearAction(runtimeNotificationsAction);
        clearAction(viewLogAction);
    }

    private void clearAction(BasicResourceAction action)
    {
        action.setTargetAction(null);
        action.setEnabled(false);
    }

    /**
     * Method to set the toolbar buttons actions.
     *
     * @param selectedNode The selected tree node plugin
     */
    public void setActions(DefaultMutableTreeNode selectedNode)
    {
        if (m_lastObject == selectedNode)
        {
            return;
        }

        clearActions();

        m_lastObject = selectedNode;

        if (selectedNode != null && selectedNode instanceof IPlugin)
        {
            IPlugin node = (IPlugin)selectedNode;

            setActions(node, IPlugin.NEW_TYPE);
            setActions(node, IPlugin.EDIT_TYPE);
            setActions(node, IPlugin.SYSTEM_TYPE);
            setActions(node, IPlugin.VIEW_TYPE);
            setActions(node, IPlugin.PROPERTIES_TYPE);

            // Don't call for PLUGIN_TYPE there should be nothing on the
            // toolbar that is tied to menu items in this group
            // setActions(node, IPlugin.PLUGIN_TYPE);
        }
    }

    /**
     * Method to set the toolbar buttons actions.
     *
     * @param contentPane The content pane for the selected table row
     */
    public void setActions(IContentPane cp)
    {
        clearActions();

        m_lastObject = cp;

        if (cp != null)
        {
            setActions(cp, IPlugin.NEW_TYPE);
            setActions(cp, IPlugin.EDIT_TYPE);
            setActions(cp, IPlugin.SYSTEM_TYPE);
            setActions(cp, IPlugin.VIEW_TYPE);
            setActions(cp, IPlugin.PROPERTIES_TYPE);

            // Don't call for PLUGIN_TYPE there should be nothing on the
            // toolbar that is tied to menu items in this group
            // setActions(cp, IPlugin.PLUGIN_TYPE);
        }
    }

    public void setActions(IPlugin node, int type)
    {
        try
        {
            setActions(node.getMenuItems(type), type);
        }
        catch (Exception e)
        {
            // bug 10377
            // This is one of the places that will catch a NullPointerException
            // when the mgmt broker is reloaded
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_WARNING,
                                       "Failed to generate the menu items for node '" + node.getClass().getName() + "'",
                                       e, false);
        }
    }

    private void setActions(IContentPane cp, int type)
    {
        setActions(cp.getMenuItems(type), type);
    }

    /**
     * Method to set the toolbar buttons actions.
     *
     * @param selectedNode The selected tree node plugin
     */
    private void setActions(JComponent[] menuItems,int type)
    {
        if (menuItems != null && menuItems.length > 0)
        {
            switch (type)
            {
                case IPlugin.NEW_TYPE:
                    // Set the Toolbar "New" button action.
                    if (menuItems[0] instanceof JMenu)
                    {
                        JMenu menu = (JMenu)menuItems[0];

                        for (int i = 0; i < menu.getItemCount(); i++)
                        {
                            JMenuItem menuItem = menu.getItem(i);

                            if (menuItem != null) // JSeparator items are returned as null
                            {
                                Action action = menuItem.getAction();

                                if (action != null)
                                {
                                    String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);

                                    if (actionValue != null)
                                    {
                                        if (actionValue.equals("file.new.folder"))
                                        {
                                            fileNewFolderAction.setTargetAction(action);
                                        }
                                        else if (actionValue.equals("file.new.configuration"))
                                        {
                                            fileNewAction.setTargetAction(action);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    else if (menuItems[0] instanceof JMenuItem)
                    {
                        fileNewAction.setTargetAction(((JMenuItem)menuItems[0]).getAction());
                    }
                    break;

                case IPlugin.EDIT_TYPE:
                    // Set the Toolbar "Cut", "Copy", and "Paste" button actions.
                    for (int i = 0; i < menuItems.length; i++)
                    {
                        if (menuItems[i] instanceof JMenuItem)
                        {
                            JMenuItem menuItem = (JMenuItem)menuItems[i];
                            Action action = menuItem.getAction();
                            if (action != null)
                            {
                                String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);
                                if (actionValue != null)
                                {
                                    if (actionValue.equals("edit.cut"))
                                    {
                                        editCutAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("edit.copy"))
                                    {
                                        editCopyAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("edit.paste"))
                                    {
                                        editPasteAction.setTargetAction(action);
                                    }
                                }
                            }
                        }
                    }
                    break;

                case IPlugin.SYSTEM_TYPE:
                    // Set the Toolbar "Delete" button action.
                    for (int i = 0; i < menuItems.length; i++)
                    {
                    	if(menuItems[i] instanceof JMenu)
                        {
                            JMenu menu = (JMenu)menuItems[i];

                            for (int j = 0; j < menu.getItemCount(); j++)
                            {
                                JMenuItem menuItem = menu.getItem(j);

                                if (menuItem != null) // JSeparator items are returned as null
                                {
                                    Action action = menuItem.getAction();

                                    if (action != null)
                                    {
                                        String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);

                                        if (actionValue != null)
                                        {
                                            if (actionValue.equals("runtime.viewlog"))
                                            {
                                            	viewLogAction.setTargetAction(action);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if (menuItems[i] instanceof JMenuItem)
                        {
                            JMenuItem menuItem = (JMenuItem)menuItems[i];
                            Action action = menuItem.getAction();
                            if (action != null)
                            {
                                String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);
                                if (actionValue != null)
                                {
                                    if (actionValue.equals("edit.delete") || actionValue.equals("folder.delete"))
                                    {
                                        editDeleteAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("runtime.viewlog"))
                                    {
                                        viewLogAction.setTargetAction(action);
                                    }
                                }
                            }
                        } 
                    }
                    break;

                case IPlugin.VIEW_TYPE:
                    // Set the Toolbar "Refresh" button action.
                    for (int i = 0; i < menuItems.length; i++)
                    {
                        if (menuItems[i] instanceof JMenuItem)
                        {
                            JMenuItem menuItem = (JMenuItem)menuItems[i];
                            Action action = menuItem.getAction();
                            if (action != null)
                            {
                                String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);
                                if (actionValue != null)
                                {
                                    if (actionValue.equals("view.refresh"))
                                    {
                                        viewRefreshAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("view.gotoinstance"))
                                    {
                                        viewGotoInstanceAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("view.gotoconfiguration"))
                                    {
                                        viewGotoConfigurationAction.setTargetAction(action);
                                    }
                                }
                            }
                        }
                    }
                    break;

                case IPlugin.PROPERTIES_TYPE:
                    // Set the Toolbar "Properties" button action.
                    for (int i = 0; i < menuItems.length; i++)
                    {
                        if (menuItems[i] instanceof JMenuItem)
                        {
                            JMenuItem menuItem = (JMenuItem)menuItems[i];
                            Action action = menuItem.getAction();
                            if (action != null)
                            {
                                String actionValue = (String)action.getValue(Action.ACTION_COMMAND_KEY);
                                if (actionValue != null)
                                {
                                    if (actionValue.equals("edit.properties"))
                                    {
                                        editPropertiesAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("ComponentPlugin.notifications"))
                                    {
                                        runtimeNotificationsAction.setTargetAction(action);
                                    }
                                    else if (actionValue.equals("ComponentPlugin.metrics"))
                                    {
                                        runtimeMetricsAction.setTargetAction(action);
                                    }
                                }
                            }
                        }
                    }
                    break;
            }
        }
    }

    /**
     * <p>Title: FileConnectAction</p>
     * <p>Description: Class to manage the File Connect action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class FileConnectAction extends BasicGuiAction
    {
        public FileConnectAction()
        {
            super("file.connect.new");
            putSmallIconValue();
        }

        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon((Icon)getValue(Action.SMALL_ICON), ResourceManager.getIcon(getClass(), "newoverlay")));
        }
        
        @Override
        public void actionPerformed(ActionEvent event)
        {
            MgmtConsole.getMgmtConsole().connectToDefaultDomain();
        }
    }

    /**
     * <p>Title: FileNewFolderAction</p>
     * <p>Description: Class to manage the File New Folder action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class FileNewFolderAction extends BasicGuiAction
    {
        public FileNewFolderAction()
        {
            super("file.new.folder");
            putSmallIconValue();
        }
        
        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon((Icon)getValue(Action.SMALL_ICON), ResourceManager.getIcon(getClass(), "newoverlay")));
        }
    }


    /**
     * <p>Title: FileNewAction</p>
     * <p>Description: Class to manage the File New action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class FileNewAction extends BasicGuiAction
    {
        public FileNewAction()
        {
            super("file.new.configuration");

            putSmallIconValue();
        }
        
        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon((Icon)getValue(Action.SMALL_ICON), ResourceManager.getIcon(getClass(), "newoverlay")));
        }
    }

    /**
     * <p>Title: EditCutAction</p>
     * <p>Description: Class to manage the Edit Cut action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class EditCutAction extends BasicGuiAction
    {
        public EditCutAction()
        {
            super("edit.cut");
        }
    }

    /**
     * <p>Title: EditCopyAction</p>
     * <p>Description: Class to manage the Edit Copy action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class EditCopyAction extends BasicGuiAction
    {
        public EditCopyAction()
        {
            super("edit.copy");
        }
    }

    /**
     * <p>Title: EditPasteAction</p>
     * <p>Description: Class to manage the Edit Paste action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class EditPasteAction extends BasicGuiAction
    {
        public EditPasteAction()
        {
            super("edit.paste");
        }
    }

    /**
     * <p>Title: EditDeleteAction</p>
     * <p>Description: Class to manage the Edit Delete action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class EditDeleteAction extends BasicGuiAction
    {
        public EditDeleteAction()
        {
            super("edit.delete");
        }
    }

    /**
     * <p>Title: ViewRefreshAction</p>
     * <p>Description: Class to manage the View Refresh action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class ViewRefreshAction extends BasicGuiAction
    {
        public ViewRefreshAction()
        {
            super("view.refresh");
        }
    }

    class ViewGotoInstanceAction extends BasicGuiAction
    {
        public ViewGotoInstanceAction()
        {
            super("view.gotoinstance");
        }
    }

    class ViewGotoConfigurationAction extends BasicGuiAction
    {
        public ViewGotoConfigurationAction()
        {
            super("view.gotoconfiguration");
        }
    }

    /**
     * <p>Title: EditPropertiesAction</p>
     * <p>Description: Class to manage the Edit Properties action.</p>
     * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Charlie Cocchiaro
     * @version 1.0
     */
    class EditPropertiesAction extends BasicGuiAction
    {
        public EditPropertiesAction()
        {
            super("edit.properties");
        }
    }

    class RuntimeMetricsAction extends BasicGuiAction
    {
        public RuntimeMetricsAction()
        {
            super("runtime.metrics");
        }
    }

    class RuntimeNotificationsAction extends BasicGuiAction
    {
        public RuntimeNotificationsAction()
        {
            super("runtime.notifications");
        }
    }

    /**
     * <p>Title: ViewLogAction</p>
     * <p>Description: Class to manage the View Log action.</p>
     * <p>Copyright: Copyright (c) 2004 Sonic Software Corporation. All Rights Reserved.</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Chandana Konda
     * @version 1.0
     */
 	class ViewLogAction extends BasicGuiAction
    {
        public ViewLogAction()
        {
            super("runtime.viewlog");
        }
    }

    /**
     * This button type is used to re-configure what action properties are
     * loaded (when compared to the standard button).
     */
    class JToolBarButton extends JButton
    {
        public JToolBarButton(Action action)
        {
            super(action);
        }

        @Override
        protected void configurePropertiesFromAction(Action a)
        {
            String[] types = { Action.SMALL_ICON,
                               Action.SHORT_DESCRIPTION,
                               Action.LONG_DESCRIPTION,
                               "enabled" };

            Helper.configurePropertiesFromAction(this, a, types);
        }
    }

    class JToolBarToggleButton extends JToggleButton
    {
        public JToolBarToggleButton(Action action)
        {
            super(action);
        }

        @Override
        protected void configurePropertiesFromAction(Action a)
        {
            String[] types = { Action.SMALL_ICON,
                               Action.SHORT_DESCRIPTION,
                               Action.LONG_DESCRIPTION,
                               "enabled" };

            Helper.configurePropertiesFromAction(this, a, types);
        }

        @Override
        protected void fireActionPerformed(ActionEvent evt)
        {
            // Because we are dealing with a CheckBox-style component, an action is
            // generated per selection state change.
            // We override here to pass the selection state on to the attached
            // action.
            ((BasicAction)getAction()).setSelected(isSelected());

            super.fireActionPerformed(evt);
        }

        @Override
        protected PropertyChangeListener createActionPropertyChangeListener(Action a)
        {
            return new ButtonActionPropertyChangeListener(this, a);
        }

    }
}
