/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui;

import java.awt.event.ActionEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;

import javax.management.ObjectName;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSeparator;
import javax.swing.event.MenuEvent;

import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JMADesktopPane;
import com.sonicsw.ma.gui.util.JProgressDialog;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.SwingWorker;
import com.sonicsw.ma.plugin.ClasspathToolInfo;
import com.sonicsw.ma.plugin.ToolInfo;
import com.sonicsw.mx.config.ConfigServerUtility;

import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.Version;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;

/**
 * Context sensitive "Tools" menu
 */
final class ToolsMenu extends JBasicMenu
{
    private static final ToolInfo[] HARDWIRED_TOOLS =
    {
        new ClasspathToolInfo("&JMS Administered Objects",
                              "JMS Administered Objects Tool",
                              "com.sonicsw.mq.gui.adminobjs.ExplorerClient"),
    };

    private PreferenceManager m_preferences;
    private JMADesktopPane    m_desktop;
    private BasicGuiAction    m_prefAction;
    private BasicGuiAction    m_exportLocalAction;
    private BasicGuiAction    m_exportRemoteAction;
    private BasicGuiAction    m_repairAction;
    private BasicGuiAction    m_typeManagerAction;
    private BasicGuiAction    m_launchSMCAction;

    ToolsMenu(PreferenceManager preferences,
              MouseListener     mouseListener,
              JMADesktopPane    desktop)
    {
        super(new ToolsAction());

        m_preferences  = preferences;
        m_desktop      = desktop;
        m_prefAction   = new ToolPrefsAction();
        m_exportLocalAction = new ToolExportDSAction(true);
        m_exportRemoteAction = new ToolExportDSAction(false);
        m_repairAction = new ToolRepairDSAction();
        m_typeManagerAction = new ToolTypeManagerAction();
        m_launchSMCAction = new LaunchSMCAction();
        setMouseListener(mouseListener);

        buildMenu(null);
    }

    public Action getPrefAction()
    {
        return (Action)m_prefAction;
    }

    @Override
    public void menuSelected(MenuEvent evt)
    {
        ArrayList        list   = new ArrayList();
        JInternalFrame[] frames = m_desktop.getAllFrames();

        for (int i = 0; i < frames.length; i++)
        {
            if (frames[i] instanceof IDomainWorkspaceWindow)
            {
                ArrayList              frameList  = new ArrayList();
                IDomainWorkspaceWindow frame      = (IDomainWorkspaceWindow)frames[i];
                JComponent[]           frameItems = frame.getContext().getLibrary().getToolsMenuItems();

                for (int j = 0; j < frameItems.length; j++)
                {
                    if (!alreadyInList(list, frameItems[j]))
                    {
                        frameList.add(frameItems[j]);
                    }
                }

                if (!frameList.isEmpty())
                {
                    if (!list.isEmpty())
                    {
                        list.add(new JSeparator());
                    }

                    for (int k = 0; k < frameList.size(); k++)
                    {
                        list.add(frameList.get(k));
                    }
                }
            }
        }

        buildMenu((JComponent[])list.toArray(new JComponent[list.size()]));

        super.menuSelected(evt);
    }

    private boolean alreadyInList(ArrayList list, JComponent c)
    {
        Iterator   i   = list.iterator();

        while (i.hasNext())
        {
            JComponent item = (JComponent)i.next();

            if ((c instanceof JMenuItem) && (item instanceof JMenuItem))
            {
                JMenuItem miC    = (JMenuItem)c;
                JMenuItem miItem = (JMenuItem)item;

                if (miC != null && miC.getActionCommand().equals(miItem.getActionCommand()))
                {
                    return true;
                }
            }
        }
        return false;
    }

    private void buildMenu(JComponent[] items)
    {
        removeAll();

        // Add in hardwired tools
        //
        for (int iTool = 0; iTool < HARDWIRED_TOOLS.length; iTool++)
        {
            add(HARDWIRED_TOOLS[iTool].getMenuItem());
        }

        add(m_launchSMCAction);
        JBasicMenu exportMenu = new JBasicMenu(new ToolExportAction());
        exportMenu.add(m_exportRemoteAction);
        exportMenu.add(m_exportLocalAction);
        add(exportMenu);

        if (MgmtConsole.DEVELOPER_MODE)
        {
            add(m_repairAction);
            add(m_typeManagerAction);
        }

        if (getItemCount() > 0)
        {
            // Build up the dynamic (domain) portion of the menu
            //
            if ((items != null) && (items.length > 0))
            {
                addSeparator();

                for (int i = 0; i < items.length; i++)
                {
                    add(items[i]);
                }
            }

            addSeparator();
        }

        add(m_prefAction);
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    static class ToolsAction extends BasicGuiAction
    {
        public ToolsAction()
        {
            super("tool");
        }
    }

    class ToolPrefsAction extends BasicGuiAction
    {
        public ToolPrefsAction()
        {
            super("tool.prefs");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JPreferencesDialog dialog = new JPreferencesDialog(MgmtConsole.getMgmtConsole(), ToolsMenu.this.m_preferences);
            dialog.setVisible(true);
        }
    }

    abstract class ToolAction extends BasicGuiAction
    {
        public ToolAction(String resourceId)
        {
            super(resourceId);
        }

        public ToolAction(String resourceId, Action targetAction)
        {
            super(resourceId, targetAction);
        }

        protected IDomainWorkspaceWindow getCurrentWorkspaceWindow()
        {
            IDomainWorkspaceWindow res = null;

            try
            {
                JInternalFrame frame = m_desktop.getSelectedFrame();

                if (frame instanceof IDomainWorkspaceWindow)
                {
                    res = (IDomainWorkspaceWindow)frame;
                }
            }
            catch (Exception e)
            {
                res = null;
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
            }

            return res;
        }
    }

    class ToolExportAction extends ToolAction
    {
        public ToolExportAction()
        {
            super("tool.exportDS");
        }

        @Override
        public boolean isEnabled()
        {
            return (getCurrentWorkspaceWindow() != null);
        }
    }

    class ToolExportDSAction extends ToolAction
    {
        private String m_exportPath = null;
        private boolean m_local;

        public ToolExportDSAction(boolean local)
        {
            super(local ? "tool.exportDS.local" : "tool.exportDS.remote");

            m_local = local;
        }

        @Override
        public boolean isEnabled()
        {
            return (getCurrentWorkspaceWindow() != null);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            if (m_local)
            {
                exportLocal();
            }
            else
            {
                exportRemote();
            }
        }

        private void exportLocal()
        {
            JFileChooser fc = new JCheckedFileChooser();
            fc.setDialogTitle("Export Directory Service Configuration");

            if (m_exportPath == null)
            {
                String domain = getCurrentWorkspaceWindow().getContext().getConnectionInfo().getDomainName();
                StringBuffer         sb  = new StringBuffer();

                sb.append("dump.").append(domain).append(".xml");

                m_exportPath = sb.toString();
            }

            try
            {
                fc.setSelectedFile(new File(m_exportPath));
            }
            catch (Exception e)
            {
                MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
            }

            if (fc.showSaveDialog(MgmtConsole.getMgmtConsole()) == JFileChooser.APPROVE_OPTION)
            {
                // Keep around the path so that the next time the action is
                // launched we go back to the same place...
                //
                m_exportPath = fc.getSelectedFile().getPath();

                Thread t = new Thread(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        exportConfig(m_exportPath);
                    }
                });

                t.start();
            }
        }

        private void exportRemote()
        {
            IDomainWorkspaceWindow window = getCurrentWorkspaceWindow();

            if (window == null)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                    "Can't export configuration if no domain selected!", true);
                return;
            }

            try
            {
                DomainConnectionModel dcm = window.getContext().getConnectionInfo();

                ((IDirectoryAdminService)dcm.getDirectoryService()).dumpContentsToXML();
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                       "Failed to export directory service configuration.", e, true);
            }
        }

        /**
         * Because the call to export the DS contents as XML may take some
         * time, we wrap the call in a loop, checking for timeouts and then
         * allow the user to supply a more appropriate timeout if the call
         * fails.
         */
        private void exportConfig(String exportPath)
        {
            IDomainWorkspaceWindow window = getCurrentWorkspaceWindow();

            if (window == null)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                    "Can't export configuration if no domain selected!", true);
                return;
            }

            boolean notFinished = true;

            DomainConnectionModel dcm     = window.getContext().getConnectionInfo();
            JMSConnectorClient    server  = (JMSConnectorClient)dcm.getMBeanServer();
            int                   timeout = dcm.getTimeout() * 1000 * 4;

            while (notFinished)
            {
                try
                {
                    ObjectName dsName  = new ObjectName(dcm.getDirectoryServiceName());
                    String     dump    = null;

                    if (server != null)
                    {
                        // We call invoke directly on the connector rather than
                        // going via the IDirectoryAdminService::exportDirectoryToXML
                        // because we need to control the timeout explicitly on this
                        // operation.
                        dump = (String)server.invoke(dsName, "exportDirectoryToXML",
                                                     new Object[] { "/" },
                                                     new String[] { String.class.getName() },
                                                     timeout);
                    }
                    else
                    {
                        // But if we connected locally using the ds.xml file then
                        // we won't have a JMSConnectorClient so we fall back on
                        // the old approach which was to call the Directory
                        // Service Proxy directly (NO TIMEOUT)
                        //
                        dump = ((IDirectoryAdminService)dcm.getDirectoryService()).exportDirectoryToXML("/");
                    }

                    if (dump == null)
                    {
                        throw new Exception("Directory Service has nothing to export!");
                    }

                    Writer writer = new FileWriter(exportPath);

                    writer.write(dump);
                    writer.flush();
                    writer.close();

                    notFinished = false;
                }
                catch (InvokeTimeoutException e)
                {
                    String value = (String)JOptionPane.showInputDialog(MgmtConsole.getMgmtConsole(),
                       "Failed to export directory service configuration.\nTimeout waiting for response.\n\nTo retry please enter a new timeout (in seconds):",
                       "Attention", JOptionPane.ERROR_MESSAGE, null, null, Integer.toString(timeout / 1000));

                    if ((value == null) || (value.trim().length() == 0))
                    {
                        notFinished = false;
                    }
                    else
                    {
                        try
                        {
                            timeout = Integer.parseInt(value.trim()) * 1000;

                            if (timeout <= 0)
                            {
                                timeout = dcm.getTimeout() * 1000 * 4;
                            }
                        }
                        catch (NumberFormatException eNFE)
                        {
                            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                "Failed to export directory service configuration.\nInvalid timeout!", e, true);
                            notFinished = false;
                        }
                    }
                }
                catch(Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                          "Failed to export directory service configuration.", e, true);
                    notFinished = false;
                }
            }
        }
    }

    class JCheckedFileChooser extends JFileChooser
    {
        @Override
        public void approveSelection()
        {
            File selectedFile = getSelectedFile();

            if ((selectedFile != null) && selectedFile.exists())
            {
                if (JOptionPane.showConfirmDialog(this, "The file " + selectedFile.getName() + " already exists. Do you want to replace the existing file?",
                             "Attention", JOptionPane.YES_NO_CANCEL_OPTION) != JOptionPane.YES_OPTION)
                {
                    return;
                }
            }

            super.approveSelection();
        }
    }

    class ToolTypeManagerAction extends ToolAction
    {
        public ToolTypeManagerAction()
        {
            super("tool.typeManager");
        }

        @Override
        public boolean isEnabled()
        {
            return (getCurrentWorkspaceWindow() != null);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                IDomainWorkspaceWindow window = getCurrentWorkspaceWindow();
                IDirectoryAdminService ds = (IDirectoryAdminService)window.getContext().getConnectionInfo().getDirectoryService();
                JTypeManagerDialog dlg = new JTypeManagerDialog(MgmtConsole.getMgmtConsole(), ds);
                dlg.setVisible(true);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                     "Type Manager", e, true);
            }
        }
    }

    class ToolRepairDSAction extends ToolAction
    {
        public ToolRepairDSAction()
        {
            super("tool.repairDS");
        }

        @Override
        public boolean isEnabled()
        {
            return (getCurrentWorkspaceWindow() != null);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                IDomainWorkspaceWindow window = getCurrentWorkspaceWindow();
                JProgressDialog dlg = new JProgressDialog(MgmtConsole.getMgmtConsole(), "Repair Directory Service", true);

                dlg.setContent(new JLabel("Please wait..."));
                dlg.setWorker(new RepairDSWorker(window.getContext().getConnectionInfo(), dlg));
                dlg.setVisible(true);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                     "Failed to repair directory service.", e, true);
            }
        }
    }

    class RepairDSWorker extends SwingWorker
    {
        JProgressDialog       m_dialog;
        DomainConnectionModel m_model;
        ConfigServerUtility   m_csu;
        boolean               m_cancelled = false;

        public RepairDSWorker(DomainConnectionModel model, JProgressDialog dialog)
        {
            super();

            m_model  = model;
            m_dialog = dialog;
        }

        @Override
        public Object construct()
        {
            Object res = null;

            m_csu = new ConfigServerUtility();

            try
            {
                m_csu.connect(m_model.getDirectoryService(), false);
                m_csu.repairDanglingBeanReferences();

                res = m_csu;
            }
            catch (Throwable e)
            {
                res = e;
            }

            return res;
        }

        @Override
        public void finished()
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

            try
            {
                if (m_dialog != null)
                {
                    m_dialog.setVisible(false);
                }

                if (m_cancelled)
                {
                    disconnect();
                    return;
                }

                if (getValue() == null)
                {
                    return;
                }

                if (getValue()instanceof Throwable)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                      "Failed to repair Directory Service", (Throwable)getValue(), true);

                    disconnect();
                }
                else
                if (!m_cancelled)
                {
                }
            }
            finally
            {
                if (m_dialog != null)
                {
                    m_dialog.dispose();
                }

                wc.release();
            }
        }

        @Override
        public void interrupt()
        {
            m_cancelled = true;

            super.interrupt();
        }

        public void disconnect()
        {
            try
            {
                m_csu.disconnect();
                m_csu = null;
            }
            catch(Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                 "Failed to disconnect " + e.getMessage(), e, false);
            }
        }
    }

    class LaunchSMCAction extends ToolAction{

        public LaunchSMCAction() {
            super("tool.jmsTestClient");
        }

        @Override
        public void actionPerformed(ActionEvent evt) {
            String sonicHome = System.getProperty( "sonicsw.home" );
            String mqHome = sonicHome + File.separator + "MQ" + getMajorMinorVersion();
            String startMCFile = mqHome + File.separator + "bin" + File.separator + "testClient." + getFileExtn();
            File mqDir = new File(mqHome);
            try {
                final Process process = Runtime.getRuntime().exec(startMCFile, null, mqDir);
            } catch (IOException e) {
                e.printStackTrace();
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
            }
        }
        public String getMajorMinorVersion(){
            return Version.getMajorVersion() + "." + Version.getMinorVersion();
        }

        private String getFileExtn() {
            String osName = System.getProperty("os.name");
            if (osName.toLowerCase().indexOf("win") != -1)
            {
                return "bat";
            }
            return "sh";
        }
    }
}
