/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui;

import java.awt.event.InputEvent;
import java.awt.Rectangle;
import java.awt.KeyboardFocusManager;
import java.awt.KeyEventDispatcher;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.StringTokenizer;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.JTree;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import com.sonicsw.ma.gui.config.ConfigDomainFolderPlugin;
import com.sonicsw.ma.gui.runtime.AbstractRuntimePlugin;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.ITableContentPane;
import com.sonicsw.ma.gui.util.DescriptionPanel;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.IMonitorPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginContext;

import com.sonicsw.mf.common.runtime.IComponentIdentity;

public class WorkspaceWindow extends JMAInternalFrame
    implements IDomainWorkspaceWindow, ChangeListener
{
    private IPluginContext m_context;
    private JTabbedPane m_tab;
    private Object m_selected = null;
    private int m_lastWorkspace = -1;
    private EventListenerList m_listenerList = new EventListenerList();
    private DescriptionPanel m_panelDescription = null;

    public WorkspaceWindow(String title, IPluginContext context)
    {
        super("workspace");
        prepareWSWindow(title, context);
    }
    
    private void prepareWSWindow(String title, IPluginContext context) {
        try
        {
            setTitle(title);
            setFrameIcon(ResourceManager.getIcon(getClass(), "Connect"));

            m_context = context;
            ((PluginContext)context).setWorkspace(this);

            m_tab = new JTabbedPane(JTabbedPane.TOP);
            m_tab.addChangeListener(this);

            getContentPane().setLayout(new BorderLayout());
            getContentPane().add(m_panelDescription = new DescriptionPanel("Test Description"), BorderLayout.NORTH);
            m_panelDescription.setBorder(BorderFactory.createEmptyBorder(0, 0, JPartitionPanel.BORDER_SIZE, 0));
            getContentPane().add(m_tab, BorderLayout.CENTER);

            Class[] classes = getPanelClasses();
            for (int i = 0; i < classes.length; i++)
            {
                JWorkspacePanel panel = (JWorkspacePanel)classes[i].newInstance();

                m_tab.add(panel, panel.getName());
                m_tab.setIconAt(i, panel.getIcon());
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
        }
    }

    @Override
    public boolean canClose()
    {
        return (MgmtConsole.getMgmtConsole().showConfirmDialog(
            "Are you sure you want to close connection:\n\t" + getTitle(),
            "Close Connection",
            JOptionPane.QUESTION_MESSAGE,
            JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION);
    }

    private Class[] getPanelClasses()
    {
    	List res = new ArrayList();
    	
    	res.add(JWorkspaceConfigPanel.class);
    	res.add(JWorkspaceRuntimePanel.class);
    	
    	if (MgmtConsole.DEVELOPER_MODE)
        {
            res.add(com.sonicsw.ma.gui.dev.JDevWorkspacePanel.class);
        }
    	
    	return (Class[])res.toArray(new Class[0]);
    }
    
    public void updateDescription(String strSelection)
    {
        m_panelDescription.setExtendedText(strSelection);
    }

    /**
     * Perform any initialization here....gets called before the window is
     * made visible but after construction.
     *
     * Only gets called once.
     */
    @Override
    protected void maInitialize()
    {
    }

    /**
     * Perform any cleanup of window resources here....gets called after a
     * window is sure that it can close. It determines this by calling the
     * JMAInternalFrame::canClose method.
     *
     * Could possible get called more than once...so cleanup code should
     * cleanup references properly.
     */
    @Override
    protected void maCleanup()
    {
        this.m_tab.removeChangeListener(this);

        for (int i = 0; i < m_tab.getTabCount(); i++)
        {
            JWorkspacePanel panel = (JWorkspacePanel) m_tab.getComponentAt(i);

            panel.dispose();
        }

        m_tab.removeAll();

        try
        {
            m_context.getConnectionInfo().disconnect();

            MgmtConsole.getMgmtConsole().getToolBar().clearActions();
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                "Failed to disconnect", e, false);
        }
        finally
        {
            if (m_context != null)
            {
                m_context.destroy();
            }

            m_context = null;
            m_tab = null;
        }
    }

    //-------------------------------------------------------------------------
    //
    // IDomainWorkspaceWindow implementation
    //
    //-------------------------------------------------------------------------

    /**
     * Get the Plugin Context current used by this Workspace Window
     */
    @Override
    public IPluginContext getContext()
    {
        return m_context;
    }

    /**
     * Get the menu items for the currently selected component
     */
    @Override
    public JComponent[] getSelectedMenuItems(int type)
    {
        Component comp = getComponentWithFocus();

        JComponent[] items = null;

        try
        {
            while (comp != null)
            {
                if (comp instanceof JTree)
                {
                    IPlugin plugin = null;

                    if (((JTree) comp).getSelectionCount() > 0)
                    {
                        DefaultMutableTreeNode node = (DefaultMutableTreeNode) ((JTree) comp).getSelectionPath().getLastPathComponent();

                        if (node instanceof IPlugin)
                        {
                            plugin = (IPlugin) node;
                        }
                    }

                    if (plugin != null)
                    {
                        items = plugin.getMenuItems(type);
                    }
                }
                else if (comp instanceof IContentPane)
                {
                    items = ((IContentPane) comp).getMenuItems(type);
                    break;
                }
                comp = comp.getParent();
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
        }
        return items;
    }

    //Returns a MouseEvent which will be used by the DispatchEvent.
    // This si to simulate Shift + F10 Windows XP keyboard shortcut to open context menu.
    private MouseEvent createMouseEvent(Component source, int x, int y, int eventType)
    {
        return new MouseEvent(source, eventType, System.currentTimeMillis(), 0, x, y, 1, true, MouseEvent.BUTTON3);
    }

    //-------------------------------------------------------------------------
    //
    // ChangeListener implementation
    //
    //-------------------------------------------------------------------------

    /**
     * Responds to changes in tab selection...we need to update the menubar,
     * toolbar and possibly the statusbar here.
     *
     * @param event  The ChangeEvent itself
     */
    @Override
    public void stateChanged(ChangeEvent event)
    {
        // Don't do anything if the tab was already selected
        int index = m_tab.getSelectedIndex();

        if(index == this.m_lastWorkspace)
        {
            return;
        }

        m_lastWorkspace = index;

        JWorkspacePanel workspace = getSelectedWorkspacePanel();
        ToolBar toolbar = MgmtConsole.getMgmtConsole().getToolBar();
        toolbar.clearActions();

        Component focusComp = getComponentWithFocus();

        if (focusComp != null)
        {
            // See if the last selected component was the JTree.
            if (focusComp == workspace.getTree())
            {
                JTree tree = (JTree) focusComp;
                TreePath treePath = tree.getSelectionPath();

                if (treePath != null)
                {
                    DefaultMutableTreeNode node = (DefaultMutableTreeNode) treePath.getLastPathComponent();

                    toolbar.setActions(node);
                }
            }
            else
            if (focusComp instanceof JRowTable)
            {
                while (focusComp != null)
                {
                    if (focusComp instanceof IContentPane)
                    {
                        // Set the actions for the toolbar and edit menu.
                        toolbar.setActions((IContentPane) focusComp);
                        break;
                    }

                    focusComp = focusComp.getParent();
                }
            }
        }


        // This si to simulate Shift + F10 Windows XP keyboard shortcut to open context menu.
        KeyboardFocusManager.getCurrentKeyboardFocusManager().addKeyEventDispatcher(new KeyEventDispatcher()
        {
            @Override
            public boolean dispatchKeyEvent(KeyEvent e)
            {
                if ((e.getKeyCode() == KeyEvent.VK_F10)  && e.getID() == KeyEvent.KEY_PRESSED &&
                        (e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) > 0)
                {
                    MouseEvent me = null;
                    final Component compWithFocus = getComponentWithFocus();
                    if (compWithFocus instanceof JTree)
                    {
                        final JTree treeComp = (JTree) compWithFocus;
                        final TreePath selectionPath = treeComp.getSelectionPath();
                        final Rectangle pathBounds = treeComp.getPathBounds(selectionPath);
                        int x = Double.valueOf(pathBounds.getX()).intValue();
                        int y = Double.valueOf(pathBounds.getY()).intValue();
                        me = createMouseEvent(treeComp, x, y, MouseEvent.MOUSE_PRESSED);
                        treeComp.dispatchEvent(me);
                        me = createMouseEvent(treeComp, x, y, MouseEvent.MOUSE_RELEASED);
                        treeComp.dispatchEvent(me);
                        e.consume();
                        return true;
                    }
                    else if (compWithFocus instanceof JRowTable)
                    {
                        final JRowTable tableComp = (JRowTable) compWithFocus;
                        Rectangle r = tableComp.getCellRect(tableComp.getSelectedRow(), tableComp.getSelectedColumn(), true);
                        int x = Double.valueOf(r.getX()).intValue();
                        int y = Double.valueOf(r.getY()).intValue();
                        me = createMouseEvent(tableComp, x, y, MouseEvent.MOUSE_PRESSED);
                        tableComp.dispatchEvent(me);
                        me = createMouseEvent(tableComp, x, y, MouseEvent.MOUSE_RELEASED);
                        tableComp.dispatchEvent(me);
                        e.consume();
                        return true;
                    }
                }
                return false;
            }
        });


        // refresh the active content pane on tab change.
        IContentPane contentPane = workspace.getContentPane();

        if (contentPane != null)
        {
            contentPane.refresh();
        }

        updateSelection(getSelected());
        m_panelDescription.setText(workspace.getName());
        m_panelDescription.getDescriptionLabel().setIcon(workspace.getIcon());

        if (workspace.getSelectedNode() != null && workspace.getSelectedNode() instanceof IPlugin)
        {
            m_panelDescription.setExtendedText(((IPlugin)workspace.getSelectedNode()).getPluginPath());
        }
        else
        {
            m_panelDescription.setExtendedText(null);
        }

        if (workspace.getFocusComponent() != null)
        {
            workspace.getFocusComponent().requestFocus();
        }
        else
        {
            workspace.getTree().requestFocus();
        }
    }

    protected void setSelectedTab(int tab)
    {
        m_tab.getModel().setSelectedIndex(tab);
    }

    public JWorkspacePanel getSelectedWorkspacePanel()
    {
        JWorkspacePanel panelWkspc = (JWorkspacePanel) m_tab.getSelectedComponent();

        if (panelWkspc.isInitialized() == false)
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

            try
            {
                panelWkspc.initialize(m_context);
            }
            finally
            {
                wc.release();
            }
        }

        return panelWkspc;
    }

    public JWorkspacePanel getWorkspacePanel(String name)
    {
        for (int i = 0; i < m_tab.getTabCount(); i++)
        {
            JWorkspacePanel panel = (JWorkspacePanel) m_tab.getComponentAt(i);

            if (panel.getName().equals(name))
            {
                return panel;
            }
        }
        return null;
    }

    public IPlugin findConfiguration(final String path)
    {
        return goToConfiguration(path,true,false,false);
    }

    public IPlugin goToConfiguration(final String path, boolean isConfig)
    {
        return goToConfiguration(path, isConfig, true);
    }

    public IPlugin goToConfiguration(String  path,
                                        boolean isConfig,
                                        boolean changeSelection)    {
        return goToConfiguration(path,isConfig,changeSelection,true);
    }


    public IPlugin goToConfiguration(String  path,
                                        boolean isConfig,
                                        boolean changeSelection,boolean isVisual)
    {
        JWorkspacePanel panelConfig = getWorkspacePanel(JWorkspaceConfigPanel.PANEL_NAME);
        JTree treeConfig = panelConfig.getTree();
        /**
         * Added a new root when ESB Explorer moved to SMC config Tab. This is a temparory fix
         */
        path = ConfigDomainFolderPlugin.PLUGIN_NAME + path;

        // ensure config tab is visible...
        if(isVisual)
        {
            m_tab.setSelectedIndex(m_tab.indexOfComponent(panelConfig));
        }

        TreeNode node = (TreeNode)((DefaultTreeModel) treeConfig.getModel()).getRoot();
        StringTokenizer tokens = new StringTokenizer(path, "/");

        while (node!=null && tokens.hasMoreTokens())
        {
            node = ((AbstractGUIPlugin)node).findChildNode(tokens.nextToken());
        }

        if (node != null)
        {
            if (isVisual) {
                TreePath pathToNode = new TreePath(((DefaultMutableTreeNode) node).getPath());
                treeConfig.expandPath(pathToNode);

                if (changeSelection)
                {
                    treeConfig.setSelectionPath(pathToNode);
                }

                treeConfig.invalidate();
            }
        }
        else
        {
            Helper.logErrorMessage("Failed to find configuration node with path '" + path + "'");
        }

        return (IPlugin)node;
    }

    public void updateCollectionState(String path)
    {
    	JWorkspacePanel panelRuntime = getWorkspacePanel(JWorkspaceRuntimePanel.PANEL_NAME);
        JTree treeRuntime = panelRuntime.getTree();
    
        TreeNode node = (TreeNode)((DefaultTreeModel) treeRuntime.getModel()).getRoot();
        
        if (node instanceof AbstractGUIPlugin) //tree never was expended, so state will be updated properly on next tree expend
        {
    	    StringTokenizer tokens = new StringTokenizer(path, "/");
    	
    	    while (tokens.hasMoreTokens() && (node != null))
    	    {
    	        node = ((AbstractGUIPlugin)node).findChildNode(tokens.nextToken());
    	    }
    	    if  (node != null)
    	    {
    	        ((AbstractRuntimePlugin)node).getState(true);
    	    }
        }
    }
    
    public void goToInstance(String path, String compRuntimeID, boolean isConfig)
    {
        JWorkspacePanel panelRuntime = getWorkspacePanel(JWorkspaceRuntimePanel.PANEL_NAME);
        JTree treeRuntime = panelRuntime.getTree();

        // ensure runtime tab is visible...
        m_tab.setSelectedIndex(m_tab.indexOfComponent(panelRuntime));

        TreeNode node = (TreeNode)((DefaultTreeModel) treeRuntime.getModel()).getRoot();
        StringTokenizer tokens = new StringTokenizer(path, "/");

        while (tokens.hasMoreTokens() && (node != null))
        {
            node = ((AbstractGUIPlugin)node).findChildNode(tokens.nextToken());
        }

        if  (node != null)
        {
            TreePath pathToNode = new TreePath(((DefaultMutableTreeNode)node).getPath());

            boolean bOldMode = panelRuntime.getTreeExpansionMode();
            panelRuntime.setTreeExpansionMode(false);
            treeRuntime.expandPath(pathToNode);
            panelRuntime.setTreeExpansionMode(bOldMode);

            if (compRuntimeID != null)//component
            {
                Enumeration children = node.children();
                while(children.hasMoreElements())
                {
                    node  = (AbstractRuntimePlugin)children.nextElement();
                    if (((AbstractRuntimePlugin)node).getRuntimeID().getCanonicalName().equals(compRuntimeID))
                    {
                          pathToNode = new TreePath(((DefaultMutableTreeNode)node).getPath());
                          break;
                    }
                }
            }

            treeRuntime.setSelectionPath(pathToNode);
        }
    }

    private String getComponentName(String runtimeID)
    {
        int index = runtimeID.indexOf(IComponentIdentity.DELIMITED_ID_PREFIX);

        return (runtimeID.substring(index + IComponentIdentity.DELIMITED_ID_PREFIX_LENGTH));
    }

    public IPlugin getSelectedPlugin()
    {
        Object obj = getSelected();

        if(obj instanceof IPlugin)
        {
            return (IPlugin)obj;
        }

        return null;
    }

    public Object getSelected()
    {
        Component comp = getComponentWithFocus();

        Object res = null;

        while (comp != null)
        {
            if (comp instanceof JTree)
            {
                if (((JTree) comp).getSelectionCount() > 0)
                {
                    res = ((JTree)comp).getSelectionPath().getLastPathComponent();
                }
                break;
            }
            else if (comp instanceof ITableContentPane)
            {
                JRowTable table = ((ITableContentPane) comp).getTable();
                int selRow = table.getSelectedRow();

                res = ((IModelTableModel)table.getModel()).getRowModel(selRow);
            }
            else if (comp instanceof IContentPane)
            {
                res = ((IContentPane)comp).getSelectedItem();
                break;
            }
            comp = comp.getParent();
        }
        return res;
    }

    private Component getComponentWithFocus()
    {
        Component comp = null;
        int nIndex = m_tab.getSelectedIndex();

        if (nIndex != -1)
        {
            JWorkspacePanel panel = (JWorkspacePanel) m_tab.getComponentAt(nIndex);

            if (panel != null)
            {
                comp = panel.getFocusComponent();
            }
        }

        return comp;
    }

    public void updateSelection()
    {
        updateSelection(getSelected());
    }

    /* package protected */ void updateSelection(Object obj)
    {
        if (obj == null)
        {
            return;
        }

        m_selected = obj;

        if (m_selected instanceof IPlugin)
        {
            fireGUIPluginChanged((IPlugin) m_selected);
        }

        if (m_selected instanceof IMonitorPlugin)
        {
            fireGUIMonitorPluginChanged((IMonitorPlugin) m_selected);
        }
    }

    //-------------------------------------------------------------------------
    //
    // GUIPluginListener support
    //
    //-------------------------------------------------------------------------

    /**
     * Adds a listener to the list that's notified each time a change
     * to the selected plugin occurs.
     *
     * @param	l		the GUIPluginListener
     */
    public void addGUIPluginListener(GUIPluginListener l)
    {
        m_listenerList.add(GUIPluginListener.class, l);
    }

    /**
     * Removes a listener from the list that's notified each time a
     * change to the selected plugin occurs.
     *
     * @param	l		the GUIPluginListener
     */
    public void removeGUIPluginListener(GUIPluginListener l)
    {
        m_listenerList.remove(GUIPluginListener.class, l);
    }

    /**
     * Returns an array of all the gui plugin listeners
     * registered on this window.
     *
     * @return all of this model's <code>GUIPluginListener</code>s
     *         or an empty array if no listeners are currently registered
     *
     * @see #addGUIPluginListener
     * @see #removeGUIPluginListener
     */
    public GUIPluginListener[] getGUIPluginListeners()
    {
        return (GUIPluginListener[]) listenerList.getListeners(GUIPluginListener.class);
    }

    /**
     * Forwards the given notification event to all
     * <code>GUIPluginListeners</code> that registered
     * themselves as listeners for this event change.
     *
     * @param e  the event to be forwarded
     *
     * @see #addGUIPluginListener
     * @see EventListenerList
     */
    public void fireGUIPluginChanged(IPlugin plugin)
    {
        // Guaranteed to return a non-null array
        Object[] listeners = m_listenerList.getListenerList();

        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2)
        {
            if (listeners[i] == GUIPluginListener.class)
            {
                ((GUIPluginListener) listeners[i + 1]).guiPluginChanged(plugin);
            }
        }
    }

    //-------------------------------------------------------------------------
    //
    // GUIMonitorPluginListener support
    //
    //-------------------------------------------------------------------------

    /**
     * Adds a listener to the list that's notified each time a change
     * to the selected plugin occurs.
     *
     * @param	l		the GUIMonitorPluginListener
     */
    public void addGUIMonitorPluginListener(GUIMonitorPluginListener l)
    {
        m_listenerList.add(GUIMonitorPluginListener.class, l);
    }

    /**
     * Removes a listener from the list that's notified each time a
     * change to the selected plugin occurs.
     *
     * @param	l		the GUIMonitorPluginListener
     */
    public void removeGUIMonitorPluginListener(GUIMonitorPluginListener l)
    {
        m_listenerList.remove(GUIMonitorPluginListener.class, l);
    }

    /**
     * Returns an array of all the GUI monitor plugin listeners
     * registered on this window.
     *
     * @return all of this model's <code>GUIMonitorPluginListener</code>s
     *         or an empty array if no listeners are currently registered
     *
     * @see #addGUIMonitorPluginListener
     * @see #removeGUIMonitorPluginListener
     */
    public GUIMonitorPluginListener[] getGUIMonitorPluginListeners()
    {
        return (GUIMonitorPluginListener[]) listenerList.getListeners(GUIMonitorPluginListener.class);
    }

    /**
     * Forwards the given notification event to all
     * <code>GUIMonitorPluginListeners</code> that registered
     * themselves as listeners for this event change.
     *
     * @param e  the event to be forwarded
     *
     * @see #addGUIMonitorPluginListener
     * @see EventListenerList
     */
    public void fireGUIMonitorPluginChanged(IMonitorPlugin plugin)
    {
        // Guaranteed to return a non-null array
        Object[] listeners = m_listenerList.getListenerList();

        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2)
        {
            if (listeners[i] == GUIMonitorPluginListener.class)
            {
                ((GUIMonitorPluginListener) listeners[i + 1]).guiMonitorPluginChanged(plugin);
            }
        }
    }

    //-------------------------------------------------------------------------
    //
    // Child Dialog Tracking Service
    //
    //-------------------------------------------------------------------------

    private HashMap m_trackedDialogMap = new HashMap();

    public JMADialog getTrackedDialog(Object id)
    {
        return (JMADialog) m_trackedDialogMap.get(id);
    }

    public void addTrackedDialog(Object id, JMADialog dialog)
    {
        m_trackedDialogMap.put(id, dialog);
    }

    public JMADialog removeTrackedDialog(Object id)
    {
        return (JMADialog) m_trackedDialogMap.remove(id);
    }

}