/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import com.sonicsw.ma.gui.JWorkspacePanel;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.util.BasicPluginAction;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IFileExistHandler;

public class ExportPluginAction extends BasicPluginAction
{
    private static final String YES = "Yes";
    private static final String YES_TO_ALL = "Yes to All";
    private static final String NO = "No";
    private static final String CANCEL = "Cancel";
    
    private static String[] OPTIONS     = new String[] { YES,             NO,        };
    private static String[] OPTIONS_ALL = new String[] { YES, YES_TO_ALL, NO, CANCEL };

    public static final String BOOT_PATH_PREFS = "preferences.export";
    public static final String BASEPATH        = "basePath";

    public ExportPluginAction(IPlugin plugin)
    {
        super(plugin, "export.plugin");
    }

    @Override
    public boolean isEnabled()
    {
        return getPlugin().getPluginContext().getConnectionInfo().isConnected();
    }
    
    @Override
    public void actionPerformed(ActionEvent event)
    {
        PreferenceManager pm = PreferenceManager.getInstance();
        String basePath = pm.getString(BOOT_PATH_PREFS, BASEPATH, "");

        JFileChooser fc = new JFileChooser();

        fc.setDialogTitle("Select Export Location");
        fc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        fc.setApproveButtonText("Export");
        fc.setSelectedFile(new File(basePath));

        if (fc.showSaveDialog(MgmtConsole.getMgmtConsole()) == JFileChooser.APPROVE_OPTION)
        {
            String path = fc.getSelectedFile().getPath();

            pm.setString(BOOT_PATH_PREFS, BASEPATH, path, true);

            export(path);
        }
    }

    private void export(final String exportPath)
    {
        Thread t = new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

                try
                {
                	IPlugin plugin = getPlugin();
                	JWorkspacePanel wp = MgmtConsole.getMgmtConsole().getSelectedWorkspaceWindow().getSelectedWorkspacePanel();
                	Object[] selItem = wp.getSelectedItems();
                	List     list = new ArrayList();
                	
                	if (selItem != null)
                    {
                		for (int i = 0; i < selItem.length; i++)
                        {
                            list.add(((IPlugin)selItem[i]).getPluginPath());
                        }
                    }
                    
                    ConfigServerUtility csu = getPlugin().getPluginContext().getConnectionInfo().getConfigServerUtility();
                    String[] paths = (String[])list.toArray(new String[0]);
                    
                    IFileExistHandler handler = new IFileExistHandler()
                    {
                        @Override
                        public int fileExists(File file, boolean isPartOfGroup)
                        {
                            // Handler callback is being called inside a non-UI thread...
                            // so we need to invoke and wait on the main UI thread :(
                            PromptRunnable r = new PromptRunnable(file, isPartOfGroup);
                        
                            try
                            {
                                SwingUtilities.invokeAndWait(r);
                            }
                            catch (Exception e)
                            {
                            }
                            
                            return r.getRetValue();
                        }
                    };
                    
                    csu.exportBeanPath(paths, exportPath, handler, true, true);
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                        "Failed to export.", e, true);
                }
                finally
                {
                    wc.release();
                }
            }
        });

        t.start();
    }
    
    private class PromptRunnable implements Runnable
    {
        private int m_retValue = IFileExistHandler.CANCEL;
        private File m_file = null;
        private boolean m_partOfGroup;
        
        public PromptRunnable(File file, boolean isPartOfGroup)
        {
            m_file = file;
            m_partOfGroup = isPartOfGroup;
        }
        
        @Override
        public void run()
        {
            String[] options = m_partOfGroup ? OPTIONS_ALL : OPTIONS;
            String   defOption = YES;
            
            int res = JOptionPane.showOptionDialog(MgmtConsole.getMgmtConsole(),
                                                   "<html>" + 
                                                   MessageFormat.format("This folder already contains a file named ''{0}''.", new Object[] { m_file.getName() }) +
                                                   "<br><br><br>" +
                                                   "Would you like to replace the existing file?<br><br>" +
                                                   "&nbsp&nbsp&nbsp&nbsp " + getSizeString(m_file.length()) + "<br>" +
                                                   "&nbsp&nbsp&nbsp&nbsp modified: " + getDateString(m_file.lastModified()) + "<br>" +
                                                   "</html>",
                                                   "Confirm File Replace",
                                                   0, // optionType
                                                   JOptionPane.QUESTION_MESSAGE, // messageType
                                                   null,
                                                   options,
                                                   defOption);
            if (options[res].equals(YES))
            {
                m_retValue = IFileExistHandler.YES;
            }
            else
            if (options[res].equals(YES_TO_ALL))
            {
                m_retValue = IFileExistHandler.YES_ALL;
            }
            else
            if (options[res].equals(NO))
            {
                m_retValue = IFileExistHandler.NO;
            }
            
            // Default op is CANCEL
        }
        
        public int getRetValue() { return m_retValue; }
    }
    
    private String getDateString(long timestamp)
    {
        StringBuffer sb = new StringBuffer();
        Calendar cal = Calendar.getInstance();
        
        cal.setTimeInMillis(timestamp);
        int year = cal.get(Calendar.YEAR);
        int month = cal.get(Calendar.MONTH);
        int day = cal.get(Calendar.DAY_OF_MONTH);
        
        cal.setTimeInMillis(System.currentTimeMillis());
        
        if ((year == cal.get(Calendar.YEAR)) &&
            (month == cal.get(Calendar.MONTH)) &&
            (day == cal.get(Calendar.DAY_OF_MONTH)))
        {
            sb.append("Today");
        }
        
        sb.append(", ").append(DateFormat.getDateTimeInstance().format(new Date(timestamp)));
        
        return sb.toString();
    }
    
    private String getSizeString(long size)
    {
        StringBuffer sb = new StringBuffer();
        long kb = size / 1024;

        if (kb > 0)
        {
            String kbS = Long.toString(kb);
        
            if (kbS.length() > 3)
            {
                sb.append(kbS.substring(0, kbS.length() - 3));
                sb.append(',');
                sb.append(kbS.substring(kbS.length() - 3));
            }
            else
            {
                sb.append(kbS);
            }
            
            sb.append(" KB");
        }
        else
        {
            sb.append(Long.toString(size)).append(" bytes");
        }    
        
        return sb.toString();
    }
}