/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.action;

import java.awt.event.ActionEvent;
import java.io.File;
import java.text.MessageFormat;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.config.ConfigFolderPlugin;
import com.sonicsw.ma.gui.util.BasicPluginAction;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IFilePlugin;
import com.sonicsw.ma.plugin.IFolderPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.IDirectoryFileSystemService;

public class PastePluginAction extends BasicPluginAction
{
    public static final String[] FORMAT_COPY = { "{0}", "Copy {1,choice,1#|1<{1,number,integer} }of {0}" };
    //public static final String[] FORMAT_COPY = { "{0}", "Copy {1,choice,1#|1<({1,number,integer}) }of {0}" };
    public static final String[] FORMAT_NEW = { "New {0}{1,choice,1#|1< {1,number,integer}}" };
    //public static final String[] FORMAT_NEW = { "New {0}{1,choice,1#|1< ({1,number,integer})}" };
    public static final String[] FORMAT_NEW_INSTANCE = { "Instance {1,choice,1#|1<{1,number,integer} }of {0}" };
    //public static final String[] FORMAT_NEW_INSTANCE = { "Instance {1,choice,1#|1<({1,number,integer}) }of {0}" };


    public PastePluginAction(IPlugin plugin)
    {
        super(plugin, "edit.paste");
    }

    public PastePluginAction(IPlugin plugin, String name)
    {
        super(plugin, name);
    }

    @Override
    public void actionPerformed(ActionEvent event)
    {
        paste(false);
    }

    /**
     * Paste the plugin from the clipboard into the DS. If convert is false
     * then this method will create a new plugin as a direct clone of the
     * item in the clipboard. If convert is true then this method will
     * create an instance of a template or create a template from an instance
     * depending on the type of data in the clipboard
     * @param convert do instance - template conversion
     */
    protected void paste(final boolean convert)
    {
        Thread t = new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

                try
                {
                    IPlugin dstPlugin = getPlugin();

                    if ((dstPlugin != null) && (dstPlugin instanceof IFolderPlugin))
                    {
                        if (MgmtConsole.getMgmtConsole().isClipboardDataFlavorSupported(MgmtConsole.configFlavor))
                        {
                            pasteConfigElement(dstPlugin, convert);
                        }
                        else
                        if (MgmtConsole.getMgmtConsole().isClipboardDataFlavorSupported(MgmtConsole.fileFlavor))
                        {
                            pasteFileElement(dstPlugin, convert);
                        }
                    }
                }
                finally
                {
                    wc.release();
                }
            }
        });

        t.start();
    }

    public static final void pasteConfigElement(IPlugin parentPlugin, boolean convert)
    {
        IConfigPlugin childPlugin  = (IConfigPlugin)MgmtConsole.getMgmtConsole().getClipboardObject();
        boolean       isCut        = (MgmtConsole.getMgmtConsole().getClipboardAction() == MgmtConsole.CLIP_ACTION_CUT);

        IConfigServer server = null;

        try
        {
            IPluginContext              context = parentPlugin.getPluginContext();
            IDirectoryFileSystemService ds      = context.getConnectionInfo().getDirectoryService();

            server = context.getConfigContext().getConfigServer().connectTransacted();

            if (isCut)
            {
                // If the plugin was cut then we are effectively doing a rename
                // Ignore the 'convert' parameter
                //
                String       parentPath = parentPlugin.getPluginPath();
                StringBuffer sbPath     = new StringBuffer(parentPath);

                if (!parentPath.endsWith("/"))
                {
                    sbPath.append("/");
                }

                sbPath.append(childPlugin.getPluginName());
                String newPath = sbPath.toString();

                if (server.pathExists(newPath))
                {
                    throw new Exception("Configuration already exists!");
                }

                childPlugin.renamePlugin(server, newPath);
            }
            else
            {
                String newPath = getValidPath(server, parentPlugin.getPluginPath(), childPlugin.getPluginName(), getNameFormat(convert, childPlugin));

                Helper.logDebugMessage("PASTE as '" + newPath + "' (" + childPlugin.getPluginPath() + ")");

                childPlugin.copyPlugin(server, newPath, new Boolean(convert));
            }

            if (server.isTransacted())
            {
                server.commit();
            }

            if (isCut)
            {
                MgmtConsole.getMgmtConsole().clearClipboard();
            }
        }
        catch (Exception e)
        {
            if (e instanceof UnsupportedOperationException)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                                           e.getMessage(), e, false);
            }
            else
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                                           "Failed to paste configuration", e, true);
            }
        }
        finally
        {
            if (server != null)
            {
                try { 
                    server.close(); 
                } catch(Exception e) { 
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
                }
            }
        }
    }

    public static String[] getNameFormat(boolean convert, IConfigPlugin plugin)
    {
        String[] res = FORMAT_COPY;

        if (convert)
        {
            ConfigBeanModel model = (ConfigBeanModel)plugin.getModel();

            if (model.isTemplate())
            {
                res = FORMAT_NEW_INSTANCE;
            }
        }

        return res;
    }

    private void pasteFileElement(IPlugin parentPlugin, boolean convert)
    {
        IFilePlugin childPlugin  = (IFilePlugin)MgmtConsole.getMgmtConsole().getClipboardObject();

        try
        {
            IConfigServer server = parentPlugin.getPluginContext().getConfigContext().getConfigServer();

            // create new name for this object - to avoid collision.
            String newPath = getValidPath(server, parentPlugin.getPluginPath(),
                                          childPlugin.getPluginName(), FORMAT_COPY);

            duplicateFileElement(childPlugin, newPath);

            if (MgmtConsole.getMgmtConsole().getClipboardAction() == MgmtConsole.CLIP_ACTION_CUT)
            {
                childPlugin.deletePlugin(getPlugin().getPluginContext().getConfigContext().getConfigServer());
                MgmtConsole.getMgmtConsole().clearClipboard();
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                "Failed to paste file", e, true);
        }
    }

    private void duplicateFileElement(IFilePlugin pluginToPaste, String strNewPath)
    {
        String strTempFilename = null;

        try
        {
            File tmpBlob = null;

            // Only if the plugin being paste has content do we want to create
            // a temporary file to which the blob is exported...
            //
            if (pluginToPaste.hasContent())
            {
                tmpBlob = File.createTempFile("sonicfs", null);
                tmpBlob.deleteOnExit();
                pluginToPaste.exportFile(tmpBlob);
            }

            // Create a new file under this folder plugin with the blob (if any)
            // and the basic filename...NOT async
            //
            ((ConfigFolderPlugin)getPlugin()).addFile(tmpBlob, getFilename(strNewPath), false);

            // Remove the temporary file immediately...if this fails, the
            // temporary file will still get deleted on exit because of the
            // deleteOnExit call made after it is created!
            //
            if (tmpBlob != null)
            {
                tmpBlob.delete();
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                        "Unable to open file: " + strTempFilename != null ? strTempFilename : "", e, true);
        }
    }

    private String getFilename(String strPath)
    {
        String strResult = strPath;
        int index = strResult.lastIndexOf("/");

        if (index != -1)
        {
            strResult = strResult.substring(index + 1);
        }

        return strResult;
    }

    @Override
    public boolean canDisplay()
    {
        return true;
    }

    /**
     * Determines whether or not the paste action can be performed.
     * Specifically, there must be something on the clipboard, that something
     * must be a plugin and the paste location (destination) must be within the
     * same domain as that of the source plugin.
     *
     * @return  true if the paste action can be performed otherwise false
     */
    @Override
    public boolean isEnabled()
    {
        IPlugin dstPlugin = getPlugin();
        IPlugin srcPlugin = (IPlugin)MgmtConsole.getMgmtConsole().getClipboardObject();

        if (!(dstPlugin instanceof IFolderPlugin) || (srcPlugin == null) || (dstPlugin == null))
        {
            return false;
        }
        
        String dstDomain = dstPlugin.getPluginContext().getConnectionInfo().getDomainName();
        String srcDomain = srcPlugin.getPluginContext().getConnectionInfo().getDomainName();
        
        if (!dstDomain.equals(srcDomain))
        {
            return false;
        }
        
        return true;
    }

    public static final String getValidPath(IConfigServer server,
                                            String        basePath,
                                            String        name,
                                            String        pattern)
    {
        return getValidPath(server, basePath, name, new String[] { pattern });
    }

    public static final String getValidPath(IConfigServer server,
                                            String        basePath,
                                            String        name,
                                            String[]      pattern)
    {
        if (!basePath.endsWith("/"))
        {
            basePath += "/";
        }

        // Infinite loop!!!!
        // We have to assume that a unique name will be found even if the loop
        // index 'i' ends up in the thousands...
        //
        for (int i = 0; ; i++)
        {
            String p = (i >= pattern.length) ? pattern[pattern.length-1] : pattern[i];
            String candidate = MessageFormat.format(p, new Object[] { name, new Integer(i) }).trim();
            String newPath = basePath + candidate;

            if (!server.pathExists(newPath))
            {
                return newPath;
            }
        }
    }
}
