/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JComponent;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JWorkspaceConfigPanel;
import com.sonicsw.ma.gui.JWorkspacePanel;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.action.AnnotationPluginAction;
import com.sonicsw.ma.gui.action.CopyPluginAction;
import com.sonicsw.ma.gui.action.CutPluginAction;
import com.sonicsw.ma.gui.action.DeletePluginAction;
import com.sonicsw.ma.gui.action.ExportPluginAction;
import com.sonicsw.ma.gui.action.GoToInstanceAction;
import com.sonicsw.ma.gui.action.GoToTemplateAction;
import com.sonicsw.ma.gui.action.PastePluginAction;
import com.sonicsw.ma.gui.action.PasteSpecialAction;
import com.sonicsw.ma.gui.action.PropSheetAction;
import com.sonicsw.ma.gui.action.RefreshPluginAction;
import com.sonicsw.ma.gui.action.RenamePluginAction;
import com.sonicsw.ma.gui.action.UpgradePluginAction;
import com.sonicsw.ma.gui.config.revert.RevertAction;
import com.sonicsw.ma.gui.domain.AgentManagerConnection;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IAnnotationProvider;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IConfigPluginFactory;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.ConfigServiceRuntimeException;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPrototype;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.IConfigType;
import com.sonicsw.mx.config.util.ConfigHelper;

import com.sonicsw.mf.common.runtime.IIdentity;

/**
 * Abstract implementation the GUIConfigPlugin that encapsulates common
 * implementation used by all GUIConfigPlugins. Most concrete GUIConfigPlugin
 * implementations should extend this class and override appropriate methods.
 */
public abstract class AbstractConfigPlugin extends AbstractGUIPlugin implements IConfigPlugin
{
    protected ConfigBeanModel m_model = null;

    public AbstractConfigPlugin(IPluginContext context, String name, Map metaMap)
    {
        super(context, name);
        setAttributes();
        getAttributes().put(metaMap);
    }
    
    private void setAttributes() {
        setAttributes(new PluginAttributes());
    }

    @Override
    protected JWorkspacePanel getWorkspacePanel()
    {
        return getPluginContext().getWorkspace().getWorkspacePanel(JWorkspaceConfigPanel.PANEL_NAME);
    }
    
    @Override
    public void refresh()
    {
        Helper.logDebugMessage("refresh config plugin - " + getClass().getName());

        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            Object objModel = getModel();

            if (objModel instanceof ConfigBeanModel)
            {
                if (((ConfigBeanModel)objModel).getData() instanceof IConfigBean)
                {
                    IConfigBean bean = (IConfigBean)((ConfigBeanModel)objModel).getData();
                    bean.refresh();
                }
            }
            super.refresh();
        }
        catch (ConfigServiceException cse)
        {
            Helper.logWarningMessage(cse.getMessage());
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, cse.getMessage(), cse, false);    // Logs the exception.
        }
        finally
        {
            wc.release();
        }
    }

    /**
     * Is this Config Plugin a template or a configuration instance
     *
     * @return true if this plugin is a template
     */
    @Override
    public boolean isTemplatePrototype()
    {
        boolean bTemplatePrototype = false;

        if (getAttributes().containsKey(ConfigServerUtility.TEMPLATE_TYPE))
        {
            String tType = (String) getAttributes().get(ConfigServerUtility.TEMPLATE_TYPE);

            if (tType.equals(ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE))
            {
                bTemplatePrototype = true;
            }
        }
        return bTemplatePrototype;
    }

    /**
     * Is this Config Plugin linked to a template. In DS terms this means
     * is the configBean subclassed to a prototype.
     *
     * @return true if this plugin is linked to its template
     */
    @Override
    public boolean isTemplateInstance()
    {
        boolean bTemplateInstance = false;

        if (getAttributes().containsKey(ConfigServerUtility.TEMPLATE_TYPE))
        {
            String tType = (String) getAttributes().get(ConfigServerUtility.TEMPLATE_TYPE);

            if (tType.equals(ConfigServerUtility.TEMPLATE_TYPE_INSTANCE))
            {
                bTemplateInstance = true;
            }
        }

        return bTemplateInstance;
    }

    @Override
    public BasicAction getDefaultAction()
    {
        JComponent[] c = getPropertiesMenuItems();

        if ((c != null) && (c.length > 0))
        {
            if (c[c.length-1] instanceof AbstractButton)
            {
                Action action = ((AbstractButton)c[c.length-1]).getAction();

                if (action instanceof BasicAction)
                {
                    return (BasicAction)action;
                }
            }
        }
        return null;
    }

    @Override
    public List getToolTipList()
    {
        List list = super.getToolTipList();

        if (isTemplatePrototype())
        {
            list.add(formatToolTipText("Template", "yes"));
        }
        else
        if (isTemplateInstance())
        {
            String template = "COULD NOT DETERMINE";

            try
            {
                ConfigBeanModel  model     = (ConfigBeanModel)getModel();
                IConfigPrototype prototype = ((IConfigBean)model.getData()).getPrototype();

                if (prototype == null)
                {
                    template = "na";
                }
                else
                {
                    String name = prototype.getName();

                    if (name.endsWith(ConfigServerUtility.DEFAULT_SUFFIX))
                    {
                        name = name.substring(0, name.lastIndexOf(ConfigServerUtility.DEFAULT_SUFFIX) - 1);
                    }

                    template = name;
                }
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                   "Unable to determine template linkage", e, false);
            }

            list.add(formatToolTipText("Linked to template", template));
        }

        return list;
    }

    /**
     * Release resources that the plugin is using
     */
    @Override
    public void dispose()
    {
        super.dispose();

        m_model = null;
    }

    //-------------------------------------------------------------------------
    //
    // Model Manipulation
    //
    //-------------------------------------------------------------------------

    /**
     * This method gets called from the 'createModel' method and is used to
     * setup the required meta attributes on the underlying config bean model.
     *
     * The set of meta attributes is fixed, that is, every element must have
     * the same set of meta attributes defined, i.e. meta attributes are not
     * optional!
     *
     * The meta attributes generated here get set on the configuration at the
     * client-side and get written back only when the configuration is commited
     * to storage.
     *
     * @param bean  The configuration for which the meta attributes are to be
     *              generated
     * @return      A map (set) of meta attributes
     */
    protected HashMap createModelMetaAttributes(boolean isTemplate, boolean isInstance)
    {
        HashMap map = new HashMap();

        PluginAttributes attributes = getAttributes();

        map.put(ConfigServerUtility.TYPE,            attributes.getType());
        map.put(ConfigServerUtility.PRODUCT_VERSION, attributes.getProductVersion());
        map.put(ConfigServerUtility.CONFIG_VERSION,  attributes.getConfigVersion());

        if (isTemplate)
        {
            map.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE);
        }
        else if (isInstance)
        {
            map.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_INSTANCE);
        }

        return map;
    }

    /**
     * Creates a valid model for this plugin. The model is location is determined
     * by the plugin's path.
     *
     * @param server      This parameter supplies the IConfigServer used to
     *                    create the new configuration model. IConfigServer
     *                    can be transactional.
     * @param isTemplate  Indicates whether or not a templated configuration is
     *                    required - if so a prototype model is created
     * @return            A valid configuration model or null if the model
     *                    could not be created
     */
    @Override
    public Object createModel(IConfigServer server, boolean isTemplate)
    {
        if (m_model != null)
        {
            return m_model;
        }

        IConfigBean bean = null;

        try
        {
            Helper.logDebugMessage("createModel(" + _getPluginPath() + ")");

            bean = server.createConfigBean(_getPluginPath(),
                                           getAttributes().getType(),
                                           getAttributes().getConfigVersion(),
                                           isTemplate);

            // Set the meta attributes on the bean
            //
            HashMap metaMap = createModelMetaAttributes(isTemplate, false);
            if ((metaMap != null) && !metaMap.isEmpty())
            {
                Helper.logDebugMessage("Setting meta attributes on configuration: " + bean.getName());
                bean.setMetaAttributes(metaMap);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                "Failed to create model: " + _getPluginPath(), e, false);
        }

        if (bean != null)
        {
            m_model = new ConfigBeanModel(bean, getPluginName());
        }

        return m_model;
    }

    /**
     * Method simply loads an existing model using the plugin's path as the
     * model's location.
     *
     * @return  The model or null if the configuration could not be found at
     *          the specified location.
     */
    @Override
    public Object getModel()
    {
        if (m_model != null)
        {
            return m_model;
        }

        IConfigBean bean = null;

        try
        {
            Helper.logDebugMessage("getModel(" + _getPluginPath() + ")");

            IConfigServer server = getPluginContext().getConfigContext().getConfigServer();
            String path = _getPluginPath();

            bean = (IConfigBean) server.loadConfigElement(path);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                "Failed to load model: " + getPath(), e, false);
        }

        if (bean != null)
        {
            m_model = new ConfigBeanModel(bean, getPluginName());
        }

        return m_model;
    }

    /**
     * Returns the type information for the configuration model.
     *
     * @return  The configuration type for the model or null if the type could
     *          not be found
     */
    protected IConfigType getModelMetaData()
    {
        IConfigType      res = null;
        PluginAttributes attributes = getAttributes();
        String           type = attributes.getType();

        if (type != null)
        {
            Helper.logDebugMessage("getModelMetaData(" + attributes.getType() + ")");

            try
            {
                IConfigServer server = getPluginContext().getConfigContext().getConfigServer();
                String version = attributes.getConfigVersion();

                res = server.loadConfigType(type, version);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                    "Failed to load meta data: " + attributes.getType(), e, false);
            }
        }
        return res;
    }

    //-------------------------------------------------------------------------
    
    protected JComponent getEditCutMenuItem()          { return new JBasicMenuItem(new CutPluginAction(this)); }
    protected JComponent getEditCopyMenuItem()         { return new JBasicMenuItem(new CopyPluginAction(this)); }
    protected JComponent getEditPasteMenuItem()        { return new JBasicMenuItem(new PastePluginAction(this)); }
    protected JComponent getEditPasteSpecialMenuItem() { return new JBasicMenuItem(new PasteSpecialAction(this)); }
    protected JComponent getEditRevertMenuItem()       { return new JBasicMenuItem(new RevertAction(this)); }

    @Override
    protected JComponent[] getPropertiesMenuItems()
    {
        return new JComponent[]
        {
            new JBasicMenuItem(new PropSheetAction(this))
        };
    }

    @Override
    protected JComponent[] getEditMenuItems()
    {
        ArrayList  list = new ArrayList();
        JComponent c    = null;

        c = getEditCutMenuItem();           if (c != null)
        {
            list.add(c);
        }
        c = getEditCopyMenuItem();          if (c != null)
        {
            list.add(c);
        }
        c = getEditPasteMenuItem();         if (c != null)
        {
            list.add(c);
        }
        c = getEditPasteSpecialMenuItem();  if (c != null)
        {
            list.add(c);
        }
        c = getEditRevertMenuItem();        if (c != null)
        {
            list.add(c);
        }

        return (JComponent[])list.toArray(new JComponent[list.size()]);
    }

    @Override
    protected JComponent[] getSystemMenuItems()
    {
        ArrayList list = new ArrayList();
        list.add(new JBasicMenuItem(new DeletePluginAction(this)));
        list.add(new JBasicMenuItem(new RenamePluginAction(this)));
        list.add(new JBasicMenuItem(new ExportPluginAction(this)));
        list.add(new JBasicMenuItem(new UpgradePluginAction(this)));
        return (JComponent[])list.toArray(new JComponent[0]);
    }

    @Override
    protected JComponent[] getViewMenuItems()
    {
        ArrayList list = new ArrayList();
        list.add(new JBasicMenuItem(new RefreshPluginAction(this)));

        // Sonic00010517 - I've tightened up the goto behavior such that you
        // can only goto from creatable, i.e. top-level configurations such
        // as the broker, container or cluster.
        String type     = getAttributes().getType();
        String cVersion = getAttributes().getConfigVersion();
        String pVersion = getAttributes().getProductVersion();
        IConfigPluginFactory factory = getPluginContext().getConfigContext().getPluginFactory(type, cVersion, pVersion);

        
        boolean isTemplate = isTemplatePrototype();
        boolean isTemplateInstance = isTemplateInstance();
        
        if ((factory != null) &&
            (factory.getAttributes().isCreatable() ||
             factory.getAttributes().isComponent()))
        {
            // Templates can't be deployed so its not possible to go to the
            // Manage tab equivalent.
            if (!isTemplate)
            {
                list.add(new JBasicMenuItem(new GoToInstanceAction(this)));
            }
            
            // If this plugin isn't wrapping an instance configuration then
            // we don't even bother displaying the menu item.
            if (isTemplateInstance)
            {
                list.add(new JBasicMenuItem(new GoToTemplateAction(this)));
            }
            
        }
        
        if (!(isTemplateInstance || isTemplate)) 
        {
            // Templates and templates instances do NOT support annotations
            addAnnotationMenu(list);
        }
   		

        return (JComponent[])list.toArray(new JComponent[list.size()]);
    }

	protected void addAnnotationMenu(List list) {
        String              type     = getAttributes().getType();
        String              cVersion = getAttributes().getConfigVersion();
        String              pVersion = getAttributes().getProductVersion();

		IAnnotationProvider provider = getPluginContext().getLibrary().getAnnotationProvider(type, cVersion, pVersion);

        if (null != provider)
        {
            list.add(new JBasicMenuItem(new AnnotationPluginAction(this)));
        }
	}
    
    
    

    public void goToInstance()
    {
        AgentManagerConnection amc  = getPluginContext().getConnectionInfo().getAgentManagerConnection();
        String                 path = getPluginPath();

        if (AbstractConfigPlugin.this instanceof AbstractConfigFolderElementPlugin)
        {
            path = ((AbstractConfigFolderElementPlugin)this)._getPluginPath();
        }

        IIdentity[] runtimeIDs = amc.getActiveComponentIdentitiesByConfigID(path);

        if (runtimeIDs.length > 0)
        {
            IIdentity identity = runtimeIDs[0]; // we only support going to the first instance in a list

            String containerRuntimeID = identity.getDomainName() + "." + identity.getContainerName();
            String containerPath  = amc.runtimeIDToConfigName(containerRuntimeID);

            getPluginContext().getWorkspace().goToInstance(containerPath, identity.getCanonicalName(), false);
        }
        else
        {
            getPluginContext().getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                  "There are no active instances of configuration " + getPluginName(), true);
        }
    }

    @Override
    public void goToTemplate()
    {
        IConfigPrototype prototype = ((IConfigBean)((ConfigBeanModel)getModel()).getData()).getPrototype();

        if (prototype != null)
        {
            String path = prototype.getName();

            if (path.endsWith(ConfigServerUtility.DEFAULT_SUFFIX) ||
                path.endsWith(ConfigServerUtility.DOMAIN_DESCRIPTOR_SUFFIX) ||
                path.endsWith(ConfigServerUtility.POLICY_DESCRIPTOR_SUFFIX))
            {
                path = path.substring(0, path.lastIndexOf('/'));
            }

            getPluginContext().getWorkspace().goToConfiguration(path, true);
        }
        else
        {
            getPluginContext().getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                "Template name isn't valid", true);
        }
    }

    @Override
    public void onRenamed(String newName)
    {
        super.onRenamed(newName);

        m_model = null;
     }

    /**
     * This method should be overridden in a subclass that needs to add default
     * attributes into the bean model.
     *
     * @param beanModel  The main plugin's bean model.
     */
    protected void buildModelInternalDefaults(IConfigBean beanModel)
    {
    }

    /**
     * This method is called when the user has entered a new name for this
     * plugin. The plugin receiving this call will perform any additional
     * operations prior to calling into the config server to rename the
     * config element
     * @param configServer  the server with which to perform the rename
     * @param newPath       the fully qualified new name for this plugin
     * @throws Exception    if for some reason the operation cannot be performed
     *                      or completed then an exception will be thrown
     */
    @Override
    public void renamePlugin(IConfigServer configServer, String newPath)
        throws Exception
    {
        // First things first, we check to see that the names aren't the same!!
        //
        if (_getPluginPath().trim().equals(newPath.trim()))
        {
            throw new Exception("Can't rename - new path is same as old path");
        }

        configServer.rename(_getPluginPath(), newPath);
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be deleted. The implementation is responsible for removing any
     * child plugin's and associated data.
     *
     * The plugin(s) are removed from the tree/content pane asynchronously
     * by notification from the Directory Service.
     *
     * @param server      (transaction) config server to use for the delete
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void deletePlugin(IConfigServer configServer)
        throws Exception
    {
        configServer.removeConfigElement(_getPluginPath());
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be copied. The implementation is responsible for creating the
     * new plugin and associated data.
     *
     * In most cases, it is the data that is the underlying data that needs
     * to be created...and this creation kicks off notification events that
     * result in plugins being added to the appropriate views.
     *
     * @param server      (transaction) config server to use for the copy
     * @param path        the new location of the plugin copy
     * @param copyData    implementation-specific data needed to control
     *                    the copy operation, e.g. instance, template copies
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void copyPlugin(IConfigServer server, String path, Object copyData)
        throws Exception
    {
        IConfigBean bean = (IConfigBean)server.loadConfigElement(_getPluginPath());

        if(bean == null)
        {
            throw new Exception("Failed to copy plugin - no configuration data available");
        }

        IConfigBean copyBean = null;
        boolean     convert  = ((copyData != null) && copyData.equals(Boolean.TRUE));

        Helper.logDebugMessage("copying Element '" + path + "'");

        if (convert)
        {
            // we want to convert the type of the config bean. If it
            // is a template create an instance, if an instance create
            // a template
            if (bean instanceof IConfigPrototype)
            {
                copyBean = (IConfigBean)((IConfigPrototype)bean).newInstance(path);
            }
            else
            {
                copyBean = (IConfigBean)bean.createPrototype(path);
            }
        }
        else
        {
            if (bean.isPrototypeInstance())
            {
                copyBean = (IConfigBean)bean.clonePrototypeInstance(path);
            }
            else
            {
                copyBean = (IConfigBean)bean.clone();
                copyBean.setName(path);
            }
        }

        if (copyBean != null)
        {
            copyBean.setMetaAttributes((HashMap)convertMetaAttributes(bean.getMetaAttributes(), convert));

            server.storeConfigElement(copyBean);
        }
    }

    protected Map convertMetaAttributes(Map metaMap, boolean convert)
    {
        if (convert)
        {
            String templateType = (String)metaMap.get(ConfigServerUtility.TEMPLATE_TYPE);

            if (templateType == null)
            {
                metaMap.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE);
            }
            else
            if (templateType.equals(ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE))
            {
                metaMap.put(ConfigServerUtility.TEMPLATE_TYPE, ConfigServerUtility.TEMPLATE_TYPE_INSTANCE);
            }
        }

        return metaMap;
    }

    @Override
    public boolean isDragSupported(int dropAction)
    {
        return false;
    }

    @Override
    public boolean isDropSupported(int dropAction)
    {
        return false;
    }

    @Override
    public void drop(IConfigPlugin dropPlugin, int dropAction)
    {
        Helper.logDebugMessage("Drop '" + dropPlugin + "' into '" + this +"' (" + dropAction + ") - not handled");
    }

    //------------------------------------------------------------------------
    // Recursive routines for automatically cloning all the subbeans under
    // a particular bean
    //

    public class CloneMapper
    {
        private String m_oldPath;
        private String m_newPath;
        private String m_type;
        private boolean m_attr;

        public CloneMapper(String oldPath, String newPath)
        {
            this(oldPath, newPath, null, false);
        }
        public CloneMapper(String  oldPath,
                           String  newPath,
                           String  type,
                           boolean metaAttributesOnFolder)
        {
            m_oldPath = oldPath;
            m_newPath = newPath;
            m_type    = type;
            m_attr    = metaAttributesOnFolder;

            Helper.logDebugMessage("CloneMapper: " + m_oldPath + " > " + m_newPath);
        }

        public String getOldPath() { return m_oldPath; }
        public String getNewPath() { return m_newPath; }
        public String getType() { return m_type; }
        public boolean isAttributesOnParent() { return m_attr; }

        @Override
        public String toString()
        {
            return new StringBuffer(getClass().getName()).append(" {old=").append(m_oldPath).append(", new=").append(m_newPath).append(", type=").append(m_type).append("}").toString();
        }
    }

    protected void cloneBean(boolean       convert,
                             CloneMapper[] map,
                             IAttributeMap oData,
                             IAttributeMap cData)
    {
        cloneBean(new HashMap(), convert, map, oData, cData);
    }

    protected void cloneBean(boolean        convert,
                             CloneMapper[]  map,
                             IAttributeList oData,
                             IAttributeList cData)
    {
        cloneBean(new HashMap(), convert, map, oData, cData);
    }

    private void cloneBean(HashMap       handled,
                           boolean       convert,
                           CloneMapper[] map,
                           IAttributeMap oData,
                           IAttributeMap cData)
    {
        for (Iterator i = oData.keySet().iterator(); i.hasNext(); )
        {
            boolean alreadyHandled = false;
            String attributeName = (String)i.next();

            Object oObj = getAttribute(oData, attributeName);
            Object cObj = getAttribute(cData, attributeName);

            if ((oObj == null) || (cObj == null))
            {
                continue;
            }

            if (oObj instanceof IConfigBean)
            {
                try
                {
                    String      beanName  = ((IConfigBean)oObj).getName();
                    IConfigBean beanValue = (IConfigBean)handled.get(beanName);

                    if (beanValue == null)
                    {
                        // cObj has changed to this new element...lets update
                        // the cloned attribute value so that any other changes
                        // affect the correct, new attribute value.
                        //
                        beanValue = createCloneElement(convert, map, (IConfigBean)oObj);

                        if (beanValue != cObj)
                        {
                            cObj = beanValue;

                            cData.setAttribute(attributeName, beanValue);

                            Helper.logDebugMessage("clone attribute map entry '" + attributeName + "' bean = '" + beanName +
                                               "' type=" + ((oObj instanceof IConfigPrototype) ? "template" : "instance") +
                                               " convert=" + convert);
                        }
                        else
                        {
                            alreadyHandled = true;
                        }

                        handled.put(beanName, cObj);
                    }
                    else
                    {
                        alreadyHandled = true;

                        // Sonic00020052 - if a configuration has already been
                        // handled then it still needs to get set into the clone
                        // BUT only if its a different bean.
                        //
                        // NOTE: This check is necessary because the ConfigLayer
                        //       doesn't do any optimization to filter out setting
                        //       the same value in an attribute!
                        //
                        if (beanValue != cObj)
                        {
                            cData.setAttribute(attributeName, beanValue);
                        }
                    }
                }
                catch (Exception e)
                {
                    Helper.logWarningMessage("Failed to create clone for sub bean - " + e.getMessage());
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // logs the exception
                }
            }

            if (!alreadyHandled)
            {
                if (oObj instanceof IAttributeMap)
                {
                    cloneBean(handled, convert, map, (IAttributeMap)oObj, (IAttributeMap)cObj);
                }
                else
                if (oObj instanceof IAttributeList)
                {
                    cloneBean(handled, convert, map, (IAttributeList)oObj, (IAttributeList)cObj);
                }
            }
        }
    }

    private void cloneBean(HashMap        handled,
                           boolean        convert,
                           CloneMapper[]  map,
                           IAttributeList oData,
                           IAttributeList cData)
    {
        for (int i = 0; i < oData.size(); i++)
        {
            boolean alreadyHandled = false;
            Integer key  = new Integer(i);
            Object  oObj = getAttribute(oData, key);
            Object  cObj = getAttribute(cData, key);

            if ((oObj == null) || (cObj == null))
            {
                continue;
            }

            if (oObj instanceof IConfigBean)
            {
                try
                {
                    String      beanName  = ((IConfigBean)oObj).getName();
                    IConfigBean beanValue = (IConfigBean)handled.get(beanName);

                    if (beanValue == null)
                    {
                        beanValue = createCloneElement(convert, map, (IConfigBean)oObj);

                        if (beanValue != cObj)
                        {
                            cObj = beanValue;

                            cData.set(i, beanValue);

                            Helper.logDebugMessage("clone attribute list item '" + key + "' bean = '" + beanName +
                                               "' type=" + ((oObj instanceof IConfigPrototype) ? "template" : "instance") +
                                               " convert=" + convert);
                        }
                        else
                        {
                            alreadyHandled = true;
                        }

                        handled.put(beanName, beanValue);
                    }
                    else
                    {
                        alreadyHandled = true;

                        // Sonic00020052 - if a configuration has already been
                        // handled then it still needs to get set into the clone
                        // BUT only if its a different bean.
                        //
                        // NOTE: This check is necessary because the ConfigLayer
                        //       doesn't do any optimization to filter out setting
                        //       the same value in an attribute!
                        //
                        if (beanValue != cObj)
                        {
                            cData.set(i, beanValue);
                        }
                    }
                }
                catch (Exception e)
                {
                    Helper.logWarningMessage("Failed to create clone for sub bean - " + e.getMessage());
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Logs the exception
                }
            }

            if (!alreadyHandled)
            {
                if (oObj instanceof IAttributeMap)
                {
                    cloneBean(handled, convert, map, (IAttributeMap)oObj, (IAttributeMap)cObj);
                }
                else
                if (oObj instanceof IAttributeList)
                {
                    cloneBean(handled, convert, map, (IAttributeList)oObj, (IAttributeList)cObj);
                }
            }
        }
    }

    /**
     * Can't assume that get (IAttributeMap/IAttributeList) is going to return
     * null if it can't find anything...the ConfigLayer throws exceptions if
     * referenced entries can't be found...it shouldn't do this really!
     */
    private Object getAttribute(Object attribute, Object key)
    {
        Object res = null;

        try
        {
            if (attribute instanceof IAttributeList)
            {
                res = ((IAttributeList)attribute).getAttribute(((Integer)key).intValue());
            }
            else
            if (attribute instanceof IAttributeMap)
            {
                res = ((IAttributeMap)attribute).getAttribute(key.toString());
            }
            else
            if (attribute == null)
            {
                Helper.logDebugMessage("getAttribute(" + key + ") with null attribute!!!");
            }
        }
        catch (ConfigServiceRuntimeException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);
        }

        return res;
    }

    protected IConfigBean createCloneElement(boolean       convert,
                                             CloneMapper[] map,
                                             IConfigBean   oObj)
        throws Exception
    {
        IConfigBean cObj    = oObj;
        CloneMapper mapper  = findCloneMapper(map, oObj);
        String      newName = generateNameFromOldName(mapper, oObj);

        if (newName != null)
        {
            if (convert)
            {
                if (oObj instanceof IConfigPrototype)
                {
                    cObj = ConfigHelper.createInstanceBean((IConfigPrototype)oObj, newName);
                }
                else
                {
                    cObj = ConfigHelper.createPrototypeBean(oObj, newName);
                }
            }
            else
            {
                cObj = ConfigHelper.copyBean(oObj, newName);
            }

            if (mapper.isAttributesOnParent())
            {
                HashMap metaMap    = cObj.getMetaAttributes();
                String  parentPath = newName.substring(0, newName.lastIndexOf('/'));

                cObj.getConfigServer().setMetaAttributes(parentPath, metaMap);
            }
        }

        return cObj;
    }

    private CloneMapper findCloneMapper(CloneMapper[] map, IConfigBean bean)
    {
        String beanPath = bean.getName();

        for (int i = 0; i < map.length; i++)
        {
            if (beanPath.startsWith(map[i].getOldPath()))
            {
                if (map[i].getType() != null)
                {
                    if (bean.getConfigType().getName().equals(map[i].getType()))
                    {
                        return map[i];
                    }
                }
                else
                {
                    return map[i];
                }
            }
        }

        return null;
    }

    private String generateNameFromOldName(CloneMapper map, IConfigBean bean)
    {
        String beanPath = bean.getName();

        if(map == null)
        {
            return null;
        }

        return map.getNewPath() + "/" + beanPath.substring(beanPath.lastIndexOf('/') + 1);
    }

    //------------------------------------------------------------------------
    //
    // Recursive routines for automatically obtaining a list of all referenced
    // beans under a particular bean
    //
    //------------------------------------------------------------------------

    protected IConfigBean[] getReferencedBeans(IConfigServer server, IAttributeMap data)
    {
        List list = getReferencedBeanList(new ArrayList(), data);

        for (int i = 0; i < list.size(); i++)
        {
            String beanPath = (String)list.get(i);

            try
            {
                list.set(i, server.loadConfigElement(beanPath));
            }
            catch (ConfigServiceException e)
            {
                Helper.logDebugMessage("Failed to getReferencedBeans: " + e.getMessage());
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Logs the error mesage
                list.set(i, null);
            }
        }

        return (IConfigBean[])list.toArray(new IConfigBean[list.size()]);
    }

    protected List getReferencedBeanList(List list, IAttributeMap data)
    {
        for (Iterator iter = data.keySet().iterator(); iter.hasNext(); )
        {
            Object obj = data.get((String) iter.next());

            if (obj instanceof IConfigBean)
            {
                String name = ((IConfigBean) obj).getName();

                if (!list.contains(name))
                {
                    list.add(name);
                }
            }

            if (obj instanceof IAttributeMap)
            {
                getReferencedBeanList(list, (IAttributeMap) obj);
            }
            else if (obj instanceof IAttributeList)
            {
                getReferencedBeanList(list, (IAttributeList) obj);
            }
        }
        return list;
    }

    protected List getReferencedBeanList(List list, IAttributeList data)
    {
        for (Iterator iter = data.iterator(); iter.hasNext(); )
        {
            Object obj = iter.next();

            if (obj instanceof IConfigBean)
            {
                String name = ((IConfigBean) obj).getName();

                if (!list.contains(name))
                {
                    list.add(name);
                }
            }

            if (obj instanceof IAttributeMap)
            {
                getReferencedBeanList(list, (IAttributeMap) obj);
            }
            else if (obj instanceof IAttributeList)
            {
                getReferencedBeanList(list, (IAttributeList) obj);
            }
        }
        return list;
    }
}