/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.config;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;
import javax.swing.JSeparator;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.action.CreateFolderAction;
import com.sonicsw.ma.gui.action.ExportPluginAction;
import com.sonicsw.ma.gui.action.FolderPluginExpandAction;
import com.sonicsw.ma.gui.action.PasteSpecialAction;
import com.sonicsw.ma.gui.action.RefreshPluginAction;
import com.sonicsw.ma.gui.file.CreateFileAction;
import com.sonicsw.ma.gui.file.FileElementPlugin;
import com.sonicsw.ma.gui.file.FileFolderDialog;
import com.sonicsw.ma.gui.file.FileFolderPluginModel;
import com.sonicsw.ma.gui.file.ImportFileAction;
import com.sonicsw.ma.gui.propsheets.JPropSheetDialog;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.IConfigContext;
import com.sonicsw.ma.plugin.IConfigPluginFactory;
import com.sonicsw.ma.plugin.IFolderPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBeanFile;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.IConfigType;
import com.sonicsw.mx.config.util.ConfigHelper;

import com.sonicsw.mf.common.config.IElementIdentity;

public class ConfigFolderPlugin extends AbstractConfigPlugin implements IFolderPlugin
{
    private static final String DISPLAY_TYPE = "Folder";

    private boolean m_preload = false;

    private static final Object[][] ATTRIBUTES =
    {
        { PluginAttributes.DISPLAY_TYPE,     DISPLAY_TYPE },
        { PluginAttributes.IS_CHILD_BEARING, Boolean.TRUE },
    };

    private FileFolderPluginModel m_model = null;

    public ConfigFolderPlugin(IPluginContext context, String id, Map data)
    {
        super(context, id, data);

        getAttributes().put(ATTRIBUTES);
    }

    @Override
    protected IContentPane createReadablePluginContentPane()
    {
        return new ConfigFolderContentPane(this);
    }

    public void importFile(File file, String dsPath)
    throws Exception
    {
        doImportFile(this, file, dsPath);
    }

    public void addFile(final File file, final String fileName, boolean bAsync)
    {
        if (bAsync)
        {
        	final AbstractGUIPlugin plugin = this;

            Thread t = new Thread(new Runnable()
            {
                @Override
                public void run()
                {
                    doAddFile(plugin, file, fileName);
                }
            });

            t.start();
        }
        else
        {
            doAddFile(this, file, fileName);
        }
    }

    @Override
    public BasicAction getDefaultAction()
    {
        return new FolderPluginExpandAction(this);
    }

    @Override
    protected JComponent[] getNewMenuItems()
    {
        JMenu newMenu = new JBasicMenu(new BasicGuiAction("file.new"));

        newMenu.add(new JBasicMenuItem(new CreateFileAction(this)));
        newMenu.add(new JBasicMenuItem(new CreateFolderAction(this)));
        newMenu.add(new JSeparator());
        newMenu.add(new JBasicMenuItem(new NewConfigurationAction(this)));

        return new JComponent[]
        {
            newMenu
        } ;
    }

    @Override
    protected JComponent[] getViewMenuItems()
    {
        return new JComponent[]
        {
            new JBasicMenuItem(new RefreshPluginAction(this)),
        };
    }

    /**
     * In the case where the File tab is not shown we need to add the Import
     * action to the menu...we want to make sure its placed before the Export
     * action...which is what this code does.
     */
    @Override
    protected JComponent[] getSystemMenuItems()
    {
    	JComponent[] menu = super.getSystemMenuItems();
 		ArrayList    list = new ArrayList();

   		for (int i = 0; i < menu.length; i++)
   		{
   			if ((menu[i] != null) && (menu[i] instanceof JMenuItem) &&
			    (((JMenuItem)menu[i]).getAction() instanceof ExportPluginAction))
   			{
                list.add(new JBasicMenuItem(new ImportFileAction(this)));
  			}

   			list.add(menu[i]);
        }

        return (JComponent[])list.toArray(new JComponent[0]);
    }

    @Override
    protected JComponent[] getEditMenuItems()
    {
        JComponent[] c = super.getEditMenuItems();
        ArrayList list = new ArrayList(Arrays.asList(super.getEditMenuItems()));

        if (MgmtConsole.DEVELOPER_MODE)
        {
            Object clipObject = MgmtConsole.getMgmtConsole().getClipboardObject();

            if ((clipObject != null) && (clipObject instanceof AbstractConfigPlugin) &&
               ((AbstractConfigPlugin)clipObject).isTemplatePrototype())
            {
                JMenu tMenu = new JMenu("Create From Template");
                tMenu.add(new JBasicMenuItem(new CreateInstanceAction(5)));
                tMenu.add(new JBasicMenuItem(new CreateInstanceAction(10)));
                tMenu.add(new JBasicMenuItem(new CreateInstanceAction(15)));
                tMenu.add(new JBasicMenuItem(new CreateInstanceAction(20)));
                list.add(tMenu);
            }
        }
        return (JComponent[])list.toArray(new JComponent[list.size()]);
    }

    class CreateInstanceAction extends BasicGuiAction
    {
        int num;

        public CreateInstanceAction(int num)
        {
            super("sdf");
            
            putNameValue();
            this.num = num;
        }

        private void putNameValue() {
            putValue(Action.NAME, (String)(num + " instances"));
        }
        
        @Override
        public void actionPerformed(java.awt.event.ActionEvent evt)
        {
            for (int i = 0; i < num; i++)
            {
                PasteSpecialAction.pasteConfigElement(ConfigFolderPlugin.this, true);
            }
        }
    }

    @Override
    public JPropSheetDialog getPropSheetDialog()
    {
        return new FileFolderDialog(MgmtConsole.getMgmtConsole(), getPluginName());
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be deleted. The implementation is responsible for removing any
     * child plugin's and associated data.
     *
     * The plugin(s) are removed from the tree/content pane asynchronously
     * by notification from the Directory Service.
     *
     * @param configServer      (transaction) config server to use for the delete
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void deletePlugin(IConfigServer configServer)
        throws Exception
    {
        configServer.deleteFolder(getPluginPath());
    }

    @Override
    public void refresh()
    {
        final int childCount = getChildCount();

        for (int i = 0; i < childCount; i++) {
            if (getChildAt(i) instanceof FileElementPlugin) {
                final FileElementPlugin fileElementPlugin = (FileElementPlugin) getChildAt(i);
                fileElementPlugin.refresh();
            }
        }

        super.refresh();
    }

    //-------------------------------------------------------------------------
    //
    // Model Manipulation
    //
    //-------------------------------------------------------------------------

    /**
     * Creates a valid model for this plugin. The model is location is determined
     * by the plugin's path.
     *
     * @param isTemplate  Indicates whether or not a templated configuration is
     *                    required - if so a prototype model is created
     * @return            A valid configuration model or null if the model
     *                    could not be created
     */
    public Object createModel(boolean isTemplate)
    {
        try
        {
            IConfigServer server = getPluginContext().getConfigContext().getConfigServer();
            String        path   = getPluginPath();

            // Delay doing this until ConfigLayer forces you to create the
            // intermediate folders...
            //
            // server.createDirectory(path);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                          "Failed to create model: " + getPluginPath(), e, false);
        }

        return null;
    }

    /**
     * Method simply loads an existing model using the plugin's path as the
     * model's location.
     *
     * @return  The model or null if the configuration could not be found at
     *          the specified location.
     */
    @Override
    public Object getModel()
    {
        if (m_model == null)
        {
            m_model = new FileFolderPluginModel(this);
        }

        return m_model;
    }

    /**
     * Returns the type information for the configuration model.
     *
     * @return  The configuration type for the model or null if the type could
     *          not be found
     */
    @Override
    protected IConfigType getModelMetaData()
    {
        return null;
    }

    //-------------------------------------------------------------------------

    protected final void setPreloadElements(boolean preload)
    {
        m_preload = preload;
    }

    protected Set loadConfigElements()
    throws ConfigServiceException
    {
        IConfigServer server = getPluginContext().getConfigContext().getConfigServer();

        String path = _getPluginPath();

        if(!((com.sonicsw.mx.config.impl.ConfigServer)server)._pathExists(path))
        {
            return Collections.EMPTY_SET;
        }

        return server.loadConfigElements(path);
    }

    protected Set metaAttributeSet()
    throws ConfigServiceException
    {
        IConfigServer server = getPluginContext().getConfigContext().getConfigServer();

        String path = _getPluginPath();

        if(!((com.sonicsw.mx.config.impl.ConfigServer)server)._pathExists(path))
        {
            return Collections.EMPTY_SET;
        }

        return server.list(path);
    }

    @Override
    public boolean expand()
    {
    	boolean res = true;
        JProgressBar bar = MgmtConsole.getMgmtConsole().getStatusBar().getProgressBar();

        try
        {
            Set elements = null;

            if(m_preload)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_PROGRESS,
                    "Preloading all config elements", null, false);

                elements = loadConfigElements();
            }

            Set childSet = metaAttributeSet();
            Iterator i = (childSet != null) ? childSet.iterator() : null;
            int nPluginCount = (childSet != null) ? childSet.size() : 0;
            int nCount = 0;

            if (bar != null)
            {
                bar.setMinimum(0);
                bar.setMaximum(nPluginCount);
                bar.setVisible(true);
            }


            List<IPlugin> plugins = new ArrayList<IPlugin>();
            while ((i != null) && i.hasNext())
            {
                // Now try and create the child plugin...
                HashMap row = (HashMap)i.next();
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_PROGRESS,
                    "Loading config element " + ++nCount + " of " + nPluginCount, null, false);
                if (bar != null)
                {
                    bar.setValue(nCount);
                }

                IPlugin plugin = createChildPlugin(null, row);
                plugins.add(plugin);
                if(m_preload && elements != null && !elements.isEmpty())
                {
                    plugin.getModel();
                }
            }

            setChildPlugins(plugins);

        }
        catch (Exception e)
        {
        	// Reparent to get the dummy node back in...
        	setParent((javax.swing.tree.DefaultMutableTreeNode)getParent());

        	// And tell the caller that we failed
        	res = false;

            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                "ConfigFolderPlugin::expand failed", e, false);
        }
        finally
        {
            if (bar != null)
            {
                bar.setVisible(false);
                MgmtConsole.getMgmtConsole().getStatusBar().releaseProgressBar();
            }
        }
        return res;
    }

    @Override
    protected IPlugin createChildPlugin(String strName, Map metaAttributes)
    {
        IPlugin plugin   = null;
        String  metaName = null;
        boolean isFolder = false;

        try
        {
            if (metaAttributes.containsKey(IConfigServer.FOLDER_NAME))
            {
                String folderName = (String) metaAttributes.remove(IConfigServer.FOLDER_NAME);

                int folderIndex = folderName.lastIndexOf('/');
                metaName = folderName.substring(folderIndex + 1);

                isFolder = true;
            }
            else if (metaAttributes.containsKey(IConfigServer.ELEMENT_IDENTITY))
            {
                IElementIdentity eId = (IElementIdentity) metaAttributes.remove(IConfigServer.ELEMENT_IDENTITY);
                String[] comp = eId.getNameComponents();

                metaName = comp[comp.length - 1];
            }
            else
            {
                return null;
            }

            // Now try and create the child plugin...
            //
            if (metaAttributes.containsKey(ConfigServerUtility.TYPE))
            {
                String type     = (String) metaAttributes.get(ConfigServerUtility.TYPE);

                //right now the root node is hard coded and the domain properties have been attached to that hard coded treenode
                //There were changes done to the esb seed data so that the MF_DOMAIN element appears in the logical namespace in the DS
                //Its just a hack (temporary fix) to skip locating a plugin factory for this configuration
                if(type !=null && type.equals("MF_DOMAIN"))
                {
                    return null;
                }
                String cVersion = (String) metaAttributes.get(ConfigServerUtility.CONFIG_VERSION);
                String pVersion = (String) metaAttributes.get(ConfigServerUtility.PRODUCT_VERSION);

                if ((type != null) && type.equals(FileElementPlugin.PLUGIN_TYPE))
                {
                    plugin = new FileElementPlugin(getPluginContext(), metaName, metaAttributes);
                }
                else
                if(cVersion != null || pVersion != null)
                {
                    IConfigContext context = getPluginContext().getConfigContext();
                    IConfigPluginFactory factory = context.getPluginFactory(type, cVersion, pVersion);

                    if (factory != null)
                    {
                        Helper.logDebugMessage("ConfigFolderPlugin::createChildPlugin(" + metaName + " " + type + " c=" + cVersion + " p=" + pVersion + ")");
                        plugin = (IPlugin) factory.createPlugin(getPluginContext(), metaName, pVersion, metaAttributes);
                    }
                    else
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                            "Failed to lookup config factory for " + type + " c=" + cVersion + " p=" + pVersion,
                            false);
                    }
                }
            }
            else if (isFolder)
            {
                plugin = new ConfigFolderPlugin(getPluginContext(), metaName, metaAttributes);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                "ConfigFolderPlugin::createChildPlugin failed", e, false);
        }
        return plugin;
    }
    
    //------------------------------------------------------------------------
    //
    // File Import/Export
    //
    //------------------------------------------------------------------------
    
    public void doImportFile(AbstractGUIPlugin plugin, File file, String dsPath)
    throws Exception
    {
        boolean overwrite = plugin.pathExists(dsPath);

        // if we are overwriting and the file already exists in DS...
        //
        if (overwrite)
        {
            int nOverwrite = JOptionPane.showConfirmDialog(MgmtConsole.getMgmtConsole(),
                                      "The file " + dsPath + " already exists.  Do you want to overwrite the existing file?",
                                      "Verify File", JOptionPane.YES_NO_OPTION);

            if (nOverwrite != JOptionPane.YES_OPTION)
            {
                return;
            }
        }
        
        IConfigBeanFile bean = null;
        JWaitCursor     wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.getMgmtConsole(),
                                                   MgmtConsole.MESSAGE_STATUS,
                                                   (String)(overwrite ? "Overwriting " : "Creating ") + dsPath + "...",
                                                   (Throwable)null,
                                                   false);
        try
        {
            IConfigServer server = plugin.getPluginContext().getConfigContext().getConfigServer();
            String        user   = plugin.getPluginContext().getConnectionInfo().getUserName();

            if (file != null)
            {
                ConfigHelper.importFile(server, dsPath, file.getAbsolutePath(), user, true);
            }
            else
            {
                ConfigHelper.createFile(server, dsPath, user);
            }
        }
        finally
        {
            wc.release();
        }
    }

    public void doAddFile(AbstractGUIPlugin plugin, final File file, final String fileName)
    {
        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());
        String strPathName = buildFilePath(plugin.getPluginPath(), fileName);

        try
        {
            doImportFile(plugin, file, strPathName);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                new StringBuffer("Failure importing ").append(strPathName).toString(), e, true);
        }
        finally
        {
            wc.release();
        }
    }
    
    public String buildFilePath(String path, String file)
    {
        StringBuffer buffer = new StringBuffer(path);

        if(buffer.charAt(buffer.length()-1) != com.sonicsw.mf.common.config.IMFDirectories.MF_DIR_SEPARATOR)
        {
            buffer.append(com.sonicsw.mf.common.config.IMFDirectories.MF_DIR_SEPARATOR);
        }

        buffer.append(file);

        return buffer.toString();
    }

}
