/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.config;

import java.awt.Component;
import java.awt.Font;
import java.text.Collator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.UIManager;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.IPluginFactory;
import com.sonicsw.ma.plugin.IPluginFilter;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.config.IElementIdentity;

public class JLightConfigTree extends JTree
{
    private IPluginContext m_context   = null;
    private IPluginFilter  m_filter    = null;
    private String         m_rootLabel = null;
    private Icon           m_rootIcon  = null;

    public JLightConfigTree(IPluginContext context,
                            IConfigServer  server,
                            String         rootPath)
    {
        this(context, server, rootPath, null);
    }

    public JLightConfigTree(IPluginContext context,
                            IConfigServer  server,
                            String         rootPath,
                            IPluginFilter  filter)
    {
        super(new LightConfigTreeNode(rootPath));

        m_context = context;
        m_filter  = filter;
        prepareJLightConfigTree();
    }
    private void prepareJLightConfigTree() {
        
        setCellRenderer(new LightConfigTreeCellRenderer());

        ((LightConfigTreeNode)getModel().getRoot()).add(new DefaultMutableTreeNode(Boolean.TRUE));

        addTreeExpansionListener(new TreeExpansionListener()
        {
            @Override
            public void treeCollapsed(TreeExpansionEvent evt)
            {
                LightConfigTreeNode node = (LightConfigTreeNode)evt.getPath().getLastPathComponent();

                if (node != getModel().getRoot())
                {
                    node.removeAllChildren();
                    node.add(new DefaultMutableTreeNode(Boolean.TRUE));
                    ((DefaultTreeModel)getModel()).nodeStructureChanged(node);
                }
            }

            @Override
            public void treeExpanded(TreeExpansionEvent evt)
            {
                LightConfigTreeNode node = (LightConfigTreeNode)evt.getPath().getLastPathComponent();

                node.removeAllChildren();

                build(m_context.getConfigContext().getConfigServer(), m_filter, node);

                ((DefaultTreeModel)getModel()).nodeStructureChanged(node);
            }
        });
    }

    private DefaultMutableTreeNode build(IConfigServer       server,
                                         IPluginFilter       filter,
                                         LightConfigTreeNode parent)
    {
        try
        {
            HashMap  parentMap  = (HashMap)parent.getUserObject();
            String   parentPath = (String)parentMap.get(IConfigServer.FOLDER_NAME);
            Set      children   = server.list(parentPath);
            Iterator i          = children.iterator();

            while (i.hasNext())
            {
                HashMap cma = (HashMap)i.next();

                if ((filter == null) || ((filter != null) && filter.isPluginVisible(this, cma)))
                {
                    LightConfigTreeNode child = new LightConfigTreeNode(cma);
                    parent.insert(child, parent.getInsertIndex(child));

                    if (cma.containsKey(IConfigServer.FOLDER_NAME) &&
                        !cma.containsKey(ConfigServerUtility.TYPE))
                    {
                        child.add(new DefaultMutableTreeNode(Boolean.TRUE));
                    }
                }
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return parent;
    }

    protected void rebuildTree(IConfigServer server, IPluginFilter filter)
    {
       m_filter = filter;

       LightConfigTreeNode rootNode = (LightConfigTreeNode)getModel().getRoot();
       rootNode.removeAllChildren();
       build(server, filter, rootNode);

       ((DefaultTreeModel)getModel()).reload(rootNode);
    }

    public boolean isNodeFolder(DefaultMutableTreeNode node)
    {
        HashMap map = (HashMap)node.getUserObject();

        return map.containsKey(IConfigServer.FOLDER_NAME);
    }

    public boolean isNodeTyped(DefaultMutableTreeNode node)
    {
        HashMap map = (HashMap)node.getUserObject();

        return map.containsKey(ConfigServerUtility.TYPE);
    }

    public void setRootLabel(String rootLabel)
    {
        m_rootLabel = rootLabel;

        ((DefaultTreeModel)getModel()).nodeChanged((TreeNode)getModel().getRoot());
    }

    public void setRootIcon(Icon rootIcon)
    {
        m_rootIcon = rootIcon;

        ((DefaultTreeModel)getModel()).nodeChanged((TreeNode)getModel().getRoot());
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    static class LightConfigTreeNode extends DefaultMutableTreeNode
    {
        public LightConfigTreeNode(String folderName)
        {
            super();
            setUserObject(folderName);
        }
        
        private void setUserObject(String folderName) {
            HashMap map = new HashMap();

            map.put(IConfigServer.FOLDER_NAME, folderName);

            setUserObject(map);
        }

        public LightConfigTreeNode(HashMap map)
        {
            super(map);
        }

        public int getInsertIndex(LightConfigTreeNode treeNode)
        {
            String treeNodeName = treeNode.toString();

            for (int i = 0; i < super.getChildCount(); i++)
            {
                TreeNode node = getChildAt(i);
                String aName = null;

                if (Collator.getInstance().compare(treeNodeName, node.toString()) <= 0)
                {
                    return i;
                }
            }

            return super.getChildCount();
        }

        @Override
        public String toString()
        {
            HashMap map = (HashMap)getUserObject();

            if (map.containsKey(IConfigServer.FOLDER_NAME))
            {
                return (String)map.get(IConfigServer.FOLDER_NAME);
            }

            return (String)((IElementIdentity)map.get(IConfigServer.ELEMENT_IDENTITY)).getName();
        }
    }

    class LightConfigTreeCellRenderer extends DefaultTreeCellRenderer
    {
        private Font highlightFont = null;

        @Override
        public Component getTreeCellRendererComponent(JTree   tree,
                                                      Object  value,
                                                      boolean selected,
                                                      boolean expanded,
                                                      boolean isLeaf,
                                                      int     rowNum,
                                                      boolean hasFocus)
        {
            JLabel  res = (JLabel)super.getTreeCellRendererComponent(tree, value, selected, expanded, isLeaf, rowNum, hasFocus);
            DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
            Object  obj = node.getUserObject();
            HashMap map = (obj instanceof HashMap) ? (HashMap)node.getUserObject() : new HashMap();

            String type     = (String)map.get(ConfigServerUtility.TYPE);
            String cVersion = (String)map.get(ConfigServerUtility.CONFIG_VERSION);
            String pVersion = (String)map.get(ConfigServerUtility.PRODUCT_VERSION);

            Class objClass = this.getClass();

            if(type != null)
            {
                IPluginFactory factory = m_context.getConfigContext().
                    getPluginFactory(type, cVersion, pVersion);
                if(factory != null)
                {
                    objClass = factory.getClass();
                }
            }

            Icon icon = (type != null) ?
                        ResourceManager.getTypeIcon(objClass, type) :
                        UIManager.getIcon(selected ? "Tree.openIcon" : "Tree.closedIcon");

            String templateType = (String)map.get(ConfigServerUtility.TEMPLATE_TYPE);
            String name = node.toString();
            Font font = tree.getFont();

            if ((templateType != null) &&
                templateType.equalsIgnoreCase(ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE))
            {
                if (highlightFont == null)
                {
                    highlightFont = tree.getFont().deriveFont(Font.ITALIC | Font.BOLD);
                }

                font = highlightFont;
            }

            if (node.isRoot())
            {
                if (m_rootLabel != null)
                {
                    name = m_rootLabel;
                }

                if (m_rootIcon != null)
                {
                    icon = m_rootIcon;
                }
            }

            res.setFont(font);
            res.setIcon(icon);
            res.setText(name.substring(name.lastIndexOf('/') + 1));

            return res;
        }
    }

}
