/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.config;

import java.awt.event.ActionEvent;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JWorkspacePanel;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.action.PastePluginAction;
import com.sonicsw.ma.gui.config.propsheets.JConfigDialog;
import com.sonicsw.ma.gui.util.BasicPluginAction;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.SwingWorker;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IConfigPluginFactory;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginFactory;
import com.sonicsw.mx.config.ConfigServerFactory;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.IDirectoryFileSystemService;

/**
 * Class to manage the New Configuration menu item
 */
public class NewConfigurationAction extends BasicPluginAction
{
    private IConfigPluginFactory m_factory;
    private String               m_version;

    /**
     * Default constructor.
     *
     * @param plugin The plugin associated with this action
     */
    public NewConfigurationAction(IPlugin plugin)
    {
        this(plugin, null, null, "file.new.configuration");
    }

    public NewConfigurationAction(IPlugin              plugin,
                                  IConfigPluginFactory factory,
                                  String               version,
                                  String               resourceName)
    {
        super(plugin, resourceName);

        m_factory = factory;
        m_version = version;
    }

    /**
     * Overridden method to handle the action event for this menu item.
     *
     * @param event The action event that occurred for this menu item action
     */
    @Override
    public void actionPerformed(ActionEvent event)
    {
        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            if (m_factory != null)
            {
                displayPropSheet(m_factory, m_version, false);
            }
            else
            {
                // Display the configuration and template choices in a dialog.
                NewConfigurationDialog dialog = new NewConfigurationDialog(getPlugin());
                dialog.setVisible(true);

                if (dialog.getCloseCommand() == JMADialog.CLOSE_OK)
                {
                    if (dialog.isConfigActiveTab()) // create a new (normal/template) configuration
                    {
                        Object[] obj = (Object[])dialog.getSelection();

                        if (obj != null)
                        {
                            // Retrieve the factory and the specific product version
                            // that the user wants to create
                            IConfigPluginFactory factory = (IConfigPluginFactory)obj[0];
                            String productVersion = (String)obj[1];

                            if (factory != null && productVersion != null)
                            {
                                displayPropSheet(factory, productVersion, dialog.getIsTemplate());
                            }
                        }
                    }
                    else  // create a new instance (of a template) configuration
                    {
                        displayPropSheet((String)dialog.getSelection());
                    }
                }
            }
        }
        finally
        {
            wc.release();
        }
    }

    @Override
    public boolean isEnabled()
    {
        return getPlugin().getPluginContext().getConnectionInfo().isConnected();
    }

    /**
     * Method to display a property sheet for creating a normal/template
     * configuration plugin.
     *
     * @param factory The plugin factory associated with the property sheet
     * @param pVersion The product version to use for the configuration
     * @param isTemplate Should we create a template or a normal configuration
     */
    protected void displayPropSheet(final IPluginFactory factory,
                                    final String         pVersion,
                                    final boolean        isTemplate)
    {
        IConfigServer server = null;

        try
        {
            // Template operations have NOT been made transactionable
            // in the ConfigLayer so we have to use a regular ConfigServer :(
            //
            IDirectoryFileSystemService ds = getContext().getConnectionInfo().getDirectoryService();

            server = ConfigServerFactory.connectTransactedConfigServer(ds);

            String name = PastePluginAction.getValidPath(server,
                                           ((AbstractGUIPlugin)getPlugin())._getPluginPath(),
                                           factory.getAttributes().getDisplayType(),
                                           PastePluginAction.FORMAT_NEW);

            name = name.substring(name.lastIndexOf('/')+1);

            // This plugin is temporary...we don't add it into the tree - instead
            // we set a dummy parent path and throw it away once it goes out of
            // scope...
            //
            IConfigPlugin plugin = (IConfigPlugin)factory.createPlugin(getContext(), name, pVersion, null);

            ((AbstractConfigPlugin)plugin).setDummyParentPath(getPlugin().getPluginPath());

            // Create the underlying data model using config server...
            //
            ConfigBeanModel model  = (ConfigBeanModel)plugin.createModel(server, isTemplate);
            JConfigDialog   dialog = (JConfigDialog)plugin.getPropSheetDialog();

            dialog.editInstance(plugin, model, true);
            dialog.setVisible(true);

            if (dialog.getCloseCommand() == JMADialog.CLOSE_CANCEL)
             {
                plugin.dispose();  // We need to cleanup the plugin and the model it wraps
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.
                MESSAGE_ERROR, "Failed to create configuration", e, true);
        }
        finally
        {
            if(server != null)
            {
                try { 
                    server.close(); 
                } catch(Exception e) {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }
        }
    }

    /**
     * Method to display a property sheet for creating an instance (of a template)
     * configuration plugin.
     *
     * @param templatePath   The path to the template configuration from which
     *                       we need to create a new instance configuration
     */
    protected void displayPropSheet(final String templatePath)
    {
        try
        {
            WorkspaceWindow workspace = getContext().getWorkspace();
            JWorkspacePanel panel = workspace.getSelectedWorkspacePanel();
            boolean oldMode = panel.getTreeExpansionMode();
            panel.setTreeExpansionMode(false);
            IConfigPlugin   srcPlugin = (IConfigPlugin)workspace.goToConfiguration(templatePath, true, false);
            panel.setTreeExpansionMode(oldMode);

            (new NewInstanceWorker(workspace, srcPlugin, (IConfigPlugin)getPlugin())).start();
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                         "Failed to create instance of template", e, true);
        }
    }

    private void logDebugMessage(String value)
    {
        // Helper.logDebugMessage("NewInstanceWorker::" + value);
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class NewInstanceWorker extends SwingWorker
    {
        boolean       m_cancelled = false;
        WorkspaceWindow m_workspace;
        IConfigServer m_server;
        IConfigPlugin m_srcPlugin;
        IConfigPlugin m_dstPlugin;
        String        m_newPath;
        JWaitCursor   m_wc;

        public NewInstanceWorker(WorkspaceWindow workspace,
                                 IConfigPlugin   srcPlugin,
                                 IConfigPlugin   dstPlugin)
            throws ConfigServiceException
        {
            logDebugMessage("NewInstanceWorker::init");

            m_workspace = workspace;
            m_srcPlugin = srcPlugin;
            m_dstPlugin = dstPlugin;

            connect();
        }
        
        @Override
        public Object construct()
        {
            logDebugMessage("NewInstanceWorker::construct: " + m_newPath);

            Helper.invoke(new Runnable()
            {
                @Override
                public void run()
                {
                    m_wc = new JWaitCursor(MgmtConsole.getMgmtConsole());
                }
            });

            Exception error = null;

            try
            {
                synchronized (m_server)
                {
                    m_srcPlugin.copyPlugin(m_server, m_newPath, Boolean.TRUE);
                }

                if ((m_server != null) && m_server.isTransacted())
                {
                    m_server.commit();
                }
            }
            catch (Exception e)
            {
                error = e;
            }
            finally
            {
            }
            return error;
        }

        @Override
        public void finished()
        {
            logDebugMessage("finished: " + m_cancelled);

            Exception error = (Exception)getValue();

            if (!m_cancelled && (error == null))
            {
                IConfigServer server = null;

                try
                {
                    IConfigPluginFactory factory =
                        m_srcPlugin.getPluginContext().getLibrary().getChildConfigPluginFactory(
                            m_srcPlugin.getAttributes().getType(),
                            m_srcPlugin.getAttributes().getConfigVersion(),
                            m_srcPlugin.getAttributes().getProductVersion());
                    IConfigPlugin newPlugin =
                        (IConfigPlugin)factory.createPlugin(m_srcPlugin.getPluginContext(),
                            m_newPath.substring(m_newPath.lastIndexOf('/') + 1),
                            m_srcPlugin.getAttributes().getProductVersion(), null);

                    ((AbstractConfigPlugin)newPlugin).setDummyParentPath(m_newPath.substring(0, m_newPath.lastIndexOf('/')));

                    server = m_srcPlugin.getPluginContext().getConfigContext().getConfigServer().connectTransacted();

                    IConfigBean     mainBean  = (IConfigBean)((ConfigBeanModel)newPlugin.getModel()).getData();
                    IConfigBean     txnBean   = (IConfigBean)server.loadConfigElement(mainBean.getName());
                    ConfigBeanModel txnModel  = new ConfigBeanModel(txnBean);
                    JConfigDialog   dialog    = (JConfigDialog)newPlugin.getPropSheetDialog();

                    dialog.editInstance(newPlugin, txnModel, true);
                    dialog.setVisible(true);
                }
                catch (Exception e)
                {
                    error = e;
                }
                finally
                {
                    if(server != null)
                    {
                        try { 
                            server.close(); 
                        } catch(Exception e) { 
                            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                        }
                    }
                }
            }

            if (error != null)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                "Failed to create instance of template", error, true);
            }

            disconnect();
        }

        @Override
        public void interrupt()
        {
            m_cancelled = true;

            logDebugMessage("interrupt");

            super.interrupt();
        }

        public final void connect() throws ConfigServiceException
        {
            logDebugMessage("connect");

            String srcPath = m_srcPlugin.getPluginPath();
            String name    = srcPath.substring(srcPath.lastIndexOf('/')+1);

            m_server = m_srcPlugin.getPluginContext().getConfigContext().getConfigServer().connectTransacted();
            m_newPath = PastePluginAction.getValidPath(m_server, m_dstPlugin.getPluginPath(), name, PastePluginAction.FORMAT_NEW_INSTANCE);
        }

        public void disconnect()
        {
            logDebugMessage("disconnect");

            if(m_server != null)
            {
                try { 
                    m_server.close(); 
                } catch(Exception e) { 
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }
            m_server = null;

            if (m_wc != null)
            {
                m_wc.release();
                m_wc = null;
            }
        }
    }
}
