/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.config;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTabbedPane;
import javax.swing.JTree;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.ExtendedImageIcon;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.JButtonPanel;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.gui.util.SwingWorker;
import com.sonicsw.ma.plugin.ConfigFactoryInfo;
import com.sonicsw.ma.plugin.IConfigPluginFactory;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.IPluginFactory;
import com.sonicsw.ma.plugin.IPluginFilter;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.common.config.IElementIdentity;

/**
 *
 * <p>Title: NewConfigurationDialog</p>
 * <p>Description: Represents a dialog for the creation of new configurations.</p>
 * <p>Copyright: Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.</p>
 */
public class NewConfigurationDialog extends    JMADialog
                                    implements ChangeListener, TreeSelectionListener
{
    protected IPlugin      m_plugin;
    protected JTabbedPane  m_tabbedPane;
    protected JTree        m_configTree;
    protected JRadioButton m_configRadioButton;
    protected JRadioButton m_templateRadioButton;
    protected JComboBox    m_versionCombo;
    protected List         m_config;
    protected List         m_template;
    protected JLightConfigTree m_tree;
    private boolean        m_bInitialized = false;

    /**
     * Constructor to create the dialog.
     *
     * @param component The component to use to display the choices
     */
    public NewConfigurationDialog(IPlugin plugin)
    {
        super(MgmtConsole.getMgmtConsole(), "new.configuration");

        m_plugin = plugin;
    }

    /**
     * Overridden method to initialize the dialog and create its' tabs.
     */
    @Override
    protected void maInitialize()
    {
        if (!m_bInitialized)
        {
            m_tabbedPane = new JTabbedPane();
            m_tabbedPane.addChangeListener(this);
            m_tabbedPane.setBorder(new javax.swing.border.EmptyBorder(JPartitionPanel.BORDER_SIZE,
                JPartitionPanel.BORDER_SIZE, 0, JPartitionPanel.BORDER_SIZE));
            getContentPane().add(m_tabbedPane);

            m_tabbedPane.add(createConfigTab(), ResourceManager.getString(getClass(),
                "NewConfigurationDialog.ConfigurationsTab"));
            m_tabbedPane.add(createConfigFromTemplateTab(), ResourceManager.getString(getClass(),
                "NewConfigurationDialog.TemplatesTab"));

            // Overridden behavior to prevent the button panel from being added to the dialog
            // until the other components are added, so that focus traversal works correcty.
            super.getContentPane().add(createButtonPanel(getButtonActions()), BorderLayout.SOUTH);
            m_bInitialized = true;
        }
    }

    /**
     * Overridden method to cleanup after the dialog is closed.
     */
    @Override
    protected void maCleanup()
    {
    }

    /**
     * Method to create the Configurations tab.
     *
     * @return The Configuration tab panel.
     */
    private JPanel createConfigTab()
    {
        JPartitionPanel outerPanel = new JPartitionPanel(true,"p,r",null,0,0,0);

        ButtonGroup buttonGroup = new ButtonGroup();
        JPartitionPanel groupBoxPanel = new JPartitionPanel(false,"p,p", ResourceManager.getString(getClass(),"NewConfigurationDialog.ConfigGroupBox"));
        outerPanel.add(groupBoxPanel);

        m_configRadioButton = new JRadioButton(ResourceManager.getString(getClass(),"NewConfigurationDialog.ConfigRadioButton"));
        groupBoxPanel.add(m_configRadioButton);
        buttonGroup.add(m_configRadioButton);

        m_templateRadioButton = new JRadioButton(ResourceManager.getString(getClass(),"NewConfigurationDialog.TemplateRadioButton"));
        groupBoxPanel.add(m_templateRadioButton);
        buttonGroup.add(m_templateRadioButton);

        ItemListener itemListener = new ConfigItemListener();
        m_configRadioButton.addItemListener(itemListener);
        m_templateRadioButton.addItemListener(itemListener);

        JPartitionPanel listPanel = new JPartitionPanel(true, "p,r,p", null);
        outerPanel.add(listPanel);

        listPanel.add(new JLabel(ResourceManager.getString(getClass(), "NewConfigurationDialog.SelectConfigLabel")));

        createConfigTree();
        listPanel.add(new ExtendedJScrollPane(m_configTree));

        m_versionCombo = new JComboBox();
        JPartitionPanel versionPanel = new JPartitionPanel(false, "p, r", null);
        versionPanel.addRow("Product Version:", m_versionCombo);

        listPanel.add(versionPanel);

        return outerPanel;
    }

    private void createConfigTree()
    {
        m_configTree = new JTree(new DefaultTreeModel(new DefaultMutableTreeNode()));
        m_configTree.setRootVisible(false);
        m_configTree.setShowsRootHandles(true);
        m_configTree.setCellRenderer(new NewConfigTreeCellRenderer());
        m_configTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        m_configTree.addTreeSelectionListener(this);
        m_configTree.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(MouseEvent evt)
            {
                if (evt.getClickCount() == 2)
                {
                    Action action = getDefaultOKAction();
                    if (action != null)
                    {
                        JButton button = ((JButtonPanel)getButtonPanel()).getButton((String)action.getValue(Action.ACTION_COMMAND_KEY));

                        if (button != null)
                        {
                            button.doClick();
                        }
                    }
                }
            }
        });

        SwingWorker sw = new SwingWorker()
        {
            @Override
            public Object construct()
            {
                JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

                try
                {
                    // Generate a list of plugin factories that belong to the New Configuration
                    // menu item for the selected plugin.
                    m_config = getConfigList(false);
                    m_template = getConfigList(true);
                }
                finally
                {
                    wc.release();
                }

                return null;
            }

            @Override
            public void finished()
            {
                if (m_configRadioButton != null)
                {
                    m_configRadioButton.setSelected(true);
                }
            }
        };
        sw.start();

        getDefaultOKAction().setEnabled(false);
    }
    
    /**
     * Overridden method to prevent the button panel from being added to the dialog
     * until the other components are added, so that focus traversal works correcty.
     *
     * @param action Array of actions for the buttons
     * @return The button panel
     */
    @Override
    protected JPanel createButtonPanel(Action[] action)
    {
        return (m_tabbedPane != null) ? super.createButtonPanel(action) : null;
    }

    class ConfigItemListener implements ItemListener
    {
        // When the user switches from normal config creation to
        // template config creation we must re-build the tree list
        // to restrict allowable configuration types.
        //
        // Because we are re-building we have to save the expanded
        // and selection state of the tree and then re-set it
        // after the re-build - not optimal but sufficient for
        // this limited types tree.
        //
        @Override
        public void itemStateChanged(ItemEvent evt)
        {
            if (evt.getStateChange() != ItemEvent.SELECTED)
            {
                return;
            }

            String[] expanded = getExpandedProducts(m_configTree);
            String[] selPath  = getSelectedNodes(m_configTree);

            rebuildTreeModel((DefaultTreeModel)m_configTree.getModel(), (evt.getItem() == m_configRadioButton) ? m_config : m_template);
            
            setExpandedProducts(m_configTree, expanded);
            setSelectedNodes(m_configTree, selPath);
        }

        // Returns a string array representing the tree's
        // selection - should be a single entry but written
        // to support multiple.
        // The string has a '.' separated format with "<root>"
        // representing the root node...again, sufficient for
        // the task at hand...ensured that this will not clash
        // with Config Type names.
        //
        // Note: we store the selection paths as Strings rather
        //       than TreePath's because we are rebuilding the
        //       tree...TreePaths link TreeNode's which will not
        //       work here because TreeNode's are re-created all
        //       the time.
        //
        private String[] getSelectedNodes(JTree tree)
        {
            List res = new ArrayList();
            TreePath[] selPath = tree.getSelectionPaths();
            
            if (selPath != null)
            {
                for (int i = 0; i < selPath.length; i++)
                {
                    TreePath path = selPath[i];
                    StringBuffer sb = new StringBuffer();
                    
                    for (int j = 0; j < path.getPathCount(); j++)
                    {
                        DefaultMutableTreeNode aNode = (DefaultMutableTreeNode)path.getPathComponent(j);
                        
                        if (aNode.isRoot())
                        {
                            sb.append("<root>");
                        }
                        else
                        {
                            sb.append(".").append(aNode.toString());
                        }
                    }

                    res.add(sb.toString());
                }
            }
            
            return (String[])res.toArray(new String[0]);
        }
        
        // Sets 'special' selection strings back into the tree...so
        // that a selection in the old tree model will appear in a
        // newly re-built tree.
        //
        private void setSelectedNodes(JTree tree, String[] path)
        {
            List res = new ArrayList();
            
            if (path != null)
            {
                DefaultMutableTreeNode node = null;

                for (int i = 0; i < path.length; i++)
                {
                    StringTokenizer st = new StringTokenizer(path[i], ".");
                    node = (DefaultMutableTreeNode)tree.getModel().getRoot();
                
                    while (st.hasMoreTokens())
                    {
                        String c = st.nextToken().trim();
                    
                        if (c.equals("<root>"))
                        {
                            continue;
                        }

                        if (node != null)
                        {
                            node = findNodeWithName(node, c);
                        }
                    }
                
                    if (node != null)
                    {
                        res.add(new TreePath(node.getPath()));
                    }
                }
            }
            
            TreePath[] selPath = (TreePath[])res.toArray(new TreePath[0]);
            
            tree.setSelectionPaths(selPath);
        }
        
        // Builds a list of the currently expanded product folders...
        // Returns a string array that will contain the actual product
        // name (folder name), e.g. MF, Security, etc.
        //
        private String[] getExpandedProducts(JTree tree)
        {
            List res = new ArrayList();
            Enumeration en = tree.getExpandedDescendants(new TreePath(tree.getModel().getRoot()));

            while ((en != null) && en.hasMoreElements())
            {
                TreePath path = (TreePath)en.nextElement();
                
                DefaultMutableTreeNode node = (DefaultMutableTreeNode)path.getLastPathComponent();
                ConfigTypeInfo cti = (ConfigTypeInfo)node.getUserObject();
                
                if (cti == null)
                {
                    continue;
                }
                
                res.add(cti.toString());
            }

            return (String[])res.toArray(new String[0]);
        }
        
        private void setExpandedProducts(JTree tree, String[] productName)
        {
            DefaultTreeModel model = (DefaultTreeModel)tree.getModel();
            
            for (int i = 0; i < productName.length; i++)
            {
                DefaultMutableTreeNode productNode = findProductNode((DefaultMutableTreeNode)tree.getModel().getRoot(), productName[i]);

                if (productNode != null)
                {
                    m_configTree.expandPath(new TreePath(productNode.getPath()));
                }
            }
        }
    }

    /**
     * Method to return a list of plugin factories for each configuration that
     * can be created for the selected plugin.
     *
     * @return List of menu items
     */
    private List getConfigList(boolean template)
    {
        List res = new ArrayList();

        Map      factories = m_plugin.getPluginContext().getLibrary().getConfigRootFactoryMap();
        Map      productMap = buildProductFamilyMap(factories);
        Iterator iter      = factories.keySet().iterator();

        while (iter.hasNext())
        {
            String key  = (String)iter.next();   // The root (product) plugin type
            List   list = (List)factories.get(key);
            
            for (int i = 0; i < list.size(); i++)
            {
                ConfigFactoryInfo info = (ConfigFactoryInfo)list.get(i);
                String productFamilyType = (String)productMap.get(key);
                
                if (productFamilyType == null)
                {
                    productFamilyType = "Unknown";
                }
                
                Iterator iter2 = info.getChildFactoryMap().values().iterator();

                while (iter2.hasNext())
                {
                    addFactory(res, template, productFamilyType, (IPluginFactory)iter2.next());
                }
            }
        }
        
        return res;
    }
    
    private void addFactory(List factoryList, boolean template, String productFamily, IPluginFactory factory)
    {
        PluginAttributes attributes = factory.getAttributes();

        // If the factory is not fixed and an instance can be directly
        // created then add it to the list...
        //
        if (!attributes.isCreatable() ||
            !(!template || (template && attributes.isTemplatable())))
        {
            return;
        }
        
        boolean added = false;
        
        // We are storing a list of factories for each type:
        // each factory in the list represents a different
        // version of the factory.
        Iterator i = factoryList.iterator();
        
        while (i.hasNext())
        {
            ConfigTypeInfo info = (ConfigTypeInfo)i.next();
            
            if (info.addFactory(factory))
            {
                added = true;
                break;
            }
        }
        
        if (!added)
        {
            factoryList.add(new ConfigTypeInfo(productFamily, factory));
        }
    }

    /**
     * Build up a mapping between each product root type, e.g. MF_ROOT_PLUGIN
     * and the PRODUCT_FAMILY_TYPE attribute that may be set on each root plugin
     * factory.
     * e.g. MQ_ROOT_PLUGIN > "MQ"
     * 
     * This approach has the added advantage of automatically assigning old versions
     * of a root type to a product family. So the family id (type) need only be assigned
     * on later versions of a plugin.
     * 
     * Going forward, changes to family string will have to be made consistently
     * across all versions...an error is output to standard error if the method
     * detects any inconsistency.
     */
    private Map buildProductFamilyMap(Map cfiMap)
    {
        Map res = new HashMap();
        Iterator i = cfiMap.keySet().iterator();
        
        while (i.hasNext())
        {
            String productType = (String)i.next();
            List   list = (List)cfiMap.get(productType);
            
            for (int j = 0; j < list.size(); j++)
            {
                ConfigFactoryInfo cfi = (ConfigFactoryInfo)list.get(j);
            
                String productFamilyType = cfi.getRootFactory().getAttributes().getProductFamilyId();
            
                if (productFamilyType != null)
                {
                    if (res.containsKey(productType) && !res.get(productType).equals(productFamilyType))
                    {
                        System.err.println("Product Family Type for '" + productType + "' is not consistent across versions!");
                    }

                    res.put(productType, productFamilyType);
                }
            }
        }
        
        return res;
    }
    
    /**
     * Rebuilds the config type tree based on the supplied ConfigTypeInfo list.
     * The tree is completely rebuilt - first any existing nodes are removed and
     * then new nodes are added back in based on the ConfigTypeInfo.
     * 
     * @param model    The tree's model.
     * @param ctiList  The list of ConfigTypeInfo that defines the available
     *                 configuration types.
     */
    private void rebuildTreeModel(DefaultTreeModel model, List ctiList)
    {
        DefaultMutableTreeNode rootNode = (DefaultMutableTreeNode)model.getRoot();

        rootNode.removeAllChildren();
        
        Iterator i = ctiList.iterator();
        
        while (i.hasNext())
        {
            ConfigTypeInfo cti = (ConfigTypeInfo)i.next();

            DefaultMutableTreeNode productNode = findProductNode(rootNode, cti.m_productDisplayType);
            
            if (productNode == null)
            {
                ConfigTypeInfo ctiProduct = new ConfigTypeInfo(cti.m_productDisplayType);
                productNode = new DefaultMutableTreeNode(ctiProduct);
                
                rootNode.insert(productNode, findInsertIndex(rootNode, ctiProduct));
            }

            productNode.insert(new DefaultMutableTreeNode(cti), findInsertIndex(productNode, cti));
            
            // If a configuration type is ALSO a primary configuration then we add it in
            // again under the root (basically a shortcut).
            if (cti.isPrimaryConfiguration())
            {
                rootNode.insert(new DefaultMutableTreeNode(cti), findInsertIndex(rootNode, cti));
            }
        }

        model.nodeStructureChanged(rootNode);
    }
    
    // Finds a product (folder) node in the restricted config-type tree.
    static final DefaultMutableTreeNode findProductNode(DefaultMutableTreeNode rootNode, String product)
    {
        for (int i = 0; i < rootNode.getChildCount(); i++)
        {
            DefaultMutableTreeNode childNode = (DefaultMutableTreeNode)rootNode.getChildAt(i);
            ConfigTypeInfo         childInfo = (ConfigTypeInfo)childNode.getUserObject();
            
            if ((childInfo != null) && childInfo.isProduct() && childInfo.toString().equals(product))
            {
                return childNode;
            }
        }
        
        return null;
    }

    // More general purpose find method.
    static final DefaultMutableTreeNode findNodeWithName(DefaultMutableTreeNode parentNode, String name)
    {
        for (int i = 0; i < parentNode.getChildCount(); i++)
        {
            DefaultMutableTreeNode childNode = (DefaultMutableTreeNode)parentNode.getChildAt(i);
            ConfigTypeInfo         childInfo = (ConfigTypeInfo)childNode.getUserObject();
            
            if ((childInfo != null) && childInfo.toString().equals(name))
            {
                return childNode;
            }
        }
        
        return null;
    }

    private int findInsertIndex(DefaultMutableTreeNode parentNode, ConfigTypeInfo cti)
    {
        int res = parentNode.getChildCount();
        
        for (int i = 0; i < parentNode.getChildCount(); i++)
        {
            DefaultMutableTreeNode childNode = (DefaultMutableTreeNode)parentNode.getChildAt(i);
            ConfigTypeInfo         childInfo = (ConfigTypeInfo)childNode.getUserObject();
            
            if (cti.isPrimaryConfiguration() && childInfo.isProduct())
            {
                return i;
            }

            if (cti.isProduct() && childInfo.isPrimaryConfiguration())
            {
                ;
            }
            else if (cti.toString().compareTo(childInfo.toString()) < 0)
            {
                return i;
            }
        }
        
        return res;
    }
    
    /**
     * Method to create the Templates tab.
     *
     * @return The Template Configuration tab panel.
     */
    private JPanel createConfigFromTemplateTab()
    {
        JPartitionPanel panel = new JPartitionPanel(true,"p,r",null);

        panel.add(new JLabel(ResourceManager.getString(getClass(),
            "NewConfigurationDialog.SelectTemplateLabel")));

        m_tree = createTemplateTree();
        panel.add(new ExtendedJScrollPane(m_tree));

        return panel;
    }

    /**
     * Method to create a JTree containing a set of templates for each
     * configuration that can be created.
     *
     * @return The populated JTree
     */
    private JLightConfigTree createTemplateTree()
    {
        IPluginContext context = m_plugin.getPluginContext();

        JLightConfigTree tree   = new JLightConfigTree(context,
                                                       context.getConfigContext().getConfigServer(),
                                                       "/",
                                                       new TemplateFilter());

        tree.setRootLabel(ConfigDomainFolderPlugin.PLUGIN_NAME);
        tree.setRootIcon(ResourceManager.getTypeIcon(ConfigDomainFolderPlugin.PLUGIN_TYPE));
        tree.expandRow(0);
        tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        tree.addTreeSelectionListener(this);
        tree.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(MouseEvent evt)
            {
                if (evt.getClickCount() == 2)
                {
                    TreePath path = m_tree.getPathForLocation(evt.getX(), evt.getY());

                    if ((path == null) || !(((TreeNode)path.getLastPathComponent()).isLeaf()))
                    {
                        return;
                    }

                    Action action = getDefaultOKAction();

                    if (action != null)
                    {
                        JButton button = ((JButtonPanel)getButtonPanel()).
                            getButton((String)action.getValue(
                            Action.ACTION_COMMAND_KEY));

                        if (button != null)
                        {
                            button.doClick();
                        }
                    }
                }
            }
        });

        return tree;
    }

    /**
     * Method to return the active tab.
     *
     * @return The label of the active tab
     */
    public boolean isConfigActiveTab()
    {
        return (m_tabbedPane.getSelectedIndex() == 0);
    }

    /**
     * Method to return the selected plugin factory
     *
     * @return Selected plugin factory
     */
    public Object getSelection()
    {
        if(isConfigActiveTab())
        {
            FactoryVersionInfo versions = (FactoryVersionInfo)m_versionCombo.getSelectedItem();

            // find the factory with the selected version number
            DefaultMutableTreeNode selNode = (DefaultMutableTreeNode)m_configTree.getSelectionPath().getLastPathComponent();
            ConfigTypeInfo cti = (ConfigTypeInfo)selNode.getUserObject();
            
            for(int i = 0; i < cti.m_factory.size(); i++)
            {
                IPluginFactory factory = (IPluginFactory)cti.m_factory.get(i);

                if(factory.getAttributes().getConfigVersion().equals(versions.m_cVersion))
                {
                    return new Object[] { factory, versions.m_pVersion };
                }
            }
            return null;
        }

        // Templates tab is active
        return m_tree.getSelectionPath().getLastPathComponent().toString();
    }

    /**
     * Method to return whether a template is selected or not.
     *
     * @return Flag indicating true or false
     */
    public boolean getIsTemplate()
    {
        return m_templateRadioButton.isSelected();
    }

    /**
     * Re-implemented method to process event caused when a tab is activated.
     *
     * @param event Event caused when a tab is activated
     */
    @Override
    public void stateChanged(ChangeEvent event)
    {
        if(isConfigActiveTab())
        {
            enableConfigTabButtons();
        }
        else
        {
            enableTemplateTabButtons();
        }
    }

    /**
     * Re-implemented method to process event caused when a template is selected.
     *
     * @param event Event caused when a template is selected
     */
    @Override
    public void valueChanged(TreeSelectionEvent event)
    {
        if (event.getSource() == m_tree)
        {
            enableTemplateTabButtons();
        }
        else
        if (event.getSource() == m_configTree)
        {
            enableConfigTabButtons();
        }
    }

    /**
     * Method to enable the Ok Button based on the configuration selection.
     */
    protected void enableConfigTabButtons()
    {
        TreePath selPath = m_configTree.getSelectionPath();
        DefaultMutableTreeNode selNode = (selPath != null) ? (DefaultMutableTreeNode)selPath.getLastPathComponent() : null;
        ConfigTypeInfo cti = (selNode != null) ? (ConfigTypeInfo)selNode.getUserObject() : null;
        
        updateVersionCombo(cti);

        getDefaultOKAction().setEnabled(!m_configTree.isSelectionEmpty() && (cti != null) && !cti.isProduct());
    }

    private void updateVersionCombo(ConfigTypeInfo cti)
    {
        boolean hasVersion = ((cti != null) && !cti.isProduct());
        
        m_versionCombo.setEnabled(hasVersion);

        if (hasVersion)
        {
            List vList = new ArrayList();
            
            for (int i = 0; i < cti.m_factory.size(); i++)
            {
                IPluginFactory factory = (IPluginFactory)cti.m_factory.get(i);

                String[] pVersions = factory.getAttributes().getProductVersions();
                String   cVersion  = factory.getAttributes().getConfigVersion();

                for (int j = 0; j < pVersions.length; j++)
                {
                    vList.add(new FactoryVersionInfo(cVersion, pVersions[j]));
                }
            }
            Collections.sort(vList, new Comparator()
            {
                @Override
                public int compare(Object o1, Object o2)
                {
                    FactoryVersionInfo fvi1 = (FactoryVersionInfo)o1;
                    FactoryVersionInfo fvi2 = (FactoryVersionInfo)o2;

                    // We want to enforce ordering of the versions -
                    // the most recent product release should be at the top
                    // of the list and the oldest at the bottom...the
                    // latest release will be selected.
                    float f1 = Float.parseFloat(fvi1.m_pVersion);
                    float f2 = Float.parseFloat(fvi2.m_pVersion);
                    return Float.compare(f2,f1);
                }
                
                @Override
                public boolean equals(Object obj)
                {
                    return (obj == this);
                }
                
                @Override
                public int hashCode() {
                    return super.hashCode();
                }
            });
            m_versionCombo.setModel(new DefaultComboBoxModel(new Vector(vList)));
        }
        else
        {
            m_versionCombo.removeAllItems();
            m_versionCombo.addItem("None");
        }
    }

    public static class FactoryVersionInfo
    {
        String m_cVersion;
        String m_pVersion;

        public FactoryVersionInfo(String cVersion, String pVersion)
        {
            m_cVersion = cVersion;
            m_pVersion = pVersion;
        }

        @Override
        public String toString()
        {
            return m_pVersion;
        }
    }

    static class ConfigTypeInfo
    {
        List m_factory = null;
        String m_productDisplayType;
        String m_configDisplayType;
        boolean m_product = false;
        
        public ConfigTypeInfo(String productDisplayType)
        {
            m_productDisplayType = productDisplayType;
            m_product = true;  // This is a product node
        }

        public ConfigTypeInfo(String productDisplayType, IPluginFactory factory)
        {
            m_productDisplayType = productDisplayType;
            
            addFactory(factory);
        }

        public final boolean addFactory(IPluginFactory factory)
        {
            boolean res = true;
            
            if (m_factory == null)
            {
                m_factory = new ArrayList();
                
                m_factory.add(factory);
            }
            else
            {
                IPluginFactory f = (IPluginFactory)m_factory.get(0);
                
                if (f.getAttributes().getType().equals(factory.getAttributes().getType()))
                {
                    m_factory.add(factory);
                }
                else
                {
                    res = false;
                }
            }
            
            return res;
        }

        public String getDisplayType()
        {
            if ((m_factory == null) || m_factory.isEmpty())
            {
                return null;
            }
            
            IPluginFactory f = (IPluginFactory)m_factory.get(0);
            
            return f.getAttributes().getDisplayType();
        }
        
        public boolean isProduct()
        {
            return m_product;
        }
        
        @Override
        public String toString()
        {
            return isProduct() ? m_productDisplayType : getDisplayType();
        }
        
        // Looks through factories for all versions to find a primary configuration
        // setting...this attribute is sparsely set...so that we don't have to go
        // through older plugin versions...added in 7.0 (Aruba) release.
        //
        public boolean isPrimaryConfiguration()
        {
            Iterator i = (m_factory != null) ? m_factory.iterator() : Collections.EMPTY_LIST.iterator();
            
            while (i.hasNext())
            {
                IPluginFactory f = (IPluginFactory)i.next();
                
                if (f.getAttributes().isPrimaryConfiguration())
                {
                    return true;
                }
            }
            return false;
        }
    }
    
    /**
     * Method to enable the Ok Button based on the template selection.
     */
    protected void enableTemplateTabButtons()
    {
        if (m_tree.isSelectionEmpty())
        {
            getDefaultOKAction().setEnabled(false);
        }
        else
        {
            DefaultMutableTreeNode node   = (DefaultMutableTreeNode)m_tree.getLastSelectedPathComponent();
            HashMap                map    = (HashMap)node.getUserObject();
            boolean                enable = map.containsKey(ConfigServerUtility.TYPE);

            getDefaultOKAction().setEnabled(enable);
        }
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class TemplateFilter implements IPluginFilter
    {
        @Override
        public boolean isPluginVisible(Object destination, Object pluginData)
        {
            try
            {
                HashMap map      = (HashMap)pluginData;
                String  type     = (String)map.get(ConfigServerUtility.TYPE);
                String  cVersion = (String)map.get(ConfigServerUtility.CONFIG_VERSION);
                String  pVersion = (String)map.get(ConfigServerUtility.PRODUCT_VERSION);
                String  name     = (String)map.get(IConfigServer.FOLDER_NAME);

                // If its a folder and there's no type or version set,
                // i.e. its not a special ElementFolder, then we want to
                // show it...
                if (name != null)
                {
                    if ((type == null) && (cVersion == null))
                    {
                        return true;
                    }
                }

                if (name == null)
                {
                    name = (String)((IElementIdentity)map.get(IConfigServer.ELEMENT_IDENTITY)).getName();
                }

                name = name.substring(name.lastIndexOf('/') + 1);

                if (name.startsWith("_"))
                {
                    return false;
                }

                IConfigPluginFactory factory = m_plugin.getPluginContext().getConfigContext().getPluginFactory(type, cVersion, pVersion);

                if (factory == null)
                {
                    return false;
                }

                if (!factory.getAttributes().isCreatable())
                {
                    return false;
                }

                // We only want to show templates...
                //
                String templateType = (String)map.get(ConfigServerUtility.TEMPLATE_TYPE);

                if ((templateType != null) && templateType.equalsIgnoreCase(ConfigServerUtility.TEMPLATE_TYPE_PROTOTYPE))
                {
                    return true;
                }
            }
            catch (Exception e)
            {
                MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error filtering components", e, false);
            }

            return false;
        }
    }

    class NewConfigTreeCellRenderer extends DefaultTreeCellRenderer
    {
        @Override
        public Component getTreeCellRendererComponent(JTree tree,
                                                      Object value,
                                                      boolean sel,
                                                      boolean expanded,
                                                      boolean leaf,
                                                      int row,
                                                      boolean hasFocus)
        {
            super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            
            ConfigTypeInfo cti = (ConfigTypeInfo)((DefaultMutableTreeNode)value).getUserObject();
            
            if (cti == null)
            {
                ;
            }
            else
            {
                if (!cti.isProduct())
                {
                    // Just use the first factory in the list for the renderer
                    IConfigPluginFactory factory = (IConfigPluginFactory)cti.m_factory.get(0);

                    PluginAttributes attr = factory.getAttributes();
                    String strType = attr.getType();
                    
                    Icon icon = ResourceManager.getTypeIcon(factory.getClass(), strType);

                    if (cti.isPrimaryConfiguration())
                    {
                        ConfigTypeInfo parentCti = getParentInfo((DefaultMutableTreeNode)value);
                        
                        if (parentCti == null)
                        {
                            icon = new ExtendedImageIcon(icon, ResourceManager.getTypeIcon(getClass(), "Shortcut"));
                            
                            setText("Shortcut to " + getText());
                        }
                    }
                    
                    setIcon(icon);
                }
            }
            
            return this;
        }
        
        private ConfigTypeInfo getParentInfo(DefaultMutableTreeNode node)
        {
            DefaultMutableTreeNode parentNode = (DefaultMutableTreeNode)node.getParent();

            return (ConfigTypeInfo)parentNode.getUserObject();
        }
    }
}