// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.ma.gui.config.metrics;

import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.config.metrics.model.MetricsModel;
import com.sonicsw.ma.gui.runtime.metrics.model.InstanceParentNode;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.JButtonPanel;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.ResourceManager;

final class InstanceEnablementSheet
    extends JMADialog
{
    private static final String INSTANCE_ALL = "* (All Instance Patterns)";
    private static final String BANNER = "<html>The table below lists the currently enabled instance pattern metrics:</html>";

    private MetricsModel m_model;
    private InstanceParentNode m_parentNode;
    private String[] m_enabledPatterns;

    private BasicAction m_removeAction = new RemoveAction();
    private BasicAction m_enableAction = new AllInstancesAction();

    private JRowTable m_table;

    public InstanceEnablementSheet(JMADialog parent,
                                   MetricsModel model,
                                   InstanceParentNode parentNode,
                                   String[] enabledPatterns)
        throws Exception
    {
        super(parent, "metrics.pattern.enablement");

        m_model = model;
        m_parentNode = parentNode;
        m_enabledPatterns = enabledPatterns;
    }

    @Override
    public void maInitialize()
    {
        getContentPane().add(makeMainPanel());

        getOKButton().setEnabled(false);

        for (int i = 0; i < m_enabledPatterns.length; i++)
        {
            String pattern = m_enabledPatterns[i];

            if (pattern.equals("*"))
            {
                pattern = INSTANCE_ALL;
            }

            ((IModelTableModel) m_table.getModel()).addRow(pattern);
        }

        enableButtons();
        initListeners();
    }

    @Override
    public void maCleanup()
    {
        if (m_table != null)
        {
            m_table.cleanup();
        }
    }

    @Override
    public Action[] getButtonActions()
    {
        return new Action[]
            {
            new OKAction(), getDefaultCancelAction()};
    }

    private JButton getOKButton()
    {
        return ((JButtonPanel) getButtonPanel()).getButton("dialog.ok");
    }

    /*
        private void resetModelData(String[] patterns)
        {
            m_table.getModelListTableModel().clear();
            patterns = (String[])Sorter.sort(patterns);
            for (int i = 0; i < patterns.length; i++)
                m_table.getModelListTableModel().insertRowMaintainSort(patterns[i]);
        }
     */

    public String[] getInstances()
    {
        String[] res = null;

        try
        {
            List instances = m_model.getInstances(m_parentNode);

            res = (String[]) instances.toArray(new String[instances.size()]);
        }
        catch (Throwable e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to get available instance identities", e, true);
            res = new String[0];
        }

        return res;
    }

    private void initListeners()
    {
        m_table.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(ListSelectionEvent evt)
            {
                enableButtons();
            }
        });

        ((IModelTableModel) m_table.getModel()).addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(TableModelEvent evt)
            {
                getOKButton().setEnabled(true);
            }
        });
    }

    private JPanel makeBannerPanel()
    {
        JPanel panel = new JPartitionPanel(false, "p,r", null, JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0);

        JLabel label = new JLabel(BANNER);
        label.setVerticalAlignment(SwingConstants.CENTER);
        label.setHorizontalAlignment(SwingConstants.LEFT);

        panel.add(new JLabel(ResourceManager.getIcon(getClass(), "InstanceLarge")));
        panel.add(label);

        return panel;
    }

    private JPanel makeMainPanel()
    {
        JPanel panel = new JPartitionPanel(false, "r,p", null); //, JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0);

        m_table = new JRowTable(new RowTableColumn[]
                                {new RowTableColumn(0, "Enabled Instance Patterns", 100)}
                                , getClass().getName());

        JButtonPanel btnPanel = new JButtonPanel(true, JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0);

        btnPanel.addButton(new PickInstancesAction());
        btnPanel.addButton(m_removeAction);
        btnPanel.addSeparator();
        btnPanel.addButton(m_enableAction);

        panel.add(new ExtendedJScrollPane(m_table));
        panel.add(btnPanel);

        JPartitionPanel wrapPanel = new JPartitionPanel(true, "p,r", null);
        wrapPanel.setBorder(wrapPanel.getPreferredBorder());
        wrapPanel.add(makeBannerPanel());
        wrapPanel.add(panel);

        return wrapPanel;
    }

    private void enableButtons()
    {
        boolean enableAll = (((IModelTableModel) m_table.getModel()).getIndexOf(INSTANCE_ALL) < 0);

        m_removeAction.setEnabled(m_table.getSelectedRowCount() > 0);
        m_enableAction.setEnabled(enableAll);
    }

    public void addPattern(String pattern)
    {
        addPattern(new String[]
                   {pattern});
    }

    public void addPattern(String[] pattern)
    {
        IModelTableModel tableModel = (IModelTableModel) m_table.getModel();

        if (pattern != null)
        {
            for (int i = 0; i < pattern.length; i++)
            {
                if (pattern[i] == null)
                {
                    continue;
                }

                String newPattern = pattern[i].trim();

                if (newPattern.equals("*"))
                {
                    newPattern = INSTANCE_ALL;
                }

                if (tableModel.findCell(0, 0, newPattern) < 0)
                {
                    tableModel.addRow(newPattern);
                }
            }
        }
    }

    private void removePatterns()
    {
        int[] selected = m_table.getSelectedRows();
        int nextIndex = -1;

        if (selected.length > 0)
        {
            nextIndex = Integer.MAX_VALUE;

            for (int i = 0; i < selected.length; i++)
            {
                if (selected[i] < nextIndex)
                {
                    nextIndex = selected[i];
                }
            }

                    // -2 because this is called before removal of element.
            nextIndex = Math.min(nextIndex, m_table.getRowCount() - 2);
        }

        IModelTableModel tableModel = (IModelTableModel) m_table.getModel();

        for (int i = selected.length - 1; i >= 0; i--)
        {
            tableModel.deleteRow(selected[i]);
        }

        if ((nextIndex >= 0) && (nextIndex < m_table.getRowCount()))
        {
            m_table.setRowSelectionInterval(nextIndex, nextIndex);
        }
    }

    private void applyPatternChanges()
    {
        List contents = ((IModelTableModel) m_table.getModel()).getContents();
        if ( contents.size() > 0 && contents.contains(INSTANCE_ALL))
        {
            contents.remove(INSTANCE_ALL);
            contents.add("*");
        }
        Object[] patterns = contents.toArray();

        // put original list in a hash set for easy lookup
        HashSet compare = new HashSet();
        for (int i = 0; i < m_enabledPatterns.length; i++)
        {
            compare.add(m_enabledPatterns[i]);
        }

            // compare with original list to find the patterns that have been
            // added
        ArrayList added = new ArrayList();
        for (int i = 0; i < patterns.length; i++)
        {
            if (!compare.contains(patterns[i]))
            {
                added.add(patterns[i]);
            }
        }

        // put new list in a hash set for easy lookup
        compare.clear();
        for (int i = 0; i < patterns.length; i++)
        {
            compare.add(patterns[i]);
        }

            // compare with original list to find the patterns that have been
            // removed
        ArrayList removed = new ArrayList();
        for (int i = 0; i < m_enabledPatterns.length; i++)
        {
            if (!compare.contains(m_enabledPatterns[i]))
            {
                removed.add(m_enabledPatterns[i]);
            }
        }



        // now apply the added/removed to the model
        try
        {
            if (!added.isEmpty())
            {
                m_model.enableInstancePatterns(m_parentNode, added);
            }
            if (!removed.isEmpty())
            {
                m_model.disableInstancePatterns(m_parentNode, removed);
            }

                /*
                            int iAdded = added.indexOf(INSTANCE_ALL);
                            if (iAdded >= 0)
                                added.set(iAdded, "*");
                            int iRemoved = removed.indexOf(INSTANCE_ALL);
                            if (iRemoved >= 0)
                                removed.set(iRemoved, "*");
                            m_model.update.updateInstancePatterns(m_parentNode, added, removed);
                 */
        }
        catch (Throwable e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to enable/disable instance(s)",
                e, true);
        }
    }

    class OKAction
        extends BasicGuiAction
    {
        public OKAction()
        {
            super("dialog.ok", getDefaultOKAction());
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            applyPatternChanges();

            super.actionPerformed(evt);
        }
    }

    private class PickInstancesAction
        extends BasicGuiAction
    {
        public PickInstancesAction()
        {
            super("InstanceEnablementSheet.add");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                JMADialog dialog = new InstancePickSheet(InstanceEnablementSheet.this,m_model);
                dialog.setVisible(true);
            }
            catch (Throwable e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to get/pick instance", e, true);
            }
        }
    }

    private class RemoveAction
        extends BasicGuiAction
    {
        public RemoveAction()
        {
            super("InstanceEnablementSheet.remove");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            removePatterns();
        }
    }

    private class AllInstancesAction
        extends BasicGuiAction
    {
        public AllInstancesAction()
        {
            super("InstanceEnablementSheet.add-all");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            InstanceEnablementSheet.this.addPattern("*");
        }
    }
}