
package com.sonicsw.ma.gui.config.propsheets;

import java.beans.PropertyVetoException;

import modelobjects.framework.model.ModelAspectAdapter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;

import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IConfigPath;


public class ConfigBeanModelAspectAdapter extends ModelAspectAdapter
{
    private IConfigPath m_path;
    private boolean     m_readOnly;

    public ConfigBeanModelAspectAdapter(IConfigPath path,
                                        Class       type,
                                        boolean     readOnly)
    {
        super(ModelAspectId.forName(path.toString()), type);

        m_path     = path;
        m_readOnly = readOnly;
    }

    @Override
    public boolean isReadonlyAspect()
    {
        return m_readOnly;
    }

    @Override
    protected Object getAspectValue(Object             modelObject,
                                    ModelObjectAdapter modelObjectAdapter)
    {
        Object          value = null;
        ConfigBeanModel bean  = (ConfigBeanModel)modelObject;

        if (isViewName())
        {
            value = bean.getViewName();
        }
        else
        {
            value = bean.getData().getAttribute(m_path);

            if (value == null)
            {
                // The model aspect (attribute) my still have a default value
                // that should be returned...even if any intermediate attribute
                // aspects do not exist...so we navigate the config type info.
                //
                IAttributeDescription ad = bean.getData().getAttributeDescription().getAttributeDescription(m_path);

                if (ad != null)
                {
                    value = ad.getProperty(IAttributeDescription.DEFAULT_PROPERTY);
                }
            }
        }

        return value;
    }

    @Override
    protected void setAspectValue(Object             modelObject,
                                  Object             newAspectValue,
                                  ModelObjectAdapter modelObjectAdapter)
    throws PropertyVetoException,
           IllegalArgumentException,
           UnsupportedOperationException
    {
        try
        {
            ConfigBeanModel bean = (ConfigBeanModel)modelObject;

            if (isViewName())
            {
                if (newAspectValue == null)
                {
                    bean.setViewName(null);
                //    throw new IllegalArgumentException("Required attribute missing: " + ConfigBeanModel.NAME);
                }
                else
                {
                    if (!(newAspectValue instanceof String))
                    {
                        throw new IllegalArgumentException("The value '" + newAspectValue.toString() + " is not a valid string.");
                    }

                    String newValue = ((String)newAspectValue).trim();

                    if (newValue.length() == 0)
                    {
                        throw new IllegalArgumentException("You must enter a value for this field.");
                    }

                    bean.setViewName(newValue);
                }
            }
            else
            {
            	// If the new value is null or its a string and the string is
            	// empty then its ok to remove the attribute
            	//
                if ((newAspectValue == null) ||
                    ((newAspectValue instanceof String) && ((String)newAspectValue).trim().length() == 0))
                {
                    bean.getData().removeAttribute(m_path);
                }
                else
                {
                    Object oldValue = bean.getData().getAttribute(m_path);

                    // We need to check to see if the value has changed.
                    // If its a complex structure, eg. map or list then
                    // we assume it has changed otherwise we do an
                    // equals check - this avoids unnecessary traffic
                    // to the DS and also avoids instance/template
                    // issues, i.e. writing the same value locally
                    // in an instance when its already in the template.
                    //
                    if (oldValue != null)
                    {
                        if (!((newAspectValue instanceof IAttributeMap) && (newAspectValue instanceof IAttributeList)) &&
                            oldValue.equals(newAspectValue))
                        {
                            return;
                        }
                    }

                    boolean isDefault = isDefaultValue(bean.getData(), m_path, newAspectValue);

                    if (createMissingOptionalAttributes(bean.getData(), m_path, isDefault))
                    {
                        bean.getData().setAttribute(m_path, newAspectValue);
                    }
                }
            }
        }
        catch (IllegalArgumentException e)
        {
            IllegalArgumentException iae = new IllegalArgumentException(m_path.toString());
            iae.initCause(e);

            throw iae;
        }
        catch (ConfigServiceException e)
        {
            IllegalArgumentException iae = new IllegalArgumentException(m_path.toString());
            iae.initCause(e);

            throw iae;
        }
    }

    protected boolean isViewName()
    {
        return m_path.equals(ConfigBeanModel.NAME);
    }

    /**
     * If we try to set an attribute value on the path "a.b.c" and "b" is an
     * optional attribute set then the operation will fail because the Config
     * Layer does not automatically create intermediate optional attributes.
     *
     * This code searches for missing optional attributes (map/list) and creates
     * an empty attribute so that the setAttribute works.
     *
     * This may still result in failure because it may not be valid to add an
     * empty attribute map or list, i.e. the optional attribute may have complex
     * min/max occurs settings.
     *
     * Questionable whether the Config Layer should always create the intermediate
     * attributes irrespective of whether the attribute is optional or not.
     *
     * @return  true if its ok to continue setting the attribute, false if
     *          the value should not be set.
     */
    private boolean createMissingOptionalAttributes(IAttributeMap data,
                                                    IConfigPath   path,
                                                    boolean       isDefault)
        throws ConfigServiceException
    {
        for (int i = 1; i < path.size(); i++)
        {
            IConfigPath subPath = path.subPath(0, i);

            if (data.getAttribute(subPath) == null)
            {

                // This intermediate attribute doesn't exist...we should only
                // auto create it if the value being set is NOT the default
                // value...
                if (isDefault)
                {
                    return false;
                }

                // We need to try and create missing maps or lists
                // but only if the attribute is optional
                //
                IAttributeDescription ad = data.getAttributeDescription().getAttributeDescription(subPath);
                Number minOccurs = (Number)ad.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);

                if ((minOccurs != null) && (minOccurs.intValue() == 0))
                {
                    if (ad.getType().equals(IAttributeMap.class))
                    {
                        IAttributeMap map = data.createAttributeMap(subPath.toString());
                        data.setAttribute(subPath, map);
                    }
                    else
                    if (ad.getType().equals(IAttributeList.class))
                    {
                        IAttributeList list = data.createAttributeList(subPath.toString());
                        data.setAttribute(subPath, list);
                    }
                }
            }
        }

        return true;
    }

    /**
     * Determines whether or not the value for the path is the 'default'.
     *
     * @param map
     * @param path
     * @param value
     * @return       Returns true if it is the default, otherwise false.
     */
    private boolean isDefaultValue(IAttributeMap map,
                                   IConfigPath   path,
                                   Object        value)
    {
        Object val = getDefaultValue(map, path);

        return ((val != null) && (value != null) && val.equals(value));
    }

    /**
     * Gets the default value (if any) for the supplied attribute (config path).
     *
     * @param map
     * @param path
     * @return      Returns the default value (if any), otherwise 'null'.
     */
    private Object getDefaultValue(IAttributeMap map, IConfigPath path)
    {
        IAttributeDescription ad = map.getAttributeDescription().getAttributeDescription(path);

        return (ad != null) ? ad.getProperty(IAttributeDescription.DEFAULT_PROPERTY) : null;
    }
}
