/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.config.propsheets;

import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelObjectPropagator;
import modelobjects.framework.model.ModelPropagationException;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigContext;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigServer;

public class ConfigBeanPropagator implements ModelObjectPropagator
{
    private IConfigPlugin m_plugin;
    private String m_renameOld;
    private String m_renameNew;

    public ConfigBeanPropagator()
    {
        this(null);
    }

    public ConfigBeanPropagator(IConfigPlugin plugin)
    {
        setPlugin(plugin);
    }

    public IConfigPlugin getPlugin()
    {
        return m_plugin;
    }

    public final void setPlugin(IConfigPlugin plugin)
    {
        m_plugin = plugin;
    }

    /**
     * @deprecated  No longer supported...use the model's (IConfigBean)
     *              <code>getConfigServer<\code> to get the valid server.
     * @return      The current config context
     */
    protected IConfigContext getConfigContext()
    {
        return m_plugin.getPluginContext().getConfigContext();
    }

    /**
     * Propagate edits to a model object.
     * @param modelObjectAdapter the ModelObjectAdapter that holds the model
     * @exception ModelPropagationException if the edits cannot be propagated
     */
    @Override
    public void propagateModelEdit(ModelObjectAdapter adapter)
    throws ModelPropagationException
    {
        ConfigBeanModel model = (ConfigBeanModel)adapter.getModelObject();

        try
        {
            if (model.getData() instanceof IConfigBean)
            {
                IConfigServer server = ((IConfigBean)model.getData()).getConfigServer();

                server.storeConfigElement((IConfigBean)model.getData());

                // After storing the element we might want to rename it...
                if (m_renameOld != null)
                {
                    server.rename(m_renameOld, m_renameNew);
                }

                if (server.isTransacted())
                {
                    server.commit();
                }
            }
        }
        catch (Exception e)
        {
            if (model.getData() instanceof IConfigBean)
            {
                IConfigServer server = ((IConfigBean)model.getData()).getConfigServer();

                try
                {
                    server.rollback();
                }
                catch (ConfigServiceException cse)
                {
                    Helper.logDebugMessage("Failed to rollback changes: " + ((IConfigBean)model.getData()).getName());
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }

            throw new ModelPropagationException(e);
        }
    }

    protected String rename(ConfigBeanModel model)
    {
        IConfigBean bean = (IConfigBean)model.getData();
        String      path = bean.getName();
        int         index = path.lastIndexOf('/');
        String      basePath = path.substring(0, index + 1);
        String      beanName = model.getViewName();

        return basePath + beanName;
    }

    // Allows the propagator to perform a delayed rename...such that the rename
    // is performed at the end of the propagation routine. Only in this way
    // can a rename be performed without affecting write/creation order.
    //
    protected void delayedRename(String oldPath, String newPath)
    {
        m_renameOld = oldPath;
        m_renameNew = newPath;
    }

    /**
     *  Propagate a newly created model object.
     *  @param modelObjectAdapter the ModelObjectAdapter that holds the model
     *  @exception ModelPropagationException if the create cannot be propagated
     */
    @Override
    public void propagateModelCreation(ModelObjectAdapter adapter)
    throws ModelPropagationException
    {
        try
        {
            ConfigBeanModel model = (ConfigBeanModel)adapter.getModelObject();

            if (model.getData() instanceof IConfigBean)
            {
                IConfigBean bean = (IConfigBean)model.getData();

                // The following code is done to pick up any change to the
                // configuration (view) name that the user might have made
                // before commiting...only set the name back into the bean
                // if it has changed...
                //
                String newName = rename(model);
                if (!newName.equals(bean.getName()))
                {
                    if (bean.getConfigServer().pathExists(newName))
                    {
                        throw new Exception("A configuration with name '" + newName + "' already exists");
                    }

                    // If the path (element) already exists in the DS (server)
                    // then what we really want to do is a rename...but we
                    // will delay it until the end of the propagate to avoid
                    // any ordering issues...
                    //
                    if (bean.getConfigServer().pathExists(bean.getName()))
                    {
                        delayedRename(bean.getName(), newName);
                    }
                    else
                    {
                        bean.setName(newName);
                    }
                }
            }

            propagateModelEdit(adapter);
        }
        catch (ModelPropagationException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new ModelPropagationException(e);
        }
    }

    /**
     * Propagate deletion of a model object.
     * @param modelObjectAdapter the ModelObjectAdapter that holds the model
     * @exception ModelPropagationException if the delete cannot be propagated
     */
    @Override
    public void propagateModelDeletion(ModelObjectAdapter adapter)
    throws ModelPropagationException
    {
        performModelDeletion((ConfigBeanModel)adapter.getModelObject());
    }

    public void performModelDeletion(ConfigBeanModel model)
    throws ModelPropagationException
    {
        try
        {
            if (model != null && model.getData() instanceof IConfigBean)
            {
                IConfigBean   bean   = (IConfigBean)model.getData();
                IConfigServer server = bean.getConfigServer();

                server.removeConfigElement(bean.getName());

                String[] names = getDeleteNames(bean);

                if (names == null)
                {
                    throw new IllegalArgumentException("No beans to delete");
                }

                server.removeConfigElements(names);

                if (server.isTransacted())
                {
                    server.commit();
                }
            }
        }
        catch (Exception e)
        {
            throw new ModelPropagationException(e);
        }
    }

    /**
     * Override in a subclass to get the propagator delete any referenced
     * children config beans that need to get cleared up at the same time that
     * the main config bean is deleted.
     *
     * @param beanModel  The main bean to be deleted.
     * @return           The list of bean names (identitys) to be deleted.
     */
    protected String[] getDeleteNames(IConfigBean beanModel)
    {
        return new String[] { beanModel.getName() };
    }

}
