/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.config.propsheets;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import modelobjects.framework.EditRule;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelDescriptor;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.swing.JCheckBoxAdapter;
import modelobjects.framework.swing.JComboBoxAdapter;
import modelobjects.framework.swing.JLabelAdapter;
import modelobjects.framework.swing.JTextAreaAdapter;
import modelobjects.framework.swing.JTextFieldAdapter;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.propsheets.JPropSheetDialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigType;
import com.sonicsw.mx.config.util.ConfigHelper;

public abstract class JConfigDialog extends JPropSheetDialog
{
    public static final Dimension MIN_SIZE = new Dimension(350, 400);
    public static final Dimension MAX_SIZE = new Dimension(550, 650);

    public static final Dimension DEF_SIZE = MIN_SIZE;
    
    public JConfigDialog(JMAFrame parent, String name, IAttributeDescription type)
    {
        super(parent, name, ConfigBeanModel.class);

        setDescriptorProperty(type);

        setValidator(new ConfigBeanValidator());
        setPropagator(new ConfigBeanPropagator());
    }

    public JConfigDialog(JConfigDialog parent, String name, IAttributeDescription type)
    {
        super(parent, name, ConfigBeanModel.class);
        setDescriptorProperty(type);
    }

    private void setDescriptorProperty(IAttributeDescription type) {
        ((ConfigBeanModelDescriptor)getDescriptor()).setType(type);
    }
    
    @Override
    public void pack()
    {
        super.pack();
        
        Rectangle bounds = getBounds();

        if(bounds.width > MAX_SIZE.width)
        {
            bounds.width = MAX_SIZE.width;
        }
        else if(bounds.width < MIN_SIZE.width)
        {
            bounds.width = MIN_SIZE.width;
        }
        
        if(bounds.height > MAX_SIZE.height)
        {
            bounds.height = MAX_SIZE.height;
        }
        else if(bounds.height < MIN_SIZE.height)
        {
            bounds.height = MIN_SIZE.height;
        }
        
        setBounds(bounds);
    }
    
    @Override
    protected ModelDescriptor makeModelDescriptor()
    {
        return new ConfigBeanModelDescriptor();
    }

    @Override
    public void editInstance(IPlugin plugin, Object model, boolean isNew)
    throws Exception
    {
        if ((getPropagator() != null) && (getPropagator() instanceof ConfigBeanPropagator))
        {
            ((ConfigBeanPropagator)getPropagator()).setPlugin((IConfigPlugin)plugin);
        }

        super.editInstance(plugin, model, isNew);

        boolean isTemplate = ((IConfigPlugin)plugin).isTemplatePrototype();

        setTitle(getDialogTitle(isNew, isTemplate));
    }

    protected String getDialogTitle(boolean isNew, boolean isTemplate)
    {
        String result;

        if(isNew)
        {
            String titlePrefix = ResourceManager.getString(getClass(),
                isTemplate ?  "JMADialog.newTemplateTitlePrefix" :
                              "JMADialog.newTitlePrefix");
            result =  titlePrefix + " " + getTitle();
        }
        else
        {
            String titlePrefix = ResourceManager.getString(getClass(),
                isTemplate ?  "JMADialog.editTemplateTitlePrefix" :
                              "JMADialog.editTitlePrefix");
            String titleSuffix = ResourceManager.getString(getClass(),
                              "JMADialog.editTitleSuffix");
            result = titlePrefix + " " + getTitle() + " " + titleSuffix;
        }
        return result;
    }

    public IAttributeDescription getModelMetaData()
    {
        return ((ConfigBeanModelDescriptor)getDescriptor()).getType();
    }

    //-------------------------------------------------------------------------
    //
    // ViewAspectAdapter helpers
    //
    //-------------------------------------------------------------------------

    public JLabelAdapter createJLabelAdapter(IConfigPath        attributePath,
                                             JLabel             viewComponent,
                                             ViewValueConverter converter)
    {
        return new JLabelAdapter(getModelAspectId(attributePath),
            viewComponent, converter, getEditForm().getModelEditMediator());
    }

    public JTextFieldAdapter createJTextFieldAdapter(IConfigPath        attributePath,
                                                     JTextField         viewComponent,
                                                     ViewValueConverter converter)
    {
        return new JTextFieldAdapter(getModelAspectId(attributePath), viewComponent,
                                     isFieldEditOnInit(attributePath) ? EditRule.FOR_INIT : EditRule.ALWAYS,
                                     converter, getEditForm().getModelEditMediator());
    }

    public JTextAreaAdapter createJTextAreaAdapter(IConfigPath        attributePath,
                                                   JTextArea          viewComponent,
                                                   ViewValueConverter converter)
    {
        return new JTextAreaAdapter(getModelAspectId(attributePath), viewComponent,
                                    isFieldEditOnInit(attributePath) ? EditRule.FOR_INIT : EditRule.ALWAYS,
                                    converter, getEditForm().getModelEditMediator());
    }

    public JComboBoxAdapter createJComboBoxAdapter(IConfigPath        attributePath,
                                                   JComboBox          viewComponent,
                                                   ViewValueConverter converter)
    {
        return new JComboBoxAdapter(getModelAspectId(attributePath), viewComponent,
                                    isFieldEditOnInit(attributePath) ? EditRule.FOR_INIT : EditRule.ALWAYS,
                                    converter, getEditForm().getModelEditMediator());
    }

    public JCheckBoxAdapter createJCheckBoxAdapter(IConfigPath        attributePath,
                                                   JCheckBox          viewComponent,
                                                   ViewValueConverter converter)
    {
        return new JCheckBoxAdapter(getModelAspectId(attributePath), viewComponent,
                                    isFieldEditOnInit(attributePath) ? EditRule.FOR_INIT : EditRule.ALWAYS,
                                    converter, getEditForm().getModelEditMediator());
    }

    public ModelAspectId getModelAspectId(IConfigPath configPath)
    {
        return ModelAspectId.forName(configPath.toString());
    }

    /**
     * Check to see if we have a mandatory item.
     */
    public boolean isMandatoryField(IConfigPath path)
    {
        boolean res = true;

        if (path == null)
        {
            res = false;
        }
        else
        if (!path.equals(ConfigBeanModel.NAME))
        {
            IAttributeDescription type = getModelMetaData();
            IAttributeDescription    elem = type.getAttributeDescription(path);

            if (elem != null)
            {
                // We check to see if we have a default value...because if it
                // has a default then its not really mandatory because the
                // schema will always supply the default if nothing is entered.
                // (Sonic00008627)
                //
                Object defValue = elem.getProperty(IAttributeDescription.DEFAULT_PROPERTY);

                if (defValue != null)
                {
                    res = false;
                }
                else
                {
                    Long min = (Long)elem.getProperty(IAttributeDescription.MIN_OCCURS_PROPERTY);
                    Long max = (Long)elem.getProperty(IAttributeDescription.MAX_OCCURS_PROPERTY);

                    res = ((min != null) && (min.longValue() == 1) &&
                           (max != null) && (max.longValue() == 1));
                }
            }
            else
            {
                res = false;
            }
        }

        return res;
    }

    private static final int VALUE_NOT_DEFINED    = -1;
    private static final int VALUE_FROM_PROTOTYPE = 0;
    private static final int VALUE_FROM_BEAN      = 1;

    public int getAttributeValueSource(IConfigPath path)
    {
        ConfigBeanModel    model = (ConfigBeanModel)getModel();
        int                res   = model.isTemplateInstance() ? VALUE_FROM_PROTOTYPE : VALUE_FROM_BEAN;
        IAttributeMetaData meta  = null;

        try
        {
            if (path != null)
            {
                if (model.getDataList() != null)
                {
                    meta = ((IAttributeList)model.getDataList()).getAttributeMetaData(path);
                }
                else
                if (model.getData() != null)
                {
                    meta = ((IAttributeMap)model.getData()).getAttributeMetaData(path);
                }

                if (path.equals(ConfigBeanModel.NAME))
                {
                    res = VALUE_NOT_DEFINED;
                }
                else
                if (meta != null)
                {
                    if (meta.isFromPrototype())
                    {
                        res = VALUE_FROM_PROTOTYPE;
                    }
                    else
                    if (meta.isFromConfigBean())
                    {
                        res = VALUE_FROM_BEAN;
                    }
                    else
                    if (meta.isFromConfigType()) 
                    {
                        // Assume default source, i.e. prototype if template, bean if bean
                    }
                }
                else
                {
                    res = VALUE_NOT_DEFINED;
                }
            }
            else
            {
                res = VALUE_NOT_DEFINED;
            }
        }
        catch (ConfigServiceException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                "Failed to find source of attribute '" + path + "'", e, false);
        }
        return res;
    }

    /**
     * Builds up tooltip text using html tags for multi-line support.
     * The tip text describes the given configuration path.
     */
    protected String getToolTipText(IConfigPath path, int source)
    {
        StringBuffer sb = new StringBuffer();

        sb.append("<html>");

        if (path != null)
        {
            // Sonic00009422 - no resources defined for this so we won't
            // support it in this release.
            String tip = getResourceString(new Object[] { path, "help" });

            if (tip != null)
            {
                String[] split = splitOnWord(tip, 30);

                for (int i = 0; i < split.length; i++)
                {
                    sb.append(split[i]);
                    sb.append("<br>");
                }
                sb.append("<br>");
            }

            String methodName = path.toString();
            int    methodIndex = methodName.lastIndexOf('.');
            if (methodIndex >= 0)
            {
                methodName = methodName.substring(methodIndex+1);
            }

            sb.append("Method: ").append(ConfigHelper.generateName(methodName));
        }

        if (source == VALUE_FROM_PROTOTYPE)
        {
            sb.append("<br>").append("Value from Template");
        }

        sb.append("</html>");

        return sb.toString();
    }

    /**
     * Splits the supplied tooltip text into 1 or more strings using any
     * non-alphanumeric character as a delimiter. Each string is limited in
     * length.
     */
    private String[] splitOnWord(String tip, int max)
    {
        ArrayList list = new ArrayList();
        String    delim = " /,.;:'<?\\~!@#$%^&*(+=";
        StringTokenizer st = new StringTokenizer(tip, delim, true);
        StringBuffer    sb = new StringBuffer();

        while (st.hasMoreTokens())
        {
            String token = st.nextToken();

            if ((sb.length() + token.length()) <= max)
            {
                sb.append(token);

                if (sb.length() == max)
                {
                    list.add(sb.toString());
                    sb = new StringBuffer();
                }
            }
            else
            {
                list.add(sb.toString());
                sb = new StringBuffer();
                sb.append(token);
            }
        }

        if (sb.length() > 0)
        {
            list.add(sb.toString());
        }

        return (String[])list.toArray(new String[list.size()]);
    }

    /**
     * Determines from the supplied path whether or not a field can only be
     * editted when a new instance is being created.
     *
     * TBD AHJ - needs to get this info from the schema - work not done yet!
     *
     * @param path  The config path to the attribute (field)
     * @return      true if the form should only allow editting when new
     *              or false if the field can always be editted.
     */
    protected boolean isFieldEditOnInit(IConfigPath path)
    {
        boolean res = false;

        if (path.equals(ConfigBeanModel.NAME))
        {
            res = true;
        }

        return res;
    }

    public String[] getStringEnumeration(IConfigPath path)
    {
        return getStringEnumeration(path, null);
    }

    public String[] getStringEnumeration(IConfigPath path, String[] removeFilter)
    {
        ArrayList res = new ArrayList();

        try
        {
            IAttributeDescription ad   = getModelMetaData().getAttributeDescription(path);
            IAttributeList        list = (IAttributeList)ad.getProperty(IAttributeDescription.EMUMERATION_FACET);
            Iterator              i    = list.iterator();

            while (i.hasNext())
            {
                String item = (String)i.next();
                boolean add = true;

                if (removeFilter != null)
                {
                    // Search removal filter for match - if we get one then we
                    // don't want it in the returned list.
                    //
                    for (int iFilter = 0; iFilter < removeFilter.length; iFilter++)
                    {
                        if ((removeFilter[iFilter] != null) &&
                            removeFilter[iFilter].equals(item))
                        {
                            add = false;
                            break;
                        }
                    }
                }

                if (add)
                {
                    res.add(item);
                }
            }
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, e.toString(), e, true);
        }

        return (String[])res.toArray(new String[res.size()]);
    }

    //-------------------------------------------------------------------------
    //
    // Resource helpers
    //
    //-------------------------------------------------------------------------

    public String getResourceString(Object key)
    {
        return getResourceString(new Object[] { key } );
    }

    public String getResourceString(Object[] key)
    {
        ArrayList list = new ArrayList();

        list.add(ResourceManager.KEY_CONFIG);

        if (getTypedResourceKey() != null)
        {
            list.add(getTypedResourceKey());
        }

        for (int i = 0; i < key.length; i++)
        {
            list.add(key[i]);
        }

        return ResourceManager.getString(getClass(), (Object[])list.toArray(new Object[list.size()]));
    }

/*
    public JLabel _getResourceLabel(Object key)
    {
        return new JLabel(ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_CONFIG, key, ResourceManager.KEY_LABEL }));
    }
*/

    public String[] _getResourceStringArray(Object key)
    {
        return ResourceManager.getStringArray(getClass(), new Object[] { ResourceManager.KEY_CONFIG, key, ResourceManager.KEY_LIST });
    }

    public JLabel getResourceLabel(IConfigPath path)
    {
        String resString = getResourceString(new Object[] { path, ResourceManager.KEY_LABEL });

        if (resString == null)
        {
            resString = "<NO RESOURCE>";
        }

        return getResourceLabel(resString, path);
    }

    /**
     * Builds a JLabel for the given config path. It uses the config path to
     * determine the attribute source and properties and the key to build a
     * resource path.
     *
     * @param path  The IConfigPath identifying the attribute.
     * @param key   The Object identifying the attribute in the resource files.
     * @return      A JLabel that represents the attribute found in the path.
     */
    public JLabel getResourceLabel(IConfigPath path, Object key)
    {
        String resString = ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_CONFIG, key, ResourceManager.KEY_LABEL });

        if (resString == null)
        {
            resString = "<NO RESOURCE>";
        }

        return getResourceLabel(resString, path);
    }

    protected JLabel getResourceLabel(String resString, IConfigPath path)
    {
        int    source = getAttributeValueSource(path);
        JLabel label  = getResourceLabel(resString, isMandatoryField(path), source);

        if (source == VALUE_FROM_PROTOTYPE)
        {
            label.setFont(label.getFont().deriveFont(Font.ITALIC | Font.BOLD));
        }

        label.setToolTipText(getToolTipText(path, source));

        return label;
    }

    protected JLabel getResourceLabel(String resString, boolean mandatory, int source)
    {
        JLabel       label = new JLabel();
        StringBuffer sb    = new StringBuffer(resString);

        if ((sb.length() > 0) && sb.charAt(sb.length() - 1) != ':')
        {
            sb.append(':');
        }

        if (mandatory)
        {
            sb.insert(0, "*");

            if (source == VALUE_NOT_DEFINED)
            {
                label.setForeground(Color.RED);
            }
        }

        label.setText(sb.toString());

        return label;
    }

    public String getResourceUnit(Object key)
    {
        return ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_UNIT, key });
    }

    public String getTypedResourceKey()
    {
        if (getModelMetaData() instanceof IConfigType)
        {
            return ((IConfigType)getModelMetaData()).getName();
        }

        return null;
    }

    /**
     * Overridden method to extract failure message from ConfigLayer mx
     * model implementation.
     */
    @Override
    protected String getModelEditFailureMessage(Throwable throwable)
    {
        // We're only interested in failures of the model validation kind here
        // because we can leave JPropSheetDialog implementation to handle
        // view-to-model conversion errors...
        //
        if (throwable instanceof ModelValidationException)
        {
            // ModelValiationException thrown by ModelObjects framework
            //
            ModelValidationException e       = (ModelValidationException)throwable;
            Throwable                handleE = e.getNestedException();

            if (handleE != null)
            {
                if (handleE instanceof IllegalArgumentException)
                {
                    if (((IllegalArgumentException)handleE).getCause() != null)
                    {
                        handleE = ((IllegalArgumentException)handleE).getCause();
                    }
                }

                if (handleE instanceof ConfigServiceException)
                {
                    if (handleE instanceof ConfigAttributeException)
                    {
                        ConfigAttributeException cae = (ConfigAttributeException)handleE;

                        if (cae.getErrorKey().equals("am-validate-complete-req-attr-missing"))
                        {
                            cae.setErrorKey(cae.getErrorKey() + "-mgmtconsole");
                        }
                    }

                    // ConfigServiceException thrown by underlying
                    // model implementation...ConfigLayer :)
                    //
                    if (((ConfigServiceException)handleE).getLinkedException() != null)
                    {
                        handleE = ((ConfigServiceException)handleE).getLinkedException();
                    }
                }
            }
            else
            {
                handleE = e;
            }

            return handleE.getMessage();
        }

        return super.getModelEditFailureMessage(throwable);
    }

}
