package com.sonicsw.ma.gui.config.revert;

import java.lang.reflect.Method;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.ConfigBeanModel;
import com.sonicsw.ma.plugin.IConfigPlugin;
import com.sonicsw.mx.config.ConfigAttributeException;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.IAttributeDescription;
import com.sonicsw.mx.config.IAttributeList;
import com.sonicsw.mx.config.IAttributeMap;
import com.sonicsw.mx.config.IAttributeMetaData;
import com.sonicsw.mx.config.IConfigBean;
import com.sonicsw.mx.config.IConfigPath;

/* A FileNode is a derivative of the File class - though we delegate to
 * the File object rather than subclassing it. It is used to maintain a
 * cache of a directory's children and therefore avoid repeated access
 * to the underlying file system during rendering.
 */
public class ConfigAttributeTreeTableNode
{
    IConfigPlugin                m_plugin;
    ConfigAttributeTreeTableNode m_parent;
    String                       m_key;
    Object                       m_value;
    Object[]                     m_children;
    String                       m_label;

    public ConfigAttributeTreeTableNode(IConfigPlugin plugin)
    {
        super();

        IConfigBean bean = (IConfigBean)((ConfigBeanModel)plugin.getModel()).getData();

        m_parent   = null;
        m_key      = bean.getName();
        m_value    = bean;
        m_children = null;
        m_plugin   = plugin;
        m_label    = null;
    }

    public ConfigAttributeTreeTableNode(ConfigAttributeTreeTableNode parent,
                                        String                       key,
                                        Object                       value)
    {
        m_parent   = parent;
        m_key      = key;
        m_value    = value;
        m_children = null;
        m_label    = null;
    }

    public ConfigAttributeTreeTableNode getParent() { return m_parent; }
    public Object getValue() { return m_value; }
    public String getKey() { return m_key; }

    @Override
    public String toString()
    {
        if (m_label == null)
        {
            IConfigBean parentBean = getParentConfigBean();
            IConfigPath path = getConfigPath(parentBean);

            if (isUnenumeratedItem())
            {
                m_label = path.subPath(path.size()-1).toString();
            }
            else
            {
                ArrayList list = new ArrayList();
                IConfigPath resPath = getResourcePath(parentBean);

                list.add(ResourceManager.KEY_CONFIG);
                list.add(parentBean.getConfigType().getName());
                for (int i = 0; i < resPath.size(); i++)
                {
                    list.add(resPath.getComponent(i));
                }
                list.add(ResourceManager.KEY_LABEL);

                m_label = ResourceManager.getString(getPlugin().getClass(),
                                                    (Object[])list.toArray(new Object[list.size()]));
                
                // Handle missing resources - toString should never return null!!
                if (m_label == null)
                {
                    m_label = "";
                }
            }
        }

        return m_label;
    }

    public IConfigPlugin getPlugin()
    {
        IConfigPlugin                plugin = null;
        ConfigAttributeTreeTableNode node   = this;

        while (!node.isRoot())
        {
            node = node.getParent();
        }

        return node.m_plugin;
    }

    public ConfigAttributeSource getConfigAttributeSource()
    {
        // Sonic00020523 - attribute source was not calculated properly.
        // If the attribute is a ConfigBean then we must get its parent.
        //
        IConfigBean parent = getParentConfigBean();
        IConfigPath adjustedPath = getConfigPath(parent);

        return new ConfigAttributeSource(parent, adjustedPath);
    }

    private IConfigPath getResourcePath(IConfigBean parentBean)
    {
        IConfigPath                  path = ConfigFactory.createConfigPath();
        ConfigAttributeTreeTableNode node = this;

        while ((node != null) && (node.getValue() != parentBean))
        {
            String part = node.getKey();

            if ((node != this) && node.isUnenumeratedItem())
            {
                IAttributeDescription ad = node.getAttributeDescription();

                part = (String)ad.getProperty(IAttributeDescription.SCHEMA_TYPE_NAME);
            }

            path.insert(0, part);

            node = node.getParent();
        }

        return path;
    }

    public IConfigPath getConfigPath()
    {
        return getConfigPath(null);
    }

    public IConfigPath getConfigPath(IConfigBean bean)
    {
        IConfigPath                  path = ConfigFactory.createConfigPath();
        ConfigAttributeTreeTableNode node = this;

        if (bean == null)
        {
            while ((node != null) && !node.isRoot())
            {
                path.insert(0, new String[] { node.getKey() });

                node = node.getParent();
            }
        }
        else
        {
            while ((node != null) && (node.getValue() != bean))
            {
                path.insert(0, new String[] { node.getKey() });

                node = node.getParent();
            }
        }

        return path;
    }

    /**
     * Hops up the tree to find the attribute's immediate (nearest) ConfigBean
     * parent.
     *
     * @return  The <code>IConfigBean</code> parent.
     */
    public IConfigBean getParentConfigBean()
    {
        IConfigBean                  bean = null;
        ConfigAttributeTreeTableNode node = this;

        if ((getValue() instanceof IConfigBean) && !node.isRoot())
        {
            node = node.getParent();
        }

        while (node != null)
        {
            if (node.getValue() instanceof IConfigBean)
            {
                bean = (IConfigBean)node.getValue();
                break;
            }

            node = node.getParent();
        }

        return bean;
    }

    public IConfigBean getRootConfigBean()
    {
        return getConfigBean(false);
    }

    public IConfigBean getConfigBean()
    {
        return getConfigBean(true);
    }

    protected IConfigBean getConfigBean(boolean firstMatch)
    {
        IConfigBean                  bean = null;
        ConfigAttributeTreeTableNode node = this;

        while (node != null)
        {
            if (node.getValue() instanceof IConfigBean)
            {
                bean = (IConfigBean)node.getValue();

                if (firstMatch)
                {
                    break;
                }
            }

            node = node.getParent();
        }

        return bean;
    }

    private IAttributeDescription getAttributeDescription()
    {
        IConfigBean parent = getParentConfigBean();
        IConfigPath path   = getConfigPath(parent);

        if (getValue() instanceof IAttributeMap)
        {
            return ((IAttributeMap)getValue()).getAttributeDescription();
        }

        if (getValue() instanceof IAttributeList)
        {
            return ((IAttributeList)getValue()).getAttributeDescription();
        }

        return parent.getAttributeDescription().getAttributeDescription(path); //path.subPath(0, path.size()-1));
    }

    /**
     * Loads the children, caching the results in the children ivar.
     */
    protected Object[] getChildren()
    {
        if (m_children != null)
        {
            return m_children;
        }

        List list = new ArrayList();

        if (m_value instanceof IAttributeMap)
        {
            Iterator i = ((com.sonicsw.mx.config.impl.AttributeMapImpl)m_value).getAllAttributeNames().iterator();

            while (i.hasNext())
            {
                try
                {
                    IAttributeMap      valueMap = (IAttributeMap)m_value;
                    String             key      = (String)i.next();
                    Object             val      = valueMap.getAttribute(key);
                    IAttributeMetaData meta     = valueMap.getAttributeMetaData(key);
                    IAttributeDescription ad    = valueMap.getAttributeDescription().getAttributeDescription(key);

                    // We don't want any non-local configurations showing up
                    // in the list, i.e. NO weak references
                    //if ((val instanceof IConfigBean) && !isLocalReference((IConfigBean)val))
                    //    continue;
                    //
                    // Sonic00020377 (AHJ) - the above code was slightly wrong.
                    // We want the placeholder, top-level node to appear, e.g.
                    // a cluster member in a cluster config but we don't want
                    // any of the member's attributes to appear.
                    //
                    if (!(val instanceof IConfigBean) && !isLocalReference(getConfigBean()))
                    {
                        continue;
                    }
                    	
                    // If an attribute is fixed then it can never change so don't show
                    Object fixed = (ad != null) ? ad.getProperty(IAttributeDescription.FIXED_PROPERTY) : null;
                    if (fixed != null)
                    {
                        continue;
                    }

                    if ((meta != null) && meta.isFromConfigBean())
                    {
                        ConfigAttributeTreeTableNode child = new ConfigAttributeTreeTableNode(this, key, val);

                        if (!isAttributeFiltered(child.getConfigPath()))
                        {
                            list.add(0, child);
                        }
                    }
                }
                catch (ConfigAttributeException e)
                {
                    e.printStackTrace();
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }
        }
        else
        if (m_value instanceof IAttributeList)
        {
            for (int i = 0; i < ((IAttributeList)m_value).size(); i++)
            {
                IAttributeList     valueList = (IAttributeList)m_value;
                String             key       = Integer.toString(i);
                Object             val       = valueList.getAttribute(i);
                IAttributeMetaData meta      = valueList.getAttributeMetaData(i);

                // We don't want any non-local configurations showing up
                // in the list, i.e. NO weak references
                if ((val instanceof IConfigBean) && !isLocalReference((IConfigBean)val))
                {
                    continue;
                }

                if ((val != null) && ((meta != null) && meta.isFromConfigBean()))
                {
                    ConfigAttributeTreeTableNode child = new ConfigAttributeTreeTableNode(this, key, val);

                    if (!isAttributeFiltered(child.getConfigPath()))
                    {
                        list.add(0, child);
                    }
                }
            }
        }

        m_children = (Object[])list.toArray(new Object[list.size()]);

        Arrays.sort(m_children, COMPARATOR);

        return m_children;
    }

    /**
     * Checks the filtered attribute config path list for a match.
     *
     * @param path  The config path to check.
     * @return      Returns true if the path was in the filter list, otherwise
     *              false is returned.
     */
    private boolean isAttributeFiltered(IConfigPath path)
    {
        IConfigPath[] filter = getFilteredAttributes(getPlugin());

        for (int i = 0; i < filter.length; i++)
        {
            if (filter[i].equals(path))
            {
                return true;
            }
        }

        return false;
    }

    private Method m_pluginMethodCache = null;

    /**
     * Uses reflection on the plugin to obtain a list of attribute paths
     * (if any) that should NOT be displayed, i.e. filtered from the revert
     * tree.
     *
     * Currently, only the ContainerPlugin and BrokerPlugin implement the
     * <code>getRevertFilteredAttributes</code> method to stop the user from
     * reverting the internal attribute name.
     *
     * @param plugin  The plugin from which the filter attribute paths are to
     *                be obtained.
     * @return        An array of filter attribute config paths...if it fails
     *                to find the method then an empty array is returned. This
     *                method should never return null.
     */
    private IConfigPath[] getFilteredAttributes(IConfigPlugin plugin)
    {
        try
        {
            if (m_pluginMethodCache == null)
            {
                m_pluginMethodCache = plugin.getClass().getMethod("getRevertFilteredAttributes", new Class[0]);
            }

            return (IConfigPath[])m_pluginMethodCache.invoke(plugin, new Object[0]);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            return new IConfigPath[0];
        }
    }

    /**
     * Decode to determine whether or not the attribute is part
     * of an unenumerated AttributeMap.
     */
    protected boolean isUnenumeratedItem()
    {
        if (!isRoot())
        {
            IAttributeDescription ad = getParent().getAttributeDescription();

            if (ad != null)
            {
                Boolean unenum = (Boolean)ad.getProperty(IAttributeDescription.UNENUM_MAP_PROPERTY);

                if ((unenum != null) && unenum.booleanValue())
                {
                    return true;
                }
            }
        }

        return false;
    }

    public boolean isRoot()
    {
        return (m_parent == null);
    }

    private boolean isLocalReference(IConfigBean refBean)
    {
        IConfigBean rootBean  = getRootConfigBean();
        String      rootPath  = rootBean.getName();
        int         rootIndex = rootPath.lastIndexOf("_Default");

        if (rootIndex >= 0)
        {
            rootPath = rootPath.substring(0, rootIndex);
        }

        return refBean.getName().startsWith(rootPath);
    }

    public boolean isLeaf()
    {
        if ((m_value instanceof IAttributeMap) ||
            (m_value instanceof IAttributeList))
        {
            return false;
        }

        return true;
    }

    private static final Comparator COMPARATOR = new NodeComparator();
    private static final Collator   COLLATOR = Collator.getInstance();

    static class NodeComparator implements Comparator
    {
        @Override
        public int compare(Object o1, Object o2)
        {
            String s1 = (o1 != null) ? o1.toString() : "";
            String s2 = (o2 != null) ? o2.toString() : "";

            return COLLATOR.compare(s1, s2);
        }
    }
}
