/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.domain;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import javax.management.Notification;
import javax.management.NotificationListener;

import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.StateHelper;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.IIdentity;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.IState;
import com.sonicsw.mf.common.runtime.ISubComponentState;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.ComponentState;
import com.sonicsw.mf.common.runtime.impl.ContainerState;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

public class AgentManagerConnection extends Thread implements NotificationListener
{
    private HashMap m_containerStatesByConfigID   = new HashMap(); // indexed by container storage config ID
    private HashMap m_containerStatesByLogicalID  = new HashMap(); // indexed by container logical config ID
    private HashMap m_containerStatesByRuntimeID  = new HashMap(); // indexed by container runtime ID
    private Vector  m_stateListeners              = new Vector();

    private IAgentManagerProxy m_agentManager;
    private IConfigServer      m_configServer;

    private boolean m_shutdown        = false;
    private boolean m_pollImmediately = false;
    private Object m_agentManagerConnLockObj = new Object();
    
    // notification names for state.system category
    public static final String EVENT_ONLINE       = "Online";
    public static final String EVENT_OFFLINE      = "Offline";
    public static final String EVENT_UNREACHABLE  = "Unreachable";
    public static final String EVENT_STARTUP      = "Startup";
    public static final String EVENT_SHUTDOWN     = "Shutdown";
    public static final String EVENT_ACTIVATE     = "Activate";
    public static final String EVENT_DEACTIVATE   = "Deactivate";
    public static final String EVENT_LOAD         = "Load";
    public static final String EVENT_UNLOAD       = "Unload";
    public static final String EVENT_FAILOVER     = "Failover";
    public static final String EVENT_FAILURE      = "Failure";
    public static final String[] HANDLED_EVENTS   = new String[]
    {
        EVENT_ONLINE,
        EVENT_OFFLINE,
        EVENT_UNREACHABLE,
        EVENT_STARTUP,
        EVENT_SHUTDOWN,
        EVENT_ACTIVATE,
        EVENT_DEACTIVATE,
        EVENT_LOAD,
        EVENT_UNLOAD,
        EVENT_FAILOVER,
        EVENT_FAILURE
    };

    public static final short OPERATION_LOAD   = 5;
    public static final short OPERATION_UNLOAD = (short)(OPERATION_LOAD + 1);

    private static final IContainerState[] EMPTY_CONTAINER_STATE_ARRAY = new IContainerState[0];
    private static final IIdentity[]       EMPTY_IDENTITY_ARRAY        = new IIdentity[0];

    public AgentManagerConnection(IAgentManagerProxy agentManager, IConfigServer configServer)
    {
        super("AgentManager monitor");

        m_agentManager = agentManager;
        m_configServer = configServer;
    }

    public void disconnect()
    {
        synchronized(m_agentManagerConnLockObj)
        {
            m_shutdown = true;

            m_agentManagerConnLockObj.notifyAll();
        }
        m_agentManager = null;
        m_configServer = null;
    }

    public void startAMConnection()
    {
        // If we aren't already running we need to start the thread.
        if(m_agentManager != null)
        {
            boolean alive = isAlive();

            if(!isAlive())
            {
                setDaemon(true);
                start();
            }
        }
    }

    @Override
    public void run()
    {
        boolean isUnreachableReported = false;

        // The polling design is not the most efficient .. if we get a flood of notifications that cause
        // the poll to occur frequently (e.g. someone shuts down a collection of containers) then we
        // end up making multiple calls to the AM, possibly immediately after each other, essentially
        // returning the same information. A more efficient design would request information on only
        // those containers state who are known to have changed - however such a design creates a level
        // of code complexity that does not seem worth it.
        //
        // To mitigate some of the effects of the inefficiency of the current design, we have a sleep
        // of 1 second, if one poll should immediately follow another - that way if multiple state
        // notifications arrive in a short period they are aggregated into a single poll the delay of
        // which should not be noticable to the change listeners (and ultimately the GUI rendering of
        // state).

        while (!isInterrupted() && !m_shutdown)
        {
            long timestamp = System.currentTimeMillis();

            try
            {
                try
                {
                    synchronized(m_agentManagerConnLockObj) { m_pollImmediately = false; }

                    if (m_agentManager != null)
                    {
                        setContainerStates((IContainerState[])m_agentManager.getCollectiveState(), true);
                    }

                    if (m_shutdown)
                    {
                        break;
                    }

                    isUnreachableReported = false; // we've had a success so we can report problems again
                }
                catch(ProxyRuntimeException e)
                {
                    // only report the fact once until we have successfully got state again
                    if (!isUnreachableReported && !m_shutdown)
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                                                   "Agent Manager is unreacheable",
                                                                   e,
                                                                   false);
                        isUnreachableReported = true;
                    }

                    // force a state of unknown since we can't get up to date info from the Agent Manager
                    synchronized(m_containerStatesByConfigID)
                    {
                        setContainerStates(getAllContainerStates(), IContainerState.STATE_UNKNOWN, timestamp);
                    }

                    // if it was a timeout condition then we can go back and try immediately (effectively
                    // ignoring the regular poll frequency)
                    if (e.getTargetException() instanceof InvokeTimeoutException)
                    {
                        continue;
                    }
                }
            }
            catch(Exception e) // this catches code issues and at least keeps this thread running!
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                                           "Error while updating domain state",
                                                           e,
                                                           false);
            }

            // get the poll interval each time in case it is changed
            int interval =
                PreferenceManager.getInstance().getInt(JPreferencesDialog.GENERAL_PREFS,
                                                       JPreferencesDialog.AM_POLL_FREQUENCY,
                                                       JPreferencesDialog.DEFAULT_AM_POLL_FREQUENCY);

            synchronized(m_agentManagerConnLockObj)
            {
                // the order of these is important
                if (!m_pollImmediately)
                {
                    try { m_agentManagerConnLockObj.wait(interval * 1000); } catch(InterruptedException ie) { }
                }

                if (m_shutdown)
                {
                    break;
                }

                if (m_pollImmediately)
                 {
                    try { m_agentManagerConnLockObj.wait(1000); } catch(InterruptedException ie) { }   // calling wait method
                }
            }
        }
    }

    @Override
    public void handleNotification(Notification notification, Object handback)
    {
        INotification mfNotification = (INotification)notification;
        String        eventName      = mfNotification.getEventName();

        if (MgmtConsole.DEVELOPER_MODE)
        {
            Helper.logDebugMessage("EVENT name=" + eventName + ", source=" +
                                   mfNotification.getSourceIdentity().getCanonicalName());
        }

        if (eventName.equals(EVENT_ACTIVATE))
        {
            return;
        }

        if (eventName.equals(EVENT_DEACTIVATE))
        {
            return;
        }

        if (eventName.equals(EVENT_SHUTDOWN))
        {
            // if this is the container hosting the AM then we can automatically set all to unknown
            IComponentIdentity source = mfNotification.getSourceIdentity();
            IContainerState containerState = getContainerStateByRuntimeID(source.getDomainName() + '.' + source.getContainerName());

            if (containerState != null)
            {
                IComponentState[] componentStates = containerState.getComponentStates();

                for (int i = 0; i < componentStates.length; i++)
                {
                    IComponentIdentity componentID = (IComponentIdentity)componentStates[i].getRuntimeIdentity();

                    if (componentID.getComponentName().equals(IAgentManagerProxy.GLOBAL_ID))
                    {
                        if (isOnlyAgentManager(componentID, getAllContainerStates()))
                        {
                            handleAgentManagerStateChange(IComponentState.STATE_OFFLINE, mfNotification.getTimeStamp());
                            return;
                        }
                    }
                }
            }
            // all others pass thru
        }

        if (eventName.equals(EVENT_FAILOVER))
        {
            // if this is the container hosting the AM then we can immediately get the whole domain state
            IComponentIdentity source = mfNotification.getSourceIdentity();

            if (source.getComponentName().equals(IAgentManagerProxy.GLOBAL_ID))
            {
                handleAgentManagerStateChange(IComponentState.STATE_ONLINE, mfNotification.getTimeStamp());
                return;
            }
            // all others pass thru
        }

        if (eventName.equals(EVENT_UNREACHABLE))
        {
            // check to see if we already know this .. if so then no need to shorten
            // the poll cycle
            String          containerRuntimeID = (String)mfNotification.getAttributes().get("Container");
            IContainerState containerState     = getContainerStateByRuntimeID(containerRuntimeID);

            if (containerState != null && containerState.getState() == IContainerState.STATE_OFFLINE)
            {
                return;
            }
        }

        pollImmediately();

        return;
    }

    /**
     * Return true if there is a runtime instance for the given container
     * configuration name.
     */
    public boolean containerExists(String logicalName)
    {
        return (m_containerStatesByLogicalID.get(logicalName) != null);
    }

    /**
     * Return true if there is an runtime instance for the given component
     * configuration.
     */
    public boolean activeComponentExists(String logicalName)
    {
        String configID = logicalToStorage(logicalName);

        if(configID == null)
        {
            return false;
        }

        synchronized(m_containerStatesByConfigID)
        {
            IContainerState[] containerStates = getAllContainerStates();

            for (int i = 0; i < containerStates.length; i++)
            {
                IComponentState[] componentStates = containerStates[i].getComponentStates();

                for (int j = 0; j < componentStates.length; j++)
                {
                    if ((componentStates[j].getRuntimeIdentity().getConfigIdentity().getName().equals(configID)) &&
                        (componentStates[j].getState() == IComponentState.STATE_ONLINE))
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Return the runtime identities of active (online) components with the
     * given configuration ID.
     */
    public IIdentity[] getActiveComponentIdentitiesByConfigID(String logicalName)
    {
        ArrayList activeComponentIDs = new ArrayList();

        String configID = logicalToStorage(logicalName);

        if(configID == null)
        {
            return EMPTY_IDENTITY_ARRAY;
        }

        synchronized(m_containerStatesByConfigID)
        {
            IContainerState[] containerStates = getAllContainerStates();

            for (int i = 0; i < containerStates.length; i++)
            {
                IComponentState[] componentStates = containerStates[i].getComponentStates();

                for (int j = 0; j < componentStates.length; j++)
                {
                    IIdentity runtimeID = componentStates[j].getRuntimeIdentity();

                    if(runtimeID.getConfigIdentity().getName().equals(configID) &&
                       componentStates[j].getState() == IComponentState.STATE_ONLINE)
                    {
                        activeComponentIDs.add(runtimeID);
                    }
                }
            }
        }
        return (IIdentity[])activeComponentIDs.toArray(EMPTY_IDENTITY_ARRAY);
    }

    /**
     * Return the runtime identities of active (online) components with the
     * given componment plugin type.
     */
    public IIdentity[] getActiveComponentIdentitiesByPluginType(String pluginType)
    {
        ArrayList activeComponentIDs = new ArrayList();

        synchronized(m_containerStatesByConfigID)
        {
            IContainerState[] containerStates = getAllContainerStates();

            for (int i = 0; i < containerStates.length; i++)
            {
                IComponentState[] componentStates = containerStates[i].getComponentStates();

                for (int j = 0; j < componentStates.length; j++)
                {
                    IIdentity runtimeID = componentStates[j].getRuntimeIdentity();

                    if (runtimeID.getConfigIdentity().getType().equals(pluginType) &&
                        componentStates[j].getState() == IComponentState.STATE_ONLINE)
                    {
                        activeComponentIDs.add(runtimeID);
                    }
                }
            }
        }
        return (IIdentity[])activeComponentIDs.toArray(EMPTY_IDENTITY_ARRAY);
    }

    /**
     * Return the configuration name for the container/component with the
     * given stringified runtime name.
     */
    public String runtimeIDToConfigName(String runtimeID)
    {
        if(runtimeID == null || runtimeID.length() == 0)
        {
            return null;
        }

        String containerRuntimeID;
        int index = runtimeID.indexOf(IComponentIdentity.DELIMITED_ID_PREFIX);

        if (index == -1)
        {
            containerRuntimeID = runtimeID;
        }
        else
        {
            containerRuntimeID = runtimeID.substring(0, index);
        }

        synchronized(m_containerStatesByConfigID)
        {
            IContainerState   containerState  = getContainerStateByRuntimeID(containerRuntimeID);

            //first check if component is a container
            IIdentity identity = containerState.getRuntimeIdentity();

            if(identity.getCanonicalName().equals(runtimeID))
            {
                return storageToLogical(identity.getConfigIdentity().getName());
            }

            //check deployed components
            IComponentState[] componentStates = containerState.getComponentStates();

            for (int j = 0; j < componentStates.length; j++)
            {
                identity = componentStates[j].getRuntimeIdentity();

                if (identity.getCanonicalName().equals(runtimeID))
                {
                    return storageToLogical(identity.getConfigIdentity().getName());
                }
            }
        }
        return null;
    }

    /**
     * Gets the container state for the given config ID.
     */
    public IContainerState getContainerStateByLogicalName(String logicalName)
    {
        return (IContainerState)m_containerStatesByLogicalID.get(logicalName);
    }

    /**
     * Get an array of all container states
     */
    private IContainerState[] getAllContainerStates()
    {
        return (IContainerState[])m_containerStatesByConfigID.values().toArray(EMPTY_CONTAINER_STATE_ARRAY);
    }

    /**
     * Gets the container state for the given runtime ID.
     */
    public IContainerState getContainerStateByRuntimeID(String runtimeID)
    {
        CanonicalName canonicalName = new CanonicalName(runtimeID);
        canonicalName = new CanonicalName(canonicalName.getDomainName(), canonicalName.getContainerName(), "");

        if (runtimeID.indexOf(IComponentIdentity.DELIMITED_ID_PREFIX) == -1)
        {
            runtimeID += IComponentIdentity.DELIMITED_ID_PREFIX + IAgentProxy.ID;
        }

        return (IContainerState)m_containerStatesByRuntimeID.get(canonicalName.getCanonicalName());
    }

    public void addStateChangeListener(StateChangeListener listener)
    {
        m_stateListeners.add(listener);
    }

    public void removeStateChangeListener(StateChangeListener listener)
    {
        m_stateListeners.remove(listener);
    }

    /**
     * Poll the AM immediately due to some event
     */
    void pollImmediately()
    {
        synchronized (m_agentManagerConnLockObj)
        {
            m_pollImmediately = true; // case where the thread is not in a wait
            m_agentManagerConnLockObj.notifyAll(); // case the thread is in the wait
        }
    }

    /**
     * This version sets the container states to that provided. If this is the full set of known
     * container states the boolean removeObsolete will be true (to remove states that are no
     * longer known).
     */
    private void setContainerStates(IContainerState[] containerStates, boolean removeObsolete)
    {
        synchronized(m_containerStatesByConfigID)
        {
            // if removeObsolete is true, then the state list represents the
            // full set of containers - thus anything in the current list that
            // is not in the new list can be removed
            if (removeObsolete)
            {
                IContainerState[] currentContainerStates = getAllContainerStates();

                for (int i = 0; i < currentContainerStates.length; i++)
                {
                    IIdentity runtimeID = currentContainerStates[i].getRuntimeIdentity();
                    String    configID  = runtimeID.getConfigIdentity().getName();
                    String    logicalID = storageToLogical(configID);

                    for (int j = 0; j <= containerStates.length; j++)
                    {
                        if (j == containerStates.length) // was not found
                        {
                            m_containerStatesByConfigID.remove(configID);
                            m_containerStatesByRuntimeID.remove(runtimeID.getCanonicalName());
                            if(logicalID != null)
                            {
                                m_containerStatesByLogicalID.remove(logicalID);
                            }
                            break;
                        }
                        // else do we have a match ?
                        if (containerStates[j].getRuntimeIdentity().getConfigIdentity().getName().equals(configID))
                        {
                            break;
                        }
                    }
                }
            }

            // replace or add as required
            for (int i = 0; i < containerStates.length; i++)
            {
                IIdentity runtimeID     = containerStates[i].getRuntimeIdentity();
                String    containerName = runtimeID.getContainerName();
                String    configID      = runtimeID.getConfigIdentity().getName();
                String    logicalID     = storageToLogical(configID);

                IContainerState existingContainerState =
                    (IContainerState)m_containerStatesByConfigID.get(configID);

                // OFFLINE - we have no existing container state or its unknown
                //
                if (existingContainerState == null ||
                    (existingContainerState.getState() == IContainerState.STATE_UNKNOWN &&
                     containerStates[i].getState() == IContainerState.STATE_OFFLINE))
                {
                    // add a new state
                    m_containerStatesByConfigID.put(configID, containerStates[i]);
                    m_containerStatesByRuntimeID.put(runtimeID.getCanonicalName(), containerStates[i]);
                    if(logicalID != null)
                    {
                        m_containerStatesByLogicalID.put(logicalID, containerStates[i]);
                    }

                    IComponentState[] componentStates = containerStates[i].getComponentStates();

                    // if there are no component states then this is because the container is offline and we can
                    // update the gui to reflect this
                    if (componentStates.length == 0)
                    {
                        fireStateChangeEvent(containerStates[i].getState(), configID, IAgentProxy.ID, containerStates[i], null);
                    }
                    else
                    {
                        // loop for all the components
                        for (int j = 0; j < componentStates.length; j++)
                        {
                            fireStateChangeEvent(componentStates[j].getState(),
                                                 configID,
                                                 ((IComponentIdentity)componentStates[j].getRuntimeIdentity()).getComponentName(),
                                                 containerStates[i],
                                                 componentStates[j]);
                        }
                    }
                }
                // NEW STATE CHANGE because:
                //   1. we have a more recent state change
                //   2. timestamp has reset to zero (recent restart) so state is unknown
                //   3. the existing state is offline or unknown but its now online
                //
                else if (existingContainerState.getTimeStamp() < containerStates[i].getTimeStamp() ||
                    (existingContainerState.getTimeStamp() > 0 && containerStates[i].getTimeStamp() == 0 &&
                    containerStates[i].getState() == IContainerState.STATE_UNKNOWN) ||
                    (existingContainerState.getState() <= IContainerState.STATE_OFFLINE &&
                    containerStates[i].getState() == IContainerState.STATE_ONLINE))
                {
                    boolean isCombinedState = false;
                    // replace the existing state
                    m_containerStatesByConfigID.put(configID, containerStates[i]);
                    m_containerStatesByRuntimeID.put(runtimeID.getCanonicalName(), containerStates[i]);
                    if(logicalID != null)
                    {
                        m_containerStatesByLogicalID.put(logicalID, containerStates[i]);
                    }

                    // and loop for all the components
                    IComponentState[] existingStates = existingContainerState.getComponentStates();
                    IComponentState[] newStates      = containerStates[i].getComponentStates();

                    // any missing from the new list ?
                    for (int j = 0; j < existingStates.length; j++)
                    {
                        String componentName = ((IComponentIdentity)existingStates[j].getRuntimeIdentity()).getComponentName();

                        for (int k = 0; k <= newStates.length; k++)
                        {
                            if (k == newStates.length) // not found - so its an unload
                            {
                                fireStateChangeEvent(OPERATION_UNLOAD, configID, IAgentProxy.ID, containerStates[i], existingStates[j]);
                                break;
                            }
                            if (((IComponentIdentity)newStates[k].getRuntimeIdentity()).getComponentName().equals(componentName))
                            {
                                break;
                            }
                        }
                    }
                    // processing of news + any missing from the existing list ?
                    for (int j = 0; j < newStates.length; j++)
                    {
                        IComponentIdentity componentID = (IComponentIdentity)newStates[j].getRuntimeIdentity();
                        String componentName = componentID.getComponentName();
                        for (int k = 0; k <= existingStates.length; k++)
                        {
                            short newSubComponentState = StateHelper.getSubComponentState(newStates[j]);
                            short existingSubComponentState = IComponentState.STATE_OFFLINE;
                            if (k < existingStates.length && existingStates.length > 0)
                            {
                                existingSubComponentState = StateHelper.getSubComponentState(existingStates[k]);
                            }
                            if(!componentName.equalsIgnoreCase("Agent") && isCombinedState == false && newSubComponentState != existingSubComponentState)
                            {
                                isCombinedState = true;
                            }
                            if (k == existingStates.length) // found - so its a load
                            {
                                fireStateChangeEvent(OPERATION_LOAD, configID, IAgentProxy.ID, containerStates[i], newStates[j]);
                                break;
                            }
                            if (((IComponentIdentity)existingStates[k].getRuntimeIdentity()).getComponentName().equals(componentName))
                            {
                                if (newStates[j].getState() != existingStates[k].getState())
                                {
                                    fireStateChangeEvent(newStates[j].getState(), configID, componentName, containerStates[i], newStates[j]);

                                    if (componentName.equals(IAgentManagerProxy.GLOBAL_ID) && isOnlyAgentManager(componentID, containerStates))
                                    {
                                        handleAgentManagerStateChange(newStates[j].getState(), containerStates[i].getTimeStamp());
                                    }
                                }
                                break;
                            }
                        }
                    }
                    if(isCombinedState){
                        fireStateChangeEvent(StateHelper.STATE_COMBINED, configID, IAgentProxy.ID, containerStates[i], null);
                    }
                }
            }

            if(MgmtConsole.DEVELOPER_MODE)
            {
                // Debug code
                StringBuffer buffer = new StringBuffer("ContainerStates\r\n");
                Set keys = m_containerStatesByLogicalID.keySet();
                for (Iterator iter = keys.iterator(); iter.hasNext(); )
                {
                    String logicalName = (String) iter.next();
                    IContainerState state = (IContainerState)
                        m_containerStatesByLogicalID.get(logicalName);

                    buffer.append(logicalName).append(" - ").append(state.toString());
                    buffer.append("\r\n");
                    IComponentState[] components = state.getComponentStates();
                    for (int i = 0; i < components.length; i++)
                    {
                        buffer.append("\t").append(components[i].getRuntimeIdentity().getCanonicalName());
                        buffer.append(" - ").append(components[i].getStateString());
                        buffer.append("\r\n");
                        buffer.append("\t\tSub Component States:\r\n");
                        Map subComponentStates = components[i].getSubComponentStates();
                        Set entrySet = subComponentStates.entrySet();
                        Iterator entryIterator = entrySet.iterator();
                        while(entryIterator.hasNext()){
                            Map.Entry entry = (Map.Entry) entryIterator.next();
                            ISubComponentState[] subCompStates = (ISubComponentState[])entry.getValue();
                            for(int ctr =0;ctr<subCompStates.length;ctr++){
                                buffer.append("\t\t\t").append(subCompStates[ctr].getSubComponentName()).append("-").append(subCompStates[ctr].getStateString());
                                buffer.append("\r\n");
                            }
                        }
                    }
                }
                Helper.logDebugMessage(buffer.toString());
            }
        }
    }

    private boolean isOnlyAgentManager(IComponentIdentity amID, IContainerState[] containerStates)
    {
        String amContainerID = amID.getDomainName() + '.' + amID.getContainerName();

        // is there another AM (this is likely a fault tolerant deployment)
        for (int j = 0; j < containerStates.length; j++)
        {
            // if its the same container as the AM we havve found, then skip
            if (containerStates[j].getRuntimeIdentity().getCanonicalName().equals(amContainerID))
            {
                continue;
            }

            IComponentState[] states = containerStates[j].getComponentStates();
            for (int k = 0; k < states.length; k++)
            {
                String componentName = ((IComponentIdentity)states[k].getRuntimeIdentity()).getComponentName();

                // if we find another AM then we don't know which AM has gone down, so we should get
                // out of the loops right away and perform another poll
                if (componentName.equals(IAgentManagerProxy.GLOBAL_ID))
                {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * This version sets the given list of container states to the state provided (with a timestamp
     * as provided).
     */
    private void setContainerStates(IContainerState[] containerStates, short newState, long timestamp)
    {
        synchronized(m_containerStatesByConfigID)
        {
            // update as required
            for (int i = 0; i < containerStates.length; i++)
            {
                String containerName = containerStates[i].getRuntimeIdentity().getContainerName();
                String configID      = containerStates[i].getRuntimeIdentity().getConfigIdentity().getName();

                IContainerState existingContainerState = (IContainerState)m_containerStatesByConfigID.get(configID);

                if (existingContainerState.getTimeStamp() <= containerStates[i].getTimeStamp() &&
                    existingContainerState.getState() != newState)
                {
                    ((ContainerState)existingContainerState).setState(newState);
                    ((ContainerState)existingContainerState).setTimestamp(timestamp);

                    IComponentState[] componentStates = containerStates[i].getComponentStates();

                    if (componentStates.length == 0)
                    {
                        fireStateChangeEvent(containerStates[i].getState(), configID, IAgentProxy.ID, containerStates[i], null);
                    }
                    else
                    {
                        for (int j = 0; j < componentStates.length; j++)
                        {
                            if (componentStates[j].getState() != newState)
                            {
                                ((ComponentState)componentStates[j]).setState(newState);

                                fireStateChangeEvent(componentStates[j].getState(), configID, ((IComponentIdentity)componentStates[j].getRuntimeIdentity()).getComponentName(), existingContainerState, componentStates[j]);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Since the Agent Manager is the source of all state information we can make certain assumptions when
     * we get a change in state of that component.
     */
    private void handleAgentManagerStateChange(short state, long timestamp)
    {
        if (state == IComponentState.STATE_OFFLINE)
        {
            // we know immediately we can't get any domain status anymore, so lets reflect that asap
            synchronized(m_containerStatesByConfigID)
            {
                setContainerStates(getAllContainerStates(), IContainerState.STATE_UNKNOWN, timestamp);
            }
        }
        else if (state == IComponentState.STATE_ONLINE)
        {
            // we can short circuit the next poll, because we know we should now be able to get domain status
            pollImmediately();
        }
    }

    private void fireStateChangeEvent(short operation,
                                      String configID,
                                      String componentName,
                                      IContainerState containerState,
                                      IComponentState componentState)
    {
        IState state = componentName.equals(IAgentProxy.ID) ? (IState)containerState : (IState)componentState;

        String logicalName = storageToLogical(configID);

        if(logicalName == null)
        {
            return;
        }

        StateChangeEvent event = new StateChangeEvent(this, operation, logicalName, componentName, state);

        if(m_stateListeners.isEmpty())
        {
            return;
        }

        if (MgmtConsole.DEVELOPER_MODE)
        {
            Helper.logDebugMessage("FIRE operation=" + event.getOperation() +
                               ", configID=" + event.getContainerConfigName() +
                               ", ID=" + event.getComponentName() +
                               ", state=" + event.getState().getState() +
                               " (" + IComponentState.STATE_TEXT[event.getState().getState()] +
                               ")");
        }

        //Fix for Bug#SNC00066410 - ConcurrentModificationException from SMC when testing LSD
        Vector m_stateListener_clone = (Vector)m_stateListeners.clone();
        Iterator iterator = m_stateListener_clone.iterator();

        while (iterator.hasNext())
        {
            try
            {
                ((StateChangeListener)iterator.next()).domainStateChanged(event);
            }
            catch(Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                                                           "Error while notifying update to domain state",
                                                           e,
                                                           false);
            }
        }
    }

    public String logicalToStorage(String name)
    {
        String ret = null;

        if(m_configServer != null)
        {
            try
            {
                ret = m_configServer.logicalToStorage(name);
            }
            catch(Exception e)
            {
                Helper.logDebugMessage("Failed to perform logicalToStorage for " + name);
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }
        }
        return ret;
    }

    public String storageToLogical(String id)
    {
        String ret = null;

        if(m_configServer != null)
        {
            try
            {
                ret = m_configServer.storageToLogical(id);
            }
            catch(Exception e)
            {
                Helper.logDebugMessage("Failed to perform storageToLogical for " + id);
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }
        }
        return ret;
    }
}