/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.domain;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Objects;
import java.util.StringTokenizer;

import javax.management.ObjectName;
import javax.swing.JDialog;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.IConfigServer;

import com.sonicsw.mf.comm.IRetryCallback;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.Version;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;
import com.sonicsw.mf.jmx.client.IConnectionListener;
import com.sonicsw.mf.jmx.client.IOrphanedReplyListener;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;
import com.sonicsw.mf.mgmtapi.runtime.MFProxyFactory;

public class DomainConnectionModel
implements IOrphanedReplyListener, IConnectionListener, IRetryCallback, Serializable
{
    public  static final int    DEF_TIMEOUT                       = 30;
    private static final int    DEF_CONNECT_TIMEOUT               = 10;
    private static final int    DEF_SUBSCRIPTION_TIMEOUT          = 600;
    private static final int    DEF_SUBSCRIPTION_RENEWAL_INTERVAL = 30;
    private static final String DEF_DOMAIN_NAME                   = "Domain1";
    private static final String DEF_CONNECTION_URL                = "tcp://localhost:2506";
    private static final String DEF_CONNECTION_NAME               = "Connection1";
    public  static final int    DEF_MAX_CONNECTIONS               = 10;
    public  static final int    DEF_SOCKET_CONNECT_TIMEOUT        = 5;

    public static String PREFS_CONNECTION_NAME     = "connectionName";
    public static String PREFS_DOMAIN_NAME         = "domainName";
    public static String PREFS_URL                 = "url";
    public static String PREFS_TIMEOUT             = "timeout";
    public static String PREFS_USER_NAME           = "userName";
    public static String PREFS_PASSWORD            = "password";            //NOSONAR field change is not required.
    public static String PREFS_USE_DRA             = "useRouting";
    public static String PREFS_LOAD_BALANCING      = "loadBalancing";
    public static String PREFS_ENABLE_COMPRESSION  = "enableCompression";
    public static String PREFS_NODE                = "managementNode";
    public static String PREFS_MAX_CONNECTIONS     = "maxConnections";

    public static String PREFS_CONNECT_TIMEOUT               = "connectTimeout";
    public static String PREFS_SUBSCRIPTION_TIMEOUT          = "notificationSubscriptionTimeout";
    public static String PREFS_SUBSCRIPTION_RENEWAL_INTERVAL = "notificationSubscriptionRenewalTimeout";
    public static String PREFS_SOCKET_CONNECT_TIMEOUT        = "socketConnectTimeout";

    private String  m_connectionName              = DEF_CONNECTION_NAME;
    private String  m_domainName                  = DEF_DOMAIN_NAME;
    private String  m_url                         = DEF_CONNECTION_URL;
    private int     m_requestTimeout              = DEF_TIMEOUT;
    private int     m_connectTimeout              = DEF_CONNECT_TIMEOUT;
    private int     m_subscriptionTimeout         = DEF_SUBSCRIPTION_TIMEOUT;
    private int     m_subscriptionRenewalInterval = DEF_SUBSCRIPTION_RENEWAL_INTERVAL;
    private String  m_userName                    = (MgmtConsole.DEVELOPER_MODE) ? "Administrator" : null;
    private String  m_password                    = (MgmtConsole.DEVELOPER_MODE) ? "Administrator" : null;
    private String  m_managementNode;
    private boolean m_loadBalancing               = true;
    private boolean m_enableCompression			  = false;
    private boolean m_useDRA                      = false;
    private boolean m_default                     = false;
    
    private int  m_socketConnectTimeout = DEF_SOCKET_CONNECT_TIMEOUT;

    private ConfigServerUtility    m_csu;
    private IAgentManagerProxy     m_amProxy;
    private AgentManagerConnection m_amConnection;
    private ArrayList              m_stateListenerRegistrationThreads;

    /**
     * Constructs a DomainConnectionModel for the default connection.
     */
    public DomainConnectionModel(PreferenceManager prefs)
    {
        this(prefs, getDefaultFromPrefs(prefs));
        setDefaultProperty(prefs);
    }

    private void setDefaultProperty(PreferenceManager prefs) {
        if (getDefaultFromPrefs(prefs) == null)
        {
            setDefault(true);
        }
    }
    
    public DomainConnectionModel(PreferenceManager prefs, String name)
    {
        initFromPrefs(prefs, (name != null) ? name : DEF_CONNECTION_NAME);

        if (MgmtConsole.DEVELOPER_MODE) {
            // Clear out default DEVELOPER_MODE password if URL points directly to DS boot file
            m_password = m_url.endsWith(".xml") ? null : m_password;
        }
    }

    public int getUniqueId()
    {
        return new StringBuffer(m_url).append("/").append(m_domainName).toString().hashCode();
    }

    /**
     * Connect to everything
     * @throws Exception
     */
    public void connect()
        throws Exception
    {
        m_csu = new ConfigServerUtility();

        m_csu.setRequestTimeout(getTimeout());
        m_csu.setConnectTimeout(getConnectTimeout());
        m_csu.setLoadBalancing(isLoadBalancing());
        m_csu.setEnableCompression(isEnableCompression());
        m_csu.setUseDRA(isUseDRA());
        m_csu.setManagementNode(getManagementNode());
        m_csu.setRequiredVersion(new String(Version.getMajorVersion() + "." + Version.getMinorVersion()));
        m_csu.setSocketConnectTimeout(getSocketConnectTimeout());
        m_csu.connect(getDomainName(), getUrl(), getUserName(), getPassword(), true);

        IRemoteMBeanServer server = m_csu.getMBeanServer();

        if(server != null)
        {
            ((JMSConnectorClient)server).setOrphanedReplyListener(this);
            ((JMSConnectorClient)server).setConnectionListener(this);
            ((JMSConnectorClient)server).registerRetryCallback(this);
            ((JMSConnectorClient)server).setNotificationSubscriptionTimeout(this.getNotificationSubscriptionTimeout());
            ((JMSConnectorClient)server).setNotificationSubscriptionRenewalInterval(this.getNotificationSubscriptionRenewalInterval());
        }

        if(!m_csu.isLocal())
        {
            m_amProxy      = createAMProxy(server);
            m_amConnection = new AgentManagerConnection(m_amProxy, m_csu.getConfigServer());

            registerStateNotificationListener();
        }
        else
        {
            m_amConnection = new AgentManagerConnection(null, m_csu.getConfigServer());
        }

        if (m_amConnection != null)
        {
            m_amConnection.startAMConnection();
        }
    }

    public void disconnect()
        throws Exception
    {
        MgmtConsole console = MgmtConsole.getMgmtConsole();

        // Close early before it would happen through the ConfigServerUtilityDisconnect
        // because we want to do a special close that avoids unregistering change
        // listeners. This is to avoid possible timeout if Management Container has
        // already been shutdown.
        //
        if (m_csu.getConfigServer() != null)
        {
            ((com.sonicsw.mx.config.impl.ConfigServer)m_csu.getConfigServer()).close(false);
        }

        if (m_stateListenerRegistrationThreads != null)
        {
            Object[] registrars = m_stateListenerRegistrationThreads.toArray();
            for (int i = 0; i < registrars.length; i++)
            {
                if (((Thread)registrars[i]).isAlive())
                {
                    ((Thread)registrars[i]).interrupt();
                }
            }
        }

        try
        {
            if(m_amConnection != null)
            {
                m_amConnection.disconnect();
            }
        }
        catch(Throwable e)
        {
            console.notifyMessage(MgmtConsole.MESSAGE_ERROR,
                "Failed to disconnect Agent Manager connection", e, false);
        }
        finally
        {
            m_amConnection = null;
        }

        m_amProxy = null;

        IRemoteMBeanServer server = getMBeanServer();

        if (server != null)
        {
            ((JMSConnectorClient)server).deregisterRetryCallback();
            ((JMSConnectorClient)server).setOrphanedReplyListener(null);
        }

        if(m_csu != null)
        {
            try
            {
                m_csu.disconnect();
            }
            catch(Throwable e)
            {
                console.notifyMessage(MgmtConsole.MESSAGE_ERROR,
                    "Failed to disconnect Config Server", e, false);
            }
            finally
            {
                m_csu = null;
            }
        }
    }

    public boolean isDefault()
    {
        return m_default;
    }

    public void setDefault(boolean value)
    {
        m_default = value;
    }

    public String getConnectionName()
    {
        return m_connectionName;
    }

    public void setConnectionName(String name)
    {
        m_connectionName = (name != null) ? name.trim() : null;
    }

    public void setTimeout(int timeout)
    {
        m_requestTimeout = timeout;
    }

    public int getTimeout()
    {
        return m_requestTimeout;
    }

    public String getDomainName()
    {
        return m_domainName;
    }

    public void setDomainName(String name)
    {
        m_domainName = (name != null) ? name.trim() : null;
    }

    public String getUrl()
    {
        return m_url;
    }

    public void setUrl(String url)
    {
        m_url = (url != null) ? url.trim() : null;
    }

    public String getUserName()
    {
        return m_userName;
    }

    public void setUserName(String name)
    {
        m_userName = (name != null) ? name.trim() : null;
    }

    public String getPassword()
    {
        return m_password;
    }

    public void setPassword(String name)
    {
        m_password = name;
    }

    public boolean isLoadBalancing()
    {
        return m_loadBalancing;
    }

    public void setLoadBalancing(boolean value)
    {
        m_loadBalancing = value;
    }

    public void setEnableCompression(boolean value) {
		this.m_enableCompression = value;
	}

	public boolean isEnableCompression() {
		return m_enableCompression;
	}

	public boolean isUseDRA()
    {
        return m_useDRA;
    }

    public void setUseDRA(boolean value)
    {
        m_useDRA = value;
    }

    public String getManagementNode()
    {
        return m_managementNode;
    }

    public void setManagementNode(String node)
    {
        m_managementNode = (node != null) ? node.trim() : null;
    }

    public String getDomainDescription()
    {
        return m_connectionName + " (" + m_domainName + "@" + m_url + ")";
    }

    public int getSocketConnectTimeout() {
        return m_socketConnectTimeout;
    }

    public void setSocketConnectTimeout(int socketConnectTimeout) {
        this.m_socketConnectTimeout = socketConnectTimeout;
    }

    public void setConnectTimeout(int timeout)
    {
        m_connectTimeout = timeout;
    }

    public int getConnectTimeout()
    {
        return m_connectTimeout;
    }

    public void setNotificationSubscriptionTimeout(int timeout)
    {
        m_subscriptionTimeout = timeout;
    }

    public int getNotificationSubscriptionTimeout()
    {
        return m_subscriptionTimeout;
    }

    public void setNotificationSubscriptionRenewalInterval(int interval)
    {
        m_subscriptionRenewalInterval = interval;
    }

    public int getNotificationSubscriptionRenewalInterval()
    {
        return m_subscriptionRenewalInterval;
    }

    public IConfigServer getConfigServer()
    {
        return m_csu.getConfigServer();
    }

    public ConfigServerUtility getConfigServerUtility()
    {
        return m_csu;
    }

    public IRemoteMBeanServer getMBeanServer()
    {
        return m_csu.getMBeanServer();
    }

    public IDirectoryFileSystemService getDirectoryService()
    {
        return m_csu.getDirectoryService();
    }

    public IAgentManagerProxy getAgentManager()
    {
        return m_amProxy;
    }

    public AgentManagerConnection getAgentManagerConnection()
    {
        return m_amConnection;
    }

    public String getAgentManagerName()
    {
        return getDomainName() + '.' + IAgentManagerProxy.GLOBAL_ID + IComponentIdentity.DELIMITED_ID_PREFIX + IAgentManagerProxy.GLOBAL_ID;
    }

    public String getDirectoryServiceName()
    {
        return getDomainName() + '.' + IDirectoryServiceProxy.GLOBAL_ID + IComponentIdentity.DELIMITED_ID_PREFIX + IDirectoryServiceProxy.GLOBAL_ID;
    }

    public boolean isConnected()
    {
        IRemoteMBeanServer server = getMBeanServer();

        if (server != null)
        {
            return server.isConnected();
        }

        return true;
    }

    /**
     * Create IAgentManagerProxy
     */
    private IAgentManagerProxy createAMProxy(IRemoteMBeanServer server)
        throws Exception
    {
        String amName = getAgentManagerName();

        return MFProxyFactory.createAgentManagerProxy((JMSConnectorClient)server, new ObjectName(amName));
    }

    public void registerStateNotificationListener()
    {
        m_stateListenerRegistrationThreads = new ArrayList();

        try
        {
            javax.management.NotificationFilterSupport filter = new javax.management.NotificationFilterSupport();
            for (int i = 0; i < AgentManagerConnection.HANDLED_EVENTS.length; i++)
            {
                filter.enableType("system.state." + AgentManagerConnection.HANDLED_EVENTS[i]);
            }

            ObjectName amName = new ObjectName(getAgentManagerName());
            new Registrar(amName, filter).start();

        }
        catch(Throwable e)
        {
            String msg =  "Failed to establish notification listener's subscription - " + getDomainDescription();

            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING, msg, e, false);
        }
    }

    private class Registrar
    extends Thread
    {
        private ObjectName amName;
        private javax.management.NotificationFilterSupport filter;

        private Registrar(ObjectName amName, javax.management.NotificationFilterSupport filter)
        {
            super("State Notification Listener Registrar - " + amName.getCanonicalName());
            this.amName = amName;
            this.filter = filter;
            super.setDaemon(true);
            DomainConnectionModel.this.m_stateListenerRegistrationThreads.add(this);
        }

        @Override
        public void run()
        {
            IRemoteMBeanServer server = DomainConnectionModel.this.getMBeanServer();

            if (server != null && !isInterrupted())
            {
                try
                {
                    server.addNotificationListener(amName, DomainConnectionModel.this.m_amConnection, filter, null);
                }
                catch(Throwable e)
                {
                    if (isInterrupted())
                    {
                        return;
                    }

                    String msg =  "Failed to establish notification listener's subscription - " + DomainConnectionModel.this.getDomainDescription();
                    MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING, msg, e, false);
                }
                finally
                {
                    DomainConnectionModel.this.m_stateListenerRegistrationThreads.remove(this);
                }
            }
        }
    }

    public void unregisterStateNotificationListener()
    {
        try
        {
            ObjectName amName = new ObjectName(getAgentManagerName());
            IRemoteMBeanServer server = DomainConnectionModel.this.getMBeanServer();

            server.removeNotificationListener(amName, m_amConnection);
        }
        catch (Throwable e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                 "Failed to unregister state notification listener", e, false);
        }
    }

    private void initFromPrefs(PreferenceManager prefs, String name)
    {
        m_connectionName = prefs.getString("connections." + name,
                                           PREFS_CONNECTION_NAME, m_connectionName);
        m_domainName     = prefs.getString("connections." + name,
                                           PREFS_DOMAIN_NAME, m_domainName);
        m_url            = prefs.getString("connections." + name,
                                           PREFS_URL, m_url);

        int defTimeout   = prefs.getInt(JPreferencesDialog.GENERAL_PREFS,
                                        PREFS_TIMEOUT, DEF_TIMEOUT);
        m_requestTimeout = prefs.getInt("connections." + name,
                                           PREFS_TIMEOUT, defTimeout);
        m_userName       = prefs.getString("connections." + name,
                                           PREFS_USER_NAME, "");
        m_loadBalancing  = prefs.getBoolean("connections." + name,
                                           PREFS_LOAD_BALANCING, true);
        m_enableCompression = prefs.getBoolean("connections."+name, 
        									PREFS_ENABLE_COMPRESSION, false);
        m_useDRA         = prefs.getBoolean("connections." + name,
                                           PREFS_USE_DRA, false);
        m_managementNode = prefs.getString("connections." + name,
                                           PREFS_NODE, "");

        m_connectTimeout  = prefs.getInt("connections." + name,
                                           PREFS_CONNECT_TIMEOUT, DEF_CONNECT_TIMEOUT);

        m_socketConnectTimeout = prefs.getInt("connections." + name,
                                              PREFS_SOCKET_CONNECT_TIMEOUT, DEF_SOCKET_CONNECT_TIMEOUT);
        m_subscriptionTimeout = prefs.getInt("connections." + name,
                                             PREFS_SUBSCRIPTION_TIMEOUT, DEF_SUBSCRIPTION_TIMEOUT);
        m_subscriptionRenewalInterval = prefs.getInt("connections." + name,
                                                     PREFS_SUBSCRIPTION_RENEWAL_INTERVAL, DEF_SUBSCRIPTION_RENEWAL_INTERVAL);

        // Determine default status by comparing connection name with the
        // stored default.
        String defConnectionName = getDefaultFromPrefs(prefs);
        if ((defConnectionName != null) && name.equals(defConnectionName))
        {
            m_default = true;
        }
    }

    public void saveToPrefs(PreferenceManager prefs)
    {
        ArrayList firstN = getFirstN(prefs);

        prefs.setString("connections." + m_connectionName,
                        PREFS_CONNECTION_NAME, m_connectionName, false);
        prefs.setString("connections." + m_connectionName,
                        PREFS_DOMAIN_NAME, m_domainName, false);
        prefs.setString("connections." + m_connectionName,
                        PREFS_URL, m_url, false);

        if (m_requestTimeout == -1)
        {
            prefs.remove("connections." + m_connectionName, PREFS_TIMEOUT, false);
        }
        else
        {
            prefs.setInt("connections." + m_connectionName, PREFS_TIMEOUT, m_requestTimeout, false);
        }

        if ((m_userName != null) && (m_userName.length() != 0))
        {
            prefs.setString("connections." + m_connectionName, PREFS_USER_NAME, m_userName, false);
        }
        else
        {
            prefs.remove("connections." + m_connectionName, PREFS_USER_NAME, false);
        }

        prefs.setBoolean("connections." + m_connectionName, PREFS_LOAD_BALANCING, m_loadBalancing, false);
        
        prefs.setBoolean("connections."+m_connectionName, PREFS_ENABLE_COMPRESSION, m_enableCompression, false);

        //Fix for bug#SNC00060121  

        m_useDRA = ((m_managementNode != null) && (m_managementNode.trim().length() != 0));
        prefs.setBoolean("connections." + m_connectionName, PREFS_USE_DRA,  m_useDRA, false);

        if(isUseDRA())
        {
            prefs.setString("connections." + m_connectionName, PREFS_NODE, m_managementNode, false);
        }
        else
        {
            prefs.setString("connections." + m_connectionName, PREFS_NODE, "", false);
        }

        if (m_connectTimeout == -1)
        {
            prefs.remove("connections." + m_connectionName, PREFS_CONNECT_TIMEOUT, false);
        }
        else
        {
            prefs.setInt("connections." + m_connectionName, PREFS_CONNECT_TIMEOUT, m_connectTimeout, false);
        }

        if (m_socketConnectTimeout == -1)
        {
            prefs.remove("connections." + m_connectionName, PREFS_SOCKET_CONNECT_TIMEOUT, false);
        }
        else
        {
            prefs.setInt("connections." + m_connectionName, PREFS_SOCKET_CONNECT_TIMEOUT, m_socketConnectTimeout, false);
        }

        if (m_subscriptionTimeout == -1)
        {
            prefs.remove("connections." + m_connectionName, PREFS_SUBSCRIPTION_TIMEOUT, false);
        }
        else
        {
            prefs.setInt("connections." + m_connectionName, PREFS_SUBSCRIPTION_TIMEOUT, m_subscriptionTimeout, false);
        }

        if (m_subscriptionRenewalInterval == -1)
        {
            prefs.remove("connections." + m_connectionName, PREFS_SUBSCRIPTION_RENEWAL_INTERVAL, false);
        }
        else
        {
            prefs.setInt("connections." + m_connectionName, PREFS_SUBSCRIPTION_RENEWAL_INTERVAL, m_subscriptionRenewalInterval, false);
        }


        if(isDefault())
        {
            saveDefaultToPrefs(prefs, m_connectionName);
        }
        else
        {
            String defName = getDefaultFromPrefs(prefs);

            if ((defName != null) && defName.equals(m_connectionName))
            {
                removeDefaultFromPrefs(prefs);
            }
        }

        int maxConnections = prefs.getInt(JPreferencesDialog.GENERAL_PREFS, PREFS_MAX_CONNECTIONS, DEF_MAX_CONNECTIONS);

        // check if this connection is in the firstN. If it isn't add it to the
        // beginning of the list. If it is then remove it from its current position first
        if(firstN.contains(m_connectionName))
        {
            firstN.remove(m_connectionName);
        }

        firstN.add(0, m_connectionName);

        // if the firstN list is too big then delete the last
        String defConName = getDefaultFromPrefs(prefs);
        while(firstN.size() > maxConnections)
        {
            String remove = (String)firstN.get(maxConnections);
            if (defConName != null && defConName.equals(remove))
            {
                remove = (String) firstN.remove(maxConnections-1);
            }
            else
            {
                remove = (String) firstN.remove(maxConnections);
            }
            prefs.removeNode("connections." + remove);
        }

        saveFirstN(prefs, firstN);

        prefs.flush("connections");
    }

    @Override
    public boolean equals(Object object)
    {
        if (object == null || this.getClass() != object.getClass())
        {
            return false;
        }

        DomainConnectionModel dcm = (DomainConnectionModel)object;

        if (!dcm.getDomainName().equals(this.getDomainName()))
        {
            return false;
        }

        return true;
    }
    
    @Override
    public int hashCode() {
        return Objects.hashCode(getDomainName());
    }

    @Override
    public String toString()
    {
        StringBuffer sb = new StringBuffer("Connection (");

        sb.append("name=").append(getConnectionName()).append(", ");
        sb.append("domain=").append(getDomainName()).append(", ");
        sb.append("url=").append(getUrl()).append(", ");
        sb.append("user=").append(getUserName()).append(", ");
        sb.append("timeout=").append(getTimeout()).append(", ");
        sb.append("load balancing=").append(isLoadBalancing());
        if (getNotificationSubscriptionTimeout() >= 0)
        {
            sb.append("notification subscription timeout=").append(getNotificationSubscriptionTimeout());
        }
        if (getNotificationSubscriptionRenewalInterval() >= 0)
        {
            sb.append("notification subscription renewal interval=").append(getNotificationSubscriptionRenewalInterval());
        }

        String managementNode = getManagementNode();

        if (isUseDRA() && (managementNode != null) && (managementNode.length() > 0))
        {
            sb.append(", management node=").append(managementNode);
        }
        //FT
        sb.append("connect_timeout=").append(getConnectTimeout()).append(", ");
        sb.append("socket_connect_timeout=").append(getSocketConnectTimeout()).append(", ");
        sb.append("enable compresiion=").append(isEnableCompression()).append(", ");
        sb.append(")");

        return sb.toString();
    }

    public static ArrayList getFirstN(PreferenceManager prefs)
    {
        ArrayList list = new ArrayList();

        String value = prefs.getString("connections", "firstN", null);

        if(value != null)
        {
            StringTokenizer st = new StringTokenizer(value, ",");

            while(st.hasMoreTokens())
            {
                list.add(st.nextToken());
            }
        }
        return list;
    }

    private static void saveFirstN(PreferenceManager prefs, ArrayList list)
    {
        StringBuffer buffer = new StringBuffer();

        for(int i = 0; i < list.size(); i++)
        {
            if(buffer.length() > 0)
            {
                buffer.append(',');
            }

            buffer.append((String)list.get(i));
        }

        prefs.setString("connections", "firstN", buffer.toString(), false);
    }

    private static String getDefaultFromPrefs(PreferenceManager prefs)
    {
        return prefs.getString("connections", "defaultConnectionName", null);
    }

    private static void saveDefaultToPrefs(PreferenceManager prefs, String name)
    {
        prefs.setString("connections", "defaultConnectionName", name, false);
    }

    private static void removeDefaultFromPrefs(PreferenceManager prefs)
    {
        prefs.remove("connections", "defaultConnectionName", false);
    }

    //-------------------------------------------------------------------------
    //
    //  IOrphanedReplyListener
    //

    /**
     * Called when a valid reply arrives after the original request had timed out.
     *
     * @see JMSConnectorClient#setOrphanedReplyListener(IOrphanedReplyListener)
     */
    @Override
    public void onSuccess(String target,
                          String operationName,
                          Object returnValue,
                          long requestReceived,
                          long requestCompleted)
    {
        int duration = (int)((requestCompleted - requestReceived) / 1000);
        String msg = "Operation '" + operationName + "' on '" + target +
                     "' completed successfully (Duration " + duration + " seconds).";

        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_INFO, msg, false);
    }

    /**
     * Called when a exception reply arrives after the original request had timed out.
     *
     * @see JMSConnectorClient#setOrphanedReplyListener(IOrphanedReplyListener)
     */
    @Override
    public void onFailure(String target,
                          String operationName,
                          Exception e,
                          long requestReceived,
                          long requestAborted)
    {
        int duration = (int)((requestAborted - requestReceived) / 1000);
        String msg = "Operation '" + operationName + "' on '" + target +
                     "' failed (Duration " + duration + " seconds).";

        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_INFO, msg, e, false);
    }

    //-------------------------------------------------------------------------
    //
    //  IConnectionListener
    //

    /**
     * Called when the connector reconnects to the underlying trasnport.
     *
     * @see JMSConnectorClient#setConnectionListener(IConnectionListener)
     */
    @Override
    public void onReconnect(String localRoutingNode)
    {
        String msg = "Reconnected to " + getDomainDescription();
        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_STATUS, msg, false);
        m_amConnection.pollImmediately();
    }

    /**
     * Called when the connector disconnects from the underlying trasnport.
     *
     * @see JMSConnectorClient#setConnectionListener(IConnectionListener)
     */
    @Override
    public void onDisconnect()
    {
        String msg = "Disconnected from " + getDomainDescription();

        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_STATUS, msg, false);
    }

    /**
     * Called when the connector detects some permanent failure in the underlying trasnport.
     *
     * @see JMSConnectorClient#setConnectionListener(IConnectionListener)
     */
    @Override
    public void onFailure(Exception e)
    {
        String msg = "Connection failure to " + getDomainDescription();

        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_STATUS, msg, e, false);
    }

    /**
     * Called when a connector is unable to renew a notification listener's subscription.
     * <p>
     * When notification listeners are added (and until they are removed), the connector will
     * attempt to periodically renew the listener's subscription as part of a feature that
     * avoids stale listeners on the server side, yet allows for temporary loss of target (e.g.
     * due to a component's container being shutdown and restarted) without having to re-add
     * the listener to the connector when the target again becomes available.
     */
    @Override
    public void onNotificationListenerRenewalFailure(Exception e)
    {
        String msg =  "Failed to renew notification listener's subscription - " + getDomainDescription();

        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING, msg, e, (e instanceof ManagementPermissionDeniedException));
    }

    //-------------------------------------------------------------------------
    //
    //  IRetryCallback
    //

    /**
     * Called when request failover occurs.
     *
     * @see JMSConnectorClient#registerRetryCallback(IRetryCallback)
     */
    @Override
    public short onRequestFailure( String failedTarget,
                                   String failedConnectionURL,
                                   Exception e,
                                   short[] allowedResponseValues )
    {
          short response = IRetryCallback.CANCEL_REQUEST;


          JFailoverDialog dlg = new JFailoverDialog( MgmtConsole.getMgmtConsole(),
                                                     failedTarget,
                                                     failedConnectionURL,
                                                     e,
                                                     allowedResponseValues);

          dlg.setVisible(true);

          if (dlg.getDefaultCloseOperation() == JDialog.DO_NOTHING_ON_CLOSE)
        {
            response = dlg.getResponseValue();
        }

          return response;
    }

    protected boolean isAllowedValue(short[] allowedValues, short value)
    {
        for(int i = 0 ; i < allowedValues.length; i++)
        {
            if (allowedValues[i] == value)
            {
                return true;
            }
        }

        return false;
    }

}
