/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.domain;

import java.awt.Dimension;

import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;

import modelobjects.framework.EditRule;
import modelobjects.framework.IntToStringConverter;
import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelObjectValidator;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.swing.JCheckBoxAdapter;
import modelobjects.framework.swing.JTextFieldAdapter;

import com.sonicsw.ma.gui.propsheets.JPropSheetTabbedDialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.ResourceManager;

import com.sonicsw.mf.comm.jms.ConnectorClient;

public class JDomainConnectionDialog extends JPropSheetTabbedDialog
{
    private JTextField     m_connectionName = new JTextField(25);
    private JTextField     m_domainName = new JTextField();
    private JTextField     m_url = new JTextField();
    private JTextField     m_userName = new JTextField();
    private JPasswordField m_password = new JPasswordField();
    private JCheckBox      m_default = new JCheckBox();
  
    private JTextField     m_managementNode = new JTextField();
    private JTextField     m_connectTimeout = new JTextField();
    private JTextField     m_socket_connect_timeout  = new JTextField();
    private JTextField     m_timeout = new JTextField();
    private JTextField     m_subscriptionTimeout = new JTextField();
    private JTextField     m_subscriptionRenewalInterval = new JTextField();
    private JCheckBox      m_loadBalancing = new JCheckBox();
    private JCheckBox	   m_enableCompression = new JCheckBox();

    public JDomainConnectionDialog(JMAFrame parent)
    {
        super(parent, "domain.connection", DomainConnectionModel.class);
        setResizableProperty();
        setValidator(new DomainConnectionValidator());
    }
    
    private void setResizableProperty() {
        setResizable(false);
    }

    @Override
    public Dimension getPreferredSize()
    {
        return preferredSize();
    }

    @Override
    protected void initUI() throws Exception
    {
        String generalTab  = ResourceManager.getString(getClass(), "dialog.domain.tab.general.label");
        String ftTab       = ResourceManager.getString(getClass(), "dialog.domain.tab.ft.label");
        String advancedTab = ResourceManager.getString(getClass(), "dialog.domain.tab.advanced.label");

        addTabPanel(generalTab,  createGeneralPanel());
        addTabPanel(advancedTab, createAdvancedPanel());
    }

    @Override
    protected void initForm() throws Exception
    {
        createJTextFieldAdapter("connectionName", m_connectionName, EditRule.ALWAYS);
        createJTextFieldAdapter("domainName",     m_domainName,     EditRule.ALWAYS);
        createJTextFieldAdapter("url",            m_url,            EditRule.ALWAYS);
        createJTextFieldAdapter("userName",       m_userName,       EditRule.ALWAYS);
        createJTextFieldAdapter("password",       m_password,       EditRule.ALWAYS);
        createJCheckBoxAdapter("default",         m_default,        EditRule.ALWAYS);
      
        createJTextFieldAdapter("managementNode",                          m_managementNode,              EditRule.ALWAYS); 
        createJTextFieldAdapter("connectTimeout",                          m_connectTimeout,              EditRule.ALWAYS, new TimeoutConverter(ConnectorClient.CONNECT_TIMEOUT_MINIMUM));
        createJTextFieldAdapter("socketConnectTimeout",                    m_socket_connect_timeout ,     EditRule.ALWAYS, new TimeoutConverter(0));
        createJTextFieldAdapter("timeout",                                 m_timeout,                     EditRule.ALWAYS, new TimeoutConverter(ConnectorClient.REQUEST_TIMEOUT_MINIMUM));
        createJTextFieldAdapter("notificationSubscriptionTimeout",         m_subscriptionTimeout,         EditRule.ALWAYS, new IntToStringConverter());
        createJTextFieldAdapter("notificationSubscriptionRenewalInterval", m_subscriptionRenewalInterval, EditRule.ALWAYS, new IntToStringConverter());
        createJCheckBoxAdapter("loadBalancing",                            m_loadBalancing,               EditRule.ALWAYS);
        createJCheckBoxAdapter("enableCompression",						   m_enableCompression,			  EditRule.ALWAYS);
    }

    private JPanel createGeneralPanel()
    {
        String paramGroup     = ResourceManager.getString(getClass(), "dialog.domain.general.parameters.label");
        String connectionName = ResourceManager.getString(getClass(), "dialog.domain.general.connection_name.label");
        String domainName     = ResourceManager.getString(getClass(), "dialog.domain.general.domain_name.label");
        String url            = ResourceManager.getString(getClass(), "dialog.domain.general.connection_url.label");
        String user           = ResourceManager.getString(getClass(), "dialog.domain.general.user_name.label");
        String password       = ResourceManager.getString(getClass(), "dialog.domain.general.password.label");
        String cbDefault      = ResourceManager.getString(getClass(), "dialog.domain.general.default.label");

        m_default.setText(cbDefault);

        JPartitionPanel panel1 = new JPartitionPanel("General Settings");
        panel1.addRow("*" + connectionName, m_connectionName);
        panel1.addRow("*" + domainName,     m_domainName);
        panel1.addRow("*" + url,            m_url);
        panel1.addRow(user,                 m_userName);
        panel1.addRow(password,             m_password);
        panel1.addRow(m_default);

        JPartitionPanel panel = new JPartitionPanel(true, "p,p", null);
        panel.add(panel1);
        return panel;
    }


    private JPanel createAdvancedPanel()
    {
        String connGroup                     = ResourceManager.getString(getClass(), "dialog.domain.advanced.group.connection.label");
        String node                          = ResourceManager.getString(getClass(), "dialog.domain.advanced.node.label");
        String loadBalancing                 = ResourceManager.getString(getClass(), "dialog.domain.advanced.load_balancing.label");
        String connect_timeout               = ResourceManager.getString(getClass(), "dialog.domain.advanced.connect_timeout.label");
        String socket_connect_timeout        = ResourceManager.getString(getClass(), "dialog.domain.ft.socket_time_out.label");
        String timeout                       = ResourceManager.getString(getClass(), "dialog.domain.advanced.timeout.label");
        String subscription_timeout          = ResourceManager.getString(getClass(), "dialog.domain.advanced.notification_subscription_timeout.label");
        String subscription_renewal_interval = ResourceManager.getString(getClass(), "dialog.domain.advanced.notification_subscription_renewal_interval.label");
        String enableCompression			 = ResourceManager.getString(getClass(), "dialog.domain.advanced.enable_compression.label");

        String unitSec = ResourceManager.getString(getClass(), new Object[] { ResourceManager.KEY_UNIT, "seconds"} );

        JPartitionPanel panel1 = new JPartitionPanel(connGroup);

        panel1.addRow(node,                          m_managementNode);
        panel1.addRow(connect_timeout,               m_connectTimeout, unitSec);
        panel1.addRow(socket_connect_timeout,        m_socket_connect_timeout,unitSec);
        panel1.addRow(timeout,                       m_timeout, unitSec);
        panel1.addRow(subscription_timeout,          m_subscriptionTimeout, unitSec);
        panel1.addRow(subscription_renewal_interval, m_subscriptionRenewalInterval, unitSec);
        panel1.addRow(loadBalancing,                 m_loadBalancing);
        panel1.addRow(enableCompression,			 m_enableCompression);

        JPartitionPanel panel = new JPartitionPanel(true, "p,p", null);
        panel.add(panel1);
        return panel;
    }

    protected JTextFieldAdapter createJTextFieldAdapter(String     propertyName,
                                                        JTextField viewComponent,
                                                        EditRule   editRule)
    {
        return createJTextFieldAdapter(propertyName, viewComponent, editRule, null);
    }

    protected JTextFieldAdapter createJTextFieldAdapter(String     propertyName,
                                                        JTextField viewComponent,
                                                        EditRule   editRule,
                                                        ViewValueConverter converter)
    {
        return new JTextFieldAdapter(ModelAspectId.forName(propertyName),
                           viewComponent, editRule, converter, getMediator());
    }

    protected JCheckBoxAdapter createJCheckBoxAdapter(String    propertyName,
                                                      JCheckBox viewComponent,
                                                      EditRule  editRule)
    {
        return new JCheckBoxAdapter(ModelAspectId.forName(propertyName),
                           viewComponent, editRule, null, getMediator());
    }

    /**
     * Override default ModelObjects error message decode to use the actually
     * exception message when a ViewValueConversionException is thrown...rather
     * than use the default ResourceManager lookup approach.
     *
     * @param throwable  The error for which a user-readable string is to be
     *                   obtained.
     * @return           The user-readable error string.
     */
    @Override
    protected String getModelEditFailureMessage(Throwable throwable)
    {
        if (throwable instanceof ViewValueConversionException)
        {
            String msg = ((ViewValueConversionException)throwable).getMessage();
            int    index = msg.lastIndexOf(',');

            return (index != -1) ? msg.substring(0, index) : msg;
        }

        return super.getModelEditFailureMessage(throwable) ;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    private class TimeoutConverter extends IntToStringConverter
    {
        private int m_minimum;

        public TimeoutConverter(long minimum)
        {
            super(false); // We don't want null values to be allowed

            m_minimum = (int)(minimum / 1000);  // convert millisecs into secs
        }

        @Override
        public Object viewValueToModel(Object viewValue)
            throws ViewValueConversionException
        {
            Integer res   = (Integer)super.viewValueToModel(viewValue);
            int     value = ((Integer)res).intValue();

            if (value < m_minimum)
            {
                throw new ViewValueConversionException("Invalid timeout - value must be >= " + m_minimum + " seconds.");
            }

            return res;
        }

        @Override
        public Object modelValueToView(Object modelValue)
            throws ViewValueConversionException
        {
           if ((modelValue instanceof Integer) && (((Integer)modelValue).intValue() == -1))
        {
            return null;
        }

           return super.modelValueToView(modelValue);
        }
    }

    private class DomainConnectionValidator implements ModelObjectValidator
    {
        @Override
        public void validateModelEdit(ModelObjectAdapter modelObjectAdapter)
            throws ModelValidationException
        {
        	String connectionName = (String)modelObjectAdapter.getAspectValue(ModelAspectId.forName("connectionName"));

            if (connectionName.trim().length() == 0)
            {
                throw new ModelValidationException("A value for 'Connection Name' isn't specified. This is a required field", new IllegalArgumentException("connectionName"));
            }
            
        }

        @Override
        public void validateModelCreation(ModelObjectAdapter modelObjectAdapter)
            throws ModelValidationException
        {
            validateModelEdit(modelObjectAdapter);
        }

        @Override
        public void validateModelDeletion(ModelObjectAdapter modelObjectAdapter)
            throws ModelValidationException
        {
        }
    }
}
