/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.file;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.JComponent;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.action.ExportPluginAction;
import com.sonicsw.ma.gui.propsheets.JPropSheetDialog;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.IConfigBeanFile;
import com.sonicsw.mx.config.IConfigServer;
import com.sonicsw.mx.config.impl.ConfigBeanFile;

public class FileElementPlugin extends AbstractFilePlugin
{
    public static final String PLUGIN_TYPE     = ConfigBeanFile.CONFIG_TYPE;
    public static final String CONFIG_VERSION  = ConfigBeanFile.CONFIG_VERSION;

    public static final Object[][] ATTRIBUTES =
    {
        { PluginAttributes.TYPE,           PLUGIN_TYPE },
        { PluginAttributes.DISPLAY_TYPE,   "File" },
        { PluginAttributes.CONFIG_VERSION, CONFIG_VERSION },
    };

    private FilePluginModel m_model = null;

    public FileElementPlugin(IPluginContext context, String id, Map data)
    {
        super(context, id, data);

        getAttributes().put(ATTRIBUTES);

        setDisplayable(false);
    }


    @Override
    public void refresh() {
        try {
            final FileElementModel model = (FileElementModel)getModel();
            model.refresh();
        } catch (ConfigServiceException e) {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                                       "Failed to refresh model: " + _getPluginPath(), e, false);
        }

        super.refresh();
    }

    @Override
    public void dispose() { m_model = null; }

    @Override
    public boolean expand()
    {
        return false;
    }

    public Icon getIcon()
    {
        return ((FileElementModel)getModel()).getTypeIcon();
    }

    @Override
    public JPropSheetDialog getPropSheetDialog()
    {
        return new FileElementDialog(MgmtConsole.getMgmtConsole(), getPluginFilename());
    }

    @Override
    public List getToolTipList()
    {
        List list  = super.getToolTipList();

        FileElementModel model = (FileElementModel)getModel();

        list.add(formatToolTipText("Size",          formatSize(model.getSize())));
        list.add(formatToolTipText("Date Modified", model.getLastModifiedTime()));

        return list;
    }

    public String getPluginFilename()
    {
        String strResult = getPluginPath();

        int index = strResult.lastIndexOf("/");

        if (index != -1)
        {
            strResult = strResult.substring(index + 1);
        }

        return strResult;
    }

    public String getPluginFilePath()
    {
        String strResult = getPluginPath();

        int index = strResult.lastIndexOf("/");

        if (index != -1)
        {
            strResult = strResult.substring(0, index + 1);
        }

        return strResult;
    }

    @Override
    protected JComponent[] getPluginMenuItems()
    {
        return new JComponent[]
        {
            new JBasicMenuItem(new OpenFileAction(this)),
        };
    }

    @Override
    protected JComponent[] getSystemMenuItems()
    {
        ArrayList list = new ArrayList(Arrays.asList(super.getSystemMenuItems()));
        
        list.add(new JBasicMenuItem(new ExportPluginAction(this)));

        return (JComponent[])list.toArray(new JComponent[list.size()]);
    }
    
    @Override
    public Object getModel()
    {
        if (m_model == null)
        {
            m_model = new FilePluginModel(this);
        }

        return m_model;
    }

    /**
     * Indicates whether or not the file element plugin has any content.
     *
     * @return  false if no content (blob) has been stored or true if the
     *          element has blob info.
     */
    @Override
    public boolean hasContent()
    {
        boolean res = false;

        try
        {
            IConfigBeanFile bean = getFileByPath(getPluginPath());

            if ((bean != null) && (bean.getContents() != null))
            {
                res = true;
            }
        }
        catch (ConfigServiceException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return res;
    }

    @Override
    public void exportFile(File file) throws Exception
    {
        FileOutputStream os = null;
        InputStream      is = null;

        JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

        try
        {
            IConfigBeanFile bean = getFileByPath(getPluginPath());

            if(bean == null)
            {
                throw new Exception("Failed to load file");
            }

            String name = getPluginName();

            try
            {
                is = bean.getContents();
            }
            catch (ConfigServiceException e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            if (is != null)
            {
                os = new FileOutputStream(file);

                byte[] buffer = new byte[1000];
                while (true)
                {
                    int length = is.read(buffer);

                    if (length == -1)
                    {
                        break;
                    }

                    os.write(buffer, 0, length);
                }
            }
        }
        finally
        {
            if(os != null)
            {
                os.flush();
                os.close();
            }

            if(is != null)
            {
                is.close();
            }

            wc.release();
        }
    }

    /**
     * This method is called when the plugin (and any data representing it)
     * is to be deleted. The implementation is responsible for removing any
     * child plugin's and associated data.
     *
     * The plugin(s) are removed from the tree/content pane asynchronously
     * by notification from the Directory Service.
     *
     * @param configServer      (transaction) config server to use for the delete
     * @throws Exception  if for some reason the operation cannot be performed
     *                    or completed then an exception will be thrown
     */
    @Override
    public void deletePlugin(IConfigServer configServer)
        throws Exception
    {
        configServer.removeConfigElement(getPluginPath());
    }
}
