// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.ma.gui.file;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.KeyboardFocusManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.EventObject;
import java.util.HashSet;
import java.util.Set;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.AbstractListModel;
import javax.swing.Action;
import javax.swing.CellEditor;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultCellEditor;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.mx.config.util.SonicFSException;
import com.sonicsw.mx.config.util.SonicFSFile;
import com.sonicsw.mx.config.util.SonicFSFileSystem;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;


public class JSonicFileChooser extends JPanel
{
    private static final boolean DEBUG = false;

    // Preferred and Minimum sizes for the dialog box
    private static int PREF_WIDTH = 500;
    private static int PREF_HEIGHT = 326;
    private static Dimension PREF_SIZE = new Dimension(PREF_WIDTH, PREF_HEIGHT);

    protected static final String[] FORMAT_NEW = { "New {0}{1,choice,1#|1< {1,number,integer}}" };

    protected static final SonicFileFilter ALL_FILTER = new AllFileFilter();
    //-------------------------------------------------------------------------
    //
    // Dialog Types
    //
    //-------------------------------------------------------------------------

    /**
     * Type value indicating that the <code>JSonicFileChooser</code> supports an
     * "Open" file operation.
     */
    public static final int OPEN_DIALOG = 0;

    /**
     * Type value indicating that the <code>JSonicFileChooser</code> supports a
     * "Save" file operation.
     */
    public static final int SAVE_DIALOG = 1;

    /**
     * Type value indicating that the <code>JSonicFileChooser</code> supports a
     * developer-specified file operation.
     */
    public static final int CUSTOM_DIALOG = 2;

    //-------------------------------------------------------------------------
    //
    // Dialog Return Values
    //
    //-------------------------------------------------------------------------

    /**
     * Return value if cancel is chosen.
     */
    public static final int OPTION_CANCEL = 1;

    /**
     * Return value if approve (yes, ok) is chosen.
     */
    public static final int OPTION_APPROVE = 0;

    /**
     * Return value if an error occured.
     */
    public static final int OPTION_ERROR = -1;

    //-------------------------------------------------------------------------

    /** Instruction to display only files. */
    public static final int FILES_ONLY = 0;

    /** Instruction to display only directories. */
    public static final int DIRECTORIES_ONLY = 1;

    /** Instruction to display both files and directories. */
    public static final int FILES_AND_DIRECTORIES = 2;

    //-------------------------------------------------------------------------

    private String m_dialogTitle = null;
    private String m_approveButtonText = null;

    private JDialog m_dialog      = null;
    private int     m_dialogType  = OPEN_DIALOG;
    private int     m_returnValue = OPTION_ERROR;
    private int     m_fileSelectionMode = FILES_ONLY;
    private Vector<SonicFileFilter>  m_filters = new Vector<SonicFileFilter>(5);
    private boolean m_includeAllFilter = true;
    private SonicFileFilter m_fileFilter = null;
    private SonicFSFile m_currentDirectory = null;
    private SonicFSFile m_selectedFile = null;
    private SonicFSFile[] m_selectedFiles;
    private SonicFSFile m_baseDir = null;

    //private ListCellEditor m_editor = null;

    private DirectoryComboBoxModel directoryComboBoxModel;
    private SonicFSFileSystem m_fs = null;
    private JComboBox      m_cbDirectory;
    private JFileList      m_lShow;
    private JFileTextField m_tSelected;
    private JComboBox      m_cbFilter;
    private JToolBar       m_toolbar;

    //-------------------------------------------------------------------------

    public static void main(String[] arg)
    {
        try { 
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName()); 
        } catch (Exception e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        com.sonicsw.mx.config.ConfigServerUtility csu = new com.sonicsw.mx.config.ConfigServerUtility();
        try { 
            csu.connect("Domain1", "localhost", "Administrator", "Administrator", false); 
        } catch (Exception e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        JSonicFileChooser chooser = new JSonicFileChooser(csu.getDirectoryService(), "test");
        chooser.setMultiSelectionEnabled(true);
        chooser.setIncludeAllFilter(false);
        chooser.setBaseDirectory("/System");
        chooser.addChoosableFileFilter(new SonicFileFilter()
        {
            @Override
            public boolean accept(SonicFSFile f)
            {
                return f.getFullName().toLowerCase().endsWith(".zip");
            }

            @Override
            public String getDescription()
            {
                return "Zip Files";
            }
        });

        chooser.addChoosableFileFilter(new SonicFileFilter()
        {
            @Override
            public boolean accept(SonicFSFile f)
            {
                return f.getFullName().toLowerCase().endsWith(".jar");
            }

            @Override
            public String getDescription()
            {
                return "JAR Files";
            }
        });

        chooser.setCurrentDirectory("sonicfs:///Containers");

        if (chooser.showOpenDialog(null) == OPTION_APPROVE)
        {
            SonicFSFile[] sel = chooser.getSelectedFiles();
            for (int i = 0; i < sel.length; i++)
            {
                debugPrintln("Selected = '" + sel[i].toURL() + "'");
            }
        }

        System.exit(0);
    }

    private static void debugPrintln(String text)
    {
        if (DEBUG)
        {
            System.err.println("JSonicFileChooser: " + text);
        }
    }

    //-------------------------------------------------------------------------

    public JSonicFileChooser(IDirectoryFileSystemService ds,
                             String                      user)
    {
      this(new TrivialCachedSonicFileSystem(ds, user), null);
    }

    public JSonicFileChooser(SonicFSFileSystem fs)
    {
        this(fs, null);
    }

    public JSonicFileChooser(SonicFSFileSystem fs,
                             SonicFSFile       currentDirectory)
    {
        super(new GridBagLayout());
        prepareJSonicFileChooser(fs, currentDirectory);
    }
    
    private void prepareJSonicFileChooser(SonicFSFileSystem fs,
                                          SonicFSFile       currentDirectory) {
        
        m_fs = fs;

        m_currentDirectory = currentDirectory;
        if (currentDirectory == null)
        {
            m_currentDirectory = m_fs.getRootFile();
        }

        directoryComboBoxModel = new DirectoryComboBoxModel();
        // Moved to showDialog()
        // directoryComboBoxModel.build(this.m_currentDirectory);

        m_toolbar = createToolBar();

        m_lShow = new JFileList(new SonicShowModel());
        m_lShow.setCellRenderer(new FileRenderer());
        m_lShow.setLayoutOrientation(JList.VERTICAL_WRAP);
        m_lShow.setVisibleRowCount(-1);
        m_lShow.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        m_lShow.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(MouseEvent evt)
            {
                if ((evt.getClickCount() != 2) || evt.isPopupTrigger())
                {
                    return;
                }

                int index = m_lShow.locationToIndex(evt.getPoint());

                if (index >= 0)
                {
                    SonicFSFile file = (SonicFSFile)m_lShow.getModel().getElementAt(index);

                    if (file.isDirectory())
                    {
                        directoryComboBoxModel.build(file);
                    }
                    else
                    {
                        JSonicFileChooser.this.approveSelection();
                    }
                }
            }
        });
        m_lShow.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(ListSelectionEvent evt)
            {
                try
                {
                    if (evt.getValueIsAdjusting())
                    {
                        return;
                    }

                    int mode = getFileSelectionMode();

                    if (isMultiSelectionEnabled())
                    {
                        Object[] res = m_lShow.getSelectedValues();
                        Set<SonicFSFile> selSet = new HashSet<SonicFSFile>();

                        // When folder is changed, the selection should not be
                        // lost but its hard/impossible to track...maybe this
                        // can be fixed in time...
                        // ...selections can be tracked but its not possible
                        // to know when to clear it down.
                        //
                        //if (m_selectedFiles != null)
                        //    selSet.addAll(Arrays.asList(m_selectedFiles));

                        for (int i = 0; i < res.length; i++)
                        {
                            SonicFSFile aFile = (SonicFSFile)res[i];

                            if ((isDirectorySelectionEnabled() && aFile.isDirectory()) ||
                                (isFileSelectionEnabled() && aFile.isFile()))
                            {
                                if (!selSet.contains(aFile))
                                {
                                    selSet.add(aFile);
                                }
                            }
                        }

                        m_selectedFiles = selSet.toArray(new SonicFSFile[0]);

                        m_tSelected.setFiles((m_selectedFiles.length == 0) ? null : m_selectedFiles);
                    }
                    else
                    {
                        SonicFSFile res = (SonicFSFile)m_lShow.getSelectedValue();

                        if (res == null)
                        {
                            return;
                        }

                        if ((isDirectorySelectionEnabled() && res.isDirectory()) ||
                            (isFileSelectionEnabled() && res.isFile()))
                        {
                            m_selectedFile = res;

                            m_tSelected.setFile(m_selectedFile);
                        }
                    }
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }
        });

        m_cbDirectory = new JComboBox(directoryComboBoxModel);
        m_cbDirectory.setRenderer(new DirectoryComboBoxRenderer());
        m_cbDirectory.setAlignmentX(JComponent.LEFT_ALIGNMENT);
        m_cbDirectory.setAlignmentY(JComponent.TOP_ALIGNMENT);
        m_cbDirectory.setMaximumRowCount(8);
        m_cbDirectory.setSelectedItem(this.m_currentDirectory);
        m_cbDirectory.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(ActionEvent evt)
            {
                SonicFSFile selDir = (SonicFSFile)((JComboBox)evt.getSource()).getSelectedItem();

                debugPrintln("Directory ComboBox Action");

                if (JSonicFileChooser.this.getCurrentDirectory().equals(selDir))
                {
                    return;
                }

                JSonicFileChooser.this.m_currentDirectory = selDir;

                ((SonicShowModel)m_lShow.getModel()).build(selDir);
                directoryComboBoxModel.build(selDir);
                
                JButton upBtn = getButtonWithCommand(m_toolbar, "Up");
                boolean bUp = !selDir.equals(getTopLevelDirectory());
                if (upBtn != null)
                {
                    upBtn.setEnabled(bUp);
                }
            }
        });

        m_tSelected = new JFileTextField();

        m_cbFilter = new JComboBox((Object[])getChoosableFileFilters());
        m_cbFilter.setMaximumRowCount(6);
        m_cbFilter.addItemListener(new ItemListener()
        {
            @Override
            public void itemStateChanged(ItemEvent evt)
            {
                // avoid reacting to ItemEvent.DESELECTED and save a network call
                if (evt.getStateChange() == ItemEvent.SELECTED)
                {
                    _setSonicFileFilter((SonicFileFilter)m_cbFilter.getSelectedItem());

                    // Need to rebuild only if we're visible.
                    if (m_dialog != null && m_dialog.isVisible())
                    {
                        ((SonicShowModel)m_lShow.getModel()).build(getCurrentDirectory());
                    }
                }
            }
        });

        GBConstraints gbc = new GBConstraints();

        gbc.set(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(5, 0, 0, 0);
        gbc.anchor = GridBagConstraints.WEST;
        add(createTopPanel(), gbc);

        gbc.set(0, 1, 1, 1, GridBagConstraints.BOTH, 1.0, 1.0);
        gbc.insets = new Insets(5, 5, 5, 5);
        add(new JScrollPane(m_lShow), gbc);

        gbc.set(0, 2, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(5, 0, 0, 0);
        gbc.anchor = GridBagConstraints.WEST;
        add(createBottomPanel(), gbc);


        resetChoosableFileFilters();

        setCurrentDirectory(currentDirectory);

        setPreferredSize(PREF_SIZE);
    }

    protected SonicFSFile getTopLevelDirectory()
    {
        SonicFSFile dir = getBaseDirectory();

        if (dir == null)
        {
            dir = m_fs.getRootFile();
        }

        return dir;
    }

    public void setBaseDirectory(SonicFSFile baseDir)
    {
        m_baseDir = baseDir.isDirectory() ? baseDir : baseDir.getParentFile();

        directoryComboBoxModel.build(getTopLevelDirectory());
    }

    public void setBaseDirectory(String baseDir)
    {
        SonicFSFile dir = null;

        try
        {
            dir = m_fs.getDetails(baseDir);
        }
        catch (SonicFSException e)
        {
            e.printStackTrace();
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            return;
        }

        setBaseDirectory(dir);
    }

    public SonicFSFile getBaseDirectory()
    {
        return m_baseDir;
    }

    public SonicFSFile getCurrentDirectory()
    {
        return m_currentDirectory;
    }

    private JToolBar createToolBar()
    {
        JToolBar toolbar = new JToolBar(JToolBar.HORIZONTAL);

        toolbar.setBorderPainted(false);
        toolbar.setFloatable(false);
        toolbar.setRollover(true);

        toolbar.add(new UpFolderAction());
        toolbar.add(new NewFolderAction());
        toolbar.add(new DetailsViewAction());

        getButtonWithCommand(toolbar, "Details").setEnabled(false);

        return toolbar;
    }

    private JPanel createTopPanel()
    {
        JPanel pDir = new JPanel(new GridBagLayout());
        GBConstraints gbc = new GBConstraints();

        gbc.set(0, 0, 1, 1, GridBagConstraints.NONE, 0.0, 0.0);
        gbc.insets = new Insets(0, 5, 0, 5);
        gbc.anchor = GridBagConstraints.WEST;
        pDir.add(new JLabel("Look in:"), gbc);

        gbc.set(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 0, 0, 0);
        gbc.anchor = GridBagConstraints.WEST;
        pDir.add(m_cbDirectory, gbc);

        gbc.set(2, 0, 1, 1, GridBagConstraints.NONE);
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 5, 0, 5);
        pDir.add(m_toolbar, gbc);

        return pDir;
    }

    private JPanel createBottomPanel()
    {
        JPanel panel = new JPanel(new GridBagLayout());
        GBConstraints gbc = new GBConstraints();

        gbc.set(0, 2, 2, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 5, 0, 5);
        gbc.anchor = GridBagConstraints.WEST;
        panel.add(createSelPanel(), gbc);

        gbc.set(2, 2, 1, 1, GridBagConstraints.NONE);
        gbc.insets = new Insets(0, 5, 5, 5);
        gbc.anchor = GridBagConstraints.EAST;
        panel.add(createButtonPanel(), gbc);

        return panel;
    }

    private JPanel createSelPanel()
    {
        JPanel panel = new JPanel(new GridBagLayout());

        GBConstraints gbc = new GBConstraints();

        gbc.set(0, 0, 1, 1, GridBagConstraints.NONE);
        gbc.insets = new Insets(0, 0, 0, 5);
        panel.add(new JLabel("File name:"), gbc);
        gbc.set(1, 0, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 0, 5, 0);
        panel.add(m_tSelected, gbc);

        gbc.set(0, 1, 1, 1, GridBagConstraints.NONE);
        gbc.insets = new Insets(0, 0, 0, 5);
        panel.add(new JLabel("File of types:"), gbc);
        gbc.set(1, 1, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 0, 0, 0);
        panel.add(m_cbFilter, gbc);

        return panel;
    }

    private JPanel createButtonPanel()
    {
        JPanel panel = new JPanel(new GridBagLayout());

        GBConstraints gbc = new GBConstraints();

        gbc.set(0, 0, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 0, 5, 0);
        panel.add(new JButton(new OkAction(getDialogType())), gbc);
        gbc.set(0, 1, 1, 1, GridBagConstraints.HORIZONTAL, 1.0, 0.0);
        gbc.insets = new Insets(0, 0, 0, 0);
        panel.add(new JButton(new CancelAction()), gbc);

        return panel;
    }

    private JButton getButtonWithCommand(JComponent c, String command)
    {
        JButton btn = null;

        if ((c instanceof JButton) && ((JButton)c).getActionCommand().equals(command))
        {
            btn = (JButton)c;
        }
        else
        {
            for (int i = 0; i < c.getComponentCount(); i++)
            {
                btn = getButtonWithCommand( (JComponent)c.getComponent(i), command);

                if (btn != null)
                {
                    break;
                }
            }
        }

        return btn;
    }

    String getFSName(SonicFSFile file)
    {
        try {
            if (file.equals(m_fs.getRootFile())) {
                return m_fs.getDirectoryService().getDomain();
            }
        } catch (ManagementPermissionDeniedException e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return file.getName();
    }

    Icon getFSIcon(SonicFSFile file, boolean isSelected) {
        if (file.isFile()) {
            Icon fileIcon = getFileIcon();
            return fileIcon;
        }

        DefaultIconProvider instance = DefaultIconProvider.getInstance();
        if (isSelected) {
            Icon openIcon = UIManager.getIcon("Tree.openIcon");
            if (openIcon == null) {
                openIcon = instance.getIcon(DefaultIconProvider.TREE_OPEN_ICON);
            }
            return openIcon;
        }
        Icon closedIcon = UIManager.getIcon("Tree.closedIcon");
        if (closedIcon == null) {
            closedIcon = instance.getIcon(DefaultIconProvider.TREE_CLOSED_ICON);
        }
        return closedIcon;
    }


    public Icon getFileIcon()
    {
        try
        {
            java.net.URL imgUrl = getClass().getClassLoader().getResource("com/sonicsw/ma/gui/image/type/MF_FILE.gif");

            return new ImageIcon(imgUrl);
        }
        catch (Exception e)
        {
            e.printStackTrace();
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return null;
    }

    public void setIncludeAllFilter(boolean state)
    {
        m_includeAllFilter = state;

        if (state)
        {
            if (!m_filters.contains(ALL_FILTER))
            {
                m_filters.add(ALL_FILTER);
                m_cbFilter.insertItemAt(ALL_FILTER, 0);
            }
        }
        else
        {
            if (m_filters.contains(ALL_FILTER))
            {
                m_filters.remove(ALL_FILTER);
                m_cbFilter.removeItem(ALL_FILTER);
            }
        }
    }

    public boolean isIncludeAllFilter()
    {
        return m_includeAllFilter;
    }

    /**
     * Gets the list of user choosable file filters.
     *
     * @return a <code>SonicFileFilter</code> array containing all the choosable
     *         file filters
     */
    public SonicFileFilter[] getChoosableFileFilters()
    {
        return m_filters.toArray(new SonicFileFilter[m_filters.size()]);
    }

    /**
     * Adds a filter to the list of user choosable file filters.
     *
     * @param filter the <code>SonicFileFilter</code> to add to the choosable file
     *               filter list
     */
    public void addChoosableFileFilter(SonicFileFilter filter)
    {
        if ((filter != null) && !m_filters.contains(filter))
        {
            m_filters.addElement(filter);
            m_cbFilter.addItem(filter);
        }
    }

    /**
     * Removes a filter from the list of user choosable file filters. Returns
     * true if the file filter was removed.
     */
    public boolean removeChoosableFileFilter(SonicFileFilter filter)
    {
        m_cbFilter.removeItem(filter);

        return m_filters.removeElement(filter);
    }

    /**
     * Resets the choosable file filter list to its starting state.
     */
    public void resetChoosableFileFilters()
    {
        setSonicFileFilter(null);

        m_filters.removeAllElements();

        if (isIncludeAllFilter())
        {
            m_filters.add(ALL_FILTER);
        }

        m_cbFilter.setModel(new DefaultComboBoxModel(getChoosableFileFilters()));
     }

    private void _setSonicFileFilter(SonicFileFilter filter)
    {
        m_fileFilter = filter;

        if (filter != null)
        {
            if (isMultiSelectionEnabled() && (m_selectedFiles != null) && (m_selectedFiles.length > 0))
            {
                Vector<SonicFSFile> fList = new Vector<SonicFSFile>();
                boolean failed = false;

                for (int i = 0; i < m_selectedFiles.length; i++)
                {
                    if (filter.accept(m_selectedFiles[i]))
                    {
                        fList.add(m_selectedFiles[i]);
                    }
                    else
                    {
                        failed = true;
                    }
                }

                if (failed)
                {
                    setSelectedFiles((fList.isEmpty()) ? null : fList.toArray(new SonicFSFile[fList.size()]));
                }
            }
            else
            if ((m_selectedFile != null) && !filter.accept(m_selectedFile))
            {
                setSelectedFile(null);
            }
        }
    }

    /**
     * Sets the current file filter. The file filter is used by the
     * file chooser to filter out files from the user's view.
     *
     * @param filter the new current file filter to use
     */
    public void setSonicFileFilter(SonicFileFilter filter)
    {
        _setSonicFileFilter(filter);

        if (filter != null)
        {
            m_cbFilter.setSelectedItem(filter);
        }
    }

    /**
     * Returns the currently selected file filter.
     *
     * @return the current file filter
     */
    public SonicFileFilter getSonicFileFilter()
    {
        return m_fileFilter;
    }

    /**
     * Sets the <code>JSonicFileChooser</code> to allow the user to just
     * select files, just select
     * directories, or select both files and directories.  The default is
     * <code>JSonicFileChooser.FILES_ONLY</code>.
     *
     * @param mode the type of files to be displayed:
     * <ul>
     * <li>JSonicFileChooser.FILES_ONLY
     * <li>JSonicFileChooser.DIRECTORIES_ONLY
     * <li>JSonicFileChooser.FILES_AND_DIRECTORIES
     * </ul>
     *
     * @exception IllegalArgumentException  if <code>mode</code> is an
     *				illegal mode
     */
    public void setFileSelectionMode(int mode)
    {
        if (m_fileSelectionMode == mode)
        {
            return;
        }

        if ((mode == FILES_ONLY) ||
            (mode == DIRECTORIES_ONLY) ||
            (mode == FILES_AND_DIRECTORIES))
        {
            int oldValue = m_fileSelectionMode;
            m_fileSelectionMode = mode;
        }
        else
        {
            throw new IllegalArgumentException("Incorrect Mode for Dialog: " + mode);
        }
    }

    /**
     * Returns the current file-selection mode.  The default is
     * <code>JSonicFileChooser.FILES_ONLY</code>.
     *
     * @return the type of files to be displayed, one of the following:
     * <ul>
     * <li>JSonicFileChooser.FILES_ONLY
     * <li>JSonicFileChooser.DIRECTORIES_ONLY
     * <li>JSonicFileChooser.FILES_AND_DIRECTORIES
     * </ul>
     * @see #setFileSelectionMode
     */
    public int getFileSelectionMode()
    {
        return m_fileSelectionMode;
    }

    /**
     * Convenience call that determines if files are selectable based on the
     * current file selection mode.
     */
    public boolean isFileSelectionEnabled()
    {
        return ((m_fileSelectionMode == FILES_ONLY) ||
                (m_fileSelectionMode == FILES_AND_DIRECTORIES));
    }

    /**
     * Convenience call that determines if directories are selectable based
     * on the current file selection mode.
     */
    public boolean isDirectorySelectionEnabled()
    {
        return ((m_fileSelectionMode == DIRECTORIES_ONLY) ||
                (m_fileSelectionMode == FILES_AND_DIRECTORIES));
    }

    public int showOpenDialog(Component parent)
        throws HeadlessException
    {
        setDialogType(OPEN_DIALOG);

        return showDialog(parent, null);
    }

    public int showSaveDialog(Component parent)
        throws HeadlessException
    {
        setDialogType(SAVE_DIALOG);

        return showDialog(parent, null);
    }

    /**
     * Returns the type of this dialog.  The default is
     * <code>JSonicFileChooser.OPEN_DIALOG</code>.
     *
     * @return   the type of dialog to be displayed:
     * <ul>
     * <li>JSonicFileChooser.OPEN_DIALOG
     * <li>JSonicFileChooser.SAVE_DIALOG
     * <li>JSonicFileChooser.CUSTOM_DIALOG
     * </ul>
     *
     * @see #setDialogType
     */
    public int getDialogType()
    {
        return m_dialogType;
    }

    /**
     * Sets the type of this dialog. Use <code>OPEN_DIALOG</code> when you
     * want to bring up a file chooser that the user can use to open a file.
     * Likewise, use <code>SAVE_DIALOG</code> for letting the user choose
     * a file for saving.
     * Use <code>CUSTOM_DIALOG</code> when you want to use the file
     * chooser in a context other than "Open" or "Save".
     * For instance, you might want to bring up a file chooser that allows
     * the user to choose a file to execute. Note that you normally would not
     * need to set the <code>JSonicFileChooser</code> to use
     * <code>CUSTOM_DIALOG</code>
     * since a call to <code>setApproveButtonText</code> does this for you.
     * The default dialog type is <code>JSonicFileChooser.OPEN_DIALOG</code>.
     *
     * @param dialogType the type of dialog to be displayed:
     * <ul>
     * <li>JSonicFileChooser.OPEN_DIALOG
     * <li>JSonicFileChooser.SAVE_DIALOG
     * <li>JSonicFileChooser.CUSTOM_DIALOG
     * </ul>
     *
     * @exception IllegalArgumentException if <code>dialogType</code> is
     *				not legal
     * @beaninfo
     *   preferred: true
     *       bound: true
     * description: The type (open, save, custom) of the JSonicFileChooser.
     *        enum:
     *              OPEN_DIALOG JSonicFileChooser.OPEN_DIALOG
     *              SAVE_DIALOG JSonicFileChooser.SAVE_DIALOG
     *              CUSTOM_DIALOG JSonicFileChooser.CUSTOM_DIALOG
     *
     * @see #getDialogType
     * @see #setApproveButtonText
     */
    // PENDING(jeff) - fire button text change property
    public void setDialogType(int dialogType)
    {
        if (m_dialogType == dialogType)
        {
            return;
        }

        if(!(dialogType == OPEN_DIALOG || dialogType == SAVE_DIALOG || dialogType == CUSTOM_DIALOG))
        {
            throw new IllegalArgumentException("Incorrect Dialog Type: " + dialogType);
        }

        int oldValue = m_dialogType;
        m_dialogType = dialogType;

        if(dialogType == OPEN_DIALOG || dialogType == SAVE_DIALOG)
        {
            setApproveButtonText(null);
        }
    }

    /**
     * Sets the string that goes in the <code>JSonicFileChooser</code> window's
     * title bar.
     *
     * @param dialogTitle the new <code>String</code> for the title bar
     *
     * @beaninfo
     *   preferred: true
     *       bound: true
     * description: The title of the JSonicFileChooser dialog window.
     *
     * @see #getDialogTitle
     *
     */
    public void setDialogTitle(String dialogTitle)
    {
        String oldValue = m_dialogTitle;
        m_dialogTitle = dialogTitle;

        if (m_dialog != null)
        {
            m_dialog.setTitle(dialogTitle);
        }
    }

    /**
     * Gets the string that goes in the <code>JSonicFileChooser</code>'s titlebar.
     *
     * @see #setDialogTitle
     */
    public String getDialogTitle()
    {
        return m_dialogTitle;
    }

    /**
     * Sets the text used in the <code>ApproveButton</code> in the
     * <code>FileChooserUI</code>.
     *
     * @beaninfo
     *   preferred: true
     *       bound: true
     * description: The text that goes in the ApproveButton.
     *
     * @param approveButtonText the text used in the <code>ApproveButton</code>
     */
    // PENDING(jeff) - have ui set this on dialog type change
    public void setApproveButtonText(String approveButtonText)
    {
        if (m_approveButtonText == approveButtonText)
        {
            return;
        }

        String oldValue = m_approveButtonText;
        m_approveButtonText = approveButtonText;
    }

    /**
     * Returns the text used in the <code>ApproveButton</code> in the
     * <code>FileChooserUI</code>.
     * If <code>null</code>, the UI object will determine the button's text.
     *
     * Typically, this would be "Open" or "Save".
     *
     * @return the text used in the <code>ApproveButton</code>
     */
    public String getApproveButtonText()
    {
        return m_approveButtonText;
    }

    /**
     * Sets the file chooser to allow multiple file selections.
     *
     * @param b true if multiple files may be selected
     * @beaninfo
     *       bound: true
     * description: Sets multiple file selection mode.
     *
     * @see #isMultiSelectionEnabled
     */
    public void setMultiSelectionEnabled(boolean b)
    {
        m_lShow.getSelectionModel().setSelectionMode(b ? ListSelectionModel.MULTIPLE_INTERVAL_SELECTION : ListSelectionModel.SINGLE_SELECTION);
    }

    /**
     * Returns true if multiple files can be selected.
     * @return true if multiple files can be selected
     * @see #setMultiSelectionEnabled
     */
    public boolean isMultiSelectionEnabled()
    {
        return (m_lShow.getSelectionModel().getSelectionMode() != ListSelectionModel.SINGLE_SELECTION);
    }

    /**
     * Called by the UI when the user hits the Approve button
     * (labeled "Open" or "Save", by default). This can also be
     * called by the programmer.
     * This method causes an action event to fire
     * with the command string equal to
     * <code>APPROVE_SELECTION</code>.
     *
     * @see #APPROVE_SELECTION
     */
    public void approveSelection()
    {
        m_returnValue = OPTION_APPROVE;

        if (m_dialog != null)
        {
            m_dialog.setVisible(false);
        }
    }

    /**
     * Called by the UI when the user chooses the Cancel button.
     * This can also be called by the programmer.
     * This method causes an action event to fire
     * with the command string equal to
     * <code>CANCEL_SELECTION</code>.
     *
     * @see #CANCEL_SELECTION
     */
    public void cancelSelection()
    {
        m_returnValue = OPTION_CANCEL;

        if (m_dialog != null)
        {
            m_dialog.setVisible(false);
        }
    }

    /**
     * Returns the selected file. This can be set either by the
     * programmer via <code>setFile</code> or by a user action, such as
     * either typing the filename into the UI or selecting the
     * file from a list in the UI.
     *
     * @return the selected file
     */
    public SonicFSFile getSelectedFile()
    {
        return m_selectedFile;
    }

    /**
     * Sets the selected file. If the file's parent directory is
     * not the current directory, changes the current directory
     * to be the file's parent directory.
     *
     * @beaninfo
     *   preferred: true
     *       bound: true
     *
     * @see #getSelectedFile
     *
     * @param file the selected file
     */
    public void setSelectedFile(SonicFSFile file)
    {
        SonicFSFile oldValue = m_selectedFile;

        m_selectedFile = file;

        if (m_selectedFile != null)
        {
            if (file.isAbsolute() && !getCurrentDirectory().isParent(m_selectedFile))
            {
                setCurrentDirectory(m_selectedFile.getParentFile());
            }

            if (!isMultiSelectionEnabled() || (m_selectedFiles == null) || (m_selectedFiles.length > 1))
            {
                ensureFileIsVisible(m_selectedFile);
            }
        }
    }

    /**
     * Returns a list of selected files if the file chooser is
     * set to allow multiple selection.
     */
    public SonicFSFile[] getSelectedFiles()
    {
        return (m_selectedFiles != null) ? (SonicFSFile[])m_selectedFiles.clone() : new SonicFSFile[0];
    }

    /**
     * Sets the list of selected files if the file chooser is
     * set to allow multiple selection.
     */
    public void setSelectedFiles(SonicFSFile[] selectedFiles)
    {
        SonicFSFile[] oldValue = m_selectedFiles;

        if ((selectedFiles != null) && (selectedFiles.length == 0))
        {
            selectedFiles = null;
        }

        m_selectedFiles = selectedFiles;

        setSelectedFile((selectedFiles != null) ? selectedFiles[0] : null);
    }

    private void ensureFileIsVisible(SonicFSFile file)
    {
        SonicShowModel model = (SonicShowModel)m_lShow.getModel();

        int i = model.indexOf(file);

        if (i >= 0)
        {
            m_lShow.ensureIndexIsVisible(i);
        }
    }

    /**
     * Sets the current directory. Passing in <code>null</code> sets the
     * file chooser to point to the user's default directory.
     * This default depends on the operating system. It is
     * typically the "My Documents" folder on Windows, and the user's
     * home directory on Unix.
     *
     * If the file passed in as <code>currentDirectory</code> is not a
     * directory, the parent of the file will be used as the currentDirectory.
     * If the parent is not traversable, then it will walk up the parent tree
     * until it finds a traversable directory, or hits the root of the
     * file system.
     *
     * @beaninfo
     *   preferred: true
     *       bound: true
     * description: The directory that the JSonicFileChooser is showing files of.
     *
     * @param dir the current directory to point to
     * @see #getCurrentDirectory
     */
    public void setCurrentDirectory(SonicFSFile dir)
    {
        SonicFSFile newDir;

        if (dir == null)
        {
            newDir = getTopLevelDirectory();
        }
        else
        {
            newDir = dir.isDirectory() ? dir : dir.getParentFile();
        }

        if (!newDir.getFullName().startsWith(getTopLevelDirectory().getFullName()))
        {
            newDir = getTopLevelDirectory();
        }

        m_currentDirectory = newDir;
    }

    public void setCurrentDirectory(String dir)
    {
        SonicFSFile newDir = null;

        try
        {
            String tmp = SonicFSFileSystem.SCHEME + "://";

            if (dir.startsWith(tmp))
            {
                dir = dir.substring(tmp.length());
            }

            newDir = m_fs.getDetails(dir);
        }
        catch (SonicFSException e)
        {
            e.printStackTrace();
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            return;
        }

        setCurrentDirectory(newDir);
    }


    /**
     * Build the combo box and file list models.
     * Should be called from showDialog() before turning visible
     */
    private void buildModels()
    {
        SonicFSFile currentDir = getCurrentDirectory();
        directoryComboBoxModel.build(currentDir);
        ((SonicShowModel) m_lShow.getModel()).build(currentDir);
    }

    /**
     * Pops a custom Sonic file chooser dialog with a custom approve button.
     * For example, the following code
     * pops up a file chooser with a "Run Application" button
     * (instead of the normal "Save" or "Open" button):
     * <pre>
     * filechooser.showDialog(parentFrame, "Run Application");
     * </pre>
     *
     * Alternatively, the following code does the same thing:
     * <pre>
     *    JSonicFileChooser chooser = new JSonicFileChooser(null);
     *    chooser.setApproveButtonText("Run Application");
     *    chooser.showDialog(parentFrame, null);
     * </pre>
     *
     * <!--PENDING(jeff) - the following method should be added to the api:
     *      showDialog(Component parent);-->
     * <!--PENDING(kwalrath) - should specify modality and what
     *      "depends" means.-->
     *
     * <p>
     *
     * The <code>parent</code> argument determines two things:
     * the frame on which the open dialog depends and
     * the component whose position the look and feel
     * should consider when placing the dialog.  If the parent
     * is a <code>Frame</code> object (such as a <code>JFrame</code>)
     * then the dialog depends on the frame and
     * the look and feel positions the dialog
     * relative to the frame (for example, centered over the frame).
     * If the parent is a component, then the dialog
     * depends on the frame containing the component,
     * and is positioned relative to the component
     * (for example, centered over the component).
     * If the parent is <code>null</code>, then the dialog depends on
     * no visible window, and it's placed in a
     * look-and-feel-dependent position
     * such as the center of the screen.
     *
     * @param   parent  the parent component of the dialog;
     *			can be <code>null</code>
     * @param   approveButtonText the text of the <code>ApproveButton</code>
     * @return  the return state of the file chooser on popdown:
     * <ul>
     * <li>JSonicFileChooser.CANCEL_OPTION
     * <li>JSonicFileChooser.APPROVE_OPTION
     * <li>JSonicFileCHooser.ERROR_OPTION if an error occurs or the
     *			dialog is dismissed
     * </ul>
     * @exception HeadlessException if GraphicsEnvironment.isHeadless()
     * returns true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     */
    public int showDialog(Component parent, String approveButtonText)
        throws HeadlessException
    {
        if (approveButtonText != null)
        {
            setApproveButtonText(approveButtonText);
            setDialogType(CUSTOM_DIALOG);
        }

        buildModels();

        m_dialog = createDialog(parent);
        m_dialog.addWindowListener(new WindowAdapter()
        {
            @Override
            public void windowClosing(WindowEvent evt)
            {
                m_returnValue = OPTION_CANCEL;
            }
        });

        m_dialog.show();
        m_dialog.dispose();
        m_dialog = null;

        return m_returnValue;
    }

    /**
     * Creates and returns a new <code>JDialog</code> wrapping
     * <code>this</code> centered on the <code>parent</code>
     * in the <code>parent</code>'s frame.
     * This method can be overriden to further manipulate the dialog,
     * to disable resizing, set the location, etc. Example:
     * <pre>
     *     class MyFileChooser extends JSonicFileChooser {
     *         protected JDialog createDialog(Component parent) throws HeadlessException {
     *             JDialog dialog = super.createDialog(parent);
     *             dialog.setLocation(300, 200);
     *             dialog.setResizable(false);
     *             return dialog;
     *         }
     *     }
     * </pre>
     *
     * @param   parent  the parent component of the dialog;
     *			can be <code>null</code>
     * @return a new <code>JDialog</code> containing this instance
     * @exception HeadlessException if GraphicsEnvironment.isHeadless()
     * returns true.
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @since 1.4
     */
    protected JDialog createDialog(Component parent)
        throws HeadlessException
    {
        Frame frame = parent instanceof Frame ? (Frame) parent
              : (Frame)SwingUtilities.getAncestorOfClass(Frame.class, parent);

        String title = getDialogTitle();//"Test"; //getUI().getDialogTitle(this);

        JDialog dialog = new JDialog(frame, title, true);

        Container contentPane = dialog.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(this, BorderLayout.CENTER);

        if (JDialog.isDefaultLookAndFeelDecorated())
        {
            boolean supportsWindowDecorations = UIManager.getLookAndFeel().getSupportsWindowDecorations();

            if (supportsWindowDecorations)
            {
                dialog.getRootPane().setWindowDecorationStyle(JRootPane.FILE_CHOOSER_DIALOG);
            }
        }

        dialog.pack();
        dialog.setLocationRelativeTo(parent);

        return dialog;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class GBConstraints extends GridBagConstraints
    {
        public GBConstraints()
        {
            super();

            this.insets = new Insets(5, 5, 0, 0);
            this.anchor = GridBagConstraints.WEST;
        }

        public void set(int gridx, int gridy, int gridwidth, int gridheight, int fill)
        {
            set(gridx, gridy, gridwidth, gridheight, fill, 0.0, 0.0);
        }

        public void set(int gridx, int gridy, int gridwidth, int gridheight, int fill, double weightx, double weighty)
        {
            this.gridx = gridx;
            this.gridy = gridy;
            this.gridwidth = gridwidth;
            this.gridheight = gridheight;
            this.fill = fill;
            this.weightx = weightx;
            this.weighty = weighty;
        }
    }

    class NewFolderAction extends AbstractAction
    {
        public NewFolderAction()
        {
            super("New Folder", UIManager.getIcon("FileChooser.newFolderIcon"));
            putActionProperties();
        }

        private void putActionProperties() {
            putValue(AbstractAction.ACTION_COMMAND_KEY, "New Folder");
            putValue(AbstractAction.SHORT_DESCRIPTION, "Create New Folder");
        }
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                String      newPath   = getValidPath(getCurrentDirectory().getFullName(), "Folder", FORMAT_NEW);
                SonicFSFile newFolder = m_fs.createDirectory(newPath);

                int insertIndex = ((SonicShowModel)m_lShow.getModel()).nearestIndexOf(newFolder);

                ((SonicShowModel)m_lShow.getModel()).add(insertIndex, newFolder);

                m_lShow.setSelectedValue(newFolder, true);
                m_lShow.editCellAt(insertIndex);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }
        }

        private String getValidPath(String   basePath,
                                    String   name,
                                    String[] pattern)
        {
            if (!basePath.endsWith("/"))
            {
                basePath += "/";
            }

            // Infinite loop!!!!
            // We have to assume that a unique name will be found even if the loop
            // index 'i' ends up in the thousands...
            //
            for (int i = 0; ; i++)
            {
                String p = (i >= pattern.length) ? pattern[pattern.length-1] : pattern[i];
                String candidate = MessageFormat.format(p, new Object[] { name, new Integer(i) }).trim();
                String newPath = basePath + candidate;

                if (!m_fs.exists(newPath))
                {
                    return newPath;
                }
            }
        }
    }

    interface ListCellEditor extends CellEditor
    {
        Component getListCellEditorComponent(JList   list,
                                             Object  value,
                                             boolean isSelected,
                                             int     row);
    }

    class FileCellEditor extends DefaultCellEditor implements ListCellEditor
    {
        private SonicFSFile m_oldFile;
        private SonicFSFile m_newFile;

        public FileCellEditor()
        {
            super(new JTextField());
        }

        @Override
        public Component getListCellEditorComponent(JList   list,
                                                    Object  value,
                                                    boolean isSelected,
                                                    int     row)
        {
            JTextField c = (JTextField)editorComponent;

            m_oldFile = (SonicFSFile)value;
            m_newFile = null;

            c.setText(((SonicFSFile)value).getName());

            return c;
        }

        @Override
        public Object getCellEditorValue()
        {
            return m_newFile;
        }

        @Override
        public boolean stopCellEditing()
        {
            boolean res = false;

            debugPrintln("stopCellEditing");

            try
            {
                if (m_oldFile != null)
                {
                    String newName = ((JTextField)editorComponent).getText();

                    if (!newName.equals(m_oldFile.getName()))
                    {
                        debugPrintln("rename");

                        m_newFile = m_fs.rename(m_oldFile, newName);
                        m_oldFile = null;
                    }
                    else
                    {
                        debugPrintln("nothing to rename");

                        m_newFile = m_oldFile;
                        m_oldFile = null;
                    }
                }

                res = true;
            }
            catch (Exception e)
            {
                e.printStackTrace();
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            if (res)
            {
                super.stopCellEditing();
            }

            return res;
        }
    }

    class UpFolderAction extends AbstractAction
    {
        public UpFolderAction()
        {
            super("Up", UIManager.getIcon("FileChooser.upFolderIcon"));
            putActionProperties();
        }
        
        private void putActionProperties() {
            putValue(AbstractAction.ACTION_COMMAND_KEY, "Up");
            putValue(AbstractAction.SHORT_DESCRIPTION,  "Up One Level");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            SonicFSFile parent = JSonicFileChooser.this.getCurrentDirectory().getParentFile();

            if ((parent != null) &&
                parent.getFullName().startsWith(getTopLevelDirectory().getFullName()))
            {
            	directoryComboBoxModel.build(parent);
            }
        }
    }

    class HomeFolderAction extends AbstractAction
    {
        public HomeFolderAction()
        {
            super("Home", UIManager.getIcon("FileChooser.homeFolderIcon"));
            putActionProperties();
        }
        
        private void putActionProperties() {
            putValue(AbstractAction.ACTION_COMMAND_KEY, "Home");
        }
        
        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }
    }

    class DetailsViewAction extends AbstractAction
    {
        public DetailsViewAction()
        {
            super("Details", UIManager.getIcon("FileChooser.detailsViewIcon"));
            putActionProperties();
        }
        
        private void putActionProperties() {
            putValue(AbstractAction.ACTION_COMMAND_KEY, "Details");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }
    }

    class ListViewAction extends AbstractAction
    {
        public ListViewAction()
        {
            super("List", UIManager.getIcon("FileChooser.listViewIcon"));
            putActionProperties();
        }

        private void putActionProperties() {
            putValue(AbstractAction.ACTION_COMMAND_KEY, "List");
        }
        
        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }
    }

    class DirectoryComboBoxRenderer extends DefaultListCellRenderer
    {
        IndentIcon ii = new IndentIcon();

        @Override
        public Component getListCellRendererComponent(JList   list,
                                                      Object  value,
                                                      int     index,
                                                      boolean isSelected,
                                                      boolean cellHasFocus)
        {
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);

            if (value == null)
            {
                setText("");
                return this;
            }

            SonicFSFile directory = (SonicFSFile)value;

            setText(JSonicFileChooser.this.getFSName(directory));
            Icon icon = JSonicFileChooser.this.getFSIcon(directory, isSelected);

            ii.icon = icon;
            ii.depth = directoryComboBoxModel.getDepth(index);
            setIcon(ii);

            return this;
        }
    }

    static final int space = 10;

    class IndentIcon implements Icon
    {
        Icon icon = null;
        int depth = 0;

        @Override
        public void paintIcon(Component c, Graphics g, int x, int y)
        {
            if (c.getComponentOrientation().isLeftToRight())
            {
                icon.paintIcon(c, g, x+depth*space, y);
            }
            else
            {
                icon.paintIcon(c, g, x, y);
            }
        }

        @Override
        public int getIconWidth()
        {
            return icon.getIconWidth() + depth*space;
        }

        @Override
        public int getIconHeight()
        {
            return icon.getIconHeight();
        }
    }

    protected class DirectoryComboBoxModel extends AbstractListModel implements ComboBoxModel
    {
        Vector<SonicFSFile> directories = new Vector<SonicFSFile>();
        int[] depths = null;
        SonicFSFile selectedDirectory = null;

        public DirectoryComboBoxModel()
        {
            // Add the current directory to the model, and make it the
            // selectedDirectory
            SonicFSFile dir = JSonicFileChooser.this.getCurrentDirectory();

            //build(getTopLevelDirectory()); //m_fs.getRootFile());
            //calculateDepths();
        }


        /**
    	 * Adds the directory to the model and sets it to be selected,
    	 * additionally clears out the previous selected directory and
    	 * the paths leading up to it, if any.
    	 */

        private void build(SonicFSFile directory) {
			if (directory == null) {
				return;
			}

			directories.clear();

			SonicFSFile root = getTopLevelDirectory();
			SonicFSFile[] baseFolders = null;

			directories.add(root);
			try {
				baseFolders = m_fs.listDetails(root.getFullName());
			} catch (SonicFSException e) {
			    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
			}

			if (baseFolders != null)
            {
                directories.addAll(Arrays.asList(baseFolders));
            }

			try {

				SonicFSFile f = directory;

				Vector<SonicFSFile> path = new Vector<SonicFSFile>(10);
				do {
					path.addElement(f);
				} while ((f = f.getParentFile()) != null);

				int pathCount = path.size();
				// Insert chain at appropriate place in vector
				for (int i = 0; i < pathCount; i++) {
					f = path.get(i);
					if (directories.contains(f)) {
						int topIndex = directories.indexOf(f);
						for (int j = i - 1; j >= 0; j--) {
							directories.insertElementAt(path.get(j), topIndex
									+ i - j);
						}
						break;
					}
				}
				calculateDepths();
				setSelectedItem(directory);
			} catch (Exception e) {
				calculateDepths();
				MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
			}

		}

        public void removeAllELements()
        {
            if (!directories.isEmpty())
            {
                int firstIndex = 0;
                int lastIndex = directories.size() - 1;

                directories.removeAllElements();

                fireIntervalRemoved(this, firstIndex, lastIndex);
            }
        }

        private void calculateDepths()
        {
            depths = new int[directories.size()];

            for (int i = 0; i < depths.length; i++)
            {
                SonicFSFile dir = directories.get(i);
                depths[i] = 0;

                if (dir.getParent() != null)
                {
                    do
                    {
                        dir = dir.getParentFile();
                        depths[i]++;
                    }
                    while (dir.getParent() != null);
                }
            }
        }

        public int getDepth(int i)
        {
            return (depths != null && i >= 0 && i < depths.length) ? depths[i] : 0;
        }

        @Override
        public void setSelectedItem(Object selectedDirectory)
        {
            this.selectedDirectory = (SonicFSFile)selectedDirectory;

            fireContentsChanged(this, -1, -1);
        }

        @Override
        public Object getSelectedItem()
        {
            return selectedDirectory;
        }

        @Override
        public int getSize()
        {
            return directories.size();
        }

        @Override
        public Object getElementAt(int index)
        {
            return directories.elementAt(index);
        }
    }

    protected class SonicShowModel extends DefaultListModel
    {
        public SonicShowModel()
        {
            // moved to buildModels()
            // build(getCurrentDirectory());
        }

        public void build(SonicFSFile parent)
        {
             clear();

             try
             {
                 SonicFSFile[] child = m_fs.listDetails(parent.getFullName());

                 for (int i = 0; i < child.length; i++)
                 {
                     if (child[i].isDirectory() ||
                         (getSonicFileFilter() == null) ||
                         getSonicFileFilter().accept(child[i]))
                    {
                        addElement(child[i]);
                    }
                 }
             }
             catch (Exception e)
             {
                 e.printStackTrace();
                 MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
             }
        }

        public int nearestIndexOf(SonicFSFile value)
        {
            for (int i = 0; i < getSize(); i++)
            {
              SonicFSFile element = (SonicFSFile) this.getElementAt(i);

                int res = SonicFSFileSystem.COMPARATOR.compare(value, element);

                // If object is already in the list then reject
                if (res == 0)
                {
                    return -1;
                }

                if (res < 0)
                {
                    return i;
                }
            }

            return getSize();
        }
    }

    protected class FileRenderer extends DefaultListCellRenderer
    {
        boolean sel = false;

        @Override
        public Component getListCellRendererComponent(JList list, Object value,
                                                      int index, boolean isSelected,
                                                      boolean cellHasFocus)
        {
            setComponentOrientation(list.getComponentOrientation());

            sel = isSelected;

            if (isSelected)
            {
                setBackground(list.getSelectionBackground());
                setForeground(list.getSelectionForeground());
            }
            else
            {
                setBackground(list.getBackground());
                setForeground(list.getForeground());
            }

            if (value instanceof Icon)
            {
                setIcon((Icon)value);
                setText("");
            }
            else
            if (value instanceof SonicFSFile)
            {
                SonicFSFile file = (SonicFSFile)value;

                setIcon(JSonicFileChooser.this.getFSIcon(file, isSelected));
                setText(JSonicFileChooser.this.getFSName(file));
            }
            else
            {
                setIcon(null);
                setText((value == null) ? "" : value.toString());
            }

            setEnabled(list.isEnabled());
            setFont(list.getFont());
            setBorder((cellHasFocus) ? UIManager.getBorder("List.focusCellHighlightBorder") : noFocusBorder);

            return this;
        }

        private Rectangle getTextRectangle()
        {
            String text = getText();
            Icon icon = isEnabled() ? getIcon() : getDisabledIcon();

            if ((icon == null) && (text == null))
            {
                return null;
            }

            Rectangle paintIconR = new Rectangle();
            Rectangle paintTextR = new Rectangle();
            Rectangle paintViewR = new Rectangle();
            Insets paintViewInsets = new Insets(0, 0, 0, 0);

            paintViewInsets = getInsets(paintViewInsets);
            paintViewR.x = paintViewInsets.left;
            paintViewR.y = paintViewInsets.top;
            paintViewR.width = getWidth() - (paintViewInsets.left + paintViewInsets.right);
            paintViewR.height = getHeight() - (paintViewInsets.top + paintViewInsets.bottom);

            Graphics g = getGraphics();
            if (g == null)
            {
                return null;
            }

            String clippedText = SwingUtilities.layoutCompoundLabel((JComponent)this, g.getFontMetrics(),
            text,
            icon,
            getVerticalAlignment(),
            getHorizontalAlignment(),
            getVerticalTextPosition(),
            getHorizontalTextPosition(),
            paintViewR,
            paintIconR,
            paintTextR,
            getIconTextGap());

            return paintTextR;
        }
    }

    public static abstract class SonicFileFilter
    {
        public abstract boolean accept(SonicFSFile f);
        public abstract String getDescription();

        @Override
        public String toString()
        {
            return getDescription();
        }
    }

    static class AllFileFilter extends SonicFileFilter
    {
        @Override
        public boolean accept(SonicFSFile f)
        {
            return true;
        }

        @Override
        public String getDescription()
        {
            return "All Files";
        }
    }

    class OkAction extends AbstractAction
    {
        public OkAction(int dialogType)
        {
            super("OK");
            putActionProperties(dialogType);
            
        }
        
        private void putActionProperties(int dialogType) {
            if (dialogType == OPEN_DIALOG)
            {
                putValue(Action.NAME, "Open");
            }
            else
            if (dialogType == SAVE_DIALOG)
            {
                putValue(Action.NAME, "Save");
            }
            else
            if (dialogType == CUSTOM_DIALOG)
            {
                putValue(Action.NAME, getApproveButtonText());
            }
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            approveSelection();
        }
    }

    class CancelAction extends AbstractAction
    {
        public CancelAction()
        {
            super("Cancel");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            cancelSelection();
        }
    }

    class JFileTextField extends JTextField
    {
        private SonicFSFile[] m_file = null;

        public JFileTextField()
        {
            super();
        }

        public void setFiles(SonicFSFile[] file)
        {
            m_file = file;

            StringBuffer sb = new StringBuffer();

            if (file != null)
            {
                for (int i = 0; i < file.length; i++)
                {
                    if (i != 0)
                    {
                        sb.append("; ");
                    }

                    sb.append(file[i].getName());
                }
            }

            setText(sb.toString());
        }

        public void setFile(SonicFSFile file)
        {
            setFiles((file == null) ? null: new SonicFSFile[] { file });
        }

        public SonicFSFile[] getFiles()
        {
            return m_file;
        }
    }

    class JFileList extends JList implements CellEditorListener, MouseMotionListener
    {
        transient protected ListCellEditor m_cellEditor;
        transient protected Component m_editorComp;
        transient protected int m_editingRow = -1;
        private boolean m_surrendersFocusOnKeystroke = false;
        private PropertyChangeListener m_editorRemover = null;
        transient protected javax.swing.Timer m_timer;
        transient protected int m_delayRow = -1;
        transient protected Point m_currPos = null;
        transient protected boolean m_selectionChanged = true;

        public JFileList(ListModel model)
        {
            super(model);
            prepareJFileList(model);
        }
        
        private void prepareJFileList(ListModel  model) {
            
            m_timer = new javax.swing.Timer(750, new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt)
                {
                    debugPrintln("Timer stop : sel changed = " + m_selectionChanged);
                    JFileList.this.removeMouseMotionListener(JFileList.this);
                    m_timer.stop();

                    if (m_selectionChanged)
                    {
                        m_selectionChanged = false;
                    }
                    else
                    //if (m_currPos != null) && (JFileList.this.locationToIndex(m_currPos) == m_delayRow))
                    {
                        JFileList.this.editCellAt(m_delayRow);
                    }
                    m_currPos = null;
                    m_delayRow = -1;
                }
            });

            getSelectionModel().addListSelectionListener(new ListSelectionListener()
            {
                @Override
                public void valueChanged(ListSelectionEvent evt)
                {
                    if (evt.getValueIsAdjusting())
                    {
                        return;
                    }

                    if ((m_editingRow >= 0) &&
                        (evt.getFirstIndex() == m_editingRow) &&
                        (evt.getLastIndex() == m_editingRow))
                    {
                        m_selectionChanged = false;
                    }
                    else
                    {
                        m_selectionChanged = true;
                    }
                }
            });

//            Set<KeyStroke> forwardFocusKeys = new TreeSet<KeyStroke>();
            Set<KeyStroke> forwardFocusKeys = new HashSet<KeyStroke>();
            forwardFocusKeys.add(KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.CTRL_MASK));

            Set<KeyStroke> backwardFocusKeys = new HashSet<KeyStroke>();
            backwardFocusKeys.add(KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.SHIFT_MASK | InputEvent.CTRL_MASK));

            setFocusTraversalKeys(KeyboardFocusManager.FORWARD_TRAVERSAL_KEYS, forwardFocusKeys);
            setFocusTraversalKeys(KeyboardFocusManager.BACKWARD_TRAVERSAL_KEYS, backwardFocusKeys);

            addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(MouseEvent evt)
                {
                    if (JFileList.this.isEditing())
                    {
                        JFileList.this.editingCanceled(null);
                    }
                    else
                    {
                        if (evt.getClickCount() != 1)
                        {
                            return;
                        }

                        int index = JFileList.this.locationToIndex(evt.getPoint());
                        Rectangle cellBounds = JFileList.this.getCellBounds(index, index);

                        if ((index >= 0) && cellBounds.contains(evt.getPoint()))
                        {
                            m_delayRow = index;
                            m_currPos = evt.getPoint();

                            debugPrintln("Timer start");
                            JFileList.this.addMouseMotionListener(JFileList.this);
                            m_timer.start();
                        }
                    }
                }
            });
        }

        @Override
        public void mouseDragged(MouseEvent evt) { mouseMoved(evt); }
        @Override
        public void mouseMoved(MouseEvent evt)
        {
            m_currPos = evt.getPoint();
        }

        @Override
        public void removeNotify()
        {
            KeyboardFocusManager.getCurrentKeyboardFocusManager().removePropertyChangeListener("permanentFocusOwner", m_editorRemover);
            m_editorRemover = null;

            super.removeNotify();
        }

        public void setSurrendersFocusOnKeystroke(boolean surrendersFocusOnKeystroke)
        {
            m_surrendersFocusOnKeystroke = surrendersFocusOnKeystroke;
        }

        public boolean getSurrendersFocusOnKeystroke()
        {
            return m_surrendersFocusOnKeystroke;
        }

        public boolean editCellAt(int row)
        {
            return editCellAt(row, null);
        }

        public boolean editCellAt(int row, EventObject e)
        {
            debugPrintln("editCellAt " + row);

            if ((m_cellEditor != null) && !m_cellEditor.stopCellEditing())
            {
                return false;
            }

            if ((row < 0) || (row >= getModel().getSize()))
            {
                return false;
            }

            if (!isCellEditable(row))
            {
                return false;
            }

            if (m_editorRemover == null)
            {
                KeyboardFocusManager fm = KeyboardFocusManager.getCurrentKeyboardFocusManager();
                m_editorRemover = new CellEditorRemover(fm);
                fm.addPropertyChangeListener("permanentFocusOwner", m_editorRemover);
            }

            ListCellEditor editor = new FileCellEditor();

            if (editor != null && editor.isCellEditable(e))
            {
                m_editorComp = prepareEditor(editor, row);

                if (m_editorComp == null)
                {
                    removeEditor();
                    return false;
                }

                // Lose the selection so that it looks right...
                getSelectionModel().removeIndexInterval(row, row);

                m_editorComp.setBounds(getCellRect(row));
                add(m_editorComp);
                m_editorComp.validate();

                setCellEditor(editor);
                setEditingRow(row);

                ((JTextField)m_editorComp).selectAll();

                editor.addCellEditorListener(this);

                return true;
            }

            return false;
        }

        private Rectangle getCellRect(int row)
        {
            Rectangle    r   = getCellBounds(row, row);
            FileRenderer cr  = (FileRenderer)getCellRenderer();
            Rectangle    tmp = cr.getTextRectangle();

            r.x += tmp.x;
            r.width -= tmp.x;

            return r;
        }

        public void removeEditor()
        {
            KeyboardFocusManager.getCurrentKeyboardFocusManager().removePropertyChangeListener("permanentFocusOwner", m_editorRemover);
            m_editorRemover = null;

            CellEditor editor = getCellEditor();

            if (editor != null)
            {
                debugPrintln("removeEditor");

                editor.removeCellEditorListener(this);

                if (m_editorComp != null)
                {
                    remove(m_editorComp);
                }

                Rectangle cellRect = getCellRect(m_editingRow);

                setCellEditor(null);
                setEditingRow(-1);
                m_editorComp = null;

                repaint(cellRect);
            }
        }

        public ListCellEditor getCellEditor()
        {
            return m_cellEditor;
        }

        public void setCellEditor(ListCellEditor anEditor)
        {
            ListCellEditor oldEditor = m_cellEditor;
            m_cellEditor = anEditor;

            firePropertyChange("cellEditor", oldEditor, anEditor);
        }

        public boolean isCellEditable(int row)
        {
            SonicFSFile file = (SonicFSFile)getModel().getElementAt(row);

            if (file == null)
            {
                return false;
            }

            if (file.getFullName().startsWith("/System"))
            {
                return false;
            }

            return true;
        }

        public void setEditingRow(int aRow)
        {
            m_editingRow = aRow;
        }

        public boolean isEditing()
        {
            return (m_cellEditor == null)? false : true;
        }

        public Component prepareEditor(ListCellEditor editor, int row)
        {
            Object value = getModel().getElementAt(row);
            boolean isSelected = this.isSelectedIndex(row);
            Component comp = editor.getListCellEditorComponent(this, value, isSelected, row);

            if (comp instanceof JComponent)
            {
                JComponent jComp = (JComponent)comp;
                if (jComp.getNextFocusableComponent() == null)
                {
                    jComp.setNextFocusableComponent(this);
                }
            }

            return comp;
        }

        public void setValueAt(Object value, int row)
        {
            DefaultListModel model = (DefaultListModel)getModel();

            model.set(row, value);
        }

        private void updateSubComponentUI(Object componentShell)
        {
            if (componentShell == null)
            {
                return;
            }

            Component component = null;

            if (componentShell instanceof Component)
            {
                component = (Component)componentShell;
            }

            if (componentShell instanceof DefaultCellEditor)
            {
                component = ((DefaultCellEditor)componentShell).getComponent();
            }

            if (component != null && component instanceof JComponent)
            {
                ((JComponent)component).updateUI();
            }
        }

        //---------------------------------------------------------------------
        //
        // CellEditorListener implementation
        //
        //---------------------------------------------------------------------

        @Override
        public void editingStopped(ChangeEvent evt)
        {
            debugPrintln("editingStopped");

            ListCellEditor editor = getCellEditor();

            if (editor != null)
            {
                debugPrintln("editingStopped - ok");

                Object value = editor.getCellEditorValue();

                setValueAt(value, m_editingRow);

                if (!m_selectionChanged)
                {
                    JFileList.this.getSelectionModel().setSelectionInterval(m_editingRow, m_editingRow);
                }

                removeEditor();
            }
        }

        /**
         * Invoked when editing is canceled. The editor object is discarded
         * and the cell is rendered once again.
         * <p>
         * Application code will not use these methods explicitly, they
         * are used internally by JTable.
         *
         * @param  e  the event received
         * @see CellEditorListener
         */
        @Override
        public void editingCanceled(ChangeEvent e)
        {
            if (!m_selectionChanged)
            {
                JFileList.this.getSelectionModel().setSelectionInterval(m_editingRow, m_editingRow);
            }

            removeEditor();
        }

        //---------------------------------------------------------------------

        // This class tracks changes in the keyboard focus state. It is used
        // when the JTable is editing to determine when to cancel the edit.
        // If focus switches to a component outside of the jtable, but in the
        // same window, this will cancel editing.
        class CellEditorRemover implements PropertyChangeListener
        {
            KeyboardFocusManager focusManager;

            public CellEditorRemover(KeyboardFocusManager fm)
            {
                this.focusManager = fm;
            }

            @Override
            public void propertyChange(PropertyChangeEvent ev)
            {
                if (!isEditing() || getClientProperty("terminateEditOnFocusLost") != Boolean.TRUE)
                {
                    return;
                }

                Component c = focusManager.getPermanentFocusOwner();

                while (c != null)
                {
                    if (c == JFileList.this)
                    {
                        // focus remains inside the table
                        return;
                    }
                    else
                    if (c instanceof Window)
                    {
                        if (c == SwingUtilities.getRoot(JFileList.this))
                        {
                            if (!getCellEditor().stopCellEditing())
                            {
                                getCellEditor().cancelCellEditing();
                            }
                        }
                        break;
                    }

                    c = c.getParent();
                }
            }
        }
    }

    /**
     * In a common use case, when the FileChooser is opened with the root as the anchoring folder,
     * the combobox model and listmodel separately ask for the details of '/' (root) in quick succession..
     * within a few milli secs (See buildModels() method).
     * The trivial caching FS is to reduce those 2 network calls to 1.
     */
    private static class TrivialCachedSonicFileSystem extends SonicFSFileSystem
    {
        /**
         * We want our cache to be very fast dying : 50 millis.
         */
        private static final long  CACHE_TTL = 50L;

        private String         lastPath;
        private SonicFSFile[]  lastPathDetails;
        private long           lastAccessTime;

        public TrivialCachedSonicFileSystem(IDirectoryFileSystemService dfs, String user)
        {
            super(dfs, user);
        }

        @Override
        public SonicFSFile[] listDetails(String path) throws SonicFSException
        {
            long now = System.currentTimeMillis();
            if (path.equals(lastPath) && (now - lastAccessTime) < CACHE_TTL)
            {
                return lastPathDetails;
            }

            lastPathDetails = super.listDetails(path);
            lastPath = path;
            lastAccessTime = System.currentTimeMillis();
            return lastPathDetails;
        }

        /**
         * Overriden as the modification operation is used by the chooser
         * @param path
         * @return
         * @throws SonicFSException
         */
        @Override
        public SonicFSFile createDirectory(String path) throws SonicFSException
        {
            clearCache();
            return super.createDirectory(path);
        }

        /**
         * Overriden as the modification operation is used by the chooser
         * @param oldFile
         * @param newName
         * @return
         * @throws SonicFSException
         */
        @Override
        public SonicFSFile rename(SonicFSFile oldFile, String newName) throws SonicFSException
        {
            clearCache();
            return super.rename(oldFile, newName);
        }

        private void clearCache()
        {
            lastPath = null;
            lastAccessTime = 0;
            lastPathDetails = null;
        }
    }

    //-------------------------------------------------------------------------
}
