/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.messagelog;

import java.awt.Color;
import java.awt.Component;
import java.io.FileWriter;
import java.io.PrintWriter;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.TableColumn;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.ColorIcon;
import com.sonicsw.ma.gui.util.ExtendedImageIcon;
import com.sonicsw.ma.gui.util.ExtendedJTable;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.ResourceManager;

public class MessageTableModel
    extends ModelListTableModel
{
    private static final int COL_TYPE = 0;
    private static final int COL_DOMAIN = 1;
    private static final int COL_MESSAGE = 2;
    private static final int COL_TIME = 3;

    public static final boolean DEFAULT_LOGFILE = false;
    public static final int DEFAULT_LOGSHOW = 100;

    private static final TableColumn[] COL = new TableColumn[]
                                             {
                                             new MessageTableColumn(0, "Type", COL_TYPE, 10),
                                             new MessageTableColumn(1, "Message", COL_MESSAGE, 70),
                                             new MessageTableColumn(2, "Time", COL_TIME, 20),
    };

    private int m_currentLogLimit;

    public MessageTableModel()
    {
        super(COL);

        m_currentLogLimit = getLogLimit();
    }

    public synchronized void addMessage(int type,
                                        String domain,
                                        String message,
                                        Throwable throwable)
    {
        MessageRowModel data = new MessageRowModel(type, domain, message, throwable);

        maintainViewSize(getLogLimit());

        insertRow(data, getRowCount());

        try
        {
            if (isLogToFile() && !MgmtConsole.isLogToFileReset())
            {
                logToFile(getLogFilePath(), data);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.resetLogToFile(true);
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR, "Error while writing to the log.", e, true);
        }
    }

    private int getLogLimit()
    {
        PreferenceManager manager = PreferenceManager.getInstance();

        return manager.getInt(JPreferencesDialog.MESSAGE_PREFS,
                              JPreferencesDialog.LOG_SCREENSHOW, DEFAULT_LOGSHOW);
    }

    private boolean isLogToFile()
    {
        PreferenceManager manager = PreferenceManager.getInstance();

        return manager.getBoolean(JPreferencesDialog.MESSAGE_PREFS,
                                  JPreferencesDialog.LOG_FILE, false);
    }

    private String getLogFilePath()
    {
        PreferenceManager manager = PreferenceManager.getInstance();

        return manager.getString(JPreferencesDialog.MESSAGE_PREFS,
                                 JPreferencesDialog.LOG_FILEPATH, null);
    }

    private void logToFile(String logFilePath, MessageRowModel data)
        throws Exception
    {
        PrintWriter writer = null;

        writer = new PrintWriter(new FileWriter(logFilePath, true));

        writer.println(data.toString());
        writer.flush();

        if (writer != null)
        {
            writer.close();
        }

    }

    private void maintainViewSize(int newLogLimit)
    {
        if (m_currentLogLimit != newLogLimit)
        {
            m_currentLogLimit = newLogLimit;
        }

            // Delete 1 more than the limit cos' we're just about
            // to add another one in!!
        while (getRowCount() >= m_currentLogLimit)
        {
            deleteRow(0);
        }
    }

    public static final String logTypeToString(int type)
    {
        String key = "message-log.level." + Integer.toString(type);
        String value = ResourceManager.getString(MessageTableModel.class, key);

        return (value != null) ? value : "-";
    }

    public static final Icon logTypeToIcon(int type)
    {
        Icon icon = null;

        switch (type)
        {
            case IApplication.MESSAGE_STATUS:
                icon = new ColorIcon(Color.BLUE, 14);
                break;
            case IApplication.MESSAGE_INFO:
                icon = new ColorIcon(Color.GREEN, 14);
                break;
            case IApplication.MESSAGE_WARNING:
                icon = new ColorIcon(Color.YELLOW, 14);
                break;
            case IApplication.MESSAGE_ERROR:
                icon = new ColorIcon(Color.RED, 14);
                break;
        }

        return icon;
    }

    //-------------------------------------------------------------------------

    static class MessageTableColumn
        extends RowTableColumn
    {
        private int m_id;

        MessageTableColumn(int modelIndex, String name, int id, int width)
        {
            super(modelIndex, name, width);
            prepareMessageTableColumn(id);
        }
        
        private void prepareMessageTableColumn(int id) {
            
            m_id = id;

            if (m_id == COL_TYPE)
            {
                setCellRenderer(new TypeTableCellRenderer());
            }

            if (m_id == COL_TIME)
            {
                setColumnClass(Long.class);
                setCellRenderer(Helper.getDateTimeTableCellRenderer());
            }
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            if (!(rowModel instanceof MessageRowModel))
            {
                return null;
            }

            MessageRowModel data = (MessageRowModel) rowModel;

            Object value = null;

            switch (m_id)
            {
                case COL_TYPE:
                    return data;

                case COL_DOMAIN:
                    if (data.m_domain != null)
                    {
                        value = data.m_domain;
                    }
                    break;

                case COL_MESSAGE:
                    if (data.m_message != null)
                    {
                        value = data.m_message;
                    }
                    break;

                case COL_TIME:
                    value = new Long(data.m_time);
                    break;
            }

            return value;
        }
    }

    static class TypeTableCellRenderer
        extends ExtendedJTable.ExtendedTableCellRenderer
    {
        TypeTableCellRenderer()
        {
            setHorizontalAlignment();
        }

        private void setHorizontalAlignment() {
            setHorizontalAlignment(SwingConstants.LEFT);
        }
        
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column)
        {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if ((value != null) && (value instanceof MessageRowModel))
            {
                MessageRowModel data = (MessageRowModel) value;

                setText(MessageTableModel.logTypeToString(data.m_type));
                setIcon(createIcon(data));
                setToolTipText(this, table, createToolTipText(data), column);
            }
            else
            {
                setText(null);
                setIcon(null);
                setToolTipText(this, table, null, column);
            }

            return this;
        }

        public String createToolTipText(MessageRowModel data)
        {
            StringBuffer sb = new StringBuffer(MessageTableModel.logTypeToString(data.m_type));

            if (data.m_throwable != null)
            {
                sb.append(" - Log available");
            }

            return sb.toString();
        }

        public ImageIcon createIcon(MessageRowModel data)
        {
            ExtendedImageIcon icon = new ExtendedImageIcon(MessageTableModel.logTypeToIcon(data.m_type));

            if (data.m_throwable != null)
            {
                icon.addOverlay(ResourceManager.getIcon(getClass(), "messageoverlay"));
            }

            return icon;
        }
    }
}