// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.ma.gui.perms;

import com.sonicsw.mf.common.security.IConfigurePermissionBits;
import com.sonicsw.mf.common.security.IConfigureScopeBits;

public class ConfigurePermissions extends AbstractPermissions implements IConfigurePermissionBits, IConfigureScopeBits {

    /**
     * <pre>
     * *	Folder:
     * o	"This folder, its subfolders and all configurations/files" [This folder + Subfolders + All configurations/files]
     * o	"This folder and its subfolders" [This folder + Subfolders]
     * o	"This folder and its configurations/files" [This folder + All configurations/files]
     * o	"This folder's configurations/files only" [All configurations/files]
     * o	"This folder's subfolders and all configurations/files" [Subfolders + All configurations/files]
     * o	"This folder's subfolders only" [Subfolders]
     * o	"This folder only" [This folder]
     * </pre>
     */
    public static PermissionsList.ScopeType [] folderScopeTypes = new PermissionsList.ScopeType[]{
            new PermissionsList.ScopeType("This folder, its subfolders and all configurations/files", THIS_FOLDER_SCOPE | ALL_FOLDERS_SCOPE | ALL_CONFIGURATIONS_AND_FILES_SCOPE),
            new PermissionsList.ScopeType("This folder and its subfolders", THIS_FOLDER_SCOPE | ALL_FOLDERS_SCOPE),
            new PermissionsList.ScopeType("This folder and its configurations/files", THIS_FOLDER_SCOPE | ALL_CONFIGURATIONS_AND_FILES_SCOPE),
            new PermissionsList.ScopeType("This folder's configurations/files only", ALL_CONFIGURATIONS_AND_FILES_SCOPE),
            new PermissionsList.ScopeType("This folder's subfolders and all configurations/files", ALL_FOLDERS_SCOPE | ALL_CONFIGURATIONS_AND_FILES_SCOPE),
            new PermissionsList.ScopeType("This folder's subfolders only", ALL_FOLDERS_SCOPE),
            new PermissionsList.ScopeType("This folder only", THIS_FOLDER_SCOPE)
    };

    /**
     * <pre>
     * *	File:
     * o	"This configuration/file only" [This configuration/file]
     * </pre>
     */
    public static PermissionsList.ScopeType [] fileOrConfigScopeTypes = new PermissionsList.ScopeType[]{
            new PermissionsList.ScopeType("This configuration/file only", THIS_CONFIGURATION_OR_FILE_SCOPE),
    };

    //pseudo permissions
    public static final int ALLOW_FULL_CONTROL = ALLOW_READ | ALLOW_WRITE | ALLOW_DELETE | ALLOW_SET_PERMISSIONS;
    public static final int DENY_FULL_CONTROL = DENY_READ | DENY_WRITE | DENY_DELETE | DENY_SET_PERMISSIONS;

    public static PermissionsList.PermissionType [] permissionTypes = {
            new PermissionsList.PermissionType("Full control", ALLOW_FULL_CONTROL, DENY_FULL_CONTROL),
            new PermissionsList.PermissionType("Read", ALLOW_READ, DENY_READ),
            new PermissionsList.PermissionType("Write", ALLOW_WRITE, DENY_WRITE),
            new PermissionsList.PermissionType("Delete", ALLOW_DELETE, DENY_DELETE),
            new PermissionsList.PermissionType("Set permissions", ALLOW_SET_PERMISSIONS, DENY_SET_PERMISSIONS)};



    public ConfigurePermissions(AbstractPermissions.Principal principal) {
        super(principal);
    }

    @Override
    public void removePermissionBit(int perm) {
        if(perm == ALLOW_FULL_CONTROL || perm == DENY_FULL_CONTROL)
         {
            return; // ignore
        }
        super.removePermissionBit(perm);
    }


    @Override
    protected void handlePermAdded(int perm) {
        if(perm == ALLOW_WRITE || perm == ALLOW_DELETE){
            _addPermissionBits(ALLOW_READ);
        }
        if(perm == DENY_READ){
            _addPermissionBits(DENY_WRITE | DENY_DELETE);
        }
    }


    @Override
    protected void handlePermRemoved(int perm) {
        if(perm == ALLOW_READ){
            _removePermissionBits(ALLOW_WRITE | ALLOW_DELETE);
        }
    }

    public static String getPermissionName(int bit) {
        PermissionsList.PermissionType [] types = permissionTypes;
        for (int i = 0; i < types.length; i++) {
              PermissionsList.PermissionType perm = types[i];
            if(bit==perm.allowBit)
            {
                return perm.name;
            }
            if(bit==perm.denyBit)
            {
                return "Deny "+perm.name;
            }
        }
        return "Unknown Permission";
    }

    @Override
    protected String getScopeDisplayName() {
        String s = nameFrom(folderScopeTypes);
        if(s!=null)
        {
            return s;
        }
        s = nameFrom(fileOrConfigScopeTypes);
        if(s!=null)
        {
            return s;
        }
        return "?";
    }

    @Override
    protected String getPermsDisplayName() {
        int perms = getPermissionsBits();
        StringBuffer buf = new StringBuffer("[ ");
        for (int i = 0; i < permissionTypes.length; i++) {
            PermissionsList.PermissionType type = permissionTypes[i];
            if((perms & type.allowBit) == type.allowBit)
            {
                buf.append(type.name+ ":Allow").append(",");
            }
            if((perms & type.denyBit) == type.denyBit)
            {
                buf.append(type.name+ ":Deny").append(",");
            }
        }
        buf.setLength(buf.length()-1);
        buf.append(" ]");
        return buf.toString();
    }


}
