// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.ma.gui.perms;

import com.sonicsw.mf.common.security.IManagePermissionBits;
import com.sonicsw.mf.common.security.IManageScopeBits;

public class ManagePermissions extends AbstractPermissions implements IManagePermissionBits, IManageScopeBits {

    /**
     * <pre>
     * *	Folder:
     * o	"This folder's and subfolders' containers and components" [Subfolders + All containers + All components]
     * o	"This folder's and subfolders' containers" [Subfolders + All containers]
     * o	"This folder's and subfolders' components" [Subfolders + All components]
     * o	"This folder's containers and components" [All containers + All components]
     * o	"This folder's containers only" [All containers]
     * o	"This folder's components only" [All components]
     * </pre>
     */
    public static PermissionsList.ScopeType [] folderScopeTypes = new PermissionsList.ScopeType[]{
            new PermissionsList.ScopeType("This folder's and subfolders' containers and components", ALL_FOLDERS_SCOPE | ALL_CONTAINERS_SCOPE | ALL_COMPONENTS_SCOPE),
            new PermissionsList.ScopeType("This folder's and subfolders' containers", ALL_FOLDERS_SCOPE | ALL_CONTAINERS_SCOPE),
            new PermissionsList.ScopeType("This folder's and subfolders' components", ALL_FOLDERS_SCOPE | ALL_COMPONENTS_SCOPE),
            new PermissionsList.ScopeType("This folder's containers and components", ALL_CONTAINERS_SCOPE | ALL_COMPONENTS_SCOPE),
            new PermissionsList.ScopeType("This folder's containers only", ALL_CONTAINERS_SCOPE),
            new PermissionsList.ScopeType("This folder's components only", ALL_COMPONENTS_SCOPE)
    };

    /**
     * <pre>
     * *	Container:
     * o	"This container and its components" [This container + All Components]
     * o	"This container only" [This container]
     * o	"This container's components only" [All components]
     * </pre>
     */
    public static PermissionsList.ScopeType [] containerScopeTypes = new PermissionsList.ScopeType[]{
            new PermissionsList.ScopeType("This container and its components", THIS_CONTAINER_SCOPE | ALL_COMPONENTS_SCOPE),
            new PermissionsList.ScopeType("This container only", THIS_CONTAINER_SCOPE),
            new PermissionsList.ScopeType("This container's components only", ALL_COMPONENTS_SCOPE)
    };

    /**
     * <pre>
     * *	Component:
     * o	"This component only" [This component]
     * </pre>
     */
    public static PermissionsList.ScopeType [] componentScopeTypes = new PermissionsList.ScopeType[]{
            new PermissionsList.ScopeType("This component only", THIS_COMPONENT_SCOPE)
    };

    //pseudo permissions
    public static final int ALLOW_FULL_CONTROL = ALLOW_LIFE_CYCLE_CONTROL | ALLOW_ENABLE_DISABLE_METRICS | ALLOW_NOTIFICATION_SUBSCRIPTION | ALLOW_SET_ATTRIBUTES | ALLOW_PERFORM_ACTIONS | ALLOW_GET_INFORMATION;
    public static final int DENY_FULL_CONTROL = DENY_LIFE_CYCLE_CONTROL | DENY_ENABLE_DISABLE_METRICS | DENY_NOTIFICATION_SUBSCRIPTION | DENY_SET_ATTRIBUTES | DENY_PERFORM_ACTIONS | DENY_GET_INFORMATION;


    public static PermissionsList.PermissionType [] permissionTypes = {
            new PermissionsList.PermissionType("Full control", ALLOW_FULL_CONTROL, DENY_FULL_CONTROL),
            new PermissionsList.PermissionType("Life cycle control", ALLOW_LIFE_CYCLE_CONTROL, DENY_LIFE_CYCLE_CONTROL),
            new PermissionsList.PermissionType("Enable/disable metrics", ALLOW_ENABLE_DISABLE_METRICS, DENY_ENABLE_DISABLE_METRICS),
            new PermissionsList.PermissionType("Subscribe to notifications", ALLOW_NOTIFICATION_SUBSCRIPTION, DENY_NOTIFICATION_SUBSCRIPTION),
            new PermissionsList.PermissionType("Set attributes", ALLOW_SET_ATTRIBUTES, DENY_SET_ATTRIBUTES),
            new PermissionsList.PermissionType("Other actions", ALLOW_PERFORM_ACTIONS, DENY_PERFORM_ACTIONS),
            new PermissionsList.PermissionType("Get information", ALLOW_GET_INFORMATION, DENY_GET_INFORMATION)};


    public ManagePermissions(AbstractPermissions.Principal principal) {
        super(principal);
    }

    @Override
    public void removePermissionBit(int perm) {
        if(perm == ALLOW_FULL_CONTROL || perm == DENY_FULL_CONTROL)
         {
            return; // ignore
        }
        super.removePermissionBit(perm);
    }

    public static String getPermissionName(int bit) {
        PermissionsList.PermissionType [] types = permissionTypes;
        for (int i = 0; i < types.length; i++) {
              PermissionsList.PermissionType perm = types[i];
            if(bit==perm.allowBit)
            {
                return perm.name;
            }
            if(bit==perm.denyBit)
            {
                return "Deny "+perm.name;
            }
        }
        return "Unknown Permission";
    }

    @Override
    protected String getScopeDisplayName() {
        String s = nameFrom(folderScopeTypes);
        if(s!=null)
        {
            return s;
        }
        s = nameFrom(containerScopeTypes);
        if(s!=null)
        {
            return s;
        }
        s = nameFrom(componentScopeTypes);
        if(s!=null)
        {
            return s;
        }
        return "?";
    }

    @Override
    protected String getPermsDisplayName() {
        int perms = getPermissionsBits();
        StringBuffer buf = new StringBuffer("[ ");
        for (int i = 0; i < permissionTypes.length; i++) {
            PermissionsList.PermissionType type = permissionTypes[i];
            if((perms & type.allowBit) == type.allowBit)
            {
                buf.append(type.name+ ":Allow").append(",");
            }
            if((perms & type.denyBit) == type.allowBit)
            {
                buf.append(type.name+ ":Deny").append(",");
            }
        }
        buf.setLength(buf.length()-1);
        buf.append(" ]");
        return buf.toString();
    }
}
