// Copyright (c) 2007 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.ma.gui.perms;

import java.util.Arrays;

import javax.swing.JComponent;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.config.AbstractConfigPlugin;
import com.sonicsw.ma.gui.config.ConfigDomainFolderPlugin;
import com.sonicsw.ma.gui.config.ConfigFolderPlugin;
import com.sonicsw.ma.gui.runtime.AbstractRuntimePlugin;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.mx.config.ConfigServerUtility;
import com.sonicsw.mx.config.ConfigServiceException;
import com.sonicsw.mx.config.impl.ConfigBeanFile;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.security.ConfigurePermissionDeniedException;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.security.IManagementSecurityConfigurator;
import com.sonicsw.mf.common.security.ManagePermissionDeniedException;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;

/**
 * @author <a href="mailto:murali@sonicsoftware.com">Murali Mohan</a>
 * @since: Sep 20, 2006
 */
public class ManagementSecurityUtils {
    public static final String MF_CONTAINER = "MF_CONTAINER";
    public static final String AUTHENTICATION_DOMAIN = "AUTHENTICATION_DOMAIN";
    public static final String[] LOG4J_CONFIG_TYPES = new String[] {"*.xml","*.properties","*.*"};

    public static boolean allowsConfigPerms(IPlugin plugin) {
        String type = plugin.getAttributes().getType();
        return plugin.getAttributes().isCreatable()    || // top level element that can be created
               plugin.getAttributes().isComponent()    || // component (covers backups that can only be created via the primary)
               ConfigBeanFile.CONFIG_TYPE.equals(type) || // file
               isFolder(plugin);                          // folder  todo.. may be this check is not sufficient
    }

    public static Readability getReadability(IPlugin plugin){
        try {
            IDirectoryFileSystemService ds = plugin.getPluginContext().getConnectionInfo().getDirectoryService();
            IManagementPermission[][] managementPermissions = ((IManagementSecurityConfigurator) ds).getManagementPermissions(new String[]{plugin.getPluginPath()}, IManagementPermission.CONFIGURE_TYPE);
        } catch (ConfigurePermissionDeniedException e){
            return new Readability(plugin,false,ManagementSecurityUtils.getErrorMessageFor(e));
        } catch (Exception e) {
            // may be readable. may be not.
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
        return new Readability(plugin,true,null);
    }

    public static boolean allowsManagePerms(IPlugin plugin) {
        String type = plugin.getAttributes().getType();
        return ((plugin instanceof AbstractConfigPlugin) && MF_CONTAINER.equals(type)) ||  // CONTAINER
                isFolder(plugin);  // folder
    }
    
    public static boolean hasManagePerms(IPlugin plugin){
    	String type = plugin.getAttributes().getType();
    	return ((plugin instanceof AbstractRuntimePlugin) && MF_CONTAINER.equals(type)) ||  // CONTAINER
        			isFolder(plugin) ||  // folder
        			plugin.getAttributes().isComponent() ||
                    MF_CONTAINER.equals(((IPlugin)((AbstractGUIPlugin)plugin).getParent()).getAttributes().getType())
        			;  
    }

    public static boolean isFolder(IPlugin plugin){
        return plugin!=null && (plugin.getClass().equals(ConfigDomainFolderPlugin.class) ||  //  root folder
                                plugin.getClass().equals(ConfigFolderPlugin.class));  // ordinary folder
    }

    public static String getSecurityDomain(IPlugin plugin){
        try {
            IDirElement domainElement = ((IManagementSecurityConfigurator)plugin.getPluginContext().getConnectionInfo().getDirectoryService()).getDomainElement(false);
            IAttributeSet attributes = domainElement.getAttributes();
            Reference authDomanRef = (Reference) attributes.getAttribute(AUTHENTICATION_DOMAIN);
            String elementName = authDomanRef.getElementName();
            String path = plugin.getPluginContext().getConnectionInfo().getDirectoryService().storageToLogical(elementName);
            if (path.endsWith(ConfigServerUtility.DEFAULT_SUFFIX) ||
                    path.endsWith(ConfigServerUtility.DOMAIN_DESCRIPTOR_SUFFIX) ||
                    path.endsWith(ConfigServerUtility.POLICY_DESCRIPTOR_SUFFIX)) {
                int lastIndex = path.lastIndexOf('/');

                path = path.substring(0, lastIndex);
            }
            return path;
        } catch (Exception e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
        return null;
    }

    public static boolean isManagementSecurityEnabled(IPlugin plugin){
        try {
            IDirElement domainElement = ((IManagementSecurityConfigurator)plugin.getPluginContext().getConnectionInfo().getDirectoryService()).getDomainElement(false);
            IAttributeSet attributes = domainElement.getAttributes();
            Reference authDomanRef = (Reference) attributes.getAttribute(AUTHENTICATION_DOMAIN);
            if (authDomanRef == null)
            {
                return false;
            }
            String elementName = authDomanRef.getElementName();
            return elementName !=null && elementName.trim().length()>0;
        } catch (Exception e) {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
        return false;
    }

    public static String[] getAllowedLog4jConfigExtensions(){
        return Arrays.copyOf(LOG4J_CONFIG_TYPES, LOG4J_CONFIG_TYPES.length);
    }

    public static JComponent[] getPermissionsMenuItems(IPlugin plugin) {
        boolean allowsConfig = ManagementSecurityUtils.allowsConfigPerms(plugin);
        boolean allowsManage = ManagementSecurityUtils.allowsManagePerms(plugin);

        if(!(allowsConfig || allowsManage))
        {
            return new JComponent[0];
        }
        String pluginPath = plugin.getPluginPath();
        return _getMenuItems(plugin, pluginPath, allowsConfig, allowsManage);
    }

    public static JComponent[] getComponentPermissionsMenuItems(IPlugin componentPlugin, String componentPath) {
        return _getMenuItems(componentPlugin, componentPath, false, true);
    }

    private static JComponent[] _getMenuItems(IPlugin plugin, String pluginPath, boolean allowsConfig, boolean allowsManage) {
        JComponent[] ret = new JComponent[1];
        JBasicMenu permsMenu = new JBasicMenu(new ManagementSecurityAction());
        ConfigurePermsAction cAction = new ConfigurePermsAction(plugin,pluginPath);
        boolean deflt = plugin!=null && isManagementSecurityEnabled(plugin);
        cAction.setEnabled(deflt && allowsConfig);
        permsMenu.add(new JBasicMenuItem(cAction));
        ManagePermsAction mAction = new ManagePermsAction(plugin,pluginPath);
        mAction.setEnabled(deflt && allowsManage);
        permsMenu.add(new JBasicMenuItem(mAction));

        ret[0] = permsMenu;
        return ret;
    }

    public static String getErrorMessageFor(ManagementPermissionDeniedException e){
        String dontHave = "You do not have permission to perform a '";
        String on = "' on ";
        String onCofig = "' on this configuration";
        String onTarget = "' on this target";
        if(e instanceof ConfigurePermissionDeniedException){
            ConfigurePermissionDeniedException ce = ((ConfigurePermissionDeniedException) e);
            String path = ce.getLogicalPath();
            int bit = ce.getRequiredPermission();
            String requiredPerm = bit==0?null:ConfigurePermissions.getPermissionName(bit).toLowerCase();
            if(path!=null && requiredPerm!=null){
                return dontHave + requiredPerm + on + path;
            }
            if(path==null && requiredPerm!=null){
                return dontHave + requiredPerm + onCofig;
            }
        }
        if(e instanceof ManagePermissionDeniedException){
            ManagePermissionDeniedException me = ((ManagePermissionDeniedException) e);
            String target = me.getTarget();
            int bit = me.getRequiredPermission();
            String requiredPerm = bit==0?null:ManagePermissions.getPermissionName(bit).toLowerCase();
            if(target!=null && requiredPerm!=null){
                if(requiredPerm.equals("Subscribe to notifications"))
                {
                    return "You do not have permission to listen to notifications from "+target;
                }
                return dontHave + requiredPerm + on + target;
            }
            if(target==null && requiredPerm!=null){
                return dontHave + requiredPerm + onTarget;
            }
        }
        return e.getMessage();
    }

    public static boolean isCausedByPermissionDenial(Exception e) {
        if(e instanceof ConfigServiceException)
        {
            return isCausedByPermissionDenial(((ConfigServiceException) e).getLinkedException());
        }
        return e instanceof ManagementPermissionDeniedException;

    }

    public static class Readability {

        /**
         * Marker Readability reused by readable plugins.
         */
        public static final Readability READABLE = new Readability(null,true,null);

        /**
         * <b>null</b> for READABLE
         */
        public final IPlugin plugin;
        public final boolean readable;
        /**
         * <b>null</b> if readable is true 
         */
        public final String message;

        private Readability(IPlugin plugin,boolean readable,String denyMessage) {

            this.plugin = plugin;
            this.readable = readable;
            this.message = denyMessage;
        }
    }
}
