package com.sonicsw.ma.gui.propsheets;

import java.awt.BorderLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.ListSelectionModel;

import modelobjects.framework.EditRule;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.util.ObjectCopier;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.config.DSFileChooser;
import com.sonicsw.ma.gui.config.FileElementFilter;
import com.sonicsw.ma.gui.config.propsheets.JConfigDialog;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.JRowTableAspectAdapter;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JPathField;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.JRowTableButtonPanel;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.IPluginFilter;
import com.sonicsw.mx.config.ConfigFactory;
import com.sonicsw.mx.config.IConfigPath;
import com.sonicsw.mx.config.IConfigServer;

/**
 * Generic panel to handle classpath attribute for
 * MF_APLLICATION_COMPONENT types:
 * such as MF_BROKER, MF_BACKUPBROKER or MF_GENERIC_COMPONENT
 */
public class ClasspathPanel extends JPanel
{
    public static final IConfigPath CLASSPATH = ConfigFactory.createConfigPath("CLASSPATH");

    private static final String TOKEN = ";";
    private static final String SONICFS_URL_PREFIX = "sonicfs:///";

    protected JConfigDialog m_configParent;
    protected JRowTable m_classpathTable;

    public ClasspathPanel(JConfigDialog parent)
    {
        super(new BorderLayout(JPartitionPanel.DEFAULT_GAP_SIZE,
                               JPartitionPanel.DEFAULT_GAP_SIZE));
        setBorder();
        m_configParent = parent;
    }

    private void setBorder() {
        setBorder(BorderFactory.createEmptyBorder(JPartitionPanel.
                                                  DEFAULT_TB_EDGE_GAP,
                                                  JPartitionPanel.
                                                  DEFAULT_LR_EDGE_GAP,
                                                  JPartitionPanel.
                                                  DEFAULT_TB_EDGE_GAP,
                                                  JPartitionPanel.
                                                  DEFAULT_LR_EDGE_GAP));
    }
    
    private ModelEditMediator getMediator()
    {
        return m_configParent.getEditForm().getModelEditMediator();
    }

    public void initUI()
    {
        add(makeClasspathTable(), BorderLayout.CENTER);
        add(makeClasspathButtons(), BorderLayout.EAST);
    }

    public void initForm()
    {
        new FormattedStringTableAspectAdapter(m_configParent.getModelAspectId(CLASSPATH),
                                              m_classpathTable, EditRule.ALWAYS,
                                              getMediator());
    }

    private JPanel makeClasspathButtons()
    {
        ClasspathButtonPanel panel = new ClasspathButtonPanel(m_classpathTable);

        panel.setAddAction(new AddAction());
        panel.setEditAction(new EditAction());
        panel.setRemoveAction(new RemoveAction(panel.getRemoveAction()));
        panel.addRemainder();
        panel.addButton(new MoveUpAction());
        panel.addButton(new MoveDownAction());

        panel.updateState(null);

        return panel;
    }

    private JComponent makeClasspathTable()
    {
        RowTableColumn[] COLUMNS = { new RowTableColumn(0, "Classpath", 100), };

        COLUMNS[0].setSortable(false);

        m_classpathTable = new JRowTable(COLUMNS, getClass().getName());

        return new ExtendedJScrollPane(m_classpathTable);
    }

    //-------------------------------------------------------------------------
    //
    // Inner Classes
    //
    //-------------------------------------------------------------------------

    class AddAction extends BasicGuiAction
    {
        public AddAction()
        {
            super("editable_component.add");
        }

        private String[] splitPath(String path)
        {
            StringTokenizer st = new StringTokenizer(path, ";");
            List res = new ArrayList();
            
            while (st.hasMoreTokens())
            {
                res.add(st.nextToken().trim());
            }
            
            return (String[])res.toArray(new String[0]);
        }
        
        @Override
        public void actionPerformed(ActionEvent evt)
        {
            AddClassPathDialog dialog = new AddClassPathDialog(ClasspathPanel.this.m_configParent, "classpath.add", true);
            dialog.setVisible(true);

            if (dialog.getCloseCommand() == JMADialog.CLOSE_OK)
            {
                // The classpath might contain more than 1 classpath entry...so we need to set each one in the table.
                String[] path = splitPath(dialog.getPath());

                IModelTableModel model = (IModelTableModel) m_classpathTable.getModel();
                
                for (int i = 0; i < path.length; i++)
                {
                    int found = model.findCell(0, 0, path[i]);
                    
                    if ((path[i].length() > 0) && (found == -1))
                    {
                        model.insertRow(path[i], m_classpathTable.getRowCount());
                    }
                }

                // Force selection on to newly added row(s)...
                //
                for (int j = 0; j < path.length; j++)
                {
                    int found = model.findCell(0, 0, path[j]);
                    
                    if (found != -1)
                    {
                        m_classpathTable.getSelectionModel().addSelectionInterval(found, found);
                    }
                }
            }
        }
    }

    class EditAction extends BasicGuiAction
    {
        public EditAction()
        {
            super("editable_component.edit");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            IModelTableModel model = (IModelTableModel) m_classpathTable.getModel();
            int sel = m_classpathTable.getSelectedRow();
            String rowModel = (String) model.getRowModel(sel);
            AddClassPathDialog dialog = new AddClassPathDialog(ClasspathPanel.this.m_configParent, "classpath.edit", false);
            dialog.setPath(rowModel);

            dialog.setVisible(true);

            if (dialog.getCloseCommand() == JMADialog.CLOSE_OK)
            {
                String path = dialog.getPath();  // No splitting here...since we should only have a single classpath entry.

                if ( (path.length() > 0) && !rowModel.equals(path))
                {
                    // Check if we have match that isn't the current selection,
                    // to prevent duplicates
                    int match = model.findCell(0, 0, path);

                    if ( (match != -1) && (match != sel))
                    {
                        Toolkit.getDefaultToolkit().beep();
                    }
                    else
                    {
                        model.getContents().set(sel, path);
                        ( (ModelListTableModel) model).fireTableRowsUpdated(sel,
                            sel);
                    }
                }
            }
        }
    }

    class RemoveAction extends BasicGuiAction
    {
        public RemoveAction(Action target)
        {
            super("editable_component.remove", target);
        }
    }

    class AddClassPathDialog extends JMADialog
    {
        JPathField m_classpath = new JPathField(32);
        Action m_okAction;
        IPlugin m_plugin = null;
        boolean m_multiSelect;
        
        public AddClassPathDialog(JMADialog parent, String resource, boolean multiSelect)
        {
            super(parent, resource);
            m_plugin = m_configParent.getPlugin();
            setResizableProperty();
            m_okAction = createOKAction();
            m_multiSelect = multiSelect;
        }

        private void setResizableProperty() {
            setResizable(true);
        }
        
        private OKAction createOKAction() {
           return new OKAction(this.getDefaultOKAction());
        }
        
        @Override
        public void maInitialize()
        {
            getContentPane().add(makeMainPanel());

            m_classpath.setButtonAction(new ClasspathAction(m_multiSelect));
        }

        @Override
        public void maCleanup()
        {
        }

        private JPanel makeMainPanel()
        {
            JPartitionPanel topPanel = new JPartitionPanel(true,"p",null);
            getContentPane().add(topPanel, BorderLayout.CENTER);

            JPartitionPanel pathPanel = new JPartitionPanel(false, "p,r", "", 5, 0, 0);
            topPanel.add(pathPanel);

            pathPanel.addRow(ResourceManager.getString(getClass(), "dialog.classpath.path.label"), m_classpath);

            return topPanel;
        }

        public String getPath()
        {
            return m_classpath.getText();
        }

        public void setPath(String path)
        {
            m_classpath.setText(path);
        }

        @Override
        public Action[] getButtonActions()
        {
            ArrayList list = new ArrayList();

            list.add(m_okAction);
            list.add(getDefaultCancelAction());

            return (Action[]) list.toArray(new Action[list.size()]);
        }

        /****
         * Inner classes of AddClassPathDialog
         */
        class OKAction extends BasicGuiAction
        {
            public OKAction(Action target)
            {
                super("dialog.ok", target);
            }

            @Override
            public void actionPerformed(ActionEvent evt)
            {
                if (getPath().length() == 0)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(IApplication.
                        MESSAGE_ERROR, "Path can't be empty", true);
                    return;
                }
                super.actionPerformed(evt);
            }
        }

        class ClasspathAction extends BasicGuiAction
        {
            String selectedFilter = "*.jar, *.war, *.zip";
            String[] m_filters = new String[] { selectedFilter, "*.*" };
            boolean m_multiSelect;
            
            ClasspathAction(boolean multiSelect)
            {
                super("archive.browse");
                
                m_multiSelect = multiSelect;
            }

            @Override
            public void actionPerformed(ActionEvent evt)
            {

                IPluginFilter filter = new FileElementFilter();

                ((FileElementFilter)filter).setSelectedFilter(selectedFilter);
                
                //If m_plugin is null, get the plugin context from mgmtconsole 
                IConfigServer server = null;
                IPluginContext context = null;
                if(m_plugin !=  null)
                {
                	server = m_plugin.getPluginContext().getConfigContext().getConfigServer();  
                	context = m_plugin.getPluginContext();
                }
                else
                {
                	server = MgmtConsole.getMgmtConsole().getPluginContext().getConfigContext().getConfigServer();
                	context = MgmtConsole.getMgmtConsole().getPluginContext();
                }
                
                JPropSheetDialog dialog = new DSFileChooser(m_configParent,
                                                            "classpath.choose",
                                                            context,
                                                            server,
                                                            filter,
                                                            m_filters,
                                                            m_multiSelect);
                try
                {
                    dialog.editInstance(m_plugin, new HashMap(), false);
                    dialog.setVisible(true);

                    if (dialog.getCloseCommand() == JMADialog.CLOSE_OK)
                    {
                        HashMap selList = (HashMap)dialog.getModel();

                        if ((selList != null) && !selList.isEmpty())
                        {
                            StringBuffer sb = new StringBuffer();
                            Iterator     i  = selList.keySet().iterator();

                            while (i.hasNext())
                            {
                                String path = (String)i.next();

                                if (path.startsWith("/"))
                                {
                                    path = path.substring(1);
                                }

                                if (sb.length() > 0)
                                {
                                    sb.append("; ");
                                }
                                
                                sb.append(SONICFS_URL_PREFIX + path);
                            }

                            m_classpath.setText(sb.toString());
                        }
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error specifying classpath.", e, true);
                }

            }
        } //end of ClasspathAction

       }

        class ClasspathButtonPanel extends JRowTableButtonPanel
        {
            public ClasspathButtonPanel(JRowTable table)
            {
                super(table, JPartitionPanel.DEFAULT_GAP_SIZE, 0, 0);
            }

            @Override
            protected void updateState(ListSelectionModel selModel)
            {
                super.updateState(selModel);

                int selCount = (selModel != null) ?
                    getSelectedRowCount(selModel) : 0;

                if (getMoveUpAction() != null)
                {
                    getMoveUpAction().setEnabled( (selCount == 1) &&
                                                  selModel != null &&
                                                 (selModel.getMinSelectionIndex() >
                                                  0));
                }

                if (getMoveDownAction() != null)
                {
                    getMoveDownAction().setEnabled( (selCount == 1) &&
                                                    selModel != null &&
                                                   (selModel.getMaxSelectionIndex() <
                                                   (getTable().getRowCount() - 1)));
                }
            }

            private Action getMoveUpAction()
            {
                return ( (getAction("editable_component.moveup") != null) ?
                        getAction("editable_component.moveup") : null);
            }

            private Action getMoveDownAction()
            {
                return ( (getButton("editable_component.movedown") != null) ?
                        getButton("editable_component.movedown").getAction() : null);
            }
        } //end of ClasspathButtonPanel

        class MoveUpAction extends BasicGuiAction
        {
            public MoveUpAction()
            {
                super("editable_component.moveup");
            }

            @Override
            public void actionPerformed(ActionEvent evt)
            {
                try
                {
                    IModelTableModel model = (IModelTableModel)m_classpathTable.getModel();
                    int[] selRow = m_classpathTable.getSelectedRows();

                    if ( (selRow != null) && (selRow.length > 0))
                    {
                        if (selRow[0] > 0)
                        {
                            String value = (String) model.getRowModel(selRow[0] - 1);
                            model.deleteRow(selRow[0] - 1);
                            model.insertRow(value, selRow[selRow.length - 1]);
                            m_classpathTable.setRowSelectionInterval(selRow[0] -
                                1, selRow[0] - 1);
                        }
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.
                        MESSAGE_ERROR, "Unable to modify classpath order.", e, true);
                }
            }
        }

        class MoveDownAction extends BasicGuiAction
        {
            public MoveDownAction()
            {
                super("editable_component.movedown");
            }

            @Override
            public void actionPerformed(ActionEvent evt)
            {
                try
                {
                    IModelTableModel model = (IModelTableModel)
                        m_classpathTable.getModel();

                    int[] selRow = m_classpathTable.getSelectedRows();

                    if ( (selRow != null) && (selRow.length > 0))
                    {
                        if (selRow[selRow.length - 1] < (model.getRowCount() - 1))
                        {
                            String value = (String) model.getRowModel(selRow[
                                selRow.length - 1] + 1);
                            model.deleteRow(selRow[selRow.length - 1] + 1);

                            int newSel = selRow[0];

                            model.insertRow(value, newSel);

                            m_classpathTable.setRowSelectionInterval(selRow[0] +
                                1, selRow[selRow.length - 1] + 1);
                        }
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.
                        MESSAGE_ERROR, "Unable to modify classpath order.", e, true);
                }
            }
        }

        class FormattedStringTableAspectAdapter
            extends JRowTableAspectAdapter
            implements ObjectCopier
        {
            public FormattedStringTableAspectAdapter(ModelAspectId
                modelAspectId,
                JRowTable table,
                EditRule editRule,
                ModelEditMediator modelEditMediator)
            {
                super(modelAspectId, table, null, editRule, modelEditMediator);
                rowModelCopier = this;
            }

            @Override
            protected void setViewAspectValue(Object viewAspectValue)
                throws IllegalArgumentException
            {
                if (viewAspectValue == null)
                {
                    return;
                }

                if (! (viewAspectValue instanceof String))
                {
                    throw new IllegalArgumentException("The view aspect value must be of type String!");
                }

                try
                {
                    String value = (String) viewAspectValue;
                    StringTokenizer tn = new StringTokenizer(value, TOKEN);
                    ArrayList contents = new ArrayList();

                    while (tn.hasMoreTokens())
                    {
                        Object path = tn.nextToken();

                        if (path != null)
                        {
                            contents.add(path);
                        }
                    }

                    super.setViewAspectValue(contents);
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.
                        MESSAGE_ERROR, "Failed to build classpath view", e, true);
                }
            }

            @Override
            public Object getViewAspectValue()
            {
                StringBuffer buffer = new StringBuffer();

                try
                {
                    Iterator i = ( (IModelTableModel) table.getModel()).getContents().iterator();

                    while (i.hasNext())
                    {
                        buffer.append( (String) i.next());

                        if (i.hasNext())
                        {
                            buffer.append(TOKEN);
                        }
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.
                        MESSAGE_ERROR, "Failed to build classpath view", e, true);
                }

                return buffer.toString();
            }

            @Override
            public Object copyObject(Object obj)
            {
                return obj;
            }
        }
}