/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.propsheets;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import javax.swing.text.JTextComponent;

import modelobjects.framework.EditForm;
import modelobjects.framework.ModelEditFailureEvent;
import modelobjects.framework.ModelEditFailureListener;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;
import modelobjects.framework.model.ModelDescriptor;
import modelobjects.framework.model.ModelObjectAdapter;
import modelobjects.framework.model.ModelObjectPropagator;
import modelobjects.framework.model.ModelObjectValidator;
import modelobjects.framework.model.ModelValidationException;
import modelobjects.framework.model.NoSuchAspectException;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;

public abstract class JPropSheetDialog extends    JMADialog
                                       implements ModelEditFailureListener
{
    public static final String CHANGES       = "There are unapplied changes.\nAre you sure you want to cancel?";
    public static final String CHANGES_TITLE =  "Confirm Cancel";
    public static final String SILENT_VALIDATION_EXCEPTION = "silentValidationException";

    private Class    m_modelClass;
    private EditForm m_editForm;
    private JPanel   m_contentPanel;
    private IPlugin  m_plugin;

    private boolean    m_formInit = false;
    private boolean    m_modelSet = false;
    private Object     m_tmpModel = null;
    private boolean    m_tmpNew   = false;


    public JPropSheetDialog(JMADialog parent, String name, Class modelClass)
    {
        super(parent, name);

        m_modelClass = modelClass;

        initPropSheet();
    }

    public JPropSheetDialog(JMAFrame parent, String name, Class modelClass)
    {
        super(parent, name);

        m_modelClass = modelClass;

        initPropSheet();
    }

    private void initPropSheet()
    {
        setModal(true);

        try
        {
            // Setup the model descriptor and edit form...
            //
            ModelDescriptor   descriptor = makeModelDescriptor();
            ModelEditMediator mediator   = new ModelEditMediator();

            m_editForm = new EditForm(descriptor, this, mediator);

            m_editForm.addModelEditFailureListener(this);
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error while initializing the property sheet.", e, true);
        }
    }

    @Override
    protected void internalInitialize()
    {
        super.internalInitialize();

        // Setup the "content" pane...
        //
        m_contentPanel = new JPanel(new BorderLayout(2, 2));

        super.getContentPane().add(m_contentPanel);
    }

    /**
     * Make a model descriptor for the property sheet.
     *
     * Override this to supply a custom-descriptor implementation.
     */
    protected ModelDescriptor makeModelDescriptor() throws Exception
    {
        return new PropSheetModelDescriptor(m_modelClass);
    }

    @Override
    public Container getContentPane()
    {
        return m_contentPanel;
    }

    public Class getModelClass()
    {
        return m_modelClass;
    }

    public EditForm getEditForm()
    {
        return m_editForm;
    }

    public IPlugin getPlugin()
    {
        return m_plugin;
    }

    public IPluginContext getPluginContext()
    {
        return (m_plugin != null) ? m_plugin.getPluginContext() : null;
    }

    /**
     * Get the current model edited by this prop sheet
     */
    public Object getModel()
    {
        Object res = m_tmpModel;

        // Covers the situation where getModel is called in initUI or initForm
        // and the model has not been set in the model object form.
        try
        {
            final ModelObjectAdapter modelObjectAdapter = m_editForm.getModel();
            if (modelObjectAdapter != null) {
                res = modelObjectAdapter.getModelObject();
            }
        }
        catch (NullPointerException e)
        {
            res = m_tmpModel;
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return res;
    }

    /**
     * Determines whether or not the model being edited is new or not.
     *
     * @return boolean True if the model is new otherwise false.
     */
    public boolean isNew()
    {
        boolean res = m_tmpNew;

        // Covers the situation where getModel is called in initUI or initForm
        // and the model has not been set in the model object form.
        //
        try
        {
            m_editForm.getModel().getModelObject();

            ModelEditMediator mediator = m_editForm.getModelEditMediator();

            res = mediator.getEditModelAsNewObject();
        }
        catch (NullPointerException e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }

        return res;
    }

    public void editInstance(IPlugin plugin, Object model, boolean isNew)
        throws Exception
    {
        m_plugin = plugin;

        // Only set the model in the edit form if the form has been initialized,
        // ie. the view aspect adapters have been registered.
        //
        if (m_formInit)
        {
            editModel(model, isNew);
        }
        else
        {
            m_tmpModel = model;
            m_tmpNew   = isNew;
        }
    }

    protected void editModel(Object model, boolean isNew)
        throws Exception
    {
        ModelDescriptor    descriptor = getDescriptor();
        ModelObjectAdapter adapter    = null;

        if (model != null)
        {
            adapter = descriptor.makeModelObjectAdapter(model);
        }

        if (isNew)
        {
            m_editForm.editModelAsNewObject(adapter);
        }
        else
        {
            m_editForm.editModelObject(adapter);
        }

        m_modelSet = true;
    }

    public ModelObjectValidator getValidator()
    {
        Iterator i = getDescriptor().getModelObjectValidators();

        return (i.hasNext()) ? (ModelObjectValidator)i.next() : null;
    }

    public final void setValidator(ModelObjectValidator validator)
    {
        Iterator i = getDescriptor().getModelObjectValidators();

        while (i.hasNext())
        {
            i.next();
            i.remove();
        }

        // v5.0
        // if (getValidator() == null)
            getDescriptor().addModelObjectValidator(validator);
    }

    public ModelObjectPropagator getPropagator()
    {
        return getDescriptor().getModelObjectPropagator();
    }

    public final void setPropagator(ModelObjectPropagator propagator)
    {
        getDescriptor().setModelObjectPropagator(propagator);
    }

    protected ModelDescriptor getDescriptor()
    {
        return getEditForm().getModelDescriptor();
    }

    protected ModelEditMediator getMediator()
    {
        return getEditForm().getModelEditMediator();
    }

    @Override
    protected boolean canClose()
    {
        int res = JOptionPane.OK_OPTION;

        if (getMediator().hasUnappliedChanges())
        {
            res = JOptionPane.showConfirmDialog(this, CHANGES, CHANGES_TITLE,
                     JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }

        return (res == JOptionPane.OK_OPTION);
    }

    //-------------------------------------------------------------------------
    //
    // Abstract methods
    //
    //-------------------------------------------------------------------------

    protected abstract void initUI() throws Exception;
    protected abstract void initForm() throws Exception;

    //-------------------------------------------------------------------------
    //
    // JMADialog implementation
    //
    //-------------------------------------------------------------------------

    @Override
    protected void maInitialize()
    {
        try
        {
            initUI();
            initForm();

            m_formInit = true;

            if (!m_modelSet)
            {
                editModel(m_tmpModel, m_tmpNew);

                m_tmpModel = null;
                m_tmpNew   = false;
            }
        }
        catch (NoSuchAspectException e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR,
                "Error initializing the property sheet. Aspect not found.", e, true);
        }
        catch (Exception e)
        {
            MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR,
                "Error initializing the property sheet.", e, true);
        }
    }

    @Override
    protected void maCleanup()
    {
        // Can't clear down the edit form yet because it might be needed in
        // order to obtain the model AFTER the dialog has been closed. Note
        // that maCleanup is called when the dialog closes (visible = false)
        // and before any reference to it goes out of scope.
        //
        //m_editForm     = null;
        m_modelClass   = null;
        m_contentPanel = null;
        m_plugin       = null;
        m_tmpModel     = null;
    }

    //-------------------------------------------------------------------------
    //
    // ModelEditFailureListener implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void modelEditFailed(ModelEditFailureEvent evt)
    {
        String     msg = getModelEditFailureMessage(evt.getException());
        JComponent c   = null;

        // Handle a ModelObjects ViewValueConverter exception...
        //
        if (evt.getException() instanceof ViewValueConversionException)
        {
            ViewValueConversionException e       = (ViewValueConversionException)evt.getException();
            ViewAspectAdapter            adapter = getMediator().getViewAspectAdapter(e.getModelAspectId());

            c = (JComponent)adapter.getViewComponent();
        }
        else
        if (evt.getException() instanceof ModelValidationException)
        {
            ModelValidationException e      = (ModelValidationException)evt.getException();
            Exception                nested = (Exception)e.getNestedException();

            // Sonic00013899
            // This is a special silent break...if the exception message is null
            // then we assume that validation failed but we don't want to log
            // a message of change focus.
            //
            if ((e.getMessage() != null) && e.getMessage().equals(SILENT_VALIDATION_EXCEPTION))
            {
                return;
            }

            if (nested instanceof IllegalArgumentException)
            {
                IllegalArgumentException iae     = (IllegalArgumentException)nested;
                ViewAspectAdapter        adapter = getMediator().getViewAspectAdapter(ModelAspectId.forName(iae.getMessage()));

                if (adapter != null)
                {
                    c = (JComponent)adapter.getViewComponent();
                }
            }
        }

        if (c != null)
        {
            JTabbedPane tab = (JTabbedPane)SwingUtilities.getAncestorOfClass(JTabbedPane.class, c);

            if (tab != null)
            {
                Container container = findTabContainer(c, tab);

                if (container != null)
                {
                    tab.setSelectedComponent(container);
                }
            }

            c.requestFocus();
            if (c instanceof JTextComponent)
            {
                JTextComponent tc = (JTextComponent)c;

                if (tc.getText().length() > 0)
                {
                    tc.select(0, tc.getText().length());
                }
                else
                {
                    tc.setCaretPosition(0);
                }
            }
        }

        if (msg == null)
        {
            msg = evt.getException().getMessage();
        }

        MgmtConsole.getMgmtConsole().notifyMessage(this, MgmtConsole.MESSAGE_WARNING,
                    msg, evt.getException(), true);
    }

    public Container findTabContainer(Container c, JTabbedPane tab)
    {
        Container old = c;
        Container res = c.getParent();

        while ((res != null) && (!(res instanceof JTabbedPane)))
        {
            old = res;
            res = res.getParent();
        }

        if (res == null || res.getParent() == null)
        {
            old = null;
        }

        return old;
    }

    private String mangleClassName(Class modelClass)
    {
        String modelName = modelClass.getName();

        modelName = modelName.substring(modelName.lastIndexOf('.') + 1);

        modelName = modelName.toLowerCase();

        return modelName;
    }

    private String buildOptionList(String[] options)
    {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < options.length; i++)
        {
            sb.append(options[i]).append("\n");
        }

        return sb.toString();
    }

    /**
     * Override this method to modify the post-processing of the
     * resource string obtained to represent a ViewValueConversion
     * exception.
     *
     * TODO (AHJ) Unsure of what the | stripping was put in for.
     * Probably want to change this so that by default the msg parameter
     * is just returned in touched and put this specific code in the
     * overridden class(es) that need it.
     *
     * @param e          The exception from which the message 'msg' was obtained
     * @param modelClass The model class representing the ViewValueConverter
     * @param msg        The message (obtained from resources) that represents the exception
     * @return
     */
    protected String postProcessViewValueMessage(ViewValueConversionException e, Class modelClass, String msg)
    {
        StringTokenizer st = new StringTokenizer(msg, "|");
        StringBuffer    sb = new StringBuffer();

        while (st.hasMoreTokens())
        {
            sb.append(st.nextToken()).append("\n");
        }

        return sb.toString();
    }

    protected String getModelEditFailureMessage(Throwable throwable)
    {
        String msg = null;

        if (throwable instanceof ViewValueConversionException)
        {
            ViewValueConversionException e       = (ViewValueConversionException)throwable;
            ViewAspectAdapter            adapter = getMediator().getViewAspectAdapter(e.getModelAspectId());
            ViewValueConverter           vvc     = adapter.getViewValueConverter();

            String[] options = vvc.getViewStringValues();
            if (options == null)
            {
                Object[] args = new Object[] { adapter.getViewAspectStringValue() };
                String msgResId = "failure.message." + vvc.getModelClass().getName().toLowerCase();
                String resource = ResourceManager.getMessage(getClass(), msgResId, args);

                if(resource != null)
                {
                    msg = postProcessViewValueMessage(e, vvc.getModelClass(), resource);
                }
                else
                {
                    // Unknown type of validation error - use generic message...
                    msg = ResourceManager.getMessage(getClass(), "failure.message.view.options.1", args);
                }
            }
            else
            {
                // Case where there's an error and there's a known set of valid values defined by
                // the ViewValueConverter...
                String part1 = ResourceManager.getString(getClass(), "failure.message.view.options.1");
                String part2 = ResourceManager.getString(getClass(), "failure.message.view.options.2");
                msg = part1 + "\n\r" + part2 + "\n\r" + buildOptionList(options);
            }
        }
        else
        if (throwable instanceof ModelValidationException)
        {
            ModelValidationException e = (ModelValidationException)throwable;
            Exception                nested = (Exception)e.getNestedException();

            if (nested instanceof IllegalArgumentException)
            {
                IllegalArgumentException iae     = (IllegalArgumentException)nested;
                ViewAspectAdapter        adapter = getMediator().getViewAspectAdapter(ModelAspectId.forName(iae.getMessage()));

                if (iae.getCause() != null)
                {
                    msg = iae.getCause().getMessage();
                }
            }
        }
        return msg;
    }

    //-------------------------------------------------------------------------

    /**
     * All dialogs in the system are currently MODAL so there's no need for an
     * "Apply" button on most dialogs.
     */
    @Override
    public Action[] getButtonActions()
    {
        ArrayList list = new ArrayList();

        list.add(getDefaultOKAction());
//        list.add(getMediator().getApplyAction());
//        list.add(getMediator().getResetAction());
        list.add(getDefaultCancelAction());

        if (getHelpId() != null)
        {
            list.add(getDefaultHelpAction());
        }

        return (Action[])list.toArray(new Action[list.size()]);
    }

    private BasicAction m_psOK = null;

    @Override
    public BasicAction getDefaultOKAction()
    {
        if (m_psOK == null)
        {
            m_psOK = new OkAction(getMediator().getApplyAction());
        }

        return m_psOK;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class OkAction extends BasicGuiAction
    {
        public OkAction(Action targetAction)
        {
            super("dialog.ok", targetAction);
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            Component        source = (Component)evt.getSource();
            JPropSheetDialog dialog = (JPropSheetDialog)SwingUtilities.getAncestorOfClass(JDialog.class, source);

            if (!dialog.getMediator().hasUnappliedChanges())
            {
                dialog.m_closeCommand = CLOSE_OK;
                dialog.dispose();
            }
        }
    }

}
