/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */

package com.sonicsw.ma.gui.runtime;

import java.util.List;
import java.util.Map;

import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.ObjectName;
import javax.swing.JComponent;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JWorkspacePanel;
import com.sonicsw.ma.gui.JWorkspaceRuntimePanel;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.action.GoToConfigurationAction;
import com.sonicsw.ma.gui.action.PropSheetAction;
import com.sonicsw.ma.gui.action.RefreshPluginAction;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.runtime.propsheets.RuntimeBean;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.plugin.AbstractGUIPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.IRuntimePlugin;
import com.sonicsw.ma.plugin.PluginAttributes;
import com.sonicsw.mx.config.ConfigServerUtility;

import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.IIdentity;
import com.sonicsw.mf.common.runtime.IState;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

/**
 * Abstract implementation the GUIRuntimePlugin that encapsulates common
 * implementation used by all GUIRuntimePlugins. Most concrete GUIRuntimePlugin
 * implementations should extend this class and override appropriate methods.
 */
public abstract class AbstractRuntimePlugin extends AbstractGUIPlugin
                                            implements IRuntimePlugin
{
    private ObjectName  m_runtimeName;
    private IIdentity   m_runtimeID;
    private RuntimeBean m_model = null;
    protected static double ESB_76_VERSION = 7.6;
    public AbstractRuntimePlugin(IPluginContext context, String id, Map data)
    {
        super(context, id);

        setAttributes();
        getAttributes().put(data);
    }

    private void setAttributes() {
        setAttributes(new PluginAttributes());
    }
    
    @Override
    public void dispose()
    {
        m_model = null;
    }

    @Override
    protected JComponent[] getPropertiesMenuItems()
    {
        return new JComponent[]
        {
            new JBasicMenuItem(new PropSheetAction(this))
        };
    }

    @Override
    protected JComponent[] getViewMenuItems()
    {
        return new JComponent[]
        {
            new JBasicMenuItem(new RefreshPluginAction(this)),
            new JBasicMenuItem(new GoToConfigurationAction(this))
        };
    }

    @Override
    public Object getModel()
    {
        if (m_model == null)
        {
            m_model = new RuntimeBean(getPluginContext().getConnectionInfo().getMBeanServer(),
                                      m_runtimeName,
                                      m_runtimeID);
        }

        return m_model;
    }

    public IState getState(boolean recalculate)
    {
        return getState();
    }

    public IState getState()
    {
        return getAttributes().getState();
    }

    public void setState()
    {
    }

    @Override
    public ObjectName getRuntimeName()
    {
        return m_runtimeName;
    }

    @Override
    public IIdentity getRuntimeID()
    {
        return m_runtimeID;
    }

    @Override
    public void setRuntimeID(IIdentity id, IState state)
    {
        m_runtimeID = id;

        if (id == null)
        {
            return;
        }

        String name = id.getCanonicalName();

        if(id instanceof IComponentIdentity)
        {
            setPluginName(((IComponentIdentity)id).getComponentName());
        }

        // if the canonical name doesn't contain ID= then we are looking at
        // the container so we should add on the ID=AGENT
        if (name.indexOf(IComponentIdentity.DELIMITED_ID_PREFIX) == -1)
        {
            name += IComponentIdentity.DELIMITED_ID_PREFIX + IAgentProxy.ID;
        }

        try
        {
            m_runtimeName = new ObjectName(name);
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.  MESSAGE_ERROR,
                "Failed to create runtime name for " + getPluginName(), e, false);
            return;
        }

        try
        {
            if(state != null && state.getState() != IComponentState.STATE_OFFLINE &&
               state.getState() != IComponentState.STATE_UNKNOWN)
            {
                // Check if we are caching MBean info on a per type or per
                // instance basis
                boolean isDynamic = getAttributes().isDynamicRuntime();

                updateMBeanInfo(isDynamic ? getAttributes() :
                                getAttributes().getParent(),
                                getPluginContext().getConnectionInfo().
                                getMBeanServer());
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.  MESSAGE_ERROR,
                "Failed to load MBeanInfo for " + getPluginName(), e, false);
        }
    }

    private void updateMBeanInfo(PluginAttributes attributes, IRemoteMBeanServer server)
        throws Exception
    {
        if(m_runtimeID.getConfigIdentity() == null)
        {
            return;
        }

        if(attributes.containsKey(PluginAttributes.MBEAN_INFO))
        {
            return;
        }

        MBeanInfo info = server.getMBeanInfo(m_runtimeName);

        // put it in the has map even if it contains null so that we
        // don't repeatedly try to get the same mbeanInfo
        attributes.set(PluginAttributes.MBEAN_INFO, info);

        // now see if we have any metrics
        if(info != null)
        {
            MBeanOperationInfo[] operations = info.getOperations();

            if(operations != null)
            {
                for(int i = 0; i < operations.length; i++)
                {
                    if(operations[i].getName().equals("getMetricsInfo"))
                    {
                        IMetricInfo[] metric =
                            (IMetricInfo[])server.invoke(m_runtimeName,
                                                         "getMetricsInfo",
                                                         new Object[] {},
                                                         new String[] {} );

                        attributes.set(PluginAttributes.METRIC_INFO, metric);
                        break;
                    }
                }
            }
        }
    }

    //-------------------------------------------------------------------------

    public MBeanNotificationInfo[] getNotificationInfo()
    {
        MBeanInfo info = (MBeanInfo)getAttributes().get(PluginAttributes.MBEAN_INFO);

        MBeanNotificationInfo[] res = null;

        if(info != null)
        {
            res = info.getNotifications();
        }

        if ((res != null) && (res.length == 0))
        {
            res = null;
        }

        return res;
    }

    protected IMetricInfo[] getMetricInfo()
    {
        IMetricInfo[] res = (IMetricInfo[])getAttributes().get(PluginAttributes.METRIC_INFO);

        if ((res != null) && (res.length == 0))
        {
            res = null;
        }

        return res;
    }

    @Override
    public Object invokeAction(String action, Object[] args, String[] sigs)
    {
        return invokeAction(action, args, sigs, -1);
    }

    /**
     * Invoke an action on the runtime component.
     * @param action the action name to invoke
     * @param args the arguments to the action
     * @param sigs the signature for each of the arguments
     * @param timeout the timeout multiple to use. If this value is
     * -1 then use the default timeout, otherwise use the specified multiple
     * of the configured default timeout
     */
    @Override
    public Object invokeAction(String action, Object[] args, String[] sigs, int timeout)
    {
        Object ret = null;

        try
        {
            long lTimeout = -1;
            DomainConnectionModel domainModel = getPluginContext().getConnectionInfo();

            if (timeout >= 0)
            {
                int defTimeout = domainModel.getTimeout();

                lTimeout = (long)(defTimeout * 1000 * timeout);
            }

            IRemoteMBeanServer server = domainModel.getMBeanServer();

            ret = invokeAction(server, getRuntimeName(), action, args, sigs, lTimeout, getClass().getClassLoader());
        }
        catch(Throwable e)
        {
            Throwable tmpE = e;

            if (e instanceof javax.management.RuntimeOperationsException)
            {
                tmpE = ((javax.management.RuntimeOperationsException)e).getTargetException();
            }

            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                                        "Failed to perform action " + action + "() on\r\n" +
                                                            m_runtimeName.getCanonicalName(),
                                                        tmpE,
                                                        true);
        }

        return ret;
    }

    public static Object invokeAction(IRemoteMBeanServer server,
                                      ObjectName         name,
                                      String             action,
                                      Object[]           args,
                                      String[]           sigs,
                                      long               timeout,
                                      ClassLoader        loader)
        throws Throwable
    {
        if (timeout < 0)
        {
            return ((com.sonicsw.mf.jmx.client.JMSConnectorClient)server).invoke(name, action, args, sigs, loader);
        }

        return ((com.sonicsw.mf.jmx.client.JMSConnectorClient)server).invoke(name, action, args, sigs, timeout, loader);
    }

    public void goToConfiguration()
    {
        boolean isStateable = getAttributes().isStateable();
        String          path = getPluginPath();

        if (isStateable)
        {
           String runtimeID = getRuntimeID().getCanonicalName();
           path = getPluginContext().getConnectionInfo().getAgentManagerConnection().runtimeIDToConfigName(runtimeID);
           int index = path.indexOf(ConfigServerUtility.DEFAULT_SUFFIX);
           if (index != -1)
        {
            path = path.substring(0, index -1);
        }
        }

        if (path != null)
        {
            getPluginContext().getWorkspace().goToConfiguration(path, true);
        }

    }

    @Override
    public List getToolTipList()
    {
        List list = super.getToolTipList();

        IState state = getAttributes().getState();

        if (state != null)
        {
            list.add(formatToolTipText("State", state.getStateString()));

            if (state instanceof IContainerState)
            {
                list.add(formatToolTipText("Host", ((IContainerState)state).getContainerHost()));
            }
        }
        return list;
    }

    @Override
    protected JWorkspacePanel getWorkspacePanel()
    {
        return getPluginContext().getWorkspace().getWorkspacePanel(JWorkspaceRuntimePanel.PANEL_NAME);
    }

    protected boolean checkVersion(double version){
        try{
            double productVersion =Double.parseDouble(getAttributes().getProductVersion());

            if(productVersion >= version)
            {
                return true;
            }

        }catch(Exception e){
            e.printStackTrace();
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
        return false;
    }
}