// Copyright (c) 2002, 2007 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.ma.gui.runtime;

import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.swing.JComponent;
import javax.swing.JMenuItem;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.propsheets.JPropSheetDialog;
import com.sonicsw.ma.gui.runtime.metrics.model.MetricsModel;
import com.sonicsw.ma.gui.runtime.notifications.model.NotificationsModel;
import com.sonicsw.ma.gui.table.IContentPane;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.BasicPluginAction;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.plugin.IMonitorPlugin;
import com.sonicsw.ma.plugin.IPlugin;
import com.sonicsw.ma.plugin.IPluginContext;
import com.sonicsw.ma.plugin.IRuntimePlugin;

import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IState;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.ComponentState;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;

public class ComponentPlugin extends AbstractRuntimePlugin
{
    private static final String TRACKID_TOOLS = "MANAGED_TOOLS_DIALOG";

    public static final String ATTRIBUTE_CONFIGID        = "ConfigID";
    public static final String ATTRIBUTE_RELEASE_VERSION = "ReleaseVersion";
    public static final String ATTRIBUTE_STATE           = "State";
    public static final String ATTRIBUTE_STATE_STRING    = "StateString";
    public static final String ATTRIBUTE_LAST_ERROR      = "LastError";
    public static final String ATTRIBUTE_ERROR_LEVEL     = "LastErrorLevel";
    public static final String ATTRIBUTE_ERROR_STRING    = "LastErrorLevelString";
    public static final String ATTRIBUTE_UPTIME          = "Uptime";
    public static final String ATTRIBUTE_CLASS_NAME      = "Classname";
    public static final String ATTRIBUTE_CLASS_PATH      = "Classpath";
    public static final String ATTRIBUTE_TRACE_MASK      = "TraceMask";
    public static final String ATTRIBUTE_TRACE_VALUES    = "TraceMaskValues";

    private NotificationsModel m_notificationsModel;
    private MetricsModel       m_metricsModel;

    public ComponentPlugin(IPluginContext context, String id, Map data)
    {
        super(context, id, data);
    }

    @Override
    public boolean expand()
    {
        return false;
    }

    @Override
    public JPropSheetDialog getPropSheetDialog()
    {
        return new ComponentPropSheet(MgmtConsole.getMgmtConsole());
    }

    //-------------------------------------------------------------------------
    //
    // IMonitorPlugin overrides
    //
    //-------------------------------------------------------------------------

    @Override
    public String[] getAllSupportedMonitors()
    {
        ArrayList res = new ArrayList();

        if (isMonitorSupported(IMonitorPlugin.MONITOR_ID_METRICS))
        {
            res.add(IMonitorPlugin.MONITOR_ID_METRICS);
        }

        if (isMonitorSupported(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS))
        {
            res.add(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS);
        }

        return (String[])res.toArray(new String[res.size()]);
    }

    @Override
    public boolean isMonitorSupported(String monitorId)
    {
        boolean res = false;

        if (monitorId != null)
        {
            if (monitorId.equals(IMonitorPlugin.MONITOR_ID_METRICS))
            {
                res = (getMetricInfo() != null);
            }
            else if (monitorId.equals(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS))
            {
                res = (getNotificationInfo() != null);
            }
        }
        return res;
    }

    @Override
    public Object getMonitorModel(String monitorId)
    {
        Object res = null;

        if (isMonitorSupported(monitorId))
        {
            if (monitorId.equals(IMonitorPlugin.MONITOR_ID_METRICS))
            {
                try
                {
                    IRemoteMBeanServer server = getPluginContext().getRuntimeContext().getMBeanServer();
                    IMetricInfo[]      info   = getMetricInfo();

//                    if (m_metricsModel == null)
                    m_metricsModel = new MetricsModel(server, getRuntimeName(), info);
//                    else
//                        m_metricsModel.update(info);
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                              "Failed to get metrics info: " + e.getMessage(), e, false);

                    m_metricsModel = null;
                }
                res = m_metricsModel;
            }
            else if (monitorId.equals(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS))
            {
                try
                {
                    IRemoteMBeanServer      server = getPluginContext().getRuntimeContext().getMBeanServer();
                    MBeanNotificationInfo[] info   = getNotificationInfo();

                    if (m_notificationsModel == null)
                    {
                        m_notificationsModel = new NotificationsModel(server, getRuntimeName(), info);
                    }
                }
                catch (Exception e)
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.MESSAGE_ERROR,
                        "Failed to get notifications info: " + e.getMessage(), e, false);

                    m_notificationsModel = null;
                }

                res = m_notificationsModel;
            }
        }
        return res;
    }

    //-------------------------------------------------------------------------

    protected boolean isComponentAvailable()
    {
        IState state = getAttributes().getState();

        return ((state != null) &&
                ((state.getState() == IComponentState.STATE_ONLINE) ||
                 (state.getState() == IComponentState.STATE_UNKNOWN)));
    }

    /**
     * Method determines if the parent container (in which the component is
     * deployed) is available, i.e. online or in an unknown (and possibly
     * online) state.
     *
     * @return  true if the container is available, otherwise false
     */
    protected boolean isParentContainerAvailable()
    {
        IPlugin container = (IPlugin)getParent();
        IState  state     = (container != null) ? container.getAttributes().getState() : null;

        return ((state != null) &&
                ((state.getState() == IComponentState.STATE_ONLINE) ||
                 (state.getState() == IComponentState.STATE_UNKNOWN)));
    }

    class ComponentOperationsAction extends BasicGuiAction
    {
        public ComponentOperationsAction(String resId)
        {
            super(resId);
        }
    }

    protected JComponent getOperationsStartMenuItem()
    {
        return new JBasicMenuItem(new RuntimeStartAction(this, "ComponentPlugin.start"));
    }

    protected JComponent getOperationsStopMenuItem()
    {
        return new JBasicMenuItem(new RuntimeStopAction(this,
                                  "ComponentPlugin.stop",
                                  "Are you sure you want to stop this component?"));
    }

    protected JComponent[] getOperationsMenuItems()
    {
        ArrayList list = new ArrayList();

        list.add(getOperationsStartMenuItem());
        list.add(getOperationsStopMenuItem());

        if (getAttributes().isReloadable())
        {
            list.add(new JBasicMenuItem(new RuntimeAction(this,
                "ComponentPlugin.reload",
                "reload",
                "Are you sure you want to reload this component?",
                4)));
        }

        list.add(new JBasicMenuItem(new ClearErrorAction(this,
            "ComponentPlugin.clearError",
            "clearError")));

        if (isComponentAvailable() &&
            isMonitorSupported(IMonitorPlugin.MONITOR_ID_METRICS))
        {
            list.add(new JBasicMenuItem(new RuntimeAction(this,
                "ComponentPlugin.resetMetrics",
                "resetMetrics",
                "Are you sure you want to reset all metrics?")));
        }

        return (JComponent[])list.toArray(new JComponent[list.size()]);
    }

    @Override
    public JComponent[] getPluginMenuItems()
    {
        JMenuItem opMenu = new JBasicMenu(new ComponentOperationsAction("runtime.operations"));
        JComponent[] ops = getOperationsMenuItems();

        for (int i = 0; i < ops.length; i++)
        {
            opMenu.add(ops[i]);
        }

        return Helper.combineMenu(new JComponent[] { opMenu },
                                  super.getPluginMenuItems(),
                                  false);
    }

    @Override
    protected JComponent[] getPropertiesMenuItems() {
        JMenuItem notificationsMenuItem = new JBasicMenuItem(new NotificationsAction(this));
        JMenuItem metricsMenuItem       = new JBasicMenuItem(new MetricsAction(this));

        JComponent[] items = new JComponent[]
        {
            notificationsMenuItem,
            metricsMenuItem
        };
        return Helper.combineMenu(items,super.getPropertiesMenuItems(),true);
    }


    private void launchMonitorDialog(final String monitorId)
    {
        try
        {
            JMonitorDialog dialog = (JMonitorDialog)getPluginContext().getWorkspace().getTrackedDialog(TRACKID_TOOLS);

            if (dialog == null)
            {
                final WorkspaceWindow ww = getPluginContext().getWorkspace();

                dialog = new JMonitorDialog(MgmtConsole.getMgmtConsole(), getPluginContext().getWorkspace());

                getPluginContext().getWorkspace().addTrackedDialog(TRACKID_TOOLS, dialog);
                dialog.addWindowListener(new WindowAdapter()
                {
                    @Override
                    public void windowClosed(WindowEvent evt)
                    {
                        ww.removeTrackedDialog(TRACKID_TOOLS);
                    }
                });

                dialog.setVisible(true);
                dialog.pluginChange(ComponentPlugin.this);
            }
            else
            {
                dialog.toFront();
            }

            dialog.setVisibleMonitor(monitorId);
        }
        catch(Throwable e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                     "Failed to launch manage tools", e, true);
        }
    }

    private class NotificationsAction extends BasicPluginAction
    {
        public NotificationsAction(IPlugin plugin)
        {
            super(plugin, "ComponentPlugin.notifications");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

            try
            {
                launchMonitorDialog(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS);
            }
            finally
            {
                wc.release();
            }
        }

        @Override
        public boolean isEnabled()
        {
            return ((ComponentPlugin)getPlugin()).isParentContainerAvailable() &&
                   ((ComponentPlugin)getPlugin()).isMonitorSupported(IMonitorPlugin.MONITOR_ID_NOTIFICATIONS);
        }
    }

    private class MetricsAction extends BasicPluginAction
    {
        public MetricsAction(IPlugin plugin)
        {
            super(plugin, "ComponentPlugin.metrics");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            JWaitCursor wc = new JWaitCursor(MgmtConsole.getMgmtConsole());

            try
            {
                launchMonitorDialog(IMonitorPlugin.MONITOR_ID_METRICS);
            }
            finally
            {
                wc.release();
            }
        }

        @Override
        public boolean isEnabled()
        {
            return ((ComponentPlugin)getPlugin()).isParentContainerAvailable() &&
                   ((ComponentPlugin)getPlugin()).isMonitorSupported(IMonitorPlugin.MONITOR_ID_METRICS);
        }
    }
    
    private class ClearErrorAction extends RuntimeAction 
    {
        public ClearErrorAction(IRuntimePlugin plugin, String name, String action)
        {
            super(plugin, name, action);
        }
        
        @Override
        public void actionPerformed(ActionEvent evt)
        {
        	super.actionPerformed(evt);
        	
        	IState state = m_plugin.getAttributes().getState();
        	if ( state instanceof ComponentState )
        	{
        		((ComponentState)state).setLastErrorDescription("");
        		((ComponentState)state).setLastErrorLevel(Level.UNKNOWN);
                IContentPane pluginContentPane = m_plugin.getPluginContentPane();
                if(pluginContentPane != null)
                {
                    pluginContentPane.refresh();
                }
        	}
        }
    }
}