/**
 * Copyright (c) 2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.ma.gui.runtime;

import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.StringTokenizer;

import javax.swing.BorderFactory;
import javax.swing.DefaultListModel;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import modelobjects.framework.EditRule;
import modelobjects.framework.LongDateToStringConverter;
import modelobjects.framework.ModelEditMediator;
import modelobjects.framework.ViewAspectAdapter;
import modelobjects.framework.ViewValueConversionException;
import modelobjects.framework.ViewValueConverter;
import modelobjects.framework.model.ModelAspectId;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.runtime.propsheets.JRuntimeTabbedDialog;
import com.sonicsw.ma.gui.runtime.propsheets.RuntimeModelDescriptor;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ColorIcon;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JPathChooser;
import com.sonicsw.ma.gui.util.StateHelper;

public class ComponentPropSheet extends JRuntimeTabbedDialog
{
    private static final String RES_COMPONENT = "component";

    private static final String SCHEMA_FILE = "file:/";

    private int m_startMask = -1;

    private JPartitionPanel m_generalPanel;

    // General fields
    //
    private JTextField m_name = new JTextField();
    private JTextField m_uptime = new JTextField();
    private JTextField m_releaseVersion = new JTextField();
    private JTextField m_className = new JTextField(32);
    private JTextArea m_classPath = new JTextArea(8, 32);
    private JTextField m_configID = new JTextField();
    private JTextField m_sharedSpace = new JTextField();

    // Status fields
    //
    private JLabel m_state = new JLabel();
    private JLabel m_stateIcon = new JLabel();
    private JLabel m_errorIcon = new JLabel();
    private JLabel m_errorLevel = new JLabel();
    private JTextArea m_lastError = new JTextArea(4, 32);

    // Trace fields
    //
    private JTraceList m_trace;

    public ComponentPropSheet(JMAFrame parent)
    {
        this(parent, "runtime.component");
    }

    public ComponentPropSheet(JMAFrame parent, String name)
    {
        super(parent, name);
    }

    @Override
    public void initUI() throws Exception
    {
        m_generalPanel = (JPartitionPanel) makeGeneralPanel();

        addTabPanel("General", JPartitionPanel.wrap(m_generalPanel));
        addTabPanel("Loading", makeLoadingPanel());
        addTabPanel("Tracing", makeTracePanel());
    }

    @Override
    public void initForm() throws Exception
    {
        // General fields
        //
        createJTextFieldAdapter(RuntimeModelDescriptor.ATTRIBUTE_NAME, m_name, null);
        createJTextFieldAdapter(ComponentPlugin.ATTRIBUTE_CLASS_NAME, m_className, null);
        createJTextAreaAdapter(ComponentPlugin.ATTRIBUTE_CLASS_PATH, m_classPath, new ClassViewValueConverter());
        createJTextFieldAdapter(ComponentPlugin.ATTRIBUTE_RELEASE_VERSION, m_releaseVersion, null);
        createJTextFieldAdapter(ComponentPlugin.ATTRIBUTE_CONFIGID, m_configID, null);
        createJTextFieldAdapter(ComponentPlugin.ATTRIBUTE_UPTIME, m_uptime,
                                new LongDateToStringConverter(true, "{0} days, {1} hours, {2} minutes"));

        // Status fields
        //
        new JStateIconLabelAdapter(ModelAspectId.forName(ComponentPlugin.ATTRIBUTE_STATE), m_stateIcon,
                                   null, getEditForm().getModelEditMediator());
        createJLabelAdapter(ComponentPlugin.ATTRIBUTE_STATE_STRING, m_state, null);

        new JErrorLevelIconLabelAdapter(ModelAspectId.forName(ComponentPlugin.ATTRIBUTE_ERROR_LEVEL), m_errorIcon,
                                        null, getEditForm().getModelEditMediator());
        createJLabelAdapter(ComponentPlugin.ATTRIBUTE_ERROR_STRING, m_errorLevel, null);
        createJTextAreaAdapter(ComponentPlugin.ATTRIBUTE_LAST_ERROR, m_lastError, null);
        m_lastError.setWrapStyleWord(true);
        m_lastError.setLineWrap(true);

        new JTraceListViewAspectAdapter(ModelAspectId.forName(ComponentPlugin.ATTRIBUTE_TRACE_VALUES), m_trace,
                                        null, getEditForm().getModelEditMediator());
        new JTraceListViewAspectAdapter(ModelAspectId.forName(ComponentPlugin.ATTRIBUTE_TRACE_MASK), m_trace,
                                        null, getEditForm().getModelEditMediator());
    }

    protected JPartitionPanel getGeneralPanel()
    {
        return m_generalPanel;
    }

    private JPanel makeGeneralPanel()
    {
        JPartitionPanel panel = new JPartitionPanel(true, "p,p,p", null, 0, 0, 0);

        panel.add(makeGeneralMainPanel());
        panel.add(makeGeneralStatusPanel());

        return panel;
    }

    private JPanel makeGeneralMainPanel()
    {
        JPartitionPanel panel = new JPartitionPanel("Identity");

        panel.addRow(getResourceLabel(RuntimeModelDescriptor.ATTRIBUTE_NAME), m_name);
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_CONFIGID), m_configID);
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_UPTIME), m_uptime);

        return panel;
    }

    private JPanel makeGeneralStatusPanel()
    {
        JPartitionPanel pState = JPartitionPanel.merge("p,r", false, 5, 0, 0, new Component[]
            {m_stateIcon, m_state});
        JPartitionPanel eState = JPartitionPanel.merge("p,r", false, 5, 0, 0, new Component[]
            {m_errorIcon, m_errorLevel});
        JPartitionPanel panel = new JPartitionPanel("Status");

        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_STATE_STRING), pState);
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_ERROR_STRING), eState);
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_LAST_ERROR), new ExtendedJScrollPane(m_lastError));

        return panel;
    }

    private JPanel makeLoadingPanel()
    {
        JPartitionPanel panel = new JPartitionPanel("Loader Details");

        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_CLASS_NAME), m_className);
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_CLASS_PATH), new ExtendedJScrollPane(m_classPath));
        panel.addRow(getResourceLabel(ComponentPlugin.ATTRIBUTE_RELEASE_VERSION), m_releaseVersion);

        return JPartitionPanel.wrap(panel);
    }

    private JPanel makeTracePanel()
    {
        m_trace = new JTraceList();

        JPartitionPanel panel = new JPartitionPanel(false, "r", null);

        panel.add(new ExtendedJScrollPane(m_trace));

        return panel;
    }

    /**
     * Converts the first character of each word in the description string
     * to upper case.
     *
     * Assumes that the string has at least one character in it!
     */
    private String capitalizeDescription(String desc)
    {
        StringBuffer sb = new StringBuffer(desc);

        // Adjust first character...
        //
        if (Character.isLowerCase(sb.charAt(0)))
        {
            sb.setCharAt(0, Character.toUpperCase(sb.charAt(0)));
        }

        for (int i = 1; i < sb.length(); i++)
        {
            if (!Character.isLetterOrDigit(sb.charAt(i - 1)) &&
                Character.isLetterOrDigit(sb.charAt(i)))
            {
                sb.setCharAt(i, Character.toUpperCase(sb.charAt(i)));
            }
        }

        return sb.toString();
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class TraceItem
    {
        private int m_mask;
        private String m_desc;
        private boolean m_selected;

        public TraceItem(int mask, String desc)
        {
            m_mask = mask;
            m_desc = capitalizeDescription(desc);
            m_selected = false;
        }

        public int getMask()
        {
            return m_mask;
        }

        public String getDescription()
        {
            return m_desc;
        }

        public boolean isSelected()
        {
            return m_selected;
        }

        public void setSelected(boolean selected)
        {
            m_selected = selected;
        }

        @Override
        public String toString()
        {
            StringBuffer sb = new StringBuffer();

            sb.append(getDescription());
            sb.append(" (");
            sb.append(getMask());
            sb.append(")");

            return sb.toString();
        }
    }

    class JTraceList
        extends JList
        implements MouseListener
    {
        public JTraceList()
        {
            super(new DefaultListModel());
            setProperties();
        }
        
        private void setProperties() {
            setCellRenderer(new TraceListCellRenderer());
            setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            addMouseListener(this);
        }

        @Override
        public void mouseClicked(MouseEvent evt)
        {
            int click = evt.getClickCount();
            Point point = evt.getPoint();
            DefaultListModel model = (DefaultListModel) getModel();
            int index = locationToIndex(point);
            TraceItem item = (TraceItem) model.getElementAt(index);

            if (((click == 1) && (index == m_trace.getSelectedIndex()))
                || (click == 2))
            {
                item.setSelected(!item.isSelected());
                model.set(index, item);
            }
        }

        @Override
        public void mouseEntered(MouseEvent e)
        {}

        @Override
        public void mouseExited(MouseEvent e)
        {}

        @Override
        public void mousePressed(MouseEvent e)
        {}

        @Override
        public void mouseReleased(MouseEvent e)
        {}
    }

    class TraceListCellRenderer
        extends JCheckBox
        implements ListCellRenderer
    {
        public TraceListCellRenderer()
        {
            super();
            setProperties();
        }
        private void setProperties() {
            
            setOpaque(true);
            setHorizontalAlignment(SwingConstants.LEFT);
            setBorder(BorderFactory.createEmptyBorder(2, 5, 2, 5));
            setBorderPainted(true);
        }

        @Override
        public Component getListCellRendererComponent(JList list,
            Object value,
            int index,
            boolean isSelected,
            boolean cellHasFocus)
        {
            TraceItem item = (TraceItem) value;

            setComponentOrientation(list.getComponentOrientation());

            setBackground(isSelected ? list.getSelectionBackground() : list.getBackground());
            setForeground(isSelected ? list.getSelectionForeground() : list.getForeground());

            setText(item.toString());
            setSelected(item.isSelected());

            setEnabled(list.isEnabled());
            setFont(list.getFont());

            return this;
        }

        //---------------------------------------------------------------------
        //
        // Methods overridden for performance reasons:
        //
        //---------------------------------------------------------------------

        @Override
        public void validate()
        {}

        @Override
        public void revalidate()
        {}

        @Override
        public void repaint(long tm, int x, int y, int width, int height)
        {}

        @Override
        public void repaint(Rectangle r)
        {}

        @Override
        public void firePropertyChange(String propertyName, byte oldValue, byte newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, char oldValue, char newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, short oldValue, short newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, int oldValue, int newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, long oldValue, long newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, float oldValue, float newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, double oldValue, double newValue)
        {}

        @Override
        public void firePropertyChange(String propertyName, boolean oldValue, boolean newValue)
        {}
    }

    class JTraceListViewAspectAdapter
        extends ViewAspectAdapter
        implements ListDataListener
    {
        private JTraceList m_list;

        public JTraceListViewAspectAdapter(ModelAspectId aspectId,
                                           JTraceList list,
                                           EditRule editRule,
                                           ModelEditMediator modelEditMediator)
        {
            super(aspectId, editRule, null, modelEditMediator);

            m_list = list;
            m_list.getModel().addListDataListener(this);
        }

        @Override
        public Component getViewComponent()
        {
            return m_list;
        }

        /**
         *  Return the model aspect value.
         */
        @Override
        public Object getModelAspectValue()
            throws ViewValueConversionException
        {
            ListModel model = m_list.getModel();
            int maskValue = 0;

            for (int i = 0; i < model.getSize(); i++)
            {
                TraceItem item = (TraceItem) model.getElementAt(i);

                if (item.isSelected())
                {
                    maskValue = maskValue | item.getMask();
                }
            }

            return new Integer(maskValue);
        }

        /**
         *  Return the view value.
         */
        @Override
        public Object getViewAspectValue()
        {
            return m_list.getModel();
        }

        /**
         *  Set the value for the view, which must be an array.
         */
        @Override
        public void setViewAspectValue(Object value)
        {
            if (value == null)
            {
                ; // Null value is valid but ignored! (Sonic00008668)
            }
            else
            if (value instanceof Integer)
            {
                int newValue = ((Integer) value).intValue();

                if (newValue >= 0)
                {
                    m_startMask = newValue;

                    enforceMask();
                }
            }
            else
            if (value instanceof String)
            {
                StringTokenizer st = new StringTokenizer((String) value, ",");

                ((DefaultListModel) m_list.getModel()).clear();

                while (st.hasMoreTokens())
                {
                    String[] split = st.nextToken().split("=");
                    TraceItem anItem = new TraceItem(Integer.parseInt(split[0]), split[1]);

                    ((DefaultListModel) m_list.getModel()).addElement(anItem);
                }

                if (m_startMask >= 0)
                {
                    enforceMask();
                }
            }
            else
            {
                throw new IllegalArgumentException("value must be a String");
            }
        }

        @Override
        public String getViewAspectStringValue()
        {
            throw new UnsupportedOperationException();
        }

        @Override
        public void setViewAspectStringValue(String stringValue)
        {
            throw new UnsupportedOperationException();
        }

        @Override
        public void setEditable(boolean editable)
        {
            // This method is getting called twice because its registered
            // with both the trace mask and the value.
            // The list will always be editable so enable it.
            m_list.setEnabled(true);
        }

        @Override
        public boolean isEditable()
        {
            return m_list.isEnabled();
        }

        @Override
        public void contentsChanged(ListDataEvent e)
        {
            fireChangeEvent();
        }

        @Override
        public void intervalAdded(ListDataEvent e)
        {
            fireChangeEvent();
        }

        @Override
        public void intervalRemoved(ListDataEvent e)
        {
            fireChangeEvent();
        }

        private void enforceMask()
        {
            DefaultListModel model = (DefaultListModel) m_list.getModel();

            for (int i = 0; i < 16; i++)
            {
                int maskValue = (int) Math.pow(2, i);

                if (maskValue > m_startMask)
                {
                    break;
                }

                if ((maskValue & m_startMask) != maskValue)
                {
                    continue;
                }

                int index = findMask(model, maskValue);

                if (index >= 0)
                {
                    TraceItem item = (TraceItem) model.getElementAt(index);

                    if (item != null)
                    {
                        item.setSelected(true);
                        model.set(index, item);
                    }
                }
            }
        }

        private int findMask(DefaultListModel model, int maskValue)
        {
            for (int i = 0; i < model.size(); i++)
            {
                TraceItem item = (TraceItem) model.getElementAt(i);

                if (maskValue == item.getMask())
                {
                    return i;
                }
            }

            return -1;
        }
    }

    public class JErrorLevelIconLabelAdapter
        extends ViewAspectAdapter
    {
        public JErrorLevelIconLabelAdapter(ModelAspectId modelAspectId,
                                           JLabel viewComponent)
        {
            super(modelAspectId);

            setViewComponent(viewComponent);
        }

        public JErrorLevelIconLabelAdapter(ModelAspectId modelAspectId,
                                           JLabel viewComponent,
                                           ViewValueConverter viewValueConverter,
                                           ModelEditMediator modelEditMediator)
        {
            super(modelAspectId, EditRule.NEVER, viewValueConverter, modelEditMediator);

            setViewComponent(viewComponent);
        }

        private void setViewComponent(JLabel viewComponent)
        {
            this.viewComponent = viewComponent;
        }

        @Override
        public Component getViewComponent()
        {
            return viewComponent;
        }

        @Override
        public boolean isEditable()
        {
            return false;
        }

        @Override
        public void setEditable(boolean editable)
        {
        }

        @Override
        public Object getViewAspectValue()
        {
            return viewComponent.getText();
        }

        @Override
        public String getViewAspectStringValue()
        {
            return viewComponent.getText();
        }

        @Override
        protected void setViewAspectValue(Object viewAspectValue)
            throws IllegalArgumentException
        {
            if (viewAspectValue == null)
            {
                viewComponent.setIcon(new ColorIcon(Color.BLUE, 14));
            }
            else if (viewAspectValue instanceof Integer)
            {
                int value = ((Integer) viewAspectValue).intValue();
                Color color = StateHelper.errorLevelToColor(value);

                if (color != null)
                {
                    viewComponent.setIcon(new ColorIcon(color, 14));
                }
            }
            else
            {
                throw new IllegalArgumentException("value must be an Integer");
            }
        }

        @Override
        protected void setViewAspectStringValue(String viewStringValue)
            throws IllegalArgumentException
        {
            if (viewStringValue == null)
            {
                viewComponent.setText("");
            }
            else
            {
                viewComponent.setText(viewStringValue);
            }
        }

        private JLabel viewComponent;
    }

    public class JStateIconLabelAdapter
        extends ViewAspectAdapter
    {
        public JStateIconLabelAdapter(ModelAspectId modelAspectId, JLabel viewComponent)
        {
            super(modelAspectId);

            setViewComponent(viewComponent);
        }

        public JStateIconLabelAdapter(ModelAspectId modelAspectId,
                                      JLabel viewComponent,
                                      ViewValueConverter viewValueConverter,
                                      ModelEditMediator modelEditMediator)
        {
            super(modelAspectId, EditRule.NEVER, viewValueConverter, modelEditMediator);

            setViewComponent(viewComponent);
        }

        private void setViewComponent(JLabel viewComponent)
        {
            this.viewComponent = viewComponent;
        }

        @Override
        public Component getViewComponent()
        {
            return viewComponent;
        }

        @Override
        public boolean isEditable()
        {
            return false;
        }

        @Override
        public void setEditable(boolean editable)
        {
        }

        @Override
        public Object getViewAspectValue()
        {
            return viewComponent.getText();
        }

        @Override
        public String getViewAspectStringValue()
        {
            return viewComponent.getText();
        }

        @Override
        protected void setViewAspectValue(Object viewAspectValue)
            throws IllegalArgumentException
        {
            if (viewAspectValue == null)
            {
                viewComponent.setIcon(null);
            }
            else if (viewAspectValue instanceof Short)
            {
                short value = ((Short) viewAspectValue).shortValue();

                Color color = StateHelper.componentStateToColor(value);

                if (color != null)
                {
                    viewComponent.setIcon(new ColorIcon(color, 14));
                }
            }
            else
            {
                throw new IllegalArgumentException("value must be a Short");
            }
        }

        @Override
        protected void setViewAspectStringValue(String viewStringValue)
            throws IllegalArgumentException
        {
            if (viewStringValue == null)
            {
                viewComponent.setText("");
            }
            else
            {
                viewComponent.setText(viewStringValue);
            }
        }

        private JLabel viewComponent;
    }

    public class ClassPathAction
        extends BasicGuiAction
    {
        public ClassPathAction()
        {
            super("ComponentPropSheet.classpath");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                String[] list = Helper.stringToArray(m_classPath.getText());
                JPathChooser chooser = new JPathChooser();

                chooser.setReadOnly(true);
                chooser.setSelectedPaths(list);
                chooser.showDialog(ComponentPropSheet.this);
            }
            catch (Exception e)
            {
                MgmtConsole.displayMessage(MgmtConsole.MESSAGE_ERROR, "Error in ClassPathAction.", e, true);
            }
        }
    }

    public class ClassViewValueConverter
        implements ViewValueConverter
    {
        public ClassViewValueConverter()
        {
        }

        @Override
        public String[] getViewStringValues()
        {
            return null;
        }

        @Override
        public Class getModelClass()
        {
            return String.class;
        }

        @Override
        public Class getViewClass()
        {
            return String.class;
        }

        @Override
        public Object viewValueToModel(Object viewValue)
            throws ViewValueConversionException
        {
            throw (new ViewValueConversionException("not supported - read only"));
        }

        @Override
        public Object modelValueToView(Object modelValue)
            throws ViewValueConversionException
        {
            if ((modelValue == null) || (!(modelValue instanceof String)))
            {
                return null;
            }
            String inValue = (String) modelValue;

            StringTokenizer st = new StringTokenizer((String) modelValue, ";");
            StringBuffer sb = new StringBuffer();

            while (st.hasMoreTokens())
            {
                String classItem = st.nextToken().replaceAll("%20", " ");

// Defect 00027016 (AHJ) Don't want to strip off file-prefix (scheme) because on some
// platforms this results in a relative path.
//                if (classItem.startsWith(SCHEMA_FILE))
//                    classItem = classItem.substring(SCHEMA_FILE.length());

                if (sb.length() > 0)
                {
                    sb.append("\n");
                }

                sb.append(classItem);
            }

            return sb.toString();
        }

        @Override
        public boolean isViewValueToModelSupported()
        {
            return true;
        }

    }
}