// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.ma.gui.runtime;

import java.awt.Color;
import java.awt.Container;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.management.openmbean.CompositeData;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.text.DefaultEditorKit;

import modelobjects.layout.PartitionLayout;

import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.action.RuntimeClearLogAction;
import com.sonicsw.ma.gui.domain.DomainConnectionModel;
import com.sonicsw.ma.gui.runtime.propsheets.JRuntimeDialog;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.BasicResourceAction;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IRuntimePlugin;

import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;

/**
 * <p>
 * Title: ViewLogInternalFrame
 * </p>
 * <p>
 * Description: Creates an internal frame to display log information
 * and perform different actions on the log information
 * </p>
 * 
 * @author Chandana Konda
 * 
 */

public class ViewLogInternalFrame extends JMAInternalFrame {
	private JScrollPane viewLogScrollpanel = null;

	private JTextArea logTextArea = null;

	private JToolBar logToolBar = null;

	private AbstractRuntimePlugin plugin = null;

	private long readLength;

	private long maxBuffer;

	private Action copyAction = null;

	private Action findAction = null;

	private Action findAgainAction = null;

	private FindToolBar findToolBar = null;

	private BasicResourceAction clearLogAction = null;

	private LogViewerSlider rangeSlider = null;

	private PreferenceManager p_manager = null;
	
	private RefreshAction refreshAction = null;
	
	public static final String KB=" Kbytes";

	public static final String LOG_PREFERENCES = "preferences.log";

	private int logFileSize = 0;
	
	public ViewLogInternalFrame(AbstractRuntimePlugin plugin) {
	    
	    super("viewlog");
	    this.plugin = plugin;

	    p_manager = PreferenceManager.getInstance();
	    readLength = new Long(p_manager.getString(LOG_PREFERENCES,
	                                              "readLength", String.valueOf(200 * 1024))).longValue();
	    maxBuffer = new Long(p_manager.getString(LOG_PREFERENCES, "maxBuffer",
	                                             String.valueOf(500 * 1024))).longValue();
	    setTitleProperty();
	    setFrameIcon();
	    
	}
	
	private void setTitleProperty() {
	    
		StringBuffer title1 = new StringBuffer();     
		title1.append(ResourceManager.getString(getClass(), "iframe." + "viewlog" + ".title"));
		if (plugin instanceof RuntimeDomainFolderPlugin)
        {
            title1.append(" - ").append(ResourceManager.getString(getClass(), "iframe." + "viewcentrallog" + ".title"));
        }
        else
		{
		        String path = (plugin != null) ? plugin.getPluginPath() : null;
		
		        if ((path != null)  && (path.trim().length() > 0))
                {
                    title1.append(" - ").append(path);
                }
		}
		setTitle(title1.toString());
	 }
	
	private void setFrameIcon() {
        setFrameIcon(ResourceManager.getApplicationIcon(getClass(), "Application18",18));
	}

	public String getPluginName() {
		return plugin.getPluginName();
	}

	public long getReadLength() {
		return readLength;
	}

	public void setReadLength(long readLength) {
		this.readLength = readLength;
	}

	public long getMaxBuffer() {
		return maxBuffer;
	}

	public void setMaxBuffer(long maxBuffer) {
		this.maxBuffer = maxBuffer;
	}

	public void setLogText(long retrieveLength) {
		if(plugin instanceof ComponentPlugin && plugin.getState().getState() != IContainerState.STATE_ONLINE)
		{
			logTextArea.setText("<Container might be offline.>");
			logFileSize = -1;
			return;
		}
		
		long fromSize = rangeSlider.getValue();
		if (plugin instanceof ComponentPlugin) {
			CompositeData data = (CompositeData) plugin
					.invokeAction("getLogExtractAndLogFileSize", new Object[] {
							new Long(fromSize), new Long(retrieveLength) },
							new String[] { Long.class.getName(),
									Long.class.getName() });
			logTextArea.setText((String) data.get("LogExtract"));
			logFileSize = (int) ((Long) data.get("LogFileSize")).longValue();
		} else {
			DomainConnectionModel domainModel = plugin.getPluginContext()
					.getConnectionInfo();
			IAgentManagerProxy agentManager = domainModel.getAgentManager();
			logTextArea.setText((String) agentManager.getLogExtract(new Long(
					fromSize), new Long(retrieveLength)));
			logFileSize = (int) ((Long) agentManager.getLogFileSize())
					.longValue();
		}
		rangeSlider.setMaximum(logFileSize);
		findToolBar.resetCounters();
		SwingUtilities.invokeLater(new Runnable() { @Override
        public void run() { logTextArea.requestFocus(); } } );
	}

	public int getLogFileSize() {
		return logFileSize;
	}

	public void disableCopyAction() {
		if (logTextArea.getSelectedText() == null)
        {
            copyAction.setEnabled(false);
        }
        else
        {
            copyAction.setEnabled(true);
        }
	}

	private JToolBar createToolBar() {
		JToolBar toolBar = new JToolBar();
		toolBar.setFloatable(false);
		toolBar.setRollover(true);
		toolBar.add(clearLogAction);
		toolBar.add(refreshAction);
		toolBar.addSeparator();
		toolBar.add(createFindAction());
		toolBar.add(createFindAgainAction());
		toolBar.addSeparator();
		toolBar.add(new PreferencesAction(this));
		return toolBar;
	}
	
	public Action createCopyAction() {		
		copyAction = logTextArea.getActionMap().get(
				DefaultEditorKit.copyAction);
		KeyStroke copyKeyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_C,
				InputEvent.CTRL_MASK);
		try {
			copyAction.putValue(Action.ACCELERATOR_KEY, copyKeyStroke);
			copyAction.putValue(Action.NAME, "Copy");
		} catch (RuntimeException e) {
			e.printStackTrace();
			MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
		}		
	return copyAction;
	}

	public Action createSelectAllAction() {
		Action selectAllAction = logTextArea.getActionMap().get(
				DefaultEditorKit.selectAllAction);
	
		selectAllAction.putValue(Action.ACCELERATOR_KEY, KeyStroke
				.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_MASK));
		selectAllAction.putValue(Action.NAME, "Select All");
		selectAllAction.setEnabled(true);
	
		return selectAllAction;
	}
	
	public Action createFindAction() {
		findAction = new BasicGuiAction("ContainerPlugin.viewlog.find") {
			@Override
            public void actionPerformed(ActionEvent e) {
				findToolBar.setVisible(true);
				findToolBar.setFocus();
			}
		};
		findAction.putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_F));
		KeyStroke key = KeyStroke.getKeyStroke(KeyEvent.VK_F,
				KeyEvent.CTRL_MASK);
		findAction.putValue(Action.ACCELERATOR_KEY, key);
		logTextArea.getInputMap(WHEN_IN_FOCUSED_WINDOW).put(key, findAction);
		logTextArea.getActionMap().put("findAction", findAction);

		return findAction;
	}

	public Action createFindAgainAction() {
		findAgainAction = new BasicGuiAction(
				"ContainerPlugin.viewlog.findagain") {
			@Override
            public void actionPerformed(ActionEvent e) {
				findToolBar.setVisible(true);
				findToolBar.setFocus();
				searchNext();
			}
		};
		findAgainAction.putValue(Action.MNEMONIC_KEY,
				new Integer(KeyEvent.VK_G));
		KeyStroke key = KeyStroke.getKeyStroke(KeyEvent.VK_G,KeyEvent.CTRL_MASK);
		findAgainAction.putValue(Action.ACCELERATOR_KEY, key);
		logTextArea.getInputMap(WHEN_IN_FOCUSED_WINDOW).put(key,
				findAgainAction);
		logTextArea.getActionMap().put("findAgainAction", findAgainAction);

		return findAgainAction;
	}

	public int searchString() {
		String searchStr = findToolBar.getFindString();
		boolean matchCase = findToolBar.getMatchCase();
		int ret = 0;
		logTextArea.getCaret().setSelectionVisible(true);
		int startIndex;
		if (matchCase)
        {
            startIndex = logTextArea.getText().indexOf(searchStr);
        }
        else
        {
            startIndex = logTextArea.getText().toLowerCase().indexOf(
					searchStr.toLowerCase());
        }
		if (startIndex != -1)
        {
            logTextArea.select(startIndex, startIndex + searchStr.length());
        }
        else {
			logTextArea.getCaret().setSelectionVisible(false);
			ret = -1;
		}
		return ret;
	}

	public int searchNext() {
		String searchStr = findToolBar.getFindString();
		boolean matchCase = findToolBar.getMatchCase();
		int pos = logTextArea.getCaretPosition();
		int startIndex = 0;
		if (matchCase)
        {
            startIndex = logTextArea.getText().indexOf(searchStr, pos);
        }
        else
        {
            startIndex = logTextArea.getText().toLowerCase().indexOf(
					searchStr.toLowerCase(), pos);
        }

		if (startIndex != -1)	
		{
			logTextArea.select(startIndex, startIndex + searchStr.length());
			logTextArea.getCaret().setSelectionVisible(true);
		}		
		return startIndex;
	}

	public int searchPrevious() {
		String searchStr = findToolBar.getFindString();
		boolean matchCase = findToolBar.getMatchCase();
		int pos = logTextArea.getCaretPosition();
		int wordStart = pos - searchStr.length();
		int startIndex = 0;
		if (matchCase) {
			pos = (logTextArea.getText().substring(wordStart, pos)
					.equals(searchStr)) ? (wordStart - 1) : pos;
			startIndex = logTextArea.getText().lastIndexOf(searchStr, pos);		
		} else {
			pos = (logTextArea.getText().substring(wordStart, pos)
					.equalsIgnoreCase(searchStr)) ? (wordStart - 1) : pos;
			startIndex = logTextArea.getText().toLowerCase().lastIndexOf(
					searchStr.toLowerCase(), pos);
		}
		if (startIndex != -1)	
		{
			logTextArea.select(startIndex, startIndex + searchStr.length());
			logTextArea.getCaret().setSelectionVisible(true);
		}
			
		return startIndex;
	}
	
	public String getSelectedText()
	{
		return logTextArea.getSelectedText() != null ? logTextArea.getSelectedText() : "";
	}

	public PreferenceManager getP_manager() {
		return p_manager;
	}

	@Override
    protected void maCleanup() {
	}

	@Override
    protected void maInitialize() {
		Container contentPane = getContentPane();
		contentPane.setLayout(new PartitionLayout(true, "p,r,p,p", 0));
		if(plugin instanceof RuntimeDomainFolderPlugin)
        {
            clearLogAction = new CentralLogClearLogAction(plugin);
        }
        else
        {
            clearLogAction = new ClearLogAction(plugin);
        }
		clearLogAction.putValue(Action.ACCELERATOR_KEY, KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_MASK));
		refreshAction = new RefreshAction(plugin);
		logTextArea = new JTextArea();
		logTextArea.setEditable(false);
		logTextArea.addMouseListener(new ReadOnlyListener(this));
		logTextArea.setMargin(new Insets(5, 5, 5, 5));
		logTextArea.getInputMap(WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_MASK),clearLogAction);

		logTextArea.getInputMap(WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke("F5"), refreshAction);
		logToolBar = createToolBar();
		contentPane.add(logToolBar);

		viewLogScrollpanel = new JScrollPane(logTextArea);
		rangeSlider = new LogViewerSlider(this);
		rangeSlider
				.setToolTipText("Drag the slider to extract log from a different position");
		findToolBar = new FindToolBar(this);
		findToolBar.setVisible(false);
		setLogText(getReadLength());
		
		contentPane.add(viewLogScrollpanel);
		contentPane.add(rangeSlider);
		contentPane.add(findToolBar);	
	}

	class ViewLogPopUpMenu extends JPopupMenu {
	    
		public ViewLogPopUpMenu() {
		    addProperties();
		}
		
		private void addProperties() {
			add(createCopyAction());
			add(createSelectAllAction());
			addSeparator();
			add(clearLogAction);
			add(refreshAction);
			addSeparator();
			add(createFindAction());
			add(createFindAgainAction());
			addSeparator();
			add(new PreferencesAction(ViewLogInternalFrame.this));
			this.addPopupMenuListener(new PopupListener());
		}
		
		
		class PopupListener implements PopupMenuListener {
			@Override
            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
				disableCopyAction();
			}

			@Override
            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
			}

			@Override
            public void popupMenuCanceled(PopupMenuEvent e) {
			}
		}
	}

	class ReadOnlyListener extends MouseAdapter {
		ViewLogInternalFrame frame = null;

		ViewLogPopUpMenu popup = null;

		public ReadOnlyListener(ViewLogInternalFrame frame) {
			popup = new ViewLogPopUpMenu();
		}

		@Override
        public void mousePressed(MouseEvent e) {
			maybeShowPopup(e);
		}

		@Override
        public void mouseReleased(MouseEvent e) {
			maybeShowPopup(e);
		}

		private void maybeShowPopup(MouseEvent e) {
			if (e.isPopupTrigger()) {
				popup.show(e.getComponent(), e.getX(), e.getY());
			}
		}
	}
	
	class ClearLogAction extends RuntimeAction {
		IRuntimePlugin plugin=null;
		public ClearLogAction(IRuntimePlugin plugin) {
			super(plugin, "ContainerPlugin.viewlog.clearlog", "clearLogFile",
					"Are you sure you want to clear the container log file?");
			this.plugin = plugin;
		}
		@Override
        public void actionPerformed(ActionEvent evt) {
			if(plugin.getAttributes().getState().getState() != IContainerState.STATE_ONLINE)
			{
				logTextArea.setText("<Container might be offline.>");
				logFileSize = -1;
				return;
			}
			super.actionPerformed(evt);	
			try {
				Thread.sleep(100);
				rangeSlider.setValue(0);
				setLogText(getLogFileSize());
			} catch (InterruptedException e) {
			}			
		}
	}
	
	class CentralLogClearLogAction extends RuntimeClearLogAction {
		public CentralLogClearLogAction(AbstractRuntimePlugin plugin)
		{
			super(plugin);
		}
		@Override
        public void actionPerformed(ActionEvent evt) {			
			super.actionPerformed(evt);
			try {
				Thread.sleep(100);
				rangeSlider.setValue(0);
				setLogText(getLogFileSize());
			} catch (InterruptedException e) {
		}
		}
	}
	class RefreshAction extends BasicGuiAction {
		public RefreshAction(IRuntimePlugin plugin) {
			super("ContainerPlugin.viewlog.refresh");
			putKeyProperty();
		}
		
		private void putKeyProperty() {
		    putValue(ACCELERATOR_KEY, KeyStroke.getKeyStroke("F5"));
		}
		@Override
        public void actionPerformed(ActionEvent evt) {			
			setLogText(getReadLength());			
		}
	}

}

class LogViewerSlider extends JSlider implements ChangeListener, MouseListener,
		MouseMotionListener {
	final JPopupMenu pop = new JPopupMenu();

	JPanel parent;

	ViewLogInternalFrame frame = null;

	JTextArea toolTipText = new JTextArea();

	public LogViewerSlider(ViewLogInternalFrame frame) {
		super(JSlider.HORIZONTAL);
		prepareLogViewSlider(frame);
	}
	
	private void prepareLogViewSlider(ViewLogInternalFrame frame) {
		this.frame = frame;
		setMaximum(frame.getLogFileSize());
		setMinimum(0);
		addMouseListener(this);
		addMouseMotionListener(this);
		addChangeListener(this);
		toolTipText.setEditable(false);
		setToolTipText(0);
		pop.add(toolTipText);
		pop.setDoubleBuffered(true);
	}

	public void setToolTipText(int sliderValue) {
		int maxValue = frame.getLogFileSize();
		
		if(maxValue <=0 ) 
		{
			toolTipText.setText("");
			return;
		}
		float logPercent = 0;	

		if (maxValue - sliderValue > frame.getReadLength())
        {
            logPercent = (frame.getReadLength() * 100) / maxValue;
        }
        else if(sliderValue == 0)
        {
            logPercent = 100;
        }
        else
        {
            logPercent = ((maxValue - sliderValue) * 100) / maxValue;
        }
		maxValue = (maxValue == 0)? maxValue :maxValue/1024;
		sliderValue = (sliderValue == 0) ? sliderValue : sliderValue /1024;
		toolTipText.setText(Math.ceil(logPercent) + "% of log in view");
		toolTipText.append("\nLog Size is " + maxValue + ViewLogInternalFrame.KB);
		toolTipText.append("\nViewing from " + sliderValue + ViewLogInternalFrame.KB);
	}

	@Override
    public void stateChanged(ChangeEvent e) {
		JSlider slider = (JSlider) e.getSource();
		setToolTipText(slider.getValue());
		if (!slider.getValueIsAdjusting()) {
			frame.setLogText(frame.getReadLength());
		}
	}

	public void showToolTip(MouseEvent me) {
		pop.show(me.getComponent(), me.getX() - 5, -30);
	}

	@Override
    public void mouseClicked(MouseEvent e) {

	}

	@Override
    public void mouseEntered(MouseEvent e) {
	}

	@Override
    public void mouseExited(MouseEvent e) {
	}

	@Override
    public void mousePressed(MouseEvent e) {
		showToolTip(e);
	}

	@Override
    public void mouseReleased(MouseEvent e) {
		pop.setVisible(false);
	}

	@Override
    public void mouseDragged(MouseEvent e) {
		showToolTip(e);
	}

	@Override
    public void mouseMoved(MouseEvent e) {
	}
}

class LogPreferencesDialog extends JRuntimeDialog {

	private JTextField maxBufferSizeTextField = null;

	private JTextField retrieveSizeTextField = null;

	private ViewLogInternalFrame frame = null;

	public LogPreferencesDialog(ViewLogInternalFrame frame) {

		super((JMAFrame) JOptionPane.getFrameForComponent(frame),
				"viewlog.preferences");
		this.frame = frame;
	}

	private int validateBufferSize(String maxSize, String retrieveSize) {
		try {
			long maxSizeValue = Long.parseLong(maxSize);
			long retrieveSizeValue = Long.parseLong(retrieveSize);
			if (maxSizeValue < 200 || maxSizeValue > 1000) {
				JOptionPane
						.showMessageDialog(
								this,
								"Max Buffer Size cannot be less than 200 Kbytes and greater than 1000 Kbytes",
								"Input Error", JOptionPane.ERROR_MESSAGE);
				return 1;
			} else if (retrieveSizeValue < 200 || retrieveSizeValue > 1000) {
				JOptionPane
						.showMessageDialog(
								this,
								"Retrieve Size should be between 200 Kbytes and 1000 Kbytes",
								"Input Error", JOptionPane.ERROR_MESSAGE);
				return 1;
			} else if (maxSizeValue < retrieveSizeValue) {
				JOptionPane.showMessageDialog(this,
						"Retrieve Size cannot be more than Max Buffer Size",
						"Input Error", JOptionPane.ERROR_MESSAGE);
				return 1;
			}
			frame.setReadLength(retrieveSizeValue * 1024);
			frame.setMaxBuffer(maxSizeValue * 1024);
			frame.setLogText(retrieveSizeValue * 1024);

			return 0;
		} catch (NumberFormatException e) {
			JOptionPane.showMessageDialog(this,
					"Enter a Valid Size Value in Kb.", "Input Error",
					JOptionPane.ERROR_MESSAGE);
			return 1;
		}
	}

	public void setPreferences() {
		frame.getP_manager().setString(ViewLogInternalFrame.LOG_PREFERENCES,
				"readLength", String.valueOf(frame.getReadLength()), true);
		frame.getP_manager().setString(ViewLogInternalFrame.LOG_PREFERENCES,
				"maxBuffer", String.valueOf(frame.getMaxBuffer()), true);
	}

	@Override
    public BasicAction getDefaultOKAction() {
		return new OkAction();
	}

	public class OkAction extends BasicGuiAction {
		public OkAction() {
			super("dialog.ok");
		}

		@Override
        public void actionPerformed(ActionEvent e) {
			int ret = validateBufferSize(maxBufferSizeTextField.getText(),
					retrieveSizeTextField.getText());
			if (ret == 0) {
				setPreferences();
				LogPreferencesDialog.this.dispose();
			}
		}
	}

	@Override
    protected void initForm() throws Exception {

	}

	@Override
    protected void initUI() throws Exception {
		JPartitionPanel panel = new JPartitionPanel();

		panel.setBorder(BorderFactory.createTitledBorder("Log Preferences"));

		maxBufferSizeTextField = new JTextField(20);
		retrieveSizeTextField = new JTextField(20);
		maxBufferSizeTextField.setText(new Long(frame.getMaxBuffer() / 1024)
				.toString());
		retrieveSizeTextField.setText(new Long(frame.getReadLength() / 1024)
				.toString());
		panel.addRow("Max Viewer Buffer Size", maxBufferSizeTextField, ViewLogInternalFrame.KB);

		panel.addRow("Retrieve Size", retrieveSizeTextField, ViewLogInternalFrame.KB);

		super.getContentPane().add(JPartitionPanel.wrap(panel));
	}
}

class PreferencesAction extends BasicGuiAction {
	ViewLogInternalFrame frame = null;

	PreferencesAction(ViewLogInternalFrame frame) {
		super("ContainerPlugin.viewlog.preferences");
		this.frame = frame;
	}

	@Override
    public void actionPerformed(ActionEvent e) {
		LogPreferencesDialog dialog = new LogPreferencesDialog(frame);
		dialog.setVisible(true);
		dialog.setModal(true);
	}
}



class FindToolBar extends JToolBar implements DocumentListener {
	private ViewLogInternalFrame frame = null;

	private JTextField textField = new JTextField(20);

	private JButton closeButton = null;

	private JButton nextButton = null;

	private JButton previousButton = null;

	private JCheckBox matchCaseCheckBox = null;
	
	private String findString = "";
	
	private JLabel statusLabel = new JLabel("");
	
	public static final int FOUND=0;
	public static final int ON_TEXT=1;
	public static final int ON_NEXT=2;
	public static final int ON_PREVIOUS=3;
	private int nextCount = 0;
	private int previousCount = 0;

	public FindToolBar(ViewLogInternalFrame viewLogFrame) {
	    prepareFindToolBar(viewLogFrame);
	}
	
	private void prepareFindToolBar(ViewLogInternalFrame viewLogFrame) {
	    
		this.frame = viewLogFrame;
		setFloatable(false);
		setRollover(true);
		setLayout(new PartitionLayout(false, "p,p,p,p,p,p,r", 5));
		setBorder(BorderFactory.createEtchedBorder());
		textField.getDocument().addDocumentListener(this);

		closeButton = new JButton(new BasicGuiAction(
				"ContainerPlugin.viewlog.close") {
			@Override
            public void actionPerformed(ActionEvent e) {
				setVisible(false);
			}
		});

		nextButton = new JButton(new NextAction());
		previousButton = new JButton(new PreviousAction());
		matchCaseCheckBox = new JCheckBox("Match Case");
		matchCaseCheckBox.addItemListener(new ItemListener(){
			@Override
            public void itemStateChanged(ItemEvent e) {
				if(!matchCaseCheckBox.isSelected() && getFindString().equalsIgnoreCase(frame.getSelectedText()))
                {
                    nextCount = previousCount = 1;
                }
                else
                {
                    nextCount = previousCount = 0;
                }
			}
		});
		add(closeButton);
		add(new JLabel("Find: "));
		add(textField);
		add(nextButton);
		add(previousButton);
		add(matchCaseCheckBox);
		add(statusLabel);
	}
	
	public void resetCounters()
	{
		previousCount = nextCount = 0;
	}

	public boolean getMatchCase() {
		return matchCaseCheckBox.isSelected();
	}

	public void setFocus() {
		textField.requestFocus();
	}

	@Override
    public void changedUpdate(DocumentEvent e) {
		findStringStatus(ON_TEXT);
	}

	@Override
    public void insertUpdate(DocumentEvent e) {
		findStringStatus(ON_TEXT);
	}

	@Override
    public void removeUpdate(DocumentEvent e) {
		findStringStatus(ON_TEXT);
	}

	public class NextAction extends BasicGuiAction {
		public NextAction() {
			super("ContainerPlugin.viewlog.next");
			putMnemonicKeyVal();
		}
		private void putMnemonicKeyVal() {
            putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_N));
		}
		@Override
        public void actionPerformed(ActionEvent evt) {
			findStringStatus(ON_NEXT);
		}
	}

	public class PreviousAction extends BasicGuiAction {
		public PreviousAction() {
			super("ContainerPlugin.viewlog.previous");
			putMnemonicKeyVal();
		}
        private void putMnemonicKeyVal() {
            putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_N));
        }
		@Override
        public void actionPerformed(ActionEvent evt) {
			findStringStatus(ON_PREVIOUS);
		}
	}
	
	private void findStringStatus(int type)
	{
		int ret = -1;
		switch(type)
		{
			case 1: ret = frame.searchString();
					if(ret == -1)
					{
						nextCount = 0;
						previousCount = 0;
						setStatus(ON_TEXT);
					}
					else
					{
						setStatus(FOUND);
						nextCount++;
						previousCount++;
					}
			break;
			case 2:
				   ret = frame.searchNext();
				   if(ret == -1)
                {
                    if(nextCount == 0)
                    {
                        setStatus(ON_TEXT);
                    }
                    else
                    {
                        setStatus(ON_NEXT);
                    }
                }
                else
				   {
					   nextCount++;
					   previousCount = 0;
					   setStatus(FOUND);
				   }
				break;
			case 3:
				ret = frame.searchPrevious();
				if(ret == -1)
                {
                    if(previousCount == 0)
                    {
                        setStatus(ON_TEXT);
                    }
                    else
                    {
                        setStatus(ON_PREVIOUS);
                    }
                }
                else
				   {
					   previousCount++;
					   nextCount = 0;
					   setStatus(FOUND);
				   }
				break;
			default : break;
		}
		
	}

	public void setStatus(int status)
	{
		switch(status)
		{
		case 1:	statusLabel.setText("     String Not Found");
				textField.setBackground(new Color(192, 72, 72));break;
		case 2: textField.setBackground(Color.WHITE);
				statusLabel.setText("     Reached end of the log");break;
		case 3: textField.setBackground(Color.WHITE);
				statusLabel.setText("     Reached beginning of the log");break;
		case 0: 
		default:textField.setBackground(Color.WHITE); 
				statusLabel.setText(""); break;
		}
	}

	public String getFindString() {
		findString = textField.getText();
		return findString;
	}
}
