package com.sonicsw.ma.gui.runtime.metrics;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Enumeration;

import javax.swing.Icon;
import javax.swing.JInternalFrame;
import javax.swing.JPopupMenu;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.runtime.JAbstractMonitorPanel;
import com.sonicsw.ma.gui.runtime.JMonitorDialog;
import com.sonicsw.ma.gui.runtime.metrics.model.AggregateMetricsModel;
import com.sonicsw.ma.gui.runtime.metrics.model.IMetricWatcher;
import com.sonicsw.ma.gui.runtime.util.AbstractNode;
import com.sonicsw.ma.gui.runtime.util.AbstractParentNode;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IMonitorPlugin;

import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;

public final class JAggregateMetricPanel extends    JAbstractMonitorPanel
                                         implements TreeModelListener
{
    public static final String MONITOR_ID_AGGREGATE_METRICS = "_AggregateMetrics";

    private JTree       m_tree;
    private BasicAction m_watchAction   = new WatchAction();
    private BasicAction m_refreshAction = new RefreshAction();

    public JAggregateMetricPanel()
    {
        m_tree = new JTree((TreeModel)null)
        {
            @Override
            public String getToolTipText(MouseEvent event)
            {
                TreePath path = getPathForLocation(event.getX(), event.getY());

                if (path == null)
                {
                    return null;
                }

                return ((AbstractNode)path.getLastPathComponent()).getDescription();
            }
        };
        m_tree.setRootVisible(false);
        m_tree.setShowsRootHandles(true);
        m_tree.setCellRenderer(new AggrMetricsTreeCellRenderer());
        m_tree.setVisibleRowCount(10);
        m_tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        m_tree.addMouseListener(new PopupMenuShower(m_tree, buildPopupMenu()));

        ExtendedJScrollPane scrollPane = new ExtendedJScrollPane(m_tree);

        JToolBar tb = new JToolBar(JToolBar.HORIZONTAL);
        tb.setFloatable(false);
        tb.setRollover(true);

        tb.add(m_watchAction);
        tb.add(m_refreshAction);

        add(tb,         BorderLayout.NORTH);
        add(scrollPane, BorderLayout.CENTER);

        enableButtons();

        m_tree.addTreeSelectionListener(new TreeSelectionListener()
        {
            @Override
            public void valueChanged(TreeSelectionEvent evt) { enableButtons(); }
        });
    }

    @Override
    public Icon getIcon() { return ResourceManager.getIcon(getClass(), "Metric"); }

    @Override
    public void maInitialize()
    {
        ToolTipManager.sharedInstance().registerComponent(m_tree);
    }

    @Override
    public void maCleanup()
    {
        AggregateMetricsModel model = (AggregateMetricsModel)m_tree.getModel();

        if (model != null)
        {
            model.removeTreeModelListener(this);
        }

        ToolTipManager.sharedInstance().unregisterComponent(m_tree);
    }

    @Override
    public String getTabTitle()
    {
        return "Aggregated Metrics";
    }

    @Override
    public String getMonitorId()
    {
       return MONITOR_ID_AGGREGATE_METRICS;
    }

    @Override
    public void changeModel(IMonitorPlugin plugin)
    {
        AggregateMetricsModel newModel = (AggregateMetricsModel)plugin.getMonitorModel(MONITOR_ID_AGGREGATE_METRICS);
        AggregateMetricsModel oldModel = (AggregateMetricsModel)m_tree.getModel();

        if (oldModel != null)
        {
            oldModel.removeTreeModelListener(this);
        }

        m_tree.setModel(newModel);

        if (newModel != null)
        {
            newModel.addTreeModelListener(this);
        }

        expandTree();
    }

    private void expandTree()
    {
        // expand all but instance nodes
        int row = 0;

        while (row < m_tree.getRowCount())
        {
            if (m_tree.getPathForRow(row).getLastPathComponent() instanceof AbstractParentNode)
            {
                m_tree.expandRow(row);
            }

            row++;
        }
    }

    private JPopupMenu buildPopupMenu()
    {
        JPopupMenu menu = new JPopupMenu();
        menu.add(new JBasicMenuItem(m_watchAction));
        return menu;
    }

    private void enableButtons()
    {
        TreePath     path   = m_tree.getSelectionPath();
        boolean      sel    = (path != null);
        AbstractNode node   = sel ? (AbstractNode)path.getLastPathComponent() : null;
        boolean      parent = sel && (path.getLastPathComponent() instanceof AbstractParentNode);
        boolean      enable = ((node != null) && node.isEnabled());

        m_watchAction.setEnabled(!parent && enable);
        m_refreshAction.setEnabled(true);
    }

    private AbstractNode getCurrentNode()
    {
        return (AbstractNode)m_tree.getSelectionPath().getLastPathComponent();
    }

    private void refresh()
    {
        AggregateMetricsModel model = (AggregateMetricsModel)m_tree.getModel();

        try
        {
            model.refreshTree();
            expandTree();
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                         "Failed to refresh", e, true);
            return;
        }
    }

    //-------------------------------------------------------------------------
    //
    // TreeModelListener Implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void treeNodesChanged(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeNodesInserted(TreeModelEvent evt)    { enableButtons(); }
    @Override
    public void treeNodesRemoved(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeStructureChanged(TreeModelEvent evt) { enableButtons(); }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------


    private class WatchAction extends BasicGuiAction
    {
        public WatchAction()
        {
            super("AggregateMetricsSheet.watch");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            IMetricWatcher watcher =
                AggregateMetricsWatchWindow.getWatchWindow(((AggregateMetricsModel)m_tree.getModel()).getComponentName());

            if (((JInternalFrame)watcher).isVisible() == false)
            {
                // Before adding this new internal frame into the desktop,
                // we need to make it dependant on the domain (workspace) frame.
                JMonitorDialog parent = (JMonitorDialog)SwingUtilities.getAncestorOfClass(JMonitorDialog.class, JAggregateMetricPanel.this);
                WorkspaceWindow domainParent = parent.getWorkspaceWindow();

                if (domainParent != null)
                {
                    domainParent.addDependantFrame((JMAInternalFrame)watcher);

                    MgmtConsole.getMgmtConsole().showInternalFrame((JInternalFrame)watcher, true);
                }
            }

            AbstractNode node = getCurrentNode();
            ArrayList nodes = new ArrayList();

            if (node instanceof AbstractParentNode)
            {
                Enumeration en = node.depthFirstEnumeration();
                while (en.hasMoreElements())
                {
                    node = (AbstractNode)en.nextElement();
                    if (!(node instanceof AbstractParentNode))
                    {
                        nodes.add(node);
                    }
                }
            }
            else
            {
                nodes.add(node);
            }

            AggregateMetricsModel model = (AggregateMetricsModel)m_tree.getModel();

            for (int i = 0; i < nodes.size(); i++)
            {
                Object          data = ((AbstractNode)nodes.get(i)).getUserObject();
                IMetricIdentity id   = (data instanceof IMetricIdentity) ? (IMetricIdentity)data : ((IMetricInfo)data).getMetricIdentity();
                watcher.addWatch(id, model);
            }
        }
    }

    private class RefreshAction extends BasicGuiAction
    {
        public RefreshAction()
        {
            super("AggregateMetricsSheet.refresh");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                refresh();
            }
            catch(Throwable e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                                    "Failed to refresh", e, true);
            }
        }
    }

}