package com.sonicsw.ma.gui.runtime.metrics;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Enumeration;

import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButton;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.WorkspaceWindow;
import com.sonicsw.ma.gui.runtime.JAbstractMonitorPanel;
import com.sonicsw.ma.gui.runtime.JMonitorDialog;
import com.sonicsw.ma.gui.runtime.metrics.model.IMetricWatcher;
import com.sonicsw.ma.gui.runtime.metrics.model.InstanceNode;
import com.sonicsw.ma.gui.runtime.metrics.model.InstanceParentNode;
import com.sonicsw.ma.gui.runtime.metrics.model.MetricsModel;
import com.sonicsw.ma.gui.runtime.metrics.model.ParentNode;
import com.sonicsw.ma.gui.runtime.util.AbstractNode;
import com.sonicsw.ma.gui.runtime.util.AbstractParentNode;
import com.sonicsw.ma.gui.runtime.util.IWatcher;
import com.sonicsw.ma.gui.util.BasicAction;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.JPartitionPanel;
import com.sonicsw.ma.gui.util.JWaitCursor;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IMonitorPlugin;

import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;

public final class JMetricsPanel extends    JAbstractMonitorPanel
                                 implements TreeModelListener
{
    private JTree       m_tree;
    private BasicAction m_enableAction = new EnableAction();
    private BasicAction m_disableAction = new DisableAction();
    private BasicAction m_enableInstancesAction = new InstancesAction();
    private BasicAction m_alertsAction = new AlertsAction();
    private BasicAction m_watchAction = new WatchButtonAction();

    public JMetricsPanel()
    {
        super();

        m_tree = new CustomJTree(((TreeModel) null));
        m_tree.setRootVisible(false);
        m_tree.setShowsRootHandles(true);
        m_tree.setCellRenderer(new MetricsTreeCellRenderer());
        m_tree.setVisibleRowCount(10);
        m_tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        m_tree.addMouseListener(new PopupMenuShower(m_tree, buildPopupMenu()));

        ExtendedJScrollPane scrollPane = new ExtendedJScrollPane(m_tree);

        JToolBar tb = new JToolBar(JToolBar.HORIZONTAL);
        tb.setFloatable(false);
        tb.setRollover(true);

        tb.add(m_enableAction);
        tb.add(m_disableAction);
        tb.addSeparator();
        tb.add(m_enableInstancesAction);
        tb.addSeparator();
        tb.add(m_alertsAction);
        tb.addSeparator();
        tb.add(new JWatchButton(m_watchAction));

        add(tb,         BorderLayout.NORTH);
        add(scrollPane, BorderLayout.CENTER);

        enableButtons();

        m_tree.addTreeSelectionListener(new TreeSelectionListener()
        {
            @Override
            public void valueChanged(TreeSelectionEvent evt) { enableButtons(); }
        });
    }

    @Override
    public Icon getIcon() { return ResourceManager.getIcon(getClass(), "Metric"); }

    @Override
    public void maInitialize()
    {
        ToolTipManager.sharedInstance().registerComponent(m_tree);
    }

    @Override
    public void maCleanup()
    {
        MetricsModel model = (MetricsModel)m_tree.getModel();

        if (model != null)
        {
            model.removeTreeModelListener(this);
        }
        m_tree.setModel(null);

        ToolTipManager.sharedInstance().unregisterComponent(m_tree);
    }

    @Override
    public String getTabTitle()
    {
        return "Metrics";
    }

    @Override
    public String getMonitorId()
    {
        return IMonitorPlugin.MONITOR_ID_METRICS;
    }

    @Override
    public void changeModel(IMonitorPlugin plugin)
    {
        MetricsModel oldModel = (MetricsModel)m_tree.getModel();

        if (oldModel != null)
        {
            oldModel.removeTreeModelListener(this);
        }
        MetricsModel newModel = (MetricsModel)plugin.getMonitorModel(IMonitorPlugin.MONITOR_ID_METRICS);
        m_tree.setModel(newModel);

        if (newModel != null)
        {
            newModel.setSource(plugin);

            newModel.addTreeModelListener(this);

            try
            {
                newModel.refreshTree();
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_ERROR,
                                         "Failed to refresh metrics model", e, false);
            }
        }

        expandTree();
    }

    private void expandTree()
    {
        // expand all but instance nodes
        int row = 0;
        while (row < m_tree.getRowCount())
        {
            if (m_tree.getPathForRow(row).getLastPathComponent() instanceof ParentNode)
            {
                m_tree.expandRow(row);
            }
            row++;
        }
    }

    private JPopupMenu buildPopupMenu()
    {
        JPopupMenu menu = new JPopupMenu();

        // Wrapping all actions in JBasicMenuItems so that any icons set in
        // the actions don't appear in the menus...the basic sub-class filters
        // out icons.
        //
        JBasicMenu watchMenu = new JBasicMenu((BasicAction)m_watchAction);
        watchMenu.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.component", IWatcher.COMPONENT_SCOPE)));
        watchMenu.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.container", IWatcher.CONTAINER_SCOPE)));
        watchMenu.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.domain",    IWatcher.DOMAIN_SCOPE)));

        menu.add(new JBasicMenuItem(m_enableAction));
        menu.add(new JBasicMenuItem(m_disableAction));
        menu.addSeparator();
        menu.add(new JBasicMenuItem(m_enableInstancesAction));
        menu.add(new JBasicMenuItem(m_alertsAction));
        menu.addSeparator();
        menu.add(watchMenu);

        return menu;
    }

    private void enableButtons()
    {
        TreePath     path   = m_tree.getSelectionPath();
        boolean      sel    = (path != null);
        AbstractNode node   = sel ? (AbstractNode)path.getLastPathComponent() : null;
        boolean      parent = sel && (path.getLastPathComponent() instanceof AbstractParentNode);
        boolean      enable = ((node != null) && node.isEnabled());

        m_enableAction.setEnabled(parent || ((node != null) && !node.isEnabled() && !(node instanceof InstanceNode)));
        m_disableAction.setEnabled(parent || ((node != null) && node.isEnabled() && !(node instanceof InstanceNode)));
        m_enableInstancesAction.setEnabled((node != null) && (node instanceof InstanceParentNode));
        m_alertsAction.setEnabled((node != null) ? isAlertsEnabled(node) : false);
        m_watchAction.setEnabled(!parent && enable);
    }

    private AbstractNode getCurrentNode()
    {
        return (AbstractNode)m_tree.getSelectionPath().getLastPathComponent();
    }

    private boolean isAlertsEnabled(AbstractNode node)
    {
        boolean     result = false;
        IMetricInfo info = null;
        if (node.getUserObject() instanceof IMetricInfo){
            info = (IMetricInfo)node.getUserObject();
        }else if (node instanceof InstanceNode) {
            InstanceParentNode parent = (InstanceParentNode)node.getParent();
            if (parent != null)
            {
                info = (IMetricInfo)parent.getUserObject();
            }
        }
        if (info != null)
        {
            result = info.supportsHighThresholdAlerts() ? true : info.supportsLowThresholdAlerts();
        }
        return result;
    }

    //-------------------------------------------------------------------------
    //
    // TreeModelListener Implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void treeNodesChanged(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeNodesInserted(TreeModelEvent evt)    { enableButtons(); }
    @Override
    public void treeNodesRemoved(TreeModelEvent evt)     { enableButtons(); }
    @Override
    public void treeStructureChanged(TreeModelEvent evt) { enableButtons(); }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    class JWatchButton extends JButton
    {
        public JWatchButton(Action action)
        {
            super(action);
        }

        @Override
        protected void configurePropertiesFromAction(Action a)
        {
            String[] types = { Action.SMALL_ICON,
                               Action.SHORT_DESCRIPTION,
                               Action.LONG_DESCRIPTION,
                               "enabled" };

            Helper.configurePropertiesFromAction(this, a, types);
        }
    }

    private class EnableAction extends BasicGuiAction
    {
        public EnableAction()
        {
            super("MetricsSheet.enable");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                ((MetricsModel)m_tree.getModel()).enableMetric(getCurrentNode());
            }
            catch(Throwable e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                                    "Failed to enable metric", e, true);
            }
        }
    }

    private class DisableAction extends BasicGuiAction
    {
        public DisableAction()
        {
            super("MetricsSheet.disable");
        }

        private boolean hasInstanceMetric(AbstractNode startNode)
        {
            boolean hasInstance = false;

            Enumeration en = startNode.breadthFirstEnumeration();

            while (en.hasMoreElements())
            {
                AbstractNode node = (AbstractNode)en.nextElement();

                if (node instanceof InstanceNode)
                {
                    hasInstance = true;
                    break;
                }
            }

            return hasInstance;
        }

        /**
         * This little bit of code enumerates through the metric model's nodes
         * looking for a InstanceParentNode and finds out if anything other
         * than a simple "*" has been set.
         *
         * @param startNode  The node below which to search for an InstanceParentNode
         * @return           true if all nodes only use a "*" pattern (or nothing)
         *                   or false if more complicated patterns were found
         */
        private boolean isInstanceEnablementSimple(AbstractNode startNode)
        {
            boolean      state = true;
            MetricsModel model = (MetricsModel)m_tree.getModel();
            Enumeration  en    = startNode.breadthFirstEnumeration();

            while (en.hasMoreElements())
            {
                AbstractNode node = (AbstractNode)en.nextElement();

                if (node instanceof InstanceParentNode)
                {
                    try
                    {
                        String[] enabledPatterns = model.getEnabledMetricsPatterns((InstanceParentNode)node);

                        if (((enabledPatterns.length != 0) && (enabledPatterns.length > 1)) ||
                            !enabledPatterns[0].equals("*"))
                        {
                            state = false;
                            break;
                        }
                    }
                    catch (Exception e)
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                    }
                }
            }

            return state;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                if (hasInstanceMetric(getCurrentNode()))
                {
                    int disableOption = MetricsModel.DISABLE_INSTANCE_NONE;

                    if (isInstanceEnablementSimple(getCurrentNode()))
                    {
                        disableOption = MetricsModel.DISABLE_INSTANCE_ALL;
                    }
                    else
                    {
                        DisableDialog dlg = new DisableDialog((JMADialog)SwingUtilities.getAncestorOfClass(JMADialog.class, JMetricsPanel.this));
                        dlg.setVisible(true);

                        if (dlg.getCloseCommand() == JMADialog.CLOSE_OK)
                        {
                            disableOption = dlg.getDisableOption();
                        }
                    }

                    ((MetricsModel)m_tree.getModel()).disableMetric(getCurrentNode(), disableOption);
                }
                else
                {
                    ((MetricsModel)m_tree.getModel()).disableMetric(getCurrentNode());
                }
            }
            catch(Throwable e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                                   "Failed to disable metric", e, true);
            }
        }
    }

    private class InstancesAction extends BasicGuiAction
    {
        public InstancesAction()
        {
            super("MetricsSheet.instances");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                MetricsModel model = (MetricsModel)m_tree.getModel();
                InstanceParentNode node = (InstanceParentNode)getCurrentNode();
                String[] enabledPatterns = model.getEnabledMetricsPatterns(node);

                JMADialog parent = (JMADialog)SwingUtilities.getAncestorOfClass(JMADialog.class,
                    JMetricsPanel.this);
                JMADialog dialog = new InstanceEnablementSheet(parent, model, node, enabledPatterns);
                dialog.setVisible(true);
            }
            catch (Throwable e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to enable/disable metric instances", e, true);
            }
        }
    }

    private class BaseWatchAction extends BasicGuiAction
    {
        public BaseWatchAction()
        {
            super("MetricsSheet.watch");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }
    }

    private class WatchButtonAction extends BaseWatchAction
    {
        public WatchButtonAction()
        {
            super();
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            super.actionPerformed(evt);

            JPopupMenu pm = new JPopupMenu("-");

            pm.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.component", IWatcher.COMPONENT_SCOPE)));
            pm.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.container", IWatcher.CONTAINER_SCOPE)));
            pm.add(new JBasicMenuItem(new WatchAction("MetricsSheet.watch.domain",    IWatcher.DOMAIN_SCOPE)));
            pm.addPopupMenuListener(new PopupMenuListener()
            {
                @Override
                public void popupMenuCanceled(PopupMenuEvent evt) {}
                @Override
                public void popupMenuWillBecomeVisible(PopupMenuEvent evt) {}
                @Override
                public void popupMenuWillBecomeInvisible(PopupMenuEvent evt)
                {
                      m_watchAction.setSelected(false);
                }
            });

            if (pm.getComponentCount() > 0)
            {
                JButton src = (JButton)evt.getSource();
                Point   pt  = Helper.getPopupMenuOrigin(src, pm);

                pm.show(src, pt.x, pt.y);
            }
        }
    }

    private class WatchAction extends BasicGuiAction
    {
        private short m_scope;

        public WatchAction(String resource, short scope)
        {
            super(resource);

            m_scope = scope;
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            Thread t = new Thread(new Runnable()
            {
                @Override
                public void run()
                {
                    JWaitCursor wc = new JWaitCursor(Helper.getJDialog(JMetricsPanel.this));

                    try
                    {
                        final IMetricWatcher watcher = MetricsWatchWindow.getWatchWindow(m_scope,
                            ((MetricsModel)m_tree.getModel()).getComponentName());

                        if (((JInternalFrame)watcher).isVisible() == false)
                        {
                            // Before adding this new internal frame into the desktop,
                            // we need to make it dependant on the domain (workspace) frame.
                            JMonitorDialog parent = (JMonitorDialog)SwingUtilities.
                                getAncestorOfClass(
                                JMonitorDialog.class, JMetricsPanel.this);
                            WorkspaceWindow domainParent = parent.getWorkspaceWindow();

                            if (domainParent != null)
                            {
                                domainParent.addDependantFrame((JMAInternalFrame)watcher);
                                ((MetricsWatchWindow)watcher).setDomainParent(domainParent);

                                MgmtConsole.getMgmtConsole().showInternalFrame((JInternalFrame)
                                    watcher, true);
                            }
                        }

                        AbstractNode node = getCurrentNode();
                        ArrayList nodes = new ArrayList();

                        if (node instanceof AbstractParentNode)
                        {
                            Enumeration en = node.depthFirstEnumeration();
                            while (en.hasMoreElements())
                            {
                                node = (AbstractNode)en.nextElement();
                                if (!(node instanceof AbstractParentNode))
                                {
                                    nodes.add(node);
                                }
                            }
                        }
                        else
                        {
                            nodes.add(node);
                        }

                        final MetricsModel model = (MetricsModel)m_tree.getModel();

                        for (int i = 0; i < nodes.size(); i++)
                        {
                            Object data = ((AbstractNode)nodes.get(i)).getUserObject();
                            final IMetricIdentity id = (data instanceof IMetricIdentity) ?
                                (IMetricIdentity)data : ((IMetricInfo)data).getMetricIdentity();

                            Helper.invoke(new Runnable()
                            {
                                @Override
                                public void run()
                                {
                                    watcher.addWatch(id, model);
                                }
                            }, true);
                        }
                    }
                    finally
                    {
                        wc.release();
                    }
                }
            });

            t.start();
        }
    }

    private class AlertsAction extends BasicGuiAction
    {
        public AlertsAction()
        {
            super("MetricsSheet.alerts");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            AbstractNode node = getCurrentNode();
            JMADialog parentDialog = (JMADialog)SwingUtilities.getAncestorOfClass(JMADialog.class,
                JMetricsPanel.this);
            JMADialog dialog = new AlertsDialog(parentDialog, (MetricsModel)m_tree.getModel(), node);
            dialog.setVisible(true);
        }
    }

    private class DisableDialog extends JMADialog
    {
        private int m_option;

        public DisableDialog(JMADialog parent)
        {
            super(parent, "manage-tools.metric.disable-instance");
            setResizableProperty();
        }
        
        private void setResizableProperty() {
            setResizable(false);
        }
        
        @Override
        public void maInitialize()
        {
            JRadioButton btn1 = new JRadioButton("Remove all instance patterns");
            JRadioButton btn2 = new JRadioButton("Remove only the '*' pattern");
            JRadioButton btn3 = new JRadioButton("Don't remove any patterns");

            btn1.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt)
                {
                    m_option = MetricsModel.DISABLE_INSTANCE_ALL;
                }
            });
            btn2.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt)
                {
                    m_option = MetricsModel.DISABLE_INSTANCE_WILDCARD;
                }
            });
            btn3.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt)
                {
                    m_option = MetricsModel.DISABLE_INSTANCE_NONE;
                }
            });

            // This is the same behavior as the MQ 5.0 release.
            btn2.doClick();

            ButtonGroup btnGroup = new ButtonGroup();
            btnGroup.add(btn1);
            btnGroup.add(btn2);
            btnGroup.add(btn3);

            JPanel btnPanel = new JPartitionPanel(true, "p,p,p", null, 0, JPartitionPanel.DEFAULT_LR_EDGE_GAP,
                                                                          JPartitionPanel.DEFAULT_TB_EDGE_GAP);
            btnPanel.add(btn1);
            btnPanel.add(btn2);
            btnPanel.add(btn3);

            JPanel wrapPanel = new JPartitionPanel(true, "p,p,r", null);
            JLabel label1 = new JLabel("You have choosen to disable the selected instance metric(s).");
            JLabel label2 = new JLabel("Do you want to:");
            wrapPanel.add(label1);
            wrapPanel.add(label2);
            wrapPanel.add(btnPanel);

            JPanel panel = new JPartitionPanel(false, "p,r", null, JPartitionPanel.DEFAULT_GAP_SIZE,
                                                                   10, 10);
            panel.add(JPartitionPanel.merge("p", true, new Component[] { new JLabel(UIManager.getIcon("OptionPane.questionIcon")) }));
            panel.add(wrapPanel);
            getContentPane().add(panel);
        }

        @Override
        public void maCleanup()
        {
        }

        public int getDisableOption()
        {
            return m_option;
        }
    }

    private static class CustomJTree extends JTree {

        public CustomJTree(TreeModel newModel) {
            super(newModel);
        }
            @Override
            public String getToolTipText(MouseEvent event)
            {
                TreePath path = getPathForLocation(event.getX(), event.getY());
                if (path == null)
                {
                    return null;
                }

                return ((AbstractNode)path.getLastPathComponent()).getDescription();
            }
    }

}