package com.sonicsw.ma.gui.runtime.metrics;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

import lt.monarch.chart.AxisMapper;
import lt.monarch.chart.ChartObject;
import lt.monarch.chart.Grid;
import lt.monarch.chart.chart2D.Axis2DX;
import lt.monarch.chart.chart2D.Axis2DY;
import lt.monarch.chart.chart2D.MultiLineLabelLayouter;
import lt.monarch.chart.chart2D.PlaneMapper2D;
import lt.monarch.chart.chart3D.BarChart3D;
import lt.monarch.chart.chart3D.ChartRotator;
import lt.monarch.chart.mapper.LabelAxisMapper;
import lt.monarch.chart.mapper.MathAxisMapper;
import lt.monarch.chart.view.ToolTipManager;

public class MetricsBarChart
    extends MetricsChart
{
    private BarChart3D m_chart = null;
    private Axis2DX m_axisX = null;
    private Axis2DY m_axisY = null;
    private AxisMapper m_yMapper = null;
    private AxisMapper m_xMapper = null;
    private Grid m_grid = null;
    private boolean m_bInitialized = false;
    private Collection m_colSeries = Collections.synchronizedList(new ArrayList());
    private ToolTipManager m_tipMgr = null;
    private ChartRotator m_rotator = null;
    private MultiLineLabelLayouter m_layouter = null;
    private PlaneMapper2D m_planeMapper = null;
    private Float m_fDefaultElevationAngle = new Float(0.0);
    private Float m_fDefaultRotationAngle = new Float(0.0);
    private BarChart3D m_chartBar = null;

    ////////////////////////////////////////////////////////////////////////////////////////////////
    // Public Methods
    //

    public MetricsBarChart(Collection colMetrics)
    {
        super(new BarChart3D());

        m_colSeries = colMetrics;
    }

    @Override
    public void init()
    {
        if (m_bInitialized == false)
        {
            // create chart layout...
            m_chart = (BarChart3D)super.getViewChart();

            createLayout();

            Collection colIncoming = m_colSeries;
            m_colSeries = Collections.synchronizedList(new ArrayList());
            Iterator iter = colIncoming.iterator();

            while (iter.hasNext())
            {
                // this value is to be charted...
                createSeries((MetricValue) iter.next());
            }

            setupChart();
            m_bInitialized = true;
        }

        super.init();
    }

    public void setElevatedAndRotated(boolean bElevatedAndRotated)
    {
        Float fElevationAngle = m_fDefaultElevationAngle;
        Float fRotationAngle = m_fDefaultRotationAngle;

        if (bElevatedAndRotated == false)
        {
            fElevationAngle = new Float(0.0);
            fRotationAngle = new Float(0.0);
        }

        m_chart.setElevationAngle(fElevationAngle.floatValue());
        m_chart.setRotationAngle(fRotationAngle.floatValue());
    }

    public void add(MetricValue value)
    {
        MetricsBarSeries series = createSeries(value);

        if (value.getCurrency().equals(new Long(0)) == false)
        {
            update();
        }
    }

    public void remove(MetricValue value)
    {
        MetricsBarSeries series = getSeries(value);
        m_chart.removeObject(series.getSeries());
        ((LabelAxisMapper)m_xMapper).unregisterKey(value.getMetric());
        m_colSeries.remove(series);
        series.close();
    }

    @Override
    public void update()
    {
        synchronized(m_colSeries)
        {
            if (m_colSeries != null)
            {
                Iterator iter = m_colSeries.iterator();

                while (iter.hasNext())
                {
                    MetricsBarSeries series = (MetricsBarSeries) iter.next();
                    series.update();
                    chart.invalidate();
                }
            }
        }
    }

    @Override
    public void close()
    {
        if (m_chart != null)
        {
            m_chart.removePlugin(m_tipMgr);
            m_chart.removePlugin(m_rotator);
            m_tipMgr = null;
            m_rotator = null;

            m_chart.cleanup();
            m_chart = null;
        }

        if (m_colSeries != null)
        {
            Iterator iter = m_colSeries.iterator();

            while (iter.hasNext())
            {
                MetricsBarSeries series = (MetricsBarSeries) iter.next();
                series.close();
            }

            m_colSeries.clear();
            m_colSeries = null;
        }

        removeAll();

        // dereference everything...
        m_grid.cleanup();
        m_grid = null;

        m_fDefaultElevationAngle = null;
        m_fDefaultRotationAngle = null;
        m_layouter = null;

        m_axisX.cleanup();
        m_axisX = null;

        m_axisY.cleanup();
        m_axisY = null;
        m_planeMapper = null;

        super.close();
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////
    // Private Methods
    //

    private MetricsBarSeries createSeries(MetricValue value)
    {
        MetricsBarSeries series = new MetricsBarSeries(value, m_axisX, m_axisY, m_yMapper,
            value.getColorId());

        m_colSeries.add(series);
        m_chart.addObject(series.getSeries());
        ((LabelAxisMapper)m_xMapper).registerKey(value.getMetric());

        return series;
    }

    private MetricsBarSeries getSeries(MetricValue value)
    {
        MetricsBarSeries series = null;

        synchronized(m_colSeries)
        {
            Iterator iter = m_colSeries.iterator();

            while (iter.hasNext())
            {
                MetricsBarSeries seriesTemp = (MetricsBarSeries)iter.next();

                if (seriesTemp.getMetric().equals(value))
                {
                    series = seriesTemp;
                    break;
                }
            }
        }

        return series;
    }

    private void setupChart()
    {
        m_fDefaultElevationAngle = new Float(m_chart.getElevationAngle());
        m_fDefaultRotationAngle = new Float(m_chart.getRotationAngle());

        m_chart.setObjects(createChartObjects());
        m_chart.setYAxis(m_axisY);
        m_chart.setXAxis(m_axisX);
        m_chart.addPlugin(m_rotator = new ChartRotator());
        m_chart.addPlugin(m_tipMgr = new ToolTipManager());
        setElevatedAndRotated(false);
    }

    private void createLayout()
    {
        m_axisX = new Axis2DX(m_xMapper = new LabelAxisMapper());
        m_axisX.setLabelLayout(m_layouter = new MultiLineLabelLayouter());
        m_axisX.setMinRowCount(2);
        m_axisX.setMaxRowCount(2);

        m_axisY = new Axis2DY(m_yMapper = new MathAxisMapper(0, 1));
        m_axisY.setTitle("Value");

        m_grid = new Grid(m_planeMapper = new PlaneMapper2D(), m_xMapper, m_yMapper);
    }

    private ChartObject[] createChartObjects()
    {
        ChartObject[] chartObjects = new ChartObject[3];

        int nIdx = 0;
        chartObjects[nIdx++] = m_grid;
        chartObjects[nIdx++] = m_axisX;
        chartObjects[nIdx++] = m_axisY;

        return chartObjects;
    }
}