package com.sonicsw.ma.gui.runtime.metrics;

import java.awt.BasicStroke;
import java.awt.Color;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Vector;

import lt.monarch.chart.AbstractChartDataModel;
import lt.monarch.chart.AxisMapper;
import lt.monarch.chart.ChartDataModel2D;
import lt.monarch.chart.ChartDataModelAnnotations;
import lt.monarch.chart.chart2D.Axis2D;
import lt.monarch.chart.chart2D.Axis2DX;
import lt.monarch.chart.chart2D.ConnectedSeries;
import lt.monarch.chart.mapper.ScrollableMathAxisMapper;
import lt.monarch.chart.marker.DiamondMarker;

import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.util.Helper;

/**
 * MetricsChartSeries
 * <p>Title: Sonic Management Console</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class MetricsChartSeries
{
    private static final float METRIC_LINE_STROKE = 0.75f;

    private MetricValue m_value = null;
    private ConnectedSeries m_series = null;
    private MetricsSeriesModel m_model = null;
    private DiamondMarker m_marker = null;
    private BasicStroke m_stroke = null;
    private Number m_objHighValue = null;
    private Number m_objLowValue = null;
    protected long m_lChartStartTime = 0;

    /**
     * MetricsChartSeries
     * @param value
     * @param xAxis
     * @param yAxis
     * @param xMapper
     * @param color
     */
    public MetricsChartSeries(MetricValue value, Axis2DX xAxis, Axis2D yAxis,
                              AxisMapper xMapper, Color color)
    {
        m_value = value;

        // create the series for this item...
        m_series = new ConnectedSeries(
            m_model = new MetricsSeriesModel(xMapper), xAxis, yAxis);

        // set the series' properties...
        m_series.style.setStroke(m_stroke = new BasicStroke(METRIC_LINE_STROKE));
        m_series.showMarkers(m_marker = new DiamondMarker());
        m_series.setLineColor(color);
        m_series.setName(getName());
    }

    /**
     * close
     */
    public void close()
    {
        m_model.close();
        m_model = null;
        m_series.cleanup();
        m_series = null;
        m_stroke = null;
        m_marker = null;
    }

    public void fireUpdateEvents()
    {
        m_model.fireUpdateEvents();
    }

    /**
     * ConnectedSeries
     * @return the ConnectedSeries
     */
    public final ConnectedSeries getSeries() { return m_series; }

    /**
     * getModel
     * @return the AbstractChartDataModel
     */
    public final AbstractChartDataModel getModel() { return m_model; }

    /**
     * getMetric
     * @return the MetricValue
     */
    public final MetricValue getMetric() { return m_value; }

    /**
     * setChartStartTime
     * @param lChartStartTime
     */
    public void setChartStartTime(long lChartStartTime) { m_lChartStartTime = lChartStartTime; }

    /**
     * update
     */
    public void update()
    {
        Number numValue = m_value.getValue();

        if (numValue != null)
        {
            // adjust the high/low, if necessary...
            checkHighYValue(numValue);
            checkLowYValue(numValue);

            // update the model...
            m_model.update();
        }
    }

    /**
     * getName
     * @return the series' name.
     */
    public final String getName()
    {
        return m_value.getMetric();
    }

    /**
     * getHighYValue
     * @return the high Y-value for this series.
     */
    public Number getHighYValue() { return m_objHighValue; }

    /**
     * getLowYValue
     * @return the low Y-value for this series.
     */
    public Number getLowYValue() { return m_objLowValue; }

    /**
     * checkHighYValue
     * @param objValue
     */
    protected void checkHighYValue(Number objValue)
    {
        if (m_objHighValue == null || objValue.doubleValue() > m_objHighValue.doubleValue())
        {
            // adjust the high value for this series...
            m_objHighValue = objValue;
        }
    }

    /**
     * checkLowYValue
     * @param objValue
     */
    protected void checkLowYValue(Number objValue)
    {
        if (m_objLowValue == null || objValue.doubleValue() < m_objLowValue.doubleValue())
        {
            // adjust the low value for this series...
            m_objLowValue = objValue;
        }
    }

    /**
     * MetricsSeriesModel
     * <p>Title: Sonic Management Console</p>
     * <p>Copyright: Copyright (c) 2003</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Jeffrey S. Pace
     * @version 1.0
     */
    private class MetricsSeriesModel
        extends AbstractChartDataModel
        implements ChartDataModel2D, ChartDataModelAnnotations
    {
        private SimpleDateFormat m_dateFormatter = new SimpleDateFormat("h:mm:ss a");

        private AxisMapper m_xMapper = null;
        private Vector m_dataX = new Vector();
        private Vector m_dataY = new Vector();
        private NumberFormat m_formatter = NumberFormat.getInstance();
        private int m_nMaxPoints = -1;
        private int m_nPollInterval = -1;

        /**
         * MetricsSeriesModel
         * @param xMapper
         */
        private MetricsSeriesModel(AxisMapper xMapper)
        {
            m_xMapper = xMapper;

            // get the max number of points we can display...
            m_nMaxPoints = PreferenceManager.getInstance().getInt(JPreferencesDialog.METRIC_PREFS,
                JPreferencesDialog.MAX_PLOT_POINTS,
                JPreferencesDialog.DEFAULT_MAX_PLOT_POINTS);

            // get the poll frequency...
            m_nPollInterval = PreferenceManager.getInstance().getInt("preferences.metrics",
                JPreferencesDialog.POLL_FREQUENCY,
                JPreferencesDialog.DEFAULT_POLL_FREQUENCY);
        }

        /**
         * close
         */
        public void close()
        {
            m_dataX.clear();
            m_dataX = null;

            m_dataY.clear();
            m_dataY = null;

            m_dateFormatter = null;
        }

        public void fireUpdateEvents()
        {
            fireModelChanged();
        }

        /**
         * getValueCount
         * @return
         */
        @Override
        public int getValueCount()
        {
            return m_dataX.size();
        }

        /**
         * getXValueAt
         * @param pos
         * @return
         */
        @Override
        public Object getXValueAt(int pos)
        {
            return new Integer(pos);
        }

        /**
         * getYValueAt
         * @param pos
         * @return
         */
        @Override
        public Object getYValueAt(int pos)
        {
            if (m_dataY == null || m_dataY.isEmpty())
            {
                // no data at that position...
                return null;
            }

            return m_dataY.get(pos);
        }

        /**
         * update
         */
        public void update()
        {
            // calculate the poll interval...
            int nPollLocation = getPollLocation();

            if (nPollLocation != -1)
            {
                // insert the data at that position...
                addToVector(m_dataX, m_value.getCurrency(), nPollLocation);
                addToVector(m_dataY, m_value.getValue(), nPollLocation);

                // update the X-range and fire off the event...
                updateRange();
                fireModelChanged();
            }
        }

        /**
         * getDescriptionAt
         * @param pos
         * @return
         */
        @Override
        public String getDescriptionAt(int pos)
        {
            // all for the tooltip...
            StringBuffer sb = new StringBuffer();

            sb.append("Metric: " + m_value.getMetricDescription() + "\r\n");
            sb.append("Value: " + m_formatter.format(getYValueAt(pos)) + "\r\n");
            sb.append("Time: " + m_dateFormatter.format(m_dataX.get(pos)) + "\r\n");
            sb.append("Poll Interval: " + pos);

            return sb.toString();
        }

        /**
         * getLabelAt
         * @param pos
         * @return
         */
        @Override
        public String getLabelAt(int pos)
        {
            return null;
        }

        /**
         * addToVector
         * @param v
         * @param objValue
         * @param nIndex
         */
        protected void addToVector(Vector v, Object objValue, int nIndex)
        {
            if (nIndex > v.size())
            {
                // set the vector to the size we need...
                v.setSize(nIndex);
            }

            Object objExistingValue = null;
            try
            {
                // attempt to get an existing value from the index...
                objExistingValue = v.get(nIndex);
            }
            catch (ArrayIndexOutOfBoundsException e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            }

            if (objExistingValue == null)
            {
                // only add the value if a value doesn't exist...
                v.add(nIndex, objValue);
            }
        }

        /**
         * updateRange
         */
        protected void updateRange()
        {
            if (!m_dataX.isEmpty())
            {
                // we have values on the x-axis, determine the current view properties...
                double dxMin = ((ScrollableMathAxisMapper)m_xMapper).getRange().getMinimum();
                double dxMax = ((ScrollableMathAxisMapper)m_xMapper).getRange().getMaximum();
                int nCount = m_dataX.size();

                if (nCount >= dxMax)
                {
                    // need to increase the maximum...
                    ((ScrollableMathAxisMapper)m_xMapper).setRange(dxMin, nCount);
                    dxMax = nCount;
                }

                if (m_nMaxPoints != -1 && m_dataX.size() > m_nMaxPoints)
                {
                    // we have more points then we should, start rotating...
                    double dNewTail = m_dataX.size() - m_nMaxPoints;

                    ((ScrollableMathAxisMapper)m_xMapper).setRange(dNewTail, dxMax);
                    m_dataX.setElementAt(null, new Double(dNewTail).intValue());
                }
            }
        }

        /**
         * getPollLocation
         * @return
         */
        private int getPollLocation()
        {
            int nPollLocation = 0;

            Long lTimeZero = new Long(m_lChartStartTime);
            Long lTimeCurrentValue = m_value.getCurrency();
            Long lTimeDelta = new Long(lTimeCurrentValue.longValue() - lTimeZero.longValue());

            if (lTimeDelta.longValue() > 0)
            {
                Long lPollInterval = new Long(1000 * m_nPollInterval);
                double dInterval = lTimeDelta.doubleValue() / lPollInterval.doubleValue();

                Helper.logDebugMessage("Poll Increment =" + dInterval);
                nPollLocation = new Double(Math.round(dInterval)).intValue();
            }

            return nPollLocation;
        }
    }
}