package com.sonicsw.ma.gui.runtime.metrics;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

import com.sonicsw.ma.gui.MgmtConsole;

import lt.monarch.chart.ChartObject;
import lt.monarch.chart.Grid;
import lt.monarch.chart.chart2D.BoxZoomer;
import lt.monarch.chart.chart2D.Chart2D;
import lt.monarch.chart.chart2D.PlaneMapper2D;
import lt.monarch.chart.chart2D.ScrollableAxis2DX;
import lt.monarch.chart.chart2D.ScrollableAxis2DY;
import lt.monarch.chart.mapper.ScrollableMathAxisMapper;
import lt.monarch.chart.view.ToolTipManager;

/**
 * MetricsLineChart
 * <p>Title: Sonic Management Console</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software Corporation</p>
 * @author Jeffrey S. Pace
 * @version 1.0
 */
public class MetricsLineChart
    extends MetricsChart
{
    public static final int MIN_VALUE = -1;
    public static final int MAX_VALUE = 1;

    private Chart2D m_chart = null;
    private ScrollableAxis2DX m_axisX = null;
    private ScrollableAxis2DY m_axisYL = null;
    private ScrollableAxis2DY m_axisYR = null;
    private ScrollableMathAxisMapper m_yMapper = null;
    private ScrollableMathAxisMapper m_xMapper = null;
    private Grid m_grid = null;
    private boolean m_bInitialized = false;
    private Collection m_colSeries = Collections.synchronizedList(new ArrayList());
    private MarkerSelector m_selector = null;
    private ToolTipManager m_tipMgr = null;
    private BoxZoomer m_zoomer = null;
    private PlaneMapper2D m_planeMapper = null;

    private long m_lChartStartTime = 0L;
    private RangeManagerInterface m_yRangeManager = null;
    private RangeManagerInterface m_xRangeManager = null;

    /**
     * MetricsLineChart
     * @param colMetrics
     */
    public MetricsLineChart(Collection colMetrics)
    {
        super(new Chart2D());

        m_colSeries = colMetrics;
    }

    /**
     * init
     */
    @Override
    public void init()
    {
        try
        {
            if (m_bInitialized == false)
            {
                // create chart layout...
                m_chart = (Chart2D)super.getViewChart();
                createLayout();

                Collection colIncoming = m_colSeries;
                m_colSeries = Collections.synchronizedList(new ArrayList());
                Iterator iter = colIncoming.iterator();

                while (iter.hasNext())
                {
                    // this value is to be charted...
                    createSeries((MetricValue)iter.next());
                }

                setupChart();
                m_bInitialized = true;
            }

            super.init();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
        }
    }

    /**
     * close
     */
    @Override
    public void close()
    {
        if (m_chart != null)
        {
            // cleanup chart atributes...
            m_chart.removePlugin(m_tipMgr);
            m_chart.removePlugin(m_selector);
            m_chart.removePlugin(m_zoomer);
            m_chart.cleanup();
            m_selector = null;
            m_tipMgr = null;
            m_zoomer = null;
            m_chart = null;
        }

        if (m_colSeries != null)
        {
            // cleanup series' information...
            synchronized(m_colSeries)
            {
                Iterator iter = m_colSeries.iterator();

                while (iter.hasNext())
                {
                    MetricsChartSeries series = (MetricsChartSeries)iter.next();
                    series.close();
                }

                m_colSeries.clear();
            }
            m_colSeries = null;
        }

        removeAll();

        // dereference everything...
        m_grid.cleanup();
        m_grid = null;

        m_yMapper = null;
        m_xMapper = null;

        m_axisX.cleanup();
        m_axisX = null;

        m_axisYL.cleanup();
        m_axisYL = null;

        m_axisYR.cleanup();
        m_axisYR = null;

        m_xRangeManager = null;
        m_yRangeManager = null;
        m_planeMapper = null;

        super.close();
    }

    /**
     * add a metric to the chart
     * @param value
     */
    public void add(MetricValue value)
    {
        MetricsChartSeries series = createSeries(value);

        if ((value != null) && (value.getCurrency().longValue() != 0) && (value.getCurrency().longValue() != -1))
        {
            // has a valid value, update the chart...
            update(value);
        }
    }

    /**
     * remove
     * @param value
     */
    public void remove(MetricValue value)
    {
        MetricsChartSeries seriesToRemove = getSeries(value);

        // remove and close down the series...
        m_chart.removeObject(seriesToRemove.getSeries());
        m_colSeries.remove(seriesToRemove);
        seriesToRemove.close();

        // reset the Y-range...
        m_yMapper.setRange(MIN_VALUE, MIN_VALUE + 1);
        m_yRangeManager.update();
    }

    /**
     * update
     * @param value
     */
    public void update(final MetricValue value)
    {
        if ((value != null) && (value.getCurrency().longValue() != 0) && (value.getCurrency().longValue() != -1))
        {
            MetricsChartSeries series = getSeries(value);

            if (series != null)
            {
                // found the series that maintains the updated metric...
                if (m_lChartStartTime == 0L ||
                    m_lChartStartTime > value.getCurrency().longValue())
                {
                    // update the start time...
                    m_lChartStartTime = value.getCurrency().longValue();
                }
                series.setChartStartTime(m_lChartStartTime);

                // update the series...
                series.update();

                // update the ranges...
                m_xRangeManager.update();
                m_yRangeManager.update();
                m_chart.invalidate();
            }
        }
    }

    /**
     * createSeries
     * @param value
     * @return
     */
    private MetricsChartSeries createSeries(MetricValue value)
    {
        // create a series for the metric...
        MetricsChartSeries series = new MetricsChartSeries(value, m_axisX,
            m_axisYL, m_xMapper, value.getColorId());

        // add it to the chart...
        m_colSeries.add(series);
        m_chart.addObject(series.getSeries());

        return series;
    }

    /**
     * getSeries locates the series for the specified metric
     * @param value
     * @return
     */
    private MetricsChartSeries getSeries(MetricValue value)
    {
        MetricsChartSeries series = null;

        synchronized(m_colSeries)
        {
            Iterator iter = m_colSeries.iterator();

            while (iter.hasNext())
            {
                MetricsChartSeries seriesTemp = (MetricsChartSeries)iter.next();

                if (seriesTemp.getMetric().equals(value))
                {
                    // found the series for the metric...
                    series = seriesTemp;
                    break;
                }
            }
        }

        return series;
    }

    /**
     * setupChart sets up the charts properties
     */
    private void setupChart()
    {
        m_chart.setObjects(createChartObjects());
        m_chart.setLeftYAxis(m_axisYL);
        m_chart.setRightYAxis(m_axisYR);
        m_chart.setXAxis(m_axisX);
        m_chart.addPlugin(m_tipMgr = new ToolTipManager());
        m_chart.addPlugin(m_selector = new MarkerSelector());

        m_zoomer = new BoxZoomer (m_xMapper, m_yMapper);
        m_zoomer.enableDigitization (m_axisX, null);
        m_chart.addPlugin (m_zoomer);
    }

    /**
     * createLayout
     */
    private void createLayout()
    {
        m_yMapper = new ScrollableMathAxisMapper(MIN_VALUE, 1);
        m_xMapper = new ScrollableMathAxisMapper(0, 1);
        m_xMapper.setVisibleRange(0, 1);

        m_axisX = new ScrollableAxis2DX(m_xMapper);
        m_axisX.setTitle("Poll Interval");

        m_axisYL = new ScrollableAxis2DY(m_yMapper);
        m_axisYL.setTitle("Value");

        m_axisYR = new ScrollableAxis2DY(m_yMapper);
        m_axisYR.setTitle("Value");

        m_grid = new Grid(m_planeMapper = new PlaneMapper2D(), m_xMapper, m_yMapper);

        m_xRangeManager = new XRangeManager(m_xMapper);
        m_yRangeManager = new YRangeManager(m_yMapper);
    }

    /**
     * createChartObjects
     * @return
     */
    private ChartObject[] createChartObjects()
    {
        ChartObject[] chartObjects = new ChartObject[4];

        int nIdx = 0;
        chartObjects[nIdx++] = m_grid;
        chartObjects[nIdx++] = m_axisX;
        chartObjects[nIdx++] = m_axisYL;
        chartObjects[nIdx++] = m_axisYR;

        return chartObjects;
    }

    /**
     * Range Management
     * <p>Title: Sonic Management Console</p>
     * <p>Copyright: Copyright (c) 2003</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Jeffrey S. Pace
     * @version 1.0
     */
    private interface RangeManagerInterface
    {
        Number getHighValue();
        Number getLowValue();
        void update();
    }

    /**
     * AbstractRangeManager
     * <p>Title: Sonic Management Console</p>
     * <p>Copyright: Copyright (c) 2003</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Jeffrey S. Pace
     * @version 1.0
     */
    private abstract class AbstractRangeManager
        implements RangeManagerInterface
    {
        public static final float BUFFER = 0.05f;

        @Override
        public abstract Number getHighValue();
        @Override
        public abstract Number getLowValue();

        @Override
        public void update()
        {
            fireUpdateEvents();
        }

        /**
         * addBuffer
         * @param dValue
         * @return
         */
        protected double addBuffer(double dValue)
        {
            return dValue + (dValue * BUFFER);
        }

        /**
         * subBuffer
         * @param dValue
         * @return
         */
        protected double subBuffer(double dValue)
        {
            return dValue - (dValue * BUFFER);
        }

        private void fireUpdateEvents()
        {
            if (m_colSeries != null)
            {
                // cleanup series' information...
                synchronized(m_colSeries)
                {
                    Iterator iter = m_colSeries.iterator();

                    while (iter.hasNext())
                    {
                        MetricsChartSeries series = (MetricsChartSeries)iter.next();
                        series.fireUpdateEvents();
                    }
                }
            }
        }
    }

    /**
     * XRangeManager
     * <p>Title: Sonic Management Console</p>
     * <p>Copyright: Copyright (c) 2003</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author Jeffrey S. Pace
     * @version 1.0
     */
    private class XRangeManager
        extends AbstractRangeManager
    {
        private ScrollableMathAxisMapper m_xMapper;

        /**
         * XRangeManager
         * @param xMapper
         */
        private XRangeManager(ScrollableMathAxisMapper xMapper)
        {
            m_xMapper = xMapper;
        }

        /**
         * getHighValue
         * @return
         */
        @Override
        public Number getHighValue()
        {
            return new Double(m_xMapper.getRange().getMaximum());
        }

        /**
         * getLowValue
         * @return
         */
        @Override
        public Number getLowValue()
        {
            return new Double(m_xMapper.getRange().getMinimum());
        }

        /**
         * update
         */
        @Override
        public void update()
        {
           m_xMapper.setRange(getLowValue().doubleValue(), getHighValue().doubleValue());

           super.update();
        }
    }

    /**
     * YRangeManager
     * <p>Title: Sonic Management Console</p>
     * <p>Copyright: Copyright (c) 2003</p>
     * <p>Company: Sonic Software Corporation</p>
     * @author not attributable
     * @version 1.0
     */
    private class YRangeManager
        extends AbstractRangeManager
    {
        private ScrollableMathAxisMapper m_yMapper;

        /**
         * YRangeManager
         * @param yMapper
         */
        private YRangeManager(ScrollableMathAxisMapper yMapper)
        {
            m_yMapper = yMapper;
        }

        /**
         * getHighValue
         * @return
         */
        @Override
        public Number getHighValue()
        {
            Number nHighValue = null;

            synchronized (m_colSeries)
            {
                if (m_colSeries != null)
                {
                    Iterator iter = m_colSeries.iterator();

                    while (iter.hasNext())
                    {
                        MetricsChartSeries series = (MetricsChartSeries)iter.next();
                        Number nSeriesHighValue = series.getHighYValue();
                        if (nSeriesHighValue != null && (nHighValue == null || nSeriesHighValue.doubleValue() > nHighValue.doubleValue()))
                        {
                            nHighValue = nSeriesHighValue;
                        }
                    }
                }
            }


            return (nHighValue != null) ? nHighValue : new Double(MAX_VALUE);
        }

        /**
         * getLowValue
         * @return
         */
        @Override
        public Number getLowValue()
        {
            Number nLowValue = null;

            synchronized (m_colSeries)
            {
                if (m_colSeries != null)
                {
                    Iterator iter = m_colSeries.iterator();

                    while (iter.hasNext())
                    {
                        MetricsChartSeries series = (MetricsChartSeries)iter.next();
                        Number nSeriesLowValue = series.getLowYValue();

                        if (nSeriesLowValue != null && (nLowValue == null || nSeriesLowValue.doubleValue() < nLowValue.doubleValue()))
                        {
                            nLowValue = nSeriesLowValue;
                        }
                    }
                }
            }

            return (nLowValue != null) ? nLowValue :  new Double(MIN_VALUE);
        }

        /**
         * update
         */
        @Override
        public void update()
        {
            updateYRange(getLowValue(), getHighValue());

            super.update();
        }

        /**
         * updateYRange
         * @param nLow
         * @param nHigh
         */
        private void updateYRange(Number nLow, Number nHigh)
        {
            // we have values on the y-axis, determine the current view properties...
            double dyMax = m_yMapper.getRange().getMaximum();
            double dyMin = m_yMapper.getRange().getMinimum();

            // create a suitable buffer around the values in the range...
            double dAdjustedHighValue = addBuffer(nHigh.doubleValue());
            double dAdjustedLowValue = subBuffer(nLow.doubleValue());

            if (dAdjustedHighValue > dyMax)
            {
                // need to increase the maximum...
                m_yMapper.setRange(dyMin, dAdjustedHighValue);
                dyMax = dAdjustedHighValue;
            }

            if (dAdjustedLowValue < dyMin || dyMin == MetricsLineChart.MIN_VALUE)
            {
                // need to increase the minimum...
                m_yMapper.setRange(dAdjustedLowValue, dyMax);
                dyMin = dAdjustedLowValue;
            }
        }
    }
}