package com.sonicsw.ma.gui.runtime.metrics;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.ObjectName;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.runtime.metrics.model.AbstractMetricsModel;
import com.sonicsw.ma.gui.runtime.metrics.model.IMetricWatcher;
import com.sonicsw.ma.gui.runtime.metrics.model.IValueUpdatedEvent;
import com.sonicsw.ma.gui.runtime.metrics.model.MetricsModel;
import com.sonicsw.ma.gui.runtime.metrics.model.ValueUpdatedEvent;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.ModelListTableModel;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ColorSeries;
import com.sonicsw.ma.gui.util.ExtendedImageIcon;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;

import com.sonicsw.mf.common.metrics.IMetricIdentity;

/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: Sonic Software Corporation </p>
 * @author not attributable
 * @version 1.0
 */
public class MetricsWatchPanel extends JSplitPane implements IMetricWatcher
{
    protected static Color[] sm_listColors =
    {
        Color.red,
        Color.green,
        Color.blue,
        Color.orange,
        Color.darkGray,
        Color.cyan,
        Color.pink,
        Color.yellow,
        Color.magenta,
        Color.black
    };

    protected short              m_scope;
    protected MetricsTable       m_table;
    protected MetricsLineChart   m_chartLine;
    protected MetricsBarChart    m_chartBar;
    protected HashMap            m_subscriptions = new HashMap();
    protected HashMap            m_modelMap      = new HashMap();
    protected RemoveWatchAction  m_removeAction  = new RemoveWatchAction();
    protected RefreshWatchAction m_refreshAction = new RefreshWatchAction();
    protected int                m_pollFrequency;
    protected int                m_nMaxPoints;
    protected ColorSeries        m_colors        = new ColorSeries(sm_listColors);
    private JTabbedPane          m_paneTabs;

    /**
     * Constructor
     */
    public MetricsWatchPanel(short scope)
    {
        super(JSplitPane.VERTICAL_SPLIT);
        m_scope = scope;

        prepareMetricsWatchPanel(scope);
    }

    private void prepareMetricsWatchPanel(short scope) {
        createGui(scope);
    }
    
    protected void createGui(short scope)
    {
        setOneTouchExpandable(true);

        m_table = createTable(scope);
        m_chartLine = new MetricsLineChart(new ArrayList());
        m_chartLine.init();
        m_chartLine.setBackground(getBackground());

        m_chartBar = new MetricsBarChart(new ArrayList());
        m_chartBar.init();
        m_chartBar.setBackground(getBackground());

        m_paneTabs = new JTabbedPane();
        m_paneTabs.addTab("Latest", ResourceManager.getIcon(getClass(), "viewbarchart"), m_chartBar);
        m_paneTabs.addTab("History", ResourceManager.getIcon(getClass(), "viewlinechart"), m_chartLine);
        m_paneTabs.setBorder(BorderFactory.createEmptyBorder(5,5,0,5));

        add(m_paneTabs, JSplitPane.TOP);
        add(m_table, JSplitPane.BOTTOM);

        // add the popup menu
        JPopupMenu popup = new JPopupMenu();
        JComponent[] popupItem = getSelectedMenuItems(IPlugin.PLUGIN_TYPE);
        for (int i = 0; i < popupItem.length; i++)
        {
            popup.add(popupItem[i]);
        }
        m_table.getTable().addMouseListener(new PopupMenuShower(m_table.getTable(), popup));
        m_table.getTable().getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(ListSelectionEvent evt)
            {
                if (evt.getValueIsAdjusting())
                {
                    return;
                }

                m_refreshAction.setEnabled(m_table.getTable().getSelectedRowCount() > 0);
                m_removeAction.setEnabled(m_table.getTable().getSelectedRowCount() > 0);
            }
        });
    }

    protected MetricsTable createTable(short scope)
    {
        return new MetricsTable(this, scope);
    }

    @Override
    public void addNotify()
    {
        setDividerLocation(.70);

        super.addNotify();
    }

    protected void maInitialize()
    {
        // set initial values from preferences
        PreferenceManager pManager = PreferenceManager.getInstance();

        m_pollFrequency = pManager.getInt(JPreferencesDialog.METRIC_PREFS,
            JPreferencesDialog.POLL_FREQUENCY,
            JPreferencesDialog.DEFAULT_POLL_FREQUENCY);

        m_nMaxPoints = pManager.getInt(JPreferencesDialog.METRIC_PREFS,
            JPreferencesDialog.MAX_PLOT_POINTS,
            JPreferencesDialog.DEFAULT_MAX_PLOT_POINTS);
    }

    protected void maCleanup()
    {
        // remove value listener subscriptions and cleanup maps
        Iterator i = m_subscriptions.entrySet().iterator();

        while (i.hasNext())
        {
            Map.Entry entry = (Map.Entry) i.next();
            ObjectName component = (ObjectName) entry.getKey();
            HashMap map = (HashMap) entry.getValue();
            MetricsModel model = (MetricsModel) ((Object[]) m_modelMap.get(component))[0];
            Iterator metricIds = map.keySet().iterator();

            while (metricIds.hasNext())
            {
                model.removeValueListener((IMetricIdentity) metricIds.next(), this);
            }

            map.clear();
        }
        m_subscriptions.clear();
        m_subscriptions = null;

        m_modelMap.clear();
        m_modelMap = null;

        m_chartLine.close();
        m_chartLine = null;

        m_chartBar.close();
        m_chartBar = null;

        m_paneTabs.removeAll();
        m_paneTabs = null;

        m_removeAction  = null;
        m_refreshAction = null;

        if (m_table != null)
        {
            m_table.maCleanup();
            m_table = null;
        }
    }

    public int getMetricCount()
    {
        return m_table.getMetricCount();
    }

    public JComponent[] getSelectedMenuItems(int type)
    {
        JComponent[] menuItems = null;

        if (type == IPlugin.PLUGIN_TYPE)
        {
            menuItems = new JComponent[]
            {
                        new JBasicMenuItem(new OptionsAction()),
                        new JSeparator(),
                        new JBasicMenuItem(m_removeAction),
            };
        }
        return menuItems;
    }

    private MetricValue[] getSelectedRowValues()
    {
        Collection colValues = m_table.getSelectedRowValues();

        return (MetricValue[]) colValues.toArray(new MetricValue[colValues.size()]);
    }

    //-------------------------------------------------------------------------
    //
    // IMetricWatcher implementation
    //
    //-------------------------------------------------------------------------

    @Override
    public void addWatch(IMetricIdentity metricId,
                         AbstractMetricsModel metricsModel)
    {
        ObjectName componentName = metricsModel.getComponentName();

        HashMap metricValues = (HashMap) m_subscriptions.get(componentName);
        if (metricValues == null)
        {
            metricValues = new HashMap();
            m_subscriptions.put(componentName, metricValues);
        }

        MetricValue metricValue = (MetricValue) metricValues.get(metricId);
        if (metricValue == null) // new watch
        {
            Object[] modelUsage = (Object[]) m_modelMap.get(componentName);
            if (modelUsage == null)
            {
                modelUsage = new Object[]
                             {metricsModel, new int[]
                             {0}
                };
                m_modelMap.put(componentName, modelUsage);
            }
            ((int[]) modelUsage[1])[0]++;

            metricValue = new MetricValue(componentName, metricId);
            metricValue.setColorId(m_colors.getNextColor());
            metricValues.put(metricId, metricValue);

            ((IModelTableModel) m_table.getModel()).addRow(metricValue);
            m_chartLine.add(metricValue);
            m_chartBar.add(metricValue);

            metricsModel.addValueListener(metricId, this);
        }
    }

    @Override
    public void valueUpdated(IValueUpdatedEvent event)
    {
        ValueUpdatedEvent vue = (ValueUpdatedEvent) event;

        HashMap metricValues = (HashMap) m_subscriptions.get(vue.getSource());
        if (metricValues != null) // helps dealing with timing issues
        {
            MetricValue metricValue = (MetricValue) metricValues.get(vue.getMetricId());

            if (metricValue != null) // helps dealing with timing issues
            {
                // Find the table row which displays this metric value and
                // update it...in this way any table selection is preserved.
                //
                int rowIndex = ((IModelTableModel) m_table.getModel()).getIndexOf(metricValue);

                metricValue.setValue(vue.getValue());
                metricValue.setCurrency(vue.getCurrencyTimestamp());

                Helper.logDebugMessage(metricValue.toDebug());

                if (rowIndex >= 0)
                {
                    ((ModelListTableModel) m_table.getModel()).rowChanged(rowIndex);
                }

                m_chartLine.update(metricValue);
                m_chartBar.update();
            }
        }
    }

    //-------------------------------------------------------------------------

    @Override
    public int getPollFrequency()
    {
        return m_pollFrequency;
    }

    @Override
    public void setPollFrequency(int pollFrequency)
    {
        m_pollFrequency = pollFrequency;
    }

    public int getMaxPoints()
    {
        return m_pollFrequency;
    }

    public void setMaxPoints(int nMaxPoints)
    {
        m_nMaxPoints = nMaxPoints;
    }

    public void removeWatch(MetricValue[] metricValue)
    {
        for (int i = 0; i < metricValue.length; i++)
        {
            if (metricValue[i] == null)
            {
                continue;
            }

            ObjectName componentName = metricValue[i].getSourceName();
            IMetricIdentity metricId = metricValue[i].getMetricIdentity();
            HashMap metricValues = (HashMap) m_subscriptions.get(componentName);

            metricValues.remove(metricId);

            if (metricValues.isEmpty())
            {
                m_subscriptions.remove(componentName);
            }

            Object[] modelUsage = (Object[]) m_modelMap.get(componentName);
            MetricsModel metricsModel = (MetricsModel) modelUsage[0];
            ((int[]) modelUsage[1])[0]--;
            if (((int[]) modelUsage[1])[0] == 0)
            {
                m_modelMap.remove(componentName);
            }

            metricsModel.removeValueListener(metricId, this);
            m_chartLine.remove(metricValue[i]);
            m_chartBar.remove(metricValue[i]);

            ((IModelTableModel) m_table.getModel()).delete(metricValue[i]);
        }
    }

    public void refreshWatch(MetricValue[] metricValue)
    {
        for (int i = 0; i < metricValue.length; i++)
        {
            if (metricValue[i] == null)
            {
                continue;
            }

            ObjectName componentName = metricValue[i].getSourceName();
            IMetricIdentity metricId = metricValue[i].getMetricIdentity();
            Object[] modelUsage = (Object[]) m_modelMap.get(componentName);
            MetricsModel metricsModel = (MetricsModel) modelUsage[0];

            metricsModel.refreshMetricValues(new IMetricIdentity[]
                                             {metricId});
        }
    }

    @Override
    public short getWatchScope()
    {
        return m_scope;
    }

    //-------------------------------------------------------------------------
    //
    // Inner classes
    //
    //-------------------------------------------------------------------------

    public class RemoveWatchAction
        extends BasicGuiAction
    {
        public RemoveWatchAction()
        {
            super("MetricsWatchWindow.remove");
            putSmallIconValue();
        }
        
        private void putSmallIconValue() {
            putValue(Action.SMALL_ICON, new ExtendedImageIcon(
                                                              ResourceManager.getIcon(getClass(), "metric"),
                                                              ResourceManager.getIcon(getClass(), "deleteoverlay")));
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            removeWatch(getSelectedRowValues());
        }
    }

    public class RefreshWatchAction
        extends BasicGuiAction
    {
        public RefreshWatchAction()
        {
            super("MetricsWatchWindow.refresh");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            refreshWatch(getSelectedRowValues());
        }
    }

    public class OptionsAction
        extends BasicGuiAction
    {
        public OptionsAction()
        {
            super("MetricsWatchWindow.options");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                JMAFrame parent = (JMAFrame) SwingUtilities.getAncestorOfClass(JMAFrame.class, MetricsWatchPanel.this);
                JMetricsWatchOptionsDialog dialog = new JMetricsWatchOptionsDialog(parent);
                dialog.editInstance(null, dialog.new Model(MetricsWatchPanel.this), false);
                dialog.setVisible(true);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to edit metrics watch properties", e, true);
            }
        }
    }
}