package com.sonicsw.ma.gui.runtime.metrics.model;

import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Timer;
import java.util.TimerTask;

import javax.management.ObjectName;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.runtime.util.AbstractNode;
import com.sonicsw.ma.gui.runtime.util.AbstractParentNode;
import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mx.util.Sorter;

import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.metrics.IAggregateMetric;
import com.sonicsw.mf.common.metrics.IAggregateMetricsData;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;

public class AggregateMetricsModel extends AbstractMetricsModel
{
    private boolean m_hasInstanceMetrics = false;

    // metric polling
    private Timer m_pollTimer;
    private boolean m_pollLostComms = false;
    private IMetricIdentity[] m_pollParams = new IMetricIdentity[] { null };
    private String m_collectionConfigID = null;

    public AggregateMetricsModel(IRemoteMBeanServer connector, ObjectName componentName, IMetricInfo[] infos, String configID)
    throws Exception
    {
        super(connector, componentName, infos);

        m_collectionConfigID = configID;
        update(infos);
    }

    @Override
    public final void update(IMetricInfo[] info)
    {
        Sorter.sort(info, this, info.length);

        Enumeration en = m_treeRootNode.depthFirstEnumeration();

        while (en.hasMoreElements())
        {
            AbstractNode node = (AbstractNode)en.nextElement();

            if (node == m_treeRootNode)
            {
                continue;
            }

            String name = null;
            boolean found = false;

            if ((node instanceof Node) || (node instanceof InstanceParentNode))
            {
                name = ((IMetricInfo)node.getUserObject()).getMetricIdentity().getName();
            }
            else
            {
                name = ((IMetricIdentity)node.getUserObject()).getName();
            }

            if (node instanceof InstanceNode)
            {
                found  = inInstancesList((InstanceParentNode)node.getParent(), name);
            }
            else
            {
                found = inList(info, name);
            }

            if (!found) //remove node and associated listeners
            {
                if ( (node instanceof Node) || (node instanceof InstanceNode))
                {
                    AbstractParentNode parent = (AbstractParentNode)node.getParent();
                    removeValueListener(node);
                    parent.remove(node);
                    super.nodeChanged(parent);
                }

                if( (node instanceof ParentNode) || (node instanceof InstanceParentNode) )
                {
                    if ( node.getChildCount() == 0 )
                    {
                        AbstractParentNode parent = (AbstractParentNode)node.getParent();
                        parent.remove(node);
                        super.nodeChanged(parent);
                    }
                }

            }

        }

        constructTree(info);
        enableChildren();
    }

    private boolean inInstancesList(InstanceParentNode parent, String name)
    {
        IMetricIdentity identity = ((IMetricInfo)parent.getUserObject()).getMetricIdentity();
        String[] names = getInstanceMetricNames(identity);
        String identityName = identity.getName();
        for (int i = 0; i < names.length; i++)
        {
            String instanceName = identityName + "." + names[i];
            if (instanceName.equals(name))
            {
                return true;
            }
        }
        return false;
    }

    private boolean inList(IMetricInfo[] list, String metricName)
    {
        boolean found = false;

        for (int i = 0; i < list.length; i++)
        {
            if (list[i].getMetricIdentity().getName().startsWith(metricName))
            {
                found = true;
                break;
            }
        }
        return found;
    }

    public String getCollectionConfigID() { return m_collectionConfigID; }


    /**
     * Construct the metrics node tree from the metrics info and enabled metrics list
     * obtained from the component.
     */
    private void constructTree(IMetricInfo[] infos)
    {
        m_hasInstanceMetrics = false;
        for (int i = 0; i < infos.length; i++)
        {
            AbstractParentNode parent = m_treeRootNode;
            String[] nameTokens = infos[i].getMetricIdentity().getNameComponents();
            StringBuffer prefix = new StringBuffer();
            for (int j = 0; j < nameTokens.length; j++)
            {
                if (j == nameTokens.length - 1)
                {
                    boolean isInstanceMetric = infos[i].isInstanceMetric();
                    if (isInstanceMetric)
                    {
                        m_hasInstanceMetrics = true;
                    }

                    if (parent.getChild(nameTokens[j]) == null)
                    {
                        parent.add(isInstanceMetric ? (AbstractNode)new InstanceParentNode(infos[i], nameTokens[j])
                                                    : (AbstractNode)new Node(infos[i], nameTokens[j]));
                    }

                    if (isInstanceMetric)
                    {
                        InstanceParentNode iParent = (InstanceParentNode)parent.getChild(nameTokens[j]);
                        IMetricIdentity id = ((IMetricInfo)iParent.getUserObject()).getMetricIdentity();
                        String[] names = getInstanceMetricNames(id);
                        for (int k = 0 ; k < names.length; k++)
                        {
                            if (iParent.getChild(names[k]) == null)
                            {
                              IMetricIdentity i_id = MetricsFactory.createMetricIdentity(id, names[k]);
                              iParent.add(new InstanceNode(i_id, names[k]));
                            }
                        }
                    }
                }
                else
                {
                    if (j > 0)
                    {
                        prefix.append('.');
                    }
                    prefix.append(nameTokens[j]);
                    AbstractParentNode childNode = (AbstractParentNode)parent.getChild(nameTokens[j]);
                    if (childNode == null)
                    {
                        childNode = new ParentNode(MetricsFactory.createMetricIdentity(prefix.toString()), nameTokens[j]);
                        parent.add(childNode);
                    }
                    parent = childNode;
                }
            }
        }
    }

    private String[] getInstanceMetricNames(IMetricIdentity id)
    {
        try
        {
            String[] names = (String[])m_connector.invoke( m_componentName,
                                                           "getInstanceMetricNames",
                                                           new Object[] {m_collectionConfigID, id },
                                                           new String[] {String.class.getName(), IMetricIdentity.class.getName() }
                                                          );

            return names;
        }
        catch(Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }

    }

    //CM should enable all metric that scheduled to be monitored
    //So we just enabling all childrens.
    private void enableChildren()
    {
        Enumeration en = m_treeRootNode.depthFirstEnumeration();
        while (en.hasMoreElements())
        {
            AbstractNode node = (AbstractNode)en.nextElement();
            if ( (node instanceof Node) || (node instanceof InstanceNode) ) // look at parent enablement
            {
                node.setEnabled(true);
            }
        }
    }


    /**
     * Refreshes the tree with new metrics that have been added since the last refresh
     * or removing those that have been deleted.
     */
    @Override
    public void refreshTree()
    throws Exception
    {
        try
        {
            Object parameters[] = { m_collectionConfigID, };
            String signatures[] = { String.class.getName(), };
            IMetricInfo[] info  =
                (IMetricInfo[])m_connector.invoke( m_componentName,
                                                   "getMonitoredMetricsInfo",
                                                   parameters,
                                                   signatures );
            update(info);
            super.nodeStructureChanged(m_treeRootNode);
        }
        catch(Exception e)
        {
            throw(e);
        }
    }

    @Override
    public void refreshMetricValues(IMetricIdentity[] id)
    {
        if ((id != null) && (id.length > 0))
        {
            if (m_pollTimer == null)
            {
                m_pollTimer = new Timer(true);
            }

            m_pollTimer.schedule(new AggregateResultUpdater(id),
                                 new Date(System.currentTimeMillis())); // schedule straight away
        }
    }


    @Override
    public void addValueListener(IMetricIdentity metricId, IValueListener listener)
    {
        synchronized(m_valueListeners)
        {
            HashSet listeners = (HashSet)m_valueListeners.get(metricId);
            if (listeners == null)
            {
                listeners = new HashSet();
                m_valueListeners.put(metricId, listeners);
            }
            listeners.add(listener);

            // When a watch (value) listener is added for a given metric then
            // we require a repaint of any tree views onto this metric node,
            // i.e. adding a watch icon.
            //
            updateNode(metricId);

            if (m_pollTimer == null)
            {
                m_pollTimer = new Timer(true);
                m_pollTimer.schedule(new AggregateResultUpdater(), new Date(System.currentTimeMillis())); // schedule straight away
            }
        }
    }

    @Override
    public void removeValueListener(IMetricIdentity metricId, IValueListener listener)
    {
        synchronized(m_valueListeners)
        {
            HashSet listeners = (HashSet)m_valueListeners.get(metricId);
            if (listeners != null)
            {
                listeners.remove(listener);
                if (listeners.isEmpty())
                {
                    m_valueListeners.remove(metricId);
                }

                 // When a watch (value) listener is removed from the metric
                 // then we require a repaint of any views onto this metric
                 // node, i.e. possible removal of watch icon.
                 //
                 updateNode(metricId);
            }

            if ((m_valueListeners.isEmpty()) && (m_pollTimer != null))
            {
                m_pollTimer.cancel();
                m_pollTimer = null;
            }
        }
    }

    private void removeValueListener(AbstractNode node)
    {
        IMetricIdentity id = null;
        if (node instanceof InstanceNode)
        {
            id = (IMetricIdentity)node.getUserObject();
        }
        else
        {
            id = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        }

        if (!hasValueListeners(id))
        {
            return;
        }

        synchronized(m_valueListeners)
        {
            HashSet listeners = (HashSet)m_valueListeners.get(id);
            if (listeners != null)
            {
                //node has been deleted so we have to remove all listeners
                listeners.clear();
                if (listeners.isEmpty())
                {
                    m_valueListeners.remove(id);
                }
                 updateNode(id);
            }

            if ((m_valueListeners.isEmpty()) && (m_pollTimer != null))
            {
                m_pollTimer.cancel();
                m_pollTimer = null;
            }
        }
    }

    //Inner classes
    private class AggregateResultUpdater extends TimerTask
    {
        private IMetricIdentity[] m_ids;
        private boolean           m_oneoff;
        private int               m_backPeriod;

        public AggregateResultUpdater()
        {
            this(null, false);
        }

        public AggregateResultUpdater(IMetricIdentity[] ids)
        {
            this(ids, true);
        }

        protected AggregateResultUpdater(IMetricIdentity[] ids, boolean oneoff)
        {
            m_ids     = ids;
            m_oneoff = oneoff;
        }

        @Override
        public void run()
        {
            if (m_pollTimer == null)
            {
                return;
            }

            if (m_ids == null)
            {
                m_pollParams = (IMetricIdentity[])m_valueListeners.keySet().toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
            }
            else
            {
                m_pollParams = m_ids;
            }

            try
            {
                //back period interval set to the latest configured poll frequency.
                m_backPeriod = 1000 * PreferenceManager.getInstance().getInt("preferences.metrics", JPreferencesDialog.AGGREGATE_POLL_FREQUENCY, JPreferencesDialog.DEFAULT_AGGREGATE_POLL_FREQUENCY);
                IAggregateMetricsData metricsData =
                    (IAggregateMetricsData)m_connector.invoke( m_componentName, "getAggregateMetricsData",
                                                               new Object[] { m_collectionConfigID, m_pollParams,
                                                                              new Long(System.currentTimeMillis()), new Long(m_backPeriod) },
                                                               new String[] { String.class.getName(), IMetricIdentity[].class.getName(),
                                                                              Long.class.getName(), Long.class.getName()}
                                                              );
                if (m_pollTimer == null)
                {
                    return;
                }
                IAggregateMetric[]  metrics = metricsData.getAggregateMetrics();

                  for (int i = 0; i < metrics.length; i++)
                  {
                      HashSet listeners = null;
                      synchronized(m_valueListeners)
                      {
                          listeners = (HashSet)m_valueListeners.get(metrics[i].getMetricIdentity());

                      }
                      if (listeners != null)
                      {
                          Iterator iterator = listeners.iterator();
                          AggregateValueUpdatedEvent vue = new AggregateValueUpdatedEvent(m_componentName, metrics[i], AggregateMetricsModel.this.getCollectionConfigID());
                          while (m_pollTimer != null && iterator.hasNext())
                          {
                              if (m_pollTimer == null)
                            {
                                return;
                            }

                              ((IValueListener)iterator.next()).valueUpdated(vue);
                          }
                      }
                  }
                 AggregateMetricsModel.this.m_pollLostComms = false;

            }
            catch(Exception e)
            {
                e.printStackTrace();
                if (!((e instanceof InvokeTimeoutCommsException) ||
                      (e instanceof InvokeTimeoutException)))
                {
                    if (m_pollLostComms == false)
                    {
                        MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                 "Failed to collect metrics from " + m_componentName, e, false);
                        m_pollLostComms = true;
                    }
                }
            }

            if (!m_oneoff)
            {
                long pollInterval = 1000 * PreferenceManager.getInstance().getInt("preferences.metrics", JPreferencesDialog.AGGREGATE_POLL_FREQUENCY, JPreferencesDialog.DEFAULT_AGGREGATE_POLL_FREQUENCY);
                long nextTime = scheduledExecutionTime() + pollInterval;

                while (nextTime <= System.currentTimeMillis())
                {
                    nextTime += pollInterval;
                }

                if (m_pollTimer == null)
                {
                    return;
                }
                m_pollTimer.schedule(new AggregateResultUpdater(), new Date(nextTime)); // schedule straight away
            }
            m_ids = null;
        }
    }
}
