package com.sonicsw.ma.gui.runtime.metrics.model;

import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import javax.management.ObjectName;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.runtime.util.AbstractNode;
import com.sonicsw.ma.gui.runtime.util.AbstractParentNode;
import com.sonicsw.ma.plugin.IMonitorPlugin;
import com.sonicsw.mx.util.Sorter;

import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.jmx.client.IRemoteMBeanServer;

public class MetricsModel extends AbstractMetricsModel
{
    private static final Long[] EMPTY_LONG_ARRAY = new Long[0];
    public static final int DISABLE_INSTANCE_NONE     = 0;
    public static final int DISABLE_INSTANCE_ALL      = 1;
    public static final int DISABLE_INSTANCE_WILDCARD = 2;

    private boolean m_hasInstanceMetrics     = false;
    private boolean m_hasMetricAlertsSupport = false;

    // metric polling
    private Timer m_pollTimer;
    private HashMap m_pollFailures = new HashMap();

    private Object[] m_pollParams = new Object[] { null, Boolean.FALSE };
    private static final String[] POLL_SIGNATURE = new String[] { IMetricIdentity[].class.getName(), Boolean.class.getName() };
    private static final String[] ALERTS_SIGNATURE = new String[] { IAlert[].class.getName() };
    private IMonitorPlugin source = null;

    public MetricsModel(IRemoteMBeanServer connector, ObjectName componentName, IMetricInfo[] infos)
    throws Exception
    {
        super(connector, componentName, infos);
        update(infos);
    }

    @Override
    public final void update(IMetricInfo[] info)
    {
        Sorter.sort(info, this, info.length);

        Enumeration en = m_treeRootNode.depthFirstEnumeration();

        while (en.hasMoreElements())
        {
            AbstractNode node = (AbstractNode)en.nextElement();

            if (node == m_treeRootNode)
            {
                continue;
            }

            String name = null;

            if ((node instanceof Node) || (node instanceof InstanceParentNode))
            {
                name = ((IMetricInfo)node.getUserObject()).getMetricIdentity().getName();
            }
            else
            {
                name = ((IMetricIdentity)node.getUserObject()).getName();
            }

            if (!inList(info, name))
            {
                try
                {
                    disableMetric(node);
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                    MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                }
            }
        }

        constructTree(info);
    }

    private boolean inList(IMetricInfo[] list, String metricName)
    {
        boolean found = false;

        for (int i = 0; i < list.length; i++)
        {
            if (list[i].getMetricIdentity().getName().startsWith(metricName))
            {
                found = true;
                break;
            }
        }

        return found;
    }

    public void enableMetric(AbstractNode node)
    throws Exception
    {
        HashSet ids = new HashSet();
        setIdentities(node, ids, true, DISABLE_INSTANCE_NONE);
        IMetricIdentity[] metricIds = (IMetricIdentity[])ids.toArray(new IMetricIdentity[ids.size()]);

        m_connector.invoke(m_componentName, "enableMetrics", new Object[] { metricIds }, METRIC_IDS_SIGNATURE);

        node.setEnabled(true);
        refreshTree();
        node.setParentEnabled();

        // call nodeChanged for the parents
        AbstractNode changeNode = node;
        while (changeNode != m_treeRootNode)
        {
            super.nodeChanged(changeNode);
            changeNode = (AbstractNode)changeNode.getParent();
        }
        // call nodeChanged for the children
        Enumeration en = node.depthFirstEnumeration();
        while (en.hasMoreElements())
        {
            AbstractNode descendant = (AbstractNode)en.nextElement();
            super.nodeChanged(descendant);
        }
        super.nodeChanged(node);

    }

    public Object getSource() {
            return source;
        }

        public void setSource(IMonitorPlugin source) {
            this.source = source;
        }
    
    public void enableInstancePatterns(InstanceParentNode node, List instanceNames)
    throws Exception
    {
        updateInstancePatterns("enableMetrics", node, instanceNames);
        if (instanceNames.contains("*"))
        {
            this.enableMetric(node);
        }
        else if (!instanceNames.isEmpty())
        {
            node.setParentEnabled();
        }
    }

    public void disableInstancePatterns(InstanceParentNode node, List instanceNames)
    throws Exception
    {
        updateInstancePatterns("disableMetrics", node, instanceNames);
    }

    private void updateInstancePatterns(String updateOperation, InstanceParentNode node, List instanceNames)
    throws Exception
    {
        IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        String[] nameComponents = metricId.getNameComponents();
        String[] instanceNameComponents = new String[nameComponents.length + 1];
        System.arraycopy(nameComponents, 0, instanceNameComponents, 0, nameComponents.length);
        int instanceNameComponentIndex = instanceNameComponents.length - 1;

        // create the new indentities
        HashSet enabledIds = new HashSet();
        Iterator iterator = instanceNames.iterator();
        while (iterator.hasNext())
        {
            instanceNameComponents = (String[])instanceNameComponents.clone();
            instanceNameComponents[instanceNameComponentIndex] = (String)iterator.next();
            enabledIds.add(MetricsFactory.createMetricIdentity(instanceNameComponents));
        }

        // now enable/disable them
        IMetricIdentity[] metricIds = (IMetricIdentity[])enabledIds.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        m_connector.invoke(m_componentName, updateOperation, new Object[] { metricIds }, METRIC_IDS_SIGNATURE);

        refreshTree();
        super.nodeStructureChanged(node);
    }

    @Override
    public List getInstances(InstanceParentNode node)
    throws Exception
    {
        IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        String[] instances = (String[])m_connector.invoke(m_componentName, "getInstanceMetricNames", new Object[] { metricId }, METRIC_ID_SIGNATURE);

        ArrayList instancesList = new ArrayList(instances.length);
        for (int i = 0; i < instances.length; i++)
        {
            instancesList.add(instances[i]);
        }

        return instancesList;
    }

    public String[] getEnabledMetricsPatterns(InstanceParentNode node)
    throws Exception
    {
        IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        IMetricIdentity[] instancePatterns = (IMetricIdentity[])m_connector.invoke(m_componentName, "getEnabledMetrics", new Object[] { new IMetricIdentity[] { metricId } }, METRIC_IDS_SIGNATURE);

        String[] patterns = new String[instancePatterns.length];
        for (int i = 0; i < instancePatterns.length; i++)
        {
            String[] patternId = instancePatterns[i].getNameComponents();
            patterns[i] = patternId[patternId.length - 1];
        }

        return patterns;
    }

    public void disableMetric(AbstractNode node)
    throws Exception
    {
        disableMetric(node, DISABLE_INSTANCE_ALL);
    }

    public void disableMetric(AbstractNode node, int instanceDisableMode)
    throws Exception
    {
        HashSet ids = new HashSet();
        setIdentities(node, ids, false, instanceDisableMode);
        IMetricIdentity[] metricIds = (IMetricIdentity[])ids.toArray(new IMetricIdentity[ids.size()]);


        m_connector.invoke(m_componentName, "disableMetrics", new Object[] { metricIds }, METRIC_IDS_SIGNATURE);

        if (node instanceof InstanceNode)
        {
            InstanceParentNode parent = (InstanceParentNode)node.getParent();
            parent.remove(node);
            super.nodeStructureChanged(parent);
        }
        else
        if (node instanceof InstanceParentNode)
        {
            node.removeAllChildren();
            refreshTree();
            super.nodeStructureChanged(node);
        }
        else
        if (node instanceof ParentNode)
        {
            ArrayList instanceParents = new ArrayList();
            Enumeration en = node.depthFirstEnumeration();
            while (en.hasMoreElements())
            {
                AbstractNode descendant = (AbstractNode)en.nextElement();
                if (descendant instanceof InstanceParentNode)
                {
                    instanceParents.add(descendant);
                }
            }
            for (int i = instanceParents.size() - 1; i >= 0; i--)
            {
                ((InstanceParentNode)instanceParents.get(i)).removeAllChildren();
                super.nodeStructureChanged((InstanceParentNode)instanceParents.get(i));
            }
            refreshTree();
            super.nodeChanged(node);
        }

        node.setEnabled(false);
        refreshTree();
        node.setParentEnabled();
        super.nodeChanged(node);
//        disableAlertsWhenMetricsDisabled(node);
    }

    private void disableAlertsWhenMetricsDisabled(AbstractNode node)
    throws Exception
    {
        ArrayList highThresholdsDelta = null;
        ArrayList lowThresholdsDelta = null;

        if (node instanceof ParentNode) {
            Enumeration en = ((ParentNode) node).children();
            while (en.hasMoreElements()) {
                disableAlertsWhenMetricsDisabled((AbstractNode) en.nextElement());
            }
        } else {
            IMetricInfo metricInfo = (IMetricInfo) node.getUserObject();
            AlertsModel m_alertsModel = ((IAlertsProvider) node).getAlertsModel();
            if (m_alertsModel == null)
            {
                return;
            }
            Long[] existingHighThresholds = (Long[]) m_alertsModel.getHighThresholdValues().toArray(EMPTY_LONG_ARRAY);
            Long[] existingLowThresholds = (Long[]) m_alertsModel.getLowThresholdValues().toArray(EMPTY_LONG_ARRAY);

            highThresholdsDelta = new ArrayList();
            if (metricInfo.supportsHighThresholdAlerts()) {
                for (int i = 0; i < existingHighThresholds.length; i++) {
                    highThresholdsDelta.add(existingHighThresholds[i]);
                }
            }
            lowThresholdsDelta = new ArrayList();
            if (metricInfo.supportsLowThresholdAlerts()) {
                for (int i = 0; i < existingLowThresholds.length; i++) {
                    lowThresholdsDelta.add(existingLowThresholds[i]);
                }
            }
            disableAlerts(node, (Long[]) highThresholdsDelta.toArray(EMPTY_LONG_ARRAY), (Long[]) lowThresholdsDelta.toArray(EMPTY_LONG_ARRAY));
        }
    }

    /**
     * Construct the metrics node tree from the metrics info and enabled metrics list
     * obtained from the component.
     */
    private void constructTree(IMetricInfo[] infos)
    {
        m_hasInstanceMetrics = false;
        m_hasMetricAlertsSupport = false;

        for (int i = 0; i < infos.length; i++)
        {
            AbstractParentNode parent = m_treeRootNode;
            String[] nameTokens = infos[i].getMetricIdentity().getNameComponents();
            StringBuffer prefix = new StringBuffer();
            for (int j = 0; j < nameTokens.length; j++)
            {
                if (j == nameTokens.length - 1)
                {
                    boolean isInstanceMetric = infos[i].isInstanceMetric();
                    if (isInstanceMetric)
                    {
                        m_hasInstanceMetrics = true;
                    }
                    if (infos[i].supportsHighThresholdAlerts() || infos[i].supportsLowThresholdAlerts())
                    {
                        m_hasMetricAlertsSupport = true;
                    }

                    if (parent.getChild(nameTokens[j]) == null)
                    {
                        parent.add(isInstanceMetric ? (AbstractNode)new InstanceParentNode(infos[i], nameTokens[j])
                                                    : (AbstractNode)new Node(infos[i], nameTokens[j]));
                    }
                }
                else
                {
                    if (j > 0)
                    {
                        prefix.append('.');
                    }
                    prefix.append(nameTokens[j]);
                    AbstractParentNode childNode = (AbstractParentNode)parent.getChild(nameTokens[j]);
                    if (childNode == null)
                    {
                        childNode = new ParentNode(MetricsFactory.createMetricIdentity(prefix.toString()), nameTokens[j]);
                        parent.add(childNode);
                    }
                    parent = childNode;
                }
            }
        }
    }

    /**
     * Refreshes the tree with enabled state of metrics and adding new instance metrics/exact patterns
     * that have been added since the last refresh or removing those that have been disabled.
     */
    @Override
    public void refreshTree()
    throws Exception
    {
        IMetricIdentity[] activeMetrics = null;
        IMetricIdentity[] enabledPatterns = null;
        IAlert[] enabledAlerts = null;

        activeMetrics = (IMetricIdentity[])m_connector.invoke(m_componentName, "getActiveMetrics", SINGLE_NULL_ARG, METRIC_IDS_SIGNATURE);
        enabledPatterns = m_hasInstanceMetrics
            ? (IMetricIdentity[])m_connector.invoke(m_componentName, "getEnabledMetrics", SINGLE_NULL_ARG, METRIC_IDS_SIGNATURE)
            : IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY;
        enabledAlerts = m_hasMetricAlertsSupport
            ? (IAlert[])m_connector.invoke(m_componentName, "getEnabledAlerts", SINGLE_NULL_ARG, METRIC_IDS_SIGNATURE)
            : new IAlert[0];

        // put them all into a HashSet for easy access
        HashSet enabledExactPatternsSet = new HashSet();
        for (int i = 0; i < enabledPatterns.length; i++)
        {
            // filter those with wildcards, since we don't want a node for them
           // if (enabledPatterns[i].getName().indexOf('*') == -1)
                enabledExactPatternsSet.add(enabledPatterns[i]);
        }
        HashSet activeMetricsSet = new HashSet(activeMetrics.length);
        for (int i = 0; i < activeMetrics.length; i++)
        {
            activeMetricsSet.add(activeMetrics[i]);
        }

        // put the alerts in a table for later use
        HashMap enabledAlertsSets = new HashMap();
        for (int i = 0; i < enabledAlerts.length; i++)
        {
            HashSet alerts = (HashSet)enabledAlertsSets.get(enabledAlerts[i].getMetricIdentity());
            if (alerts == null)
            {
                alerts = new HashSet();
                enabledAlertsSets.put(enabledAlerts[i].getMetricIdentity(), alerts);
            }
            alerts.add(enabledAlerts[i]);
        }

        // walk the tree
        setLeafNodeStates(m_treeRootNode, enabledExactPatternsSet, activeMetricsSet, enabledAlertsSets);

        // set initial parent enablement and alerts
        Enumeration en = m_treeRootNode.depthFirstEnumeration();
        ParentNode lastParentNode = null;
        while (en.hasMoreElements())
        {
            AbstractNode node = (AbstractNode)en.nextElement();
            if (node instanceof Node) // look at parent enablement
            {
                ParentNode parent = (ParentNode)node.getParent();
                if (parent != null && parent != lastParentNode)
                {
                    lastParentNode = parent;
                    node.setParentEnabled();
                }
            }
        }
    }

    /**
     * Traverses node branch to create set of IMetricIdentity for leafs(Node)
     * and InstanceParentNode( with <>.* pattern).
     *
     * Used in enableMetric, disableMetric calls
     *
     * @param node                 The node under which to find the identities.
     * @param identities           The set in-which to store the identities.
     * @param enable               true if this is for metric enabling otherwise
     *                             false (disableMetric).
     * @param instanceDisableMode  A disable mode setting that applies only
     *                             when the node parameter has instance metrics
     *                             somewhere on its sub branches...used to
     *                             determine how to build up the instance
     *                             pattern identities.
     *
     *                             DISABLE_INSTANCE_NONE - don't include any patterns
     *                             DISABLE_INSTANCE_ALL  - include all patterns in set
     *                             DISABLE_INSTANCE_WILDCARD - include only the '*' pattern
     */
    private void setIdentities(AbstractNode node,
                               HashSet      identities,
                               boolean      enable,
                               int          instanceDisableMode)
    {
        if (node instanceof Node) // simple add
        {
            IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
            identities.add(metricId);
        }
        else if (node instanceof ParentNode) // walk the children
        {
            Enumeration en = ((ParentNode)node).children();
            while (en.hasMoreElements())
            {
                setIdentities((AbstractNode)en.nextElement(), identities, enable, instanceDisableMode);
            }
        }
        else if (node instanceof InstanceParentNode) // build new identity with * pattern
        {
            // When enabling a metric all we want to do for an instance parent
            // is add the '*' wildcard pattern...
            if (enable)
            {
                IMetricIdentity parentId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
                IMetricIdentity patternId = MetricsFactory.createMetricIdentity(parentId, "*");
                identities.add(patternId);
            }
            else
            {
                // But when disabling an instance metric we can do one of three
                // things...
                switch (instanceDisableMode)

                {
                    case DISABLE_INSTANCE_NONE:
                        break;  // Do nothing
                    case DISABLE_INSTANCE_ALL:
                    {   // Disable (remove) all patterns currently set.
                        try
                        {
                            IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
                            IMetricIdentity[] instancePatterns = (IMetricIdentity[])m_connector.invoke(m_componentName, "getEnabledMetrics", new Object[] { new IMetricIdentity[] { metricId } }, METRIC_IDS_SIGNATURE);

                            for (int i = 0; i < instancePatterns.length; i++)
                            {
                                identities.add(instancePatterns[i]);
                            }
                        }
                        catch (Exception e)
                        {
                            // Severe problem if we fail to get the enabled metrics.
                            MgmtConsole.getMgmtConsole().notifyMessage(MgmtConsole.ERROR, e.getMessage(), e, false);    // Log the error msg.
                        }
                        break;
                    }
                    case DISABLE_INSTANCE_WILDCARD:
                    default:
                    {   // Disable (remove) the '*' wildcard pattern.
                        IMetricIdentity parentId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
                        IMetricIdentity patternId = MetricsFactory.createMetricIdentity(parentId, "*");
                        identities.add(patternId);
                        break;
                    }
                }
            }
        }
        else if (node instanceof InstanceNode)
        {
            InstanceParentNode parent    = (InstanceParentNode)node.getParent();
            IMetricIdentity    parentId  = ((IMetricInfo)parent.getUserObject()).getMetricIdentity();
            IMetricIdentity    patternId = MetricsFactory.createMetricIdentity(parentId, node.toString());

            identities.add(patternId);
        }
    }

    public void enableAlerts(AbstractNode node, Long[] highThresholdAlerts, Long[] lowThresholdAlerts)
    throws Exception
    {
        IMetricIdentity metricId = node instanceof InstanceNode ? (IMetricIdentity)node.getUserObject()
                                                                : ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        Object[] params = new Object[]
        {
            createAlertArray(metricId, highThresholdAlerts, lowThresholdAlerts)
        };
        m_connector.invoke(m_componentName, "enableAlerts", params, ALERTS_SIGNATURE);

        // update the model on the assumption that the list given are new alerts
        HashSet alerts = ((IAlertsProvider)node).getAlertsModel().getAlerts();
        for (int i = 0; i < highThresholdAlerts.length; i++)
        {
            alerts.add(MetricsFactory.createAlert(metricId, true, highThresholdAlerts[i].longValue()));
        }
        for (int i = 0; i < lowThresholdAlerts.length; i++)
        {
            alerts.add(MetricsFactory.createAlert(metricId, false, lowThresholdAlerts[i].longValue()));
        }

        super.nodeChanged(node);
    }

    public void disableAlerts(AbstractNode node, Long[] highThresholdAlerts, Long[] lowThresholdAlerts)
    throws Exception
    {
        IMetricIdentity metricId = node instanceof InstanceNode ? (IMetricIdentity)node.getUserObject()
                                                                : ((IMetricInfo)node.getUserObject()).getMetricIdentity();
        Object[] params = new Object[]
        {
            createAlertArray(metricId, highThresholdAlerts, lowThresholdAlerts)
        };
        m_connector.invoke(m_componentName, "disableAlerts", params, ALERTS_SIGNATURE);

        // update the model on the assumption that the list given are new alerts
        HashSet alerts = ((IAlertsProvider)node).getAlertsModel().getAlerts();
        for (int i = 0; i < highThresholdAlerts.length; i++)
        {
            removeAlert(alerts, true, highThresholdAlerts[i].longValue());
        }
        for (int i = 0; i < lowThresholdAlerts.length; i++)
        {
            removeAlert(alerts, false, lowThresholdAlerts[i].longValue());
        }

        super.nodeChanged(node);
    }

    private IAlert[] createAlertArray(IMetricIdentity metricId, Long[] highThresholdAlerts, Long[] lowThresholdAlerts)
    {
        int alertCount = (highThresholdAlerts == null ? 0 : highThresholdAlerts.length) +
                         (lowThresholdAlerts == null ? 0 : lowThresholdAlerts.length);

        IAlert[] alerts = new IAlert[alertCount];
        if (lowThresholdAlerts != null)
        {
            for (int i = lowThresholdAlerts.length - 1; i >= 0; i--)
            {
                alerts[--alertCount] = MetricsFactory.createAlert(metricId, false, lowThresholdAlerts[i].longValue());
            }
        }

        if (highThresholdAlerts != null)
        {
            for (int i = highThresholdAlerts.length - 1; i >= 0; i--)
            {
                alerts[--alertCount] = MetricsFactory.createAlert(metricId, true, highThresholdAlerts[i].longValue());
            }
        }

        return alerts;
    }

    private void removeAlert(HashSet alerts, boolean isHighThreshold, long threshold)
    {
        Iterator iterator = alerts.iterator();
        while (iterator.hasNext())
        {
            IAlert alert = (IAlert)iterator.next();
            if (alert.isHighThreshold() == isHighThreshold && alert.getThresholdValue() == threshold)
            {
                alerts.remove(alert);
                return;
            }
        }
    }

    private void setLeafNodeStates(AbstractNode node, HashSet enabledExactPatterns, HashSet activeMetrics, HashMap enabledAlertsSets)
    {
        // NOTE: remove handled enabled metrics as we go so that any iterating gets more
        //       efficient as we continue
        if (node instanceof Node) // simple enable/disable
        {
            IMetricIdentity metricId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
            boolean enabled = activeMetrics.remove(metricId);
            ((Node)node).setEnabled(enabled);
            addAlerts(node, metricId, enabledAlertsSets);
        }
        else
        if (node instanceof ParentNode) // walk the children
        {
            Enumeration en = ((ParentNode)node).children();
            while (en.hasMoreElements())
            {
                setLeafNodeStates((AbstractNode)en.nextElement(), enabledExactPatterns, activeMetrics, enabledAlertsSets);
            }
        }
        else
        if (node instanceof InstanceNode) // remove it if it is no longer enabled
        {
            if (!(activeMetrics.remove(node.getUserObject()) || enabledExactPatterns.remove(node.getUserObject())))
            {
                node.removeFromParent();
            }
        }
        else
        if (node instanceof InstanceParentNode) // are there any new instances that must be created under it
        {
            node.removeAllChildren();
            IMetricIdentity parentId = ((IMetricInfo)node.getUserObject()).getMetricIdentity();
            addAlerts(node, parentId, enabledAlertsSets);

            HashMap children = new HashMap();

            buildInstanceNodes(node, parentId, children, enabledExactPatterns, false);
            buildInstanceNodes(node, parentId, children, activeMetrics, true);

            // if there are children then add them in
            if (!children.isEmpty())
            {
                // resort then add back in
                Object[] nodes = Sorter.sort(children.values().toArray(), this, children.size());
                for (int i = 0; i < nodes.length; i++)
                {
                    node.add((InstanceNode)nodes[i]);
                    addAlerts((InstanceNode)nodes[i], (IMetricIdentity)((InstanceNode)nodes[i]).getUserObject(), enabledAlertsSets);
                }
            }

            this.nodeStructureChanged(node);
        }
    }

    private void buildInstanceNodes(AbstractNode parentNode, IMetricIdentity parent, HashMap children, Set ids, boolean enabled)
    {
        IMetricIdentity[] metricsIds = (IMetricIdentity[])ids.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);

        for (int i = 0; i < metricsIds.length; i++)
        {
            if (metricsIds[i].isInstanceOf(parent)) {
                if(containsWildcard(metricsIds[i])){
                    parentNode.setEnabled(true);
                    parentNode.setParentEnabled();
                }  else {
                    /* Non wild-card match */
                    ids.remove(metricsIds[i]); // since weve now handled this
                    String[] nameComponents = metricsIds[i].getNameComponents();
                    InstanceNode newChild = new InstanceNode(metricsIds[i], nameComponents[nameComponents.length - 1]);
                    newChild.setEnabled(enabled);
                    children.put(metricsIds[i], newChild);
                }
            }
        }
    }
    private boolean containsWildcard(IMetricIdentity metricID)
    {
    	String wildcard = "*";
    	String metricName = metricID.getAbsoluteName();
    	int index = metricName.indexOf(wildcard);
    	if (index == -1) {
    		return false;
    	}

    	String[] nameComponents = metricID.getNameComponents();
    	if (!"queue".equals(nameComponents[0])) {
    		return true;
    	}

        if(metricName.indexOf("%.*%.*")>0){
            String modifiedStr = metricName.replaceAll("%\\.\\*%\\.\\*", "_");
            index = modifiedStr.indexOf(wildcard);
        }


        return (index != -1);
    }

    private void addAlerts(AbstractNode node, IMetricIdentity metricId, HashMap enabledAlertsSets)
    {
        if (node instanceof IAlertsProvider)
        {
            HashSet alerts = (HashSet)enabledAlertsSets.remove(metricId);
            if (alerts != null)
            {
                ((IAlertsProvider)node).getAlertsModel().setAlerts(alerts);
            }
        }
    }

    @Override
    public void refreshMetricValues(IMetricIdentity[] id)
    {
        if ((id != null) && (id.length > 0))
        {
            if (m_pollTimer == null)
            {
                m_pollTimer = new Timer(true);
            }

            m_pollTimer.schedule(new ValueUpdater(id),
                                 new Date(System.currentTimeMillis())); // schedule straight away
        }
    }

    @Override
    public void addValueListener(IMetricIdentity metricId, IValueListener listener)
    {
        synchronized(m_valueListeners)
        {
            HashSet listeners = (HashSet)m_valueListeners.get(metricId);
            if (listeners == null)
            {
                listeners = new HashSet();
                m_valueListeners.put(metricId, listeners);
            }
            listeners.add(listener);

            // When a watch (value) listener is added for a given metric then
            // we require a repaint of any tree views onto this metric node,
            // i.e. adding a watch icon.
            //
            updateNode(metricId);

            if (m_pollTimer == null)
            {
                m_pollTimer = new Timer(true);
                m_pollTimer.schedule(new ValueUpdater(), new Date(System.currentTimeMillis())); // schedule straight away
            }
        }
    }

    @Override
    public void removeValueListener(IMetricIdentity metricId, IValueListener listener)
    {
        synchronized(m_valueListeners)
        {
            HashSet listeners = (HashSet)m_valueListeners.get(metricId);
            if (listeners != null)
            {
                listeners.remove(listener);
                if (listeners.isEmpty())
                {
                    m_valueListeners.remove(metricId);
                }

                 // When a watch (value) listener is removed from the metric
                 // then we require a repaint of any views onto this metric
                 // node, i.e. possible removal of watch icon.
                 //
                 updateNode(metricId);
            }

            if ((m_valueListeners.isEmpty()) && (m_pollTimer != null))
            {
                m_pollTimer.cancel();
                m_pollTimer = null;
            }
        }
    }

//---Inner classes

    private class ValueUpdater extends TimerTask
    {
        private IMetricIdentity[] m_id;
        private boolean           m_oneoff;

        public ValueUpdater()
        {
            this(null, false);
        }

        public ValueUpdater(IMetricIdentity[] id)
        {
            this(id, true);
        }

        protected ValueUpdater(IMetricIdentity[] id, boolean oneoff)
        {
            m_id     = id;
            m_oneoff = oneoff;
        }

        @Override
        public void run()
        {
            if (m_pollTimer == null)
            {
                return;
            }

            if (m_id == null)
            {
                m_pollParams[0] = m_valueListeners.keySet().toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
            }
            else
            {
                m_pollParams[0] = m_id;
            }

            try
            {
                IMetricsData metricsData = (IMetricsData)m_connector.invoke(m_componentName, "getMetricsData", m_pollParams, POLL_SIGNATURE);

                if (m_pollTimer == null)
                {
                    return;
                }

                // propagate the new value
                IMetric[] metrics = metricsData.getMetrics();

                for (int i = 0; i < metrics.length; i++)
                {
                    HashSet listeners = null;
                    synchronized(m_valueListeners)
                    {
                        listeners = (HashSet)m_valueListeners.get(metrics[i].getMetricIdentity());
                    }
                    if (listeners != null)
                    {
                        Iterator iterator = listeners.iterator();
                        ValueUpdatedEvent vue = new ValueUpdatedEvent(m_componentName, metrics[i]);
                        while (m_pollTimer != null && iterator.hasNext())
                        {
                            if (m_pollTimer == null)
                            {
                                return;
                            }

                            ((IValueListener)iterator.next()).valueUpdated(vue);
                        }
                    }
                }

                m_pollFailures.remove(m_canonicalName);
            }
            catch(Exception e)
            {
                Exception lastException = (Exception)m_pollFailures.get(m_canonicalName);
                if (lastException == null || e.getClass() != lastException.getClass())
                {
                    MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                                                               "Failed to collect metrics from " + m_componentName, e, false);
                    m_pollFailures.put(m_canonicalName, e);
                }
            }

            if (!m_oneoff)
            {
                long pollInterval = 1000 * PreferenceManager.getInstance().getInt("preferences.metrics", JPreferencesDialog.POLL_FREQUENCY, JPreferencesDialog.DEFAULT_POLL_FREQUENCY);
                long nextTime = scheduledExecutionTime() + pollInterval;

                while (nextTime <= System.currentTimeMillis())
                {
                    nextTime += pollInterval;
                }

                m_pollTimer.schedule(new ValueUpdater(), new Date(nextTime)); // schedule straight away
            }

            // Clear down the id array to speed garbage collection?
            //
            m_id = null;
        }
    }
}
