package com.sonicsw.ma.gui.runtime.notifications;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.TableColumn;

import com.sonicsw.ma.gui.IApplication;
import com.sonicsw.ma.gui.IWorkspaceWindow;
import com.sonicsw.ma.gui.JPreferencesDialog;
import com.sonicsw.ma.gui.MgmtConsole;
import com.sonicsw.ma.gui.PreferenceManager;
import com.sonicsw.ma.gui.runtime.notifications.model.AbstractNotificationsModel;
import com.sonicsw.ma.gui.runtime.notifications.model.ForwardedNotificationsModel;
import com.sonicsw.ma.gui.runtime.notifications.model.INotificationWatcher;
import com.sonicsw.ma.gui.table.IModelTableModel;
import com.sonicsw.ma.gui.table.RowTableColumn;
import com.sonicsw.ma.gui.util.BasicGuiAction;
import com.sonicsw.ma.gui.util.ExtendedJScrollPane;
import com.sonicsw.ma.gui.util.Helper;
import com.sonicsw.ma.gui.util.JBasicMenu;
import com.sonicsw.ma.gui.util.JBasicMenuItem;
import com.sonicsw.ma.gui.util.JMADialog;
import com.sonicsw.ma.gui.util.JMAFrame;
import com.sonicsw.ma.gui.util.JMAInternalFrame;
import com.sonicsw.ma.gui.util.JRowTable;
import com.sonicsw.ma.gui.util.PopupMenuShower;
import com.sonicsw.ma.gui.util.ResourceManager;
import com.sonicsw.ma.plugin.IPlugin;

import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.jmx.client.MFNotification;

class ForwardedNotificationsWatchWindow
    extends JMAInternalFrame
    implements INotificationWatcher, NotificationListener, IWorkspaceWindow
{
    private static final HashMap m_watchWindows = new HashMap();

    private short m_scope = DOMAIN_SCOPE;
    private String m_scopeID;
    private JRowTable m_table;

    private HashMap m_subscriptions = new HashMap();
    private HashMap m_modelMap = new HashMap();

    private int m_maxNotifications = 10;

    private static final int PREFERRED_HEIGHT = 300;

    static ForwardedNotificationsWatchWindow getWatchWindow(ObjectName scopeName)
    {
        String name = null;

        // first determine the name to use based on the scope
        CanonicalName cName = new CanonicalName(scopeName.toString());
        name = cName.getCanonicalName();
        ForwardedNotificationsWatchWindow window = (ForwardedNotificationsWatchWindow) m_watchWindows.get(name);
        if (window == null)
        {
            window = new ForwardedNotificationsWatchWindow(name);
            m_watchWindows.put(name, window);
        }
        return window;
    }

    /**
     * Constructor
     */
    private ForwardedNotificationsWatchWindow(String scopeID)
    {
        super("notifications.watcher");
        setProperties(scopeID);
        m_scopeID = scopeID;
    }
    
    private void setProperties(String scopeID) {
        setTitle("Forwarded " + getTitle() + " - " + scopeID);
        setFrameIcon(ResourceManager.getIcon(getClass(), "Alert"));
    }
    
    @Override
    protected void maInitialize()
    {
        // set initial values from preferences
        PreferenceManager pManager = PreferenceManager.getInstance();

        m_maxNotifications = pManager.getInt("preferences.notifications",
                                             JPreferencesDialog.MAX_NOTIFICATIONS,
                                             JPreferencesDialog.DEFAULT_MAX_NOTIFICATIONS);

        initPanels();
    }

    @Override
    protected void maCleanup()
    {
        // remove listener subscriptions and cleanup maps
        Iterator iterator = m_subscriptions.entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry) iterator.next();
            String collectionID = (String) entry.getKey();
            ForwardedNotificationsModel model = (ForwardedNotificationsModel) ((Object[]) m_modelMap.get(collectionID))[0];

            HashMap map = (HashMap) entry.getValue();
            Iterator notifications = map.keySet().iterator();
            try
            {
                while (notifications.hasNext())
                {
                    model.removeNotificationListener((String) notifications.next(), this);
                }

                map.clear();
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to remove notification subscriptions", e, true);
            }
        }
        m_subscriptions.clear();
        m_modelMap.clear();

        m_watchWindows.remove(m_scopeID);

        if (m_table != null)
        {
            m_table.cleanup();
        }
    }

    @Override
    public boolean canClose()
    {
        boolean res = true;

        if ((m_subscriptions != null) && (!m_subscriptions.isEmpty()))
        {
            res = (JOptionPane.showConfirmDialog(this,
                                                 "Are you sure you want to close watcher:\n" + m_scopeID,
                                                 "Close Notifications Watcher",
                                                 JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION);
        }

        return res;
    }

    @Override
    public JComponent[] getSelectedMenuItems(int type)
    {
        JComponent[] menuItems = null;

        if (type == IPlugin.PLUGIN_TYPE)
        {
            JBasicMenu clearMenu = new JBasicMenu(new ClearAction());

            clearMenu.add(new JBasicMenuItem(new ClearSelectedAction()));
            clearMenu.add(new JBasicMenuItem(new ClearSelectedTypeAction()));
            clearMenu.add(new JBasicMenuItem(new ClearAllAction()));

            menuItems = new JComponent[]
                        {
                        new JBasicMenuItem(new OptionsAction()),
                        new JSeparator(),
                        clearMenu,
                        new JBasicMenuItem(new RemoveWatchAction()),
                        new JSeparator(),
                        new JBasicMenuItem(new PropertiesAction()),
            };
        }
        return menuItems;
    }

    @Override
    public int getMaxNotifications()
    {
        return m_maxNotifications;
    }

    @Override
    public void setMaxNotifications(int maxNotifications)
    {
        m_maxNotifications = maxNotifications;
        enforceMaxNotifications(m_maxNotifications);
    }

    private void initPanels()
    {
        super.getContentPane().setLayout(new BorderLayout());
        super.getContentPane().add(makeTablePanel(), BorderLayout.CENTER);
    }

    private JPanel makeTablePanel()
    {
        JPanel panel = new JPanel(new BorderLayout());

        panel.add(makeTable(), BorderLayout.CENTER);

        return panel;
    }

    private JComponent makeTable()
    {
        TableColumn[] tableColumnAdapters = new TableColumn[]
                                            {
                                            new SourceColumn(),
                                            new HostnameColumn(),
                                            new NotificationColumn(),
                                            new SeverityColumn(),
                                            new LogTypeColumn(),
                                            new SequenceColumn(),
                                            new TimeColumn()
        };

        m_table = new JRowTable(tableColumnAdapters, getClass().getName());
        m_table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        // put the table in a scroll pane now so that when we add the
        // PopupMenuShower to the table it will also apply to the scroll pane
        JComponent ret = new ExtendedJScrollPane(m_table);

        // add the popup menu
        JPopupMenu popup = new JPopupMenu();

        JComponent[] menuItems = getSelectedMenuItems(IPlugin.PLUGIN_TYPE);

        for (int i = 0; i < menuItems.length; i++)
        {
            popup.add(menuItems[i]);
        }

        m_table.addMouseListener(new LocalMenuShower(m_table, popup));

        return ret;
    }

    @Override
    public synchronized void addWatch(MBeanNotificationInfo info,
                                      AbstractNotificationsModel model)
    {
        addWatch(new MBeanNotificationInfo[]
                 {info}
                 , model);
    }

    @Override
    public synchronized void addWatch(MBeanNotificationInfo[] info,
                                      AbstractNotificationsModel model)
    {
        String componentName = model.getComponentName().getCanonicalName();
        String collectionID = ((ForwardedNotificationsModel) model).getCollectionID();

        Object[] modelUsage = (Object[]) m_modelMap.get(collectionID);
        HashMap subscriptions = (HashMap) m_subscriptions.get(collectionID);

        if (subscriptions == null)
        {
            subscriptions = new HashMap();
            m_subscriptions.put(collectionID, subscriptions);
        }

        try
        {
            for (int i = info.length - 1; i >= 0; i--)
            {
                String prefix = AbstractNotificationsModel.getTypePrefix(info[i].getNotifTypes());

                if (subscriptions.containsKey(prefix))
                 {
                    continue; // were already subscribed
                }

                subscriptions.put(prefix, info[i]);

                if (modelUsage == null)
                {
                    modelUsage = new Object[]
                                 {model, new int[]
                                 {0}
                    };
                    m_modelMap.put(collectionID, modelUsage);
                }
                ((int[]) modelUsage[1])[0]++;

                model.addNotificationListener(prefix, this);
            }
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to create notification subscription(s)", e, true);
        }
    }

    private synchronized void removeWatch(NotificationValue notificationValue)
    {
        Notification notification = notificationValue.getValue();
        String collectionID = notificationValue.getCollectionID();
        boolean alreadyRemoved = false;

        HashMap subscriptions = (HashMap) m_subscriptions.get(collectionID);

        if (subscriptions != null)
        {
            if (subscriptions.remove(notification.getType()) == null)
            {
                alreadyRemoved = true;
            }
            if (subscriptions.isEmpty())
            {
                m_subscriptions.remove(collectionID);
            }
        }

        Object[] modelUsage = (Object[]) m_modelMap.get(collectionID);
        if (modelUsage != null)
        {
            ForwardedNotificationsModel notificationsModel = (ForwardedNotificationsModel) modelUsage[0];
            if (!alreadyRemoved)
            {
                ((int[]) modelUsage[1])[0]--;
                if (((int[]) modelUsage[1])[0] == 0)
                {
                    m_modelMap.remove(collectionID);
                }
            }

            try
            {
                notificationsModel.removeNotificationListener(notification.getType(), this);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to remove notification subscriptions",
                    e,
                    true);
            }
        }
    }

    private synchronized void clearSelectedNotification(NotificationValue notificationValue)
    {
        ((IModelTableModel) m_table.getModel()).delete(notificationValue);
    }

    private synchronized void clearSelectedTypeNotifications(NotificationValue notificationValue)
    {
        IModelTableModel tableModel = (IModelTableModel) m_table.getModel();
        String notificationType = notificationValue.getValue().getType();
        for (int i = tableModel.getRowCount() - 1; i >= 0; i--)
        {
            NotificationValue rowValue = (NotificationValue) tableModel.getRowModel(i);
            if (rowValue.getValue().getType().equals(notificationType))
            {
                tableModel.deleteRow(i);
            }
        }
    }

    private synchronized void clearAllNotifications()
    {
        ((IModelTableModel) m_table.getModel()).clear();
    }

    @Override
    public short getWatchScope()
    {
        return m_scope;
    }

    @Override
    public synchronized void handleNotification(Notification notification, Object handback)
    {
        HashMap subscriptions = (HashMap) m_subscriptions.get(handback);
        if (subscriptions == null)
        {
            return;
        }

        if (!subscriptions.containsKey(notification.getType()))
        {
            return;
        }

        NotificationValue notificationValue = new NotificationValue(notification, (String) handback);

        enforceMaxNotifications(m_maxNotifications - 1);
        ((IModelTableModel) m_table.getModel()).addRow(notificationValue);
    }

    private NotificationValue getSelectedRowValue()
    {
        int row = m_table.getSelectedRow();

        if (row == -1)
        {
            return null;
        }

        return (NotificationValue) ((IModelTableModel) m_table.getModel()).getRowModel(row);
    }

    private synchronized void enforceMaxNotifications(int maxNotifications)
    {
        IModelTableModel tableModel = (IModelTableModel) m_table.getModel();

        while (tableModel.getRowCount() > maxNotifications)
        {
            // find the oldest and delete that
            NotificationValue oldest = null;
            Iterator iterator = tableModel.getContents().iterator();
            while (iterator.hasNext())
            {
                NotificationValue current = (NotificationValue) iterator.next();
                if (oldest == null || oldest.getValue().getTimeStamp() > current.getValue().getTimeStamp())
                {
                    oldest = current;
                }
            }
            tableModel.delete(oldest);
        }
    }

    private static class SourceColumn
        extends RowTableColumn
    {
        SourceColumn()
        {
            super(0, "Source", 15);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getSource();
        }
    }

    private static class HostnameColumn
        extends RowTableColumn
    {
        HostnameColumn()
        {
            super(1, "Hostname", 15);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getHost();
        }
    }

    private static class NotificationColumn
        extends RowTableColumn
    {
        NotificationColumn()
        {
            super(2, "Notification", 15);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getNotification();
        }
    }

    private static class SeverityColumn
        extends RowTableColumn
    {
        SeverityColumn()
        {
            super(3, "Severity", 10);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getSeverity();
        }
    }

    private static class LogTypeColumn
        extends RowTableColumn
    {
        LogTypeColumn()
        {
            super(4, "Log Type", 10);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getLogType();
        }
    }

    private static class SequenceColumn
        extends RowTableColumn
    {
        SequenceColumn()
        {
            super(5, "Sequence", 10);
        }

        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getSequence();
        }
    }

    private static class TimeColumn
        extends RowTableColumn
    {
        TimeColumn()
        {
            super(6, "Date & Time", 15);
            setCellRenderer();
        }
        
        private void setCellRenderer() {
            setCellRenderer(Helper.getDateTimeTableCellRenderer());
        }
        
        @Override
        public Object getColumnValue(Object rowModel)
        {
            return ((NotificationValue) rowModel).getTime();
        }
    }

    private void showPropertiesDialog()
    {
        try
        {
            JMAFrame parent = (JMAFrame) SwingUtilities.getAncestorOfClass(JMAFrame.class, this);
            JMADialog dialog = new NotificationAttributesSheet(parent, getSelectedRowValue().getValue(), m_scope, m_table);
            dialog.setVisible(true);
        }
        catch (Exception e)
        {
            MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                "Failed to show notification properties",
                e,
                true);
        }
    }

    private class PropertiesAction
        extends BasicGuiAction
    {
        public PropertiesAction()
        {
            super("NotificationsWatchWindow.properties");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            showPropertiesDialog();
        }

        @Override
        public boolean isEnabled()
        {
            NotificationValue notificationValue = getSelectedRowValue();

            if (notificationValue != null)
            {
                if (notificationValue.getValue()instanceof MFNotification)
                {
                    return true;
                }
            }
            return false;
        }
    }

    private class ClearAction
        extends BasicGuiAction
    {
        public ClearAction()
        {
            super("NotificationsWatchWindow.clear");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
        }

        @Override
        public boolean isEnabled()
        {
            return (m_table.getRowCount() > 0);
        }
    }

    private class RemoveWatchAction
        extends BasicGuiAction
    {
        public RemoveWatchAction()
        {
            super("NotificationsWatchWindow.remove");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ForwardedNotificationsWatchWindow.this.removeWatch(ForwardedNotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            NotificationValue notificationValue = getSelectedRowValue();

            if (notificationValue != null)
            {
                Notification notification = notificationValue.getValue();
                String collectionID = notificationValue.getCollectionID();
                HashMap subscriptions = (HashMap) m_subscriptions.get(collectionID);
                if (subscriptions != null && subscriptions.get(notification.getType()) != null)
                {
                    return true;
                }
            }
            return false;
        }
    }

    private class ClearSelectedAction
        extends BasicGuiAction
    {
        public ClearSelectedAction()
        {
            super("NotificationsWatchWindow.clearSelected");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ForwardedNotificationsWatchWindow.this.clearSelectedNotification(ForwardedNotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            return (getSelectedRowValue() != null);
        }
    }

    private class ClearSelectedTypeAction
        extends BasicGuiAction
    {
        public ClearSelectedTypeAction()
        {
            super("NotificationsWatchWindow.clearSelectedType");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ForwardedNotificationsWatchWindow.this.clearSelectedTypeNotifications(ForwardedNotificationsWatchWindow.this.getSelectedRowValue());
        }

        @Override
        public boolean isEnabled()
        {
            return (getSelectedRowValue() != null);
        }
    }

    private class ClearAllAction
        extends BasicGuiAction
    {
        public ClearAllAction()
        {
            super("NotificationsWatchWindow.clearAll");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            ForwardedNotificationsWatchWindow.this.clearAllNotifications();
        }

        @Override
        public boolean isEnabled()
        {
            return (m_table.getRowCount() > 0);
        }
    }

    private class OptionsAction
        extends BasicGuiAction
    {
        public OptionsAction()
        {
            super("NotificationsWatchWindow.options");
        }

        @Override
        public void actionPerformed(ActionEvent evt)
        {
            try
            {
                JMAFrame parent = (JMAFrame)SwingUtilities.getAncestorOfClass(JMAFrame.class,
                    ForwardedNotificationsWatchWindow.this);
                WatchPropSheet dialog = new WatchPropSheet(parent);
                dialog.editInstance(null, dialog.new Model(ForwardedNotificationsWatchWindow.this), false);
                dialog.setVisible(true);
            }
            catch (Exception e)
            {
                MgmtConsole.getMgmtConsole().notifyMessage(IApplication.MESSAGE_WARNING,
                    "Failed to edit notification watch properties",
                    e, true);
            }
        }
    }

    private class LocalMenuShower
        extends PopupMenuShower
    {
        LocalMenuShower(JTable table, JPopupMenu popup)
        {
            super(table, popup);
        }

        @Override
        protected void showMenuIfPopupTrigger(MouseEvent evt)
        {
            JPopupMenu popup = getPopupMenu();

            if (popup != null && popup.isPopupTrigger(evt))
            {
                Component comp = evt.getComponent();

                if (comp instanceof JTable)
                {
                    JTable table = (JTable) comp;
                    Point pt = new Point(evt.getX(), evt.getY());

                    // if we don't have a selection then attempt to set the
                    // row at the current location
                    if (table.getSelectedRowCount() == 0)
                    {
                        int row = table.rowAtPoint(pt);

                        if (row >= 0)
                        {
                            table.setRowSelectionInterval(row, row);
                        }
                    }

                    if (!table.isRowSelected(table.rowAtPoint(pt)))
                    {
                        return;
                    }
                }
                else if (comp instanceof JViewport)
                {
                    getTable().clearSelection();
                }

                for (int i = 0; i < popup.getComponentCount(); i++)
                {
                    Component item = popup.getComponent(i);

                    if (item instanceof AbstractButton)
                    {
                        Action action = ((AbstractButton) item).getAction();

                        if (action != null)
                        {
                            item.setEnabled(action.isEnabled());
                        }
                    }
                }

                if (popup.getComponentCount() > 0)
                {
                    popup.show(evt.getComponent(), evt.getX(), evt.getY());
                    ensurePopupFullyOnScreen(evt);
                }
            }
        }

        /**
         * Method to detect a double-click on a selected item.
         *
         * @param event The mouse event that caused the action
         */
        @Override
        public void mouseClicked(MouseEvent event)
        {
            if (event.getClickCount() == 2)
            {
                if (event.getComponent()instanceof JTable && getTable().getSelectedRowCount() == 1)
                {
                    showPropertiesDialog();
                }
            }
        }
    }
}